/*   **********************************************************************  **
 **   Copyright notice                                                       **
 **                                                                          **
 **   (c) 2003-2006 RSSOwl Development Team                                  **
 **   http://www.rssowl.org/                                                 **
 **                                                                          **
 **   All rights reserved                                                    **
 **                                                                          **
 **   This program and the accompanying materials are made available under   **
 **   the terms of the Eclipse Public License 1.0 which accompanies this     **
 **   distribution, and is available at:                                     **
 **   http://www.rssowl.org/legal/epl-v10.html                               **
 **                                                                          **
 **   A copy is found in the file epl-v10.html and important notices to the  **
 **   license from the team is found in the textfile LICENSE.txt distributed **
 **   in this package.                                                       **
 **                                                                          **
 **   This copyright notice MUST APPEAR in all copies of the file!           **
 **                                                                          **
 **   Contributors:                                                          **
 **     RSSOwl - initial API and implementation (bpasero@rssowl.org)         **
 **                                                                          **
 **  **********************************************************************  */

package net.sourceforge.rssowl.util.shop;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.NTCredentials;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;

import java.util.Hashtable;

/**
 * Factory class for proxy connection concerns in RSSOwl.
 * 
 * @author <a href="mailto:bpasero@rssowl.org">Benjamin Pasero </a>
 * @version 1.2.3
 */
public class ProxyShop {

  /** The proxy settings (host, port, user, pw) in a hashtable */
  public static Hashtable proxySettings = new Hashtable();

  /** This utility class constructor is hidden */
  private ProxyShop() {
  // Protect default constructor
  }

  /**
   * Check if the Proxy is protected with Authentication.
   * 
   * @return TRUE if the user has supplied Proxy Credentials.
   */
  public static boolean isProxyProtected() {
    return proxySettings.containsKey("username") || proxySettings.containsKey("password");
  }

  /**
   * Get wether the proxy should be used
   * 
   * @return boolean TRUE if proxy should be used.
   */
  public static boolean isUseProxy() {
    return proxySettings.get("proxySet").equals("true");
  }

  /**
   * Set the Domain to use with the proxy
   * 
   * @param domain The Domain to use with the proxy (NTLM)
   */
  public static void setDomain(String domain) {
    proxySettings.put("domain", domain);
  }

  /**
   * Set the Host to use as proxy
   * 
   * @param host The host to use as proxy
   */
  public static void setHost(String host) {
    proxySettings.put("proxyHost", host);
  }

  /**
   * Set the password to use with the proxy
   * 
   * @param password The password to use with the proxy
   */
  public static void setPassword(String password) {
    proxySettings.put("password", password);
  }

  /**
   * Set the port to use with the proxy
   * 
   * @param port The port to use with the proxy
   */
  public static void setPort(String port) {
    proxySettings.put("proxyPort", port);
  }

  /**
   * Set up the proxy for a given HttpClient
   * 
   * @param client The HttpClient to setup the Proxy Connection
   */
  public static void setUpProxy(HttpClient client) {

    /** Get Host and Port */
    String proxyHost = (String) proxySettings.get("proxyHost");
    String proxyPortString = (String) proxySettings.get("proxyPort");

    /** In case Port is not a number, use default value (-1) */
    int proxyPort = -1;
    try {
      proxyPort = Integer.parseInt(proxyPortString);
    } catch (NumberFormatException e) {
      proxyPort = -1;
    }

    /** Get username, password and Domain if available */
    String proxyUsername = (String) proxySettings.get("username");
    String proxyPassword = (String) proxySettings.get("password");
    String proxyDomain = (String) proxySettings.get("domain");

    /** Set Proxy to HttpClient */
    client.getParams().setAuthenticationPreemptive(true);
    client.getHostConfiguration().setProxy(proxyHost, proxyPort);

    /** Set Username and Password if available */
    if (proxyUsername.length() != 0 || proxyPassword.length() != 0) {
      AuthScope proxyAuthScope = new AuthScope(proxyHost, proxyPort);

      /** Use NTLM Credentials if Domain is set */
      if (StringShop.isset(proxyDomain))
        client.getState().setProxyCredentials(proxyAuthScope, new NTCredentials(proxyUsername, proxyPassword, proxyHost, proxyDomain));

      /** Use normal Credentials if Domain is not set */
      else
        client.getState().setProxyCredentials(proxyAuthScope, new UsernamePasswordCredentials(proxyUsername, proxyPassword));
    }
  }

  /**
   * Set wether to use the proxy at all
   * 
   * @param use If TRUE, use the proxy
   */
  public static void setUseProxy(String use) {
    proxySettings.put("proxySet", use);
  }

  /**
   * Set the username to use with the proxy
   * 
   * @param username The username to use with the proxy
   */
  public static void setUsername(String username) {
    proxySettings.put("username", username);
  }
}