/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-connection.h:
 * GObject wrapper for D-Bus method calls to org.freedesktop.Telepathy.Connection
 * 
 * Copyright (C) 2006, 2007 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifndef __TI_CONNECTION_H__
#define __TI_CONNECTION_H__

#include <glib-object.h>

#include "ti-channel.h"

/*
 * Type macros.
 */

typedef struct _TIConnection TIConnection;
typedef struct _TIConnectionClass TIConnectionClass;

#define TI_TYPE_CONNECTION               ti_connection_get_type()
#define TI_CONNECTION(object)            (G_TYPE_CHECK_INSTANCE_CAST ((object), TI_TYPE_CONNECTION, TIConnection))
#define TI_CONNECTION_CLASS(klass)       (G_TYPE_CHECK_CLASS_CAST ((klass), TI_TYPE_CONNECTION, TIConnectionClass))
#define TT_IS_CONNECTION(object)         (G_TYPE_CHECK_INSTANCE_TYPE ((object), TI_TYPE_CONNECTION))
#define TT_IS_CONNECTION_CLASS(klass)    (G_TYPE_CHECK_CLASS_TYPE ((klass), TI_TYPE_CONNECTION))
#define TI_CONNECTION_GET_CLASS(object)  (G_TYPE_INSTANCE_GET_CLASS ((object), TI_TYPE_CONNECTION, TIConnectionClass))

struct _TIConnection {
  GObject parent;
  /* instance members */
};

GType ti_connection_get_type (void);

/**
 * Creates a new connection
 */
TIConnection* ti_connection_new (const gchar* service_name, const gchar* obj_path);

/**
 * Get Service Name
 *
 */
const gchar* ti_connection_get_service_name (TIConnection* connection);

/**
 * Get Object Path
 *
 */
const gchar* ti_connection_get_object_path (TIConnection* connection);

/**
 * Get Interfaces
 *
 * @return A NULL-terminated array of strings containing interfaces' names. Must be freed after use.
 */
gchar** ti_connection_get_interfaces (TIConnection* connection);

/**
 * Get Status
 *
 * @return An integer representing the current status
 */
guint ti_connection_get_status (TIConnection* connection);

/**
 * Get Protocol
 *
 * @return Protocol name. Must be freed after use.
 */
gchar* ti_connection_get_protocol (TIConnection* connection);

/**
 * Connect
 *
 * @return FALSE if an error occurred.
 */
gboolean ti_connection_connect (TIConnection* connection);

/**
 * Disconnect
 *
 * @return FALSE if an error occurred.
 */
gboolean ti_connection_disconnect (TIConnection* connection);

/**
 * Hold Single Handle
 *
 * @param handle_type Handle type.
 * @param handle_number Handle number.
 */
void ti_connection_hold_single_handle (TIConnection* connection, guint handle_type, guint handle_number, GError** error);

/**
 * Hold Handles
 *
 * @param handle_type Handle type.
 * @param handles Handle numbers.
 */
void ti_connection_hold_handles (TIConnection* connection, guint handle_type, GArray* handles, GError** error);

/**
 * Inspect Single Handle
 *
 * @param handle_type Handle type.
 * @param handles Handle numbers.
 */
gchar* ti_connection_inspect_single_handle (TIConnection* connection, guint handle_type, guint handle_number, GError** error);

/**
 * Inspect Handles
 *
 * @param handle_type Handle type.
 * @param handles Handle numbers.
 */
gchar** ti_connection_inspect_handles (TIConnection* connection, guint handle_type, GArray* handles, GError** error);

/**
 * Release Handles
 *
 * @param handle_type Handle type.
 * @param handles Handle numbers.
 */
void ti_connection_release_handles (TIConnection* connection, guint handle_type, GArray* handles, GError** error);

/**
 * Request Handles
 *
 * @param type Handle type.
 * @param names Handle names.
 * @return An array of G_TYPE_UINT elements, with the same order as in "names" parameter. Must be freed after use.
 */
GArray* ti_connection_request_handles (TIConnection* connection, guint type, gchar** names, GError** error);

/**
 * Request Channel
 *
 * @return Newly created requested channel or NULL, in case of an error. Must be unref'ed after use.
 */
TIChannel* ti_connection_request_channel (TIConnection* connection,
                                          gchar* type, guint handle_type, guint handle, gboolean suppress_handler,
                                          GError** error);

/**
 * List Channels
 *
 */
GPtrArray* ti_connection_list_channels (TIConnection* connection, GError** error);

/**
 * AddStatus method from Presence interface
 *
 */
void ti_connection_ipresence_add_status (TIConnection* connection, const gchar* status, GHashTable* opt_params, GError** error);

/**
 * ClearStatus method from Presence interface
 *
 */
void ti_connection_ipresence_clear_status (TIConnection* connection, GError** error);

/**
 * GetStatuses method from Presence interface
 *
 * @return The valid statuses. Must be freed after use.
 */
GHashTable* ti_connection_ipresence_get_statuses (TIConnection* connection, GError** error);

/**
 * RemoveStatus method from Presence interface
 *
 */
void ti_connection_ipresence_remove_status (TIConnection* connection, const gchar* status, GError** error);

/**
 * RequestPresence method from Presence interface
 *
 */
void ti_connection_ipresence_request_presence (TIConnection* connection, GArray* handles, GError** error);

/**
 * RequestAliases method from Aliasing interface
 *
 */
gchar** ti_connection_ialiasing_request_aliases (TIConnection* connection, GArray* handles, GError** error);

/**
 * SetAliases method from Aliasing interface
 *
 */
void ti_connection_ialiasing_set_aliases (TIConnection* connection, GHashTable* aliases, GError** error);

/**
 * GetAliasFlags method from Aliasing interface
 *
 */
guint ti_connection_ialiasing_get_alias_flags (TIConnection* connection, GError** error);

/**
 * GetCapabilities method from Capabilities interface
 *
 */
GPtrArray* ti_connection_icapabilities_get_capabilities (TIConnection* connection, GArray* handles, GError** error);

/**
 * AdvertiseCapabilities method from Capabilities interface
 *
 */
void ti_connection_icapabilities_advertise_capabilities (TIConnection* connection, GPtrArray* add_caps, gchar** remove_caps, GError** error);

#endif
