/*
   Copyright (C) 2001-2007 Thomas Capricelli <orzel@freehackers.org>
*/

/* Qt */
#include <qpainter.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qspinbox.h>
#include <qpushbutton.h>
#include <qdom.h>
#include <qfile.h>

/* KDE */
#include <kdebug.h>
#include <klistview.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kdebug.h>

/* Opale */
#include "data.h"
#include "fileoptions.h"
#include "listboxeditorimpl.h"
#include "replacedialog.h"
#include "recordDialog.h"
#include "pattern.h"
#include "repeat.h"



static OpaleRecord * item2record( QListViewItem *it )
{
//	Q_ASSERT(it->inherits("OpaleRecord"));
	return ( OpaleRecord* )it ;
}




/*
 * OpalDatae
 */
OpaleData::OpaleData( QWidget* parent, const char* name )
	: KListView( parent, name )
{
	reset();
	setAlternateBackground( QColor(190,190,255));
	addColumn(i18n("Date"));		// 0
	addColumn(i18n("Description"));
	addColumn(i18n("Amount"));		// 2
	addColumn(i18n("Checked"));
	addColumn(i18n("Method"));		// 4
	addColumn(i18n("Future?"));
	addColumn(i18n("Balance"));		// 6
	addColumn(i18n("Category"));
	setSorting(-1);
	setColumnAlignment( 2, AlignRight);
	setColumnAlignment( 3, AlignHCenter);
	setColumnAlignment( 5, AlignRight);
	setColumnAlignment( 6, AlignRight);
	setColumnAlignment( 7, AlignRight);

	connect( this, SIGNAL(doubleClicked(QListViewItem *)), this, SLOT(editRecord(QListViewItem *)));
	connect( this, SIGNAL(returnPressed(QListViewItem *)), this, SLOT(editRecord(QListViewItem *)));
	connect( this, SIGNAL(clicked( QListViewItem *, const QPoint &, int)), this, SLOT(itemclicked( QListViewItem *, const QPoint &, int)));
}


void OpaleData::reset(void)
{
	// new empty..
	b_modified=false;
	s_currency =  i18n("Euro");
	s_account_name =  i18n("Bank's name");

	categorySList.clear();
	categorySList.append(i18n("Misc"));
	emit categoriesUpdated();
	methodSList.clear();
	methodSList.append(i18n("Check"));
	emit methodsUpdated();
	patterns.clear(); syncPatterns();
	clear();
	emit listUpdated();
}



bool OpaleData::loadXML(const QDomDocument &doc )
{
	QDomNode n;
	int i=0;

	kdDebug(OPALE_DEBUG_STREAM) << "begin of loadXML" << endl;
	/* check document type */
	if ( doc.doctype().name() != "opale") {
		kdDebug(OPALE_DEBUG_STREAM) << "doc.doctype().name() != opaleL" << endl;
		return false;
		}

	/* check mimetype */
	kdDebug(OPALE_DEBUG_STREAM) << "check mimetype" << endl;
	QDomElement account = doc.documentElement();
	if ( account.attribute( "mime" ) != "application/x-opale" ) {
		kdDebug(OPALE_DEBUG_STREAM) << "account.attribute( mime ) != application/x-opale" << endl;
		return false;
		}


	/* read account information */
	s_currency = account.attribute("currency");
	if ("" == s_currency ) s_currency="Euros";
	s_account_name = account.attribute("account_name");

	/* read the method section */
	kdDebug(OPALE_DEBUG_STREAM) << "read the method section... " << endl;
	QDomElement methodlist = account.namedItem("howlist").toElement();
	methodSList.clear();
	n = methodlist.firstChild();
	i=0;
	while (!n.isNull()) {
		QDomElement e = n.toElement();
		if (i!=e.attribute("num").toInt()) {
			kdDebug(OPALE_DEBUG_STREAM) << "num=" << e.attribute("num").toInt()
				<< " in method :" << e.attribute("name").latin1() << " element,"
				<< i << " was expected, file is corrupted" << endl;
			return false;
		}
		i++;
		methodSList.append(e.attribute("name"));
		n = n.nextSibling();
	}
	kdDebug(OPALE_DEBUG_STREAM) << i << " elements" << endl;
	emit methodsUpdated();

	/* read the category section */
	kdDebug(OPALE_DEBUG_STREAM) << "read the category section... " << endl;
	QDomElement categorylist = account.namedItem("categorylist").toElement();
	categorySList.clear();
	n = categorylist.firstChild();
	i=0;
	while (!n.isNull()) {
		QDomElement e = n.toElement();
		if (i!=e.attribute("num").toInt()) {
			kdDebug(OPALE_DEBUG_STREAM) << "num=" << e.attribute("num").toInt()
				<< " in categorys=" << e.attribute("name").latin1() << " element,"
				<< i << " was expected, file is corrupted" << endl;
			return false;
		}
		i++;
		categorySList.append(e.attribute("name"));
		n = n.nextSibling();
	}
	kdDebug(OPALE_DEBUG_STREAM) << i << " elements" << endl;
	emit categoriesUpdated();

	/* read the patterns section */
	kdDebug(OPALE_DEBUG_STREAM) << "read the patterns section..." << endl;
	QDomElement patternslist = account.namedItem("patternlist").toElement();

	patterns.clear();
	n = patternslist.firstChild();
	i = 0;
	while (!n.isNull()) {
		patterns.append( new OpalePattern(this, n.toElement()) );
		n = n.nextSibling();
		i++;
	}
	syncPatterns(); // one single emit at the end
	kdDebug(OPALE_DEBUG_STREAM) << i << " elements" << endl;

	/* read the data section */
	kdDebug(OPALE_DEBUG_STREAM) << "read the data section..." << endl;
	QDomElement records = account.namedItem("recordlist").toElement();

	clear();
	n = records.firstChild();
	i=0;
	while (!n.isNull()) {
		new OpaleRecord(this, n.toElement());
		n = n.nextSibling();
		i++;
	}
	emit listUpdated();
	kdDebug(OPALE_DEBUG_STREAM) << i << " elements" << endl;

	kdDebug(OPALE_DEBUG_STREAM) << "end of loadXML" << endl;

	setModified(false); emit listUpdated();
	gotoNow();
	return true;
}


/*

<?xml version="1.0" encoding="UTF-8"?><!DOCTYPE opale >
<account
	mime="application/x-opale"
	editor="Opale"
	currency="Euro" 
	account_name="bank_name" >

	<howlist>
		<how name="Check" num="0" />
		<how name="asdsf" num="1" />
		<how name="asdf" num="2" />
	</howlist>

	<categorylist>
		<category name="Check" num="0" />
		<category name="asdsf" num="1" />
		<category name="asdf" num="2" />
	</categorylist>

	<data>
		<record amount="12300" when="2008-06-16" what="Burps pour la shbrob" how="3" category="1" />
		<record amount="12300" when="2008-06-16" what="Burps pour la shbrob" how="3" category="1" />
		<record amount="12300" when="2008-06-16" what="Burps pour la shbrob" how="3" category="1" />
	</data>
</account>


 */

QDomDocument OpaleData::saveXML()
{
	QDomDocument doc("opale");
	doc.appendChild( doc.createProcessingInstruction( "xml", "version=\"1.0\" encoding=\"UTF-8\"" ) );
	QDomElement account = doc.createElement( "account" ); 
	account.setAttribute( "mime",	"application/x-opale" );
	account.setAttribute( "editor",	"Opale" );
	account.setAttribute( "account_name", s_account_name );
	account.setAttribute( "currency", s_currency);
	doc.appendChild( account );

	/* record the method names */
	QDomElement methodlist = doc.createElement( "howlist" ); 

	for ( uint i=0; i< methodSList.count(); i++) {
		QDomElement record = doc.createElement( "how" );
		record.setAttribute("name", methodSList[i]);
		record.setAttribute("num", i);
		methodlist.appendChild(record);
        }
	account.appendChild(methodlist);

	/* record the category names */
	QDomElement categorylist = doc.createElement( "categorylist" ); 

	for ( uint i=0; i< categorySList.count(); i++) {
		QDomElement record = doc.createElement( "category" );
		record.setAttribute("name", categorySList[i]);
		record.setAttribute("num", i);
		categorylist.appendChild(record);
        }
	account.appendChild(categorylist);

	/* record the patterns */
	QDomElement patternslist = doc.createElement( "patternlist" ); 
	for (OpalePattern *p = patterns.first(); p; p=patterns.next()) {
		patternslist.appendChild( p->domElement(doc));
	}
	account.appendChild(patternslist);


	/* record the data */
	QDomElement qde = doc.createElement( "recordlist" ); 
	QListViewItemIterator it(this);
	while ( it.current() ) {
		qde.appendChild( item2record(it.current())->domElement(doc));
		++it;
	}
	account.appendChild(qde);

	setModified(false);
	return doc;
}

void OpaleData::setModified(bool f)
{
	b_modified=f;
	setSorting(0, false);
	sort();
	setSorting(-1);
}

void OpaleData::fillDataStats(StatsData &statsData, QDate begin , QDate end)
{
	statsData.methodPositive.resize(methodSList.count());
	statsData.categoryPositive.resize( categorySList.count() );
	statsData.methodNegative.resize(methodSList.count());
	statsData.categoryNegative.resize( categorySList.count() );

	// clear them all
	unsigned int i;
	for (i=0; i< statsData.methodPositive.count(); i++)
		statsData.methodPositive[i] = 0;
	for (i=0; i< statsData.methodNegative.count(); i++)
		statsData.methodNegative[i] = 0;
	for (i=0; i< statsData.categoryPositive.count(); i++)
		statsData.categoryPositive[i] = 0;
	for (i=0; i< statsData.categoryNegative.count(); i++)
		statsData.categoryNegative[i] = 0;

	QListViewItemIterator it(lastItem());
	// We start from the end of the list, which corresponds to the
	// start in the time
	while ( it.current() ) {
		OpaleRecord *r = item2record(it.current());
		
		// the end;
		if (r->date>end) break;

		// begin
		if (r->date>begin)   {
			//fill
			amount_t m = r->m_amount;
			if ( m>0 ) {
				statsData.methodPositive[r->method]+= m;
				statsData.categoryPositive[r->category]+= m;
			} else {
				m = -m;
				statsData.methodNegative[r->method]+= m;
				statsData.categoryNegative[r->category]+= m;
			}
		}
		// keep on
		--it;
	} // while


}

void OpaleData::fillDataGraph(QValueList<GraphData> &chartData  , QDate begin, QDate end, amount_t &a_min, amount_t &a_max)
{
	GraphData gdata;
	bool onemore=false;

	a_min = a_max = 0; 
	chartData.clear();

	QListViewItemIterator it(lastItem());
	// We start from the end of the list, which corresponds to the
	// start in the time
	while ( it.current() ) {
		OpaleRecord *r = item2record(it.current());

		// add the previous one, so that we are sure to have one
		// point _before_ the actual "begin".
		if (r->date>begin) chartData.append(gdata);

		// compute
		r->addToChart(gdata);

		// min/max
		if (begin < r->date &&  r->date <= end) {
#ifdef DISPLAY_BEST_CASE 
			if (gdata.best>a_max) a_max = gdata.best;
#else
			if (gdata.balance>a_max) a_max = gdata.balance;
#endif
			if (gdata.worst<a_min) a_min = gdata.worst;
		}

		// the end (after everything, so that we got one more point)
		if (r->date>end) {
			if (onemore)
				break;
			else onemore=true;
		}

		--it;
	}

	// safeguard
	if (a_max == a_min ) {
		a_min = -10 * 100;
		a_max = +10 * 100;
	}
}

void OpaleData::updateTotal(Total &total)
{
	total.global = 0l;	total.utt = 0l;
	total.uttsafe = 0l;	total.checked = 0l;

	QListViewItemIterator it(lastItem());
	while ( it.current() ) {
		item2record(it.current())->updateTotal( total );
		--it;
	}
}


void OpaleData::fillCombos(QComboBox *method, QComboBox *category)
{
	QStringList::iterator it;

	for (  it = methodSList.begin(); it != methodSList.end(); ++it )
		method->insertItem( (*it) );
	for (  it = categorySList.begin(); it != categorySList.end(); ++it )
		category->insertItem( (*it) );
}


int OpaleData::checkMethodCombos(QComboBox *method)
{
// XXX orzel : re-enable auto-add
/*
	int i = methodSList.findIndex(method->currentText());
	if ( i==-1 ) {
		kdDebug(OPALE_DEBUG_STREAM) << "checkMethodCombos : adding" << endl;
		i = methodSList.count();
		methodSList.append(method->currentText());
	}
	return i;
	*/
	int i = method->currentItem();
	Q_ASSERT(i>=0); Q_ASSERT(i<(int)methodSList.count());
	return i;
}

int OpaleData::checkCategoryCombos(QComboBox *category)
{
/* XXX orzel : reenable
	int i = categorySList.findIndex(category->currentText());
	if ( i==-1 ) {
		kdDebug(OPALE_DEBUG_STREAM) << "checkCategoryCombos : adding" << endl;
		i = categorySList.count();
		categorySList.append(category->currentText());
	}
	return i;
*/

	int i = category->currentItem();
	Q_ASSERT(i>=0); Q_ASSERT(i<(int)categorySList.count());
	return i;
}


void OpaleData::touched(void)
{
	setModified(true); emit listUpdated();
}

void OpaleData::gotoNow( void )
{
	int i=0;
	QDate now = QDate::currentDate();
	QListViewItemIterator it(this);
	QListViewItem *itnow=0, *beyond=0;
	while ( it.current() && i<10) {
		if ( item2record(it.current())->date < now ) i++;
		else itnow=it.current();
		beyond = it.current();
		++it;
	}
	if ( beyond ) ensureItemVisible(beyond );
	if ( itnow ) ensureItemVisible(itnow);
}


void OpaleData::addPattern(OpalePattern *newp)
{
	patterns.append( newp);
	syncPatterns();
}


void OpaleData::syncPatterns(void)
{
	QStringList	namelist;
	for (uint i=0; i< patterns.count(); i++)
		namelist.append( patterns.at(i)->name);
	emit newPatterns ( namelist );
}

int OpaleData::editFileOptions(void)
{
	
	int result;

	fileOptionsDialog dialog(parentWidget(), "edit_file_options", true);

	dialog.edit_currency->setText(currency());
	dialog.edit_account_name->setText(account_name());
	result = dialog.exec();

	if (result) {
	//	setInfos(dialog.edit_currency->text(), dialog.edit_account_name->text());
		s_currency = dialog.edit_currency->text();
		s_account_name = dialog.edit_account_name->text();
		touched();
	}
	return result;
}

int OpaleData::editCategories(void)
{
	// create dialog
	ListBoxEditor dialog( this, editCategory, parentWidget(), categorySList);
	dialog.setCaption(  tr2i18n(  "Edit Categories List" ) );

//useless	connect(dialog.preview, SIGNAL(selectionChanged(void)), this, SIGNAL(dataUpdated(void)));

	int result;

	result = dialog.exec();

	Q_ASSERT(result); // no cancel button

	// fill categorySList
	categorySList.clear();
	for ( uint i=0; i<dialog.preview->count(); i++ ) {
		Q_ASSERT( dialog.preview->text(i) != QString());
		categorySList.append(dialog.preview->text(i));
	}

	touched();
	emit categoriesUpdated();
	return result;
}

int OpaleData::editMethods(void)
{
	// create dialog
	ListBoxEditor dialog( this, editMethod, parentWidget(), methodSList);
	dialog.setCaption(tr2i18n("Edit List"));

	int result;

	result = dialog.exec();

	Q_ASSERT(result); // no cancel button

	// fill methodSList
	methodSList.clear();
	for ( uint i=0; i<dialog.preview->count(); i++ ) {
		Q_ASSERT( dialog.preview->text(i) != QString());
		methodSList.append(dialog.preview->text(i));
	}

	touched();
	emit methodsUpdated();
	return result;
}


bool OpaleData::canDelete( uint idx, editBoxWhat_t editBoxWhat )
{
	QStringList &list = (editCategory==editBoxWhat)?categorySList:methodSList;

	// check non empty
	qDebug("OpaleData::canDelete : count is %ld", list.count());
	if ( list.count()<=1 ){
		KMessageBox::error(0, i18n("You can't delete the last item!"), i18n("Not enough data"));
		return false;
		}

	// check if does exist
	bool found=false;
	QListViewItemIterator it(this);
	if ( editCategory==editBoxWhat ){

		while ( it.current() ) {
			if ( idx == item2record(it.current())->category )
			{ found=true; break; }
			++it;
		}
		for ( uint i = 0; !found && i< patterns.count(); ++ i )
			if ( idx==patterns.at(i)->category ) { found=true; break; }
	} else {
		while ( it.current() ) {
			if ( idx == item2record(it.current())->method )
			{ found=true; break; }
			++it;
		}
		for ( uint i = 0; !found && i< patterns.count(); ++ i )
			if ( idx==patterns.at(i)->method ) { found=true; break; }
	}

	uint newidx=0;
	if ( found ) {

		// check what to replace
		replaceDialog dialog(0);

		if ( editCategory==editBoxWhat ){
			dialog.setCaption(  tr2i18n(  "Category replace dialog" ) );
			for ( uint i=0; i< categorySList.count(); i++)
				if ( i!=idx ) dialog.combo->insertItem( categorySList[i] );
		} else {
			dialog.setCaption(  tr2i18n(  "Method replace dialog" ) );
			for ( uint i=0; i< methodSList.count(); i++)
				if ( i!=idx ) dialog.combo->insertItem( methodSList[i] );
		}

		if ( !dialog.exec() )
			return false;
		newidx = dialog.combo->currentItem();
		if (newidx>=idx)newidx++;
	}



	// actually delete it
	if ( editCategory==editBoxWhat )
		removeCategory(idx, newidx);
	else
		removeMethod(idx, newidx);

	return true;
}


void OpaleData::removeMethod(uint num, uint newnum)
{
	Q_ASSERT(num!=newnum);
	Q_ASSERT(num<methodSList.count());

	if ( newnum>num ) newnum--; // newnum after removing num..

	// Update records...
	QListViewItemIterator it(this);
	while ( it.current() ) {
		OpaleRecord *r = item2record(it.current());
		if ( r->method==num ) { r->method=newnum; }
		else if ( r->method>num ) r->method--;
		Q_ASSERT(r->method<methodSList.count()-1);
		++it;
	}
	// ... and patterns
	for ( uint i = 0; i< patterns.count(); ++ i ) {
		OpalePattern *p;
		p = patterns.at(i);
		if ( p->method==num ) { p->method=newnum;}
		else if ( p->method>num ) p->method--;
		Q_ASSERT(p->method<methodSList.count()-1);
		}
	// change the list
	// XXX : orzel : Qt4 provides QList::removeAt()
	for ( uint i=num+1; i< methodSList.count(); i++)
		methodSList[i-1] =  methodSList[i];
	methodSList.pop_back();

	emit methodsUpdated();
	touched();
}

void OpaleData::removeCategory(uint num, uint newnum)
{
	Q_ASSERT(num!=newnum);
	Q_ASSERT(num<categorySList.count());

	if ( newnum>num ) newnum--; // newnum after removing num..

	// Update records...
	QListViewItemIterator it(this);
	while ( it.current() ) {
		OpaleRecord *r = item2record(it.current());
		if ( r->category==num ) { r->category=newnum;}
		else if ( r->category>num ) r->category--;
		Q_ASSERT(r->category<categorySList.count()-1);
		++it;
	}
	// ... and patterns
	for ( uint i = 0; i< patterns.count(); ++ i ) {
		OpalePattern *p;
		p = patterns.at(i);
		if ( p->category==num ) { p->category=newnum;}
		else if ( p->category>num ) p->category--;
		Q_ASSERT(p->category<categorySList.count()-1);
	}
	// change the list
	// XXX : orzel : Qt4 provides QList::removeAt()
	for ( uint i=num+1; i< categorySList.count(); i++)
		categorySList[i-1] =  categorySList[i];
	categorySList.pop_back();

	emit categoriesUpdated();
	touched();
}


bool OpaleData::loadXML( const QString filename )
{
	QFile file( filename );
	if ( !file.open( IO_ReadOnly ) ) {
		QMessageBox::critical(0, "Opale", i18n("Can't open the file") );
		return false;
	}

	QDomDocument doc;
	QString errorMsg;
	int errorLine, errorColumn;
	if ( !doc.setContent(&file, &errorMsg, &errorLine, &errorColumn ) ) {
		QString msg= "Error parsing the input file\n";
		msg += "error text : \""+errorMsg+"\"\n";
		msg += QString("      line : %1\n" ).arg(errorLine);
		msg += QString("    column : %1\n" ).arg(errorColumn);
		QMessageBox::critical( 0, "Opale", msg );
		file.close();
		return false;
	}
	bool b=loadXML(doc);
	file.close();
	return b;
}

bool OpaleData::saveXML(QString filename)
{
	Q_ASSERT(filename!=QString::null); // shouldn't happen either

	QFile file( filename );
	if ( !file.open( IO_Truncate | IO_WriteOnly ) ) {
		QMessageBox::critical(0, "Opale", "Can't open the file for writing" );
		return false;
	}

	QTextStream stream( &file );
	stream.setEncoding(   QTextStream::UnicodeUTF8 );
	stream << saveXML().toString(1);

	file.close();

	return true;
}



void OpaleData::fillList(void)
{
	// find item for 'today'
	QListViewItem *item = firstChild();
	if ( !item ) return;
	QListViewItem *item2;
	for ( ;item; item = item->nextSibling() ) {
		if ( item->text( 5 )==QString( "" ) ) break; // '5' is "Future?"
		//if ( item->text( 3 )!=QString( "" ) ) break; // '3' is "Checked"
	}
	if ( !item ) { // nothing found
		item= firstChild();
		return;
		}
	// ensure visibility
	item2=item->itemAbove();
	for (  int i=0; item2 && i<7; i++,item2=item->itemAbove() )
		item=item2;
	ensureItemVisible( item );

}

void OpaleData::editRecord(QListViewItem *item)
{
	if (!item) // nothing selected
		return;
	OpaleRecord *record = item2record( item );
	int result;
	bool checked = record->checked();

	if (!record) {
		kdDebug(OPALE_DEBUG_STREAM) << "OpaleData::editRecord(KListViewItem *item):unexpected item selected ?!" << endl;
		return;
	}

	inputRecordDialog dialog(this, i18n("Modify Record").latin1(), true);
	dialog.PushButton1->setFocus(); // for some reason the datepicker get the focus..
	fillCombos(dialog.edit_method, dialog.edit_category);
	record->to_dialog(&dialog);

	result = dialog.exec();
	if (result) {
		record->from_dialog(&dialog);
		record->setChecked(checked);
		setSorting(0); sort(); setSorting(-1);
		touched();
	}
}


void OpaleData::new_pattern(void)
{
	kdDebug(OPALE_DEBUG_STREAM) << "OpaleData::new_pattern() called" << endl;

	OpalePattern pattern(this);
	int result;

	inputPatternDialog dialog(this, i18n("Modify Pattern").latin1(), true);
	fillCombos(dialog.edit_method, dialog.edit_category);

	pattern.to_dialog(&dialog);

	result = dialog.exec();
	if (result)
		addPattern(new OpalePattern(this, &dialog ) );
}



void OpaleData::repeat_record(void)
{
	// presently cut is really a 'delete'
	kdDebug(OPALE_DEBUG_STREAM) << "OpaleData::repeat_record() called" << endl;

	OpaleRecord *orig_record = item2record(currentItem());

	/* gui */
	int result;

	inputRepeatDialog dialog(this, i18n("Repeat Record").latin1(), true);
	result = dialog.exec();

	if (!result) return; // nothing to do


	int i;
	int n = dialog.spinBoxNb->value();
	int periodNb = dialog.spinBoxPeriod->value();
	int periodUnit = dialog.editPeriod->currentItem();

	for (i=0; i< n; i++) {
		OpaleRecord *r = new OpaleRecord(this, *orig_record);
		r->unCheck();
		r->addTime( (i+1)*periodNb, periodUnit);
	}
	touched();
}



void OpaleData::delete_record(void)
{
	// presently cut is really a 'delete'
	kdDebug(OPALE_DEBUG_STREAM) << "OpaleData::cut() called" << endl;

	delete currentItem();

	touched();
}

void OpaleData::check_record(void)
{
	QListViewItem *item = selectedItem();
	if ( !item ) return; // nothing selected...

	OpaleRecord *record = item2record( item );

	if (!record) {
		kdDebug(OPALE_DEBUG_STREAM) << "OpaleData::spacePressedRecord(QListViewItem *item):unexpected item selected ?!" << endl;
		return;
	}
	record->switchCheck();
	touched();
}


void OpaleData::itemclicked( QListViewItem * item, const QPoint & , int c )
{
	if ( !item )
		return;
	if ( 3==c ) {
		((OpaleRecord*)item)->switchCheck();
		touched();
	}
}

#include "data.moc"

