/*
   Copyright (C) 2001-2007 Thomas Capricelli <orzel@freehackers.org>
*/

/* System */
#include <math.h>

/* Qt */
#include <qpainter.h>
#include <qvbox.h>
#include <qpushbutton.h>

/* KDE */
#include <kdebug.h>

/* Opale */
#include "graph.h"
#include "data.h"

OpaleGraph::OpaleGraph( OpaleData *data, QWidget *parentWidget, const char *widgetName)
	: KMainWindow( parentWidget, widgetName)
{
/*
	if (lp->count() < 5)
		kdDebug(OPALE_DEBUG_STREAM) <<  "strange, lp->count is small : " << lp->count() << endl;
*/

	// layout
	QHBox * hb = new QHBox(this);
	QVBox * vb = new QVBox(hb);

	// do it
	chart = new GraphWidget(data, QDate::currentDate().addMonths(-3), QDate::currentDate().addMonths(+5),hb, "opale_chart");
	hb->setStretchFactor(vb,1);
	hb->setStretchFactor(chart,8);

//	kdDebug(OPALE_DEBUG_STREAM) <<  "Trying to find reason for ba:ck bgd"<< endl;
//	kdDebug(OPALE_DEBUG_STREAM) << "brush : " << chart->backgroundBrush() << endl;
//	kdDebug(OPALE_DEBUG_STREAM) << "color : " << chart->backgroundColor() << endl;
	chart->setBackgroundMode( Qt::FixedColor  );
	chart->setPaletteBackgroundColor( Qt::white  );

	connect( new QPushButton( i18n("Zoom in"), vb), SIGNAL(clicked(void)), chart ,SLOT(zoomin(void)));
	connect( new QPushButton( i18n("Zoom out"), vb), SIGNAL(clicked(void)), chart ,SLOT(zoomout(void)));
	connect( new QPushButton( i18n("Go left"), vb), SIGNAL(clicked(void)), chart ,SLOT(goleft(void)));
	connect( new QPushButton( i18n("Go right"), vb), SIGNAL(clicked(void)), chart ,SLOT(goright(void)));

	// polishing
	setCentralWidget(hb);
	setMinimumSize(500,400);

	connect( data, SIGNAL(listUpdated(void)), chart, SLOT(updateChart(void)));

}

OpaleGraph::~OpaleGraph()
{
}

/*
 * GraphWidget stuff
 */

#define OPALE_GRAPH_MIN_LEN (60*60*24*5) 	// 5 days
#define ZOOM_FACTOR float(1.8)
#define MOVE_FACTOR float(0.3)

/*
 * moving/zooming
 */


void GraphWidget::goleft(void)  { setMiddleLen(middle - uint( float(len) * MOVE_FACTOR),len ); }
void GraphWidget::goright(void) { setMiddleLen(middle + uint( float(len) * MOVE_FACTOR),len ); }
void GraphWidget::zoomout(void) { setMiddleLen(middle,  uint( float(len) * ZOOM_FACTOR ) ); }

void GraphWidget::zoomin(void)
{
	uint _len = uint ( float( len )/ZOOM_FACTOR ) ;
	if ( _len < OPALE_GRAPH_MIN_LEN)
		_len = OPALE_GRAPH_MIN_LEN;
	setMiddleLen( middle, _len);
}


// TODO qt4 : use Julian stuff everywhere instead of setTime_t
void GraphWidget::setBeginEnd(uint uistart, uint uiend)
{
	QDateTime qdt;
	qdt.setTime_t(uistart); start = qdt.date();
	qdt.setTime_t(uiend); end = qdt.date();

	pstart = uistart;
	pend = uiend;

	// updating len/middle
	len = uiend - uistart;
	middle = (uiend+uistart)/2;
	if (len< MINIMUM_LENGTH) {
		len = MINIMUM_LENGTH;
		uistart = middle - MINIMUM_LENGTH/2;
		uiend = middle + MINIMUM_LENGTH/2;
		qdt.setTime_t(uistart); start = qdt.date();
		qdt.setTime_t(uiend); end = qdt.date();
	}

	datelen = start.daysTo(end);
	Q_ASSERT(datelen);

	// updating chart
	updateChart();
}


GraphWidget::GraphWidget(OpaleData *_data, QDate s, QDate e, QWidget *parentWidget, const char *widgetName)
	:DrawWidget( parentWidget,widgetName ) , data(_data) , start( s ), end( e )
{
	setBeginEnd( QDateTime(start).toTime_t(), QDateTime(end).toTime_t() ); // update middle/len/datelen
}

GraphWidget::~GraphWidget()
{
}

void GraphWidget::mousePressEvent( QMouseEvent *evt )
{
//	qDebug( " mousePressEvent() x is %d", evt->x( ) );
	mousePos = -1;
	int x=evt->x();
	if (x<=x1 || x>=x2 )
		return;
	if ( evt->button() == Qt::LeftButton ) 
		mousePos = x;
	else if ( evt->button() == Qt::RightButton ){
//		qDebug("GraphWidget::mousePressEvent, x=%d, ", x);
		setMiddleLen( pixel2time(x) ,  uint( float(len) * ZOOM_FACTOR ) ); // zoomout centered on x()
	}
}

void GraphWidget::mouseDoubleClickEvent( QMouseEvent *evt )
{
	int x=evt->x();
	if (x<=x1 || x>=x2 )
		return;
	if ( evt->button() == Qt::LeftButton ) {
//		qDebug("GraphWidget::mouseDoubleClickEvent, x=%d", x);
		setMiddleLen( pixel2time(x) ,  uint( float(len) / ZOOM_FACTOR ) ); // zoomout centered on x()
	}
}

void GraphWidget::mouseReleaseEvent( QMouseEvent *evt )
{
	if ( mousePos==-1 )
		return;
	int x=evt->x();
	if (abs( x-mousePos )<5 )
		return;
//	qDebug( " mouseReleaseEvent() x is %d", evt->x( ) );
	if ( x<mousePos ){
		x=mousePos;
		mousePos=evt->x();
	}
	setBeginEnd( pixel2time(mousePos), pixel2time(x));
}



void GraphWidget::updateChart(void)
{
	data->fillDataGraph(chartData, start, end, a_min, a_max);
	// own repaint
	redraw();
}


QDate GraphWidget::pixel2date(int p)
{
	return start.addDays( (p-x1)*datelen/(x2-x1));
}

int GraphWidget::date2pixel(QDate d )
{
	int p=1+x1+start.daysTo(d)*(x2-x1)/datelen;
//	to debug pixel2date :
//	qDebug("date, point, date : %s,%d,%s", d.toString().ascii(), p, pixel2date( p ).toString().ascii() );
	return p;
}

int GraphWidget::pixel2time(int x )
{
	return middle-len/2+(x-x1)*(len/(x2-x1));
}

void GraphWidget::redraw(void)
{
	int w=width(), h=height();
	pix.resize(w,h);

	QPainter p(&pix);

	// scales
	int i = a_max-a_min;
//	qDebug("\nScaling, delta      = %d", i);
	i/=7; i = int (logf(float(i))/ logf(10.));
//	qDebug("Scaling, i          = %d", i);
	int amountLine = 1; while (i-- ) amountLine*=10; // 10^i
//	qDebug("Scaling, amountLine = %d", amountLine);
	i = a_max-a_min;
	if      ( i/amountLine >60 ) amountLine*=6;
	else if ( i/amountLine >40 ) amountLine*=4;
	else if ( i/amountLine >20 ) amountLine*=2;
	
	int amountDotted = amountLine/2;

	enum {
		dm_shortyear,     // 04/05/06/08
		dm_year,          // 2004, 2005, 2006
		dm_yearmonth,     // 01,02,03,04 on first line, years on second
		dm_month,         // 01,02,03,04 only
		dm_monthdays,     // days and month on 2 lines
	} dateMode;

	dateMode = dm_monthdays;
	if ( datelen >90 ) // 3 months
		dateMode = dm_month;
	if ( datelen > 365/2 ) // 6 months
		dateMode = dm_yearmonth;
	if ( datelen > 3*365 ) // 3 years
		dateMode = dm_year;
	if ( datelen > 22*365 ) // 22 years
		dateMode = dm_shortyear;

	// let's go
	p.eraseRect(0,0,w,h); // clear

	int textw = 90;  // text width
	int texth = 30;  // text high

#define VSPACE 10
#define HSPACE 10
	x1 = HSPACE+textw+HSPACE;
	x2 = w-HSPACE;
	y1 = VSPACE;
	y2 = h-VSPACE-texth-VSPACE;

	if (dm_yearmonth==dateMode || dm_monthdays==dateMode) y2-=texth+VSPACE; // one more line
	int x,y,z;

	//
	// draw amount grid
	//

	p.setPen(QPen(Qt::gray, 2, Qt::SolidLine));
	int a=amountLine; // 0 is allways displayed
	while (a<a_max) {
		y = amount2pixel(a);
		p.drawLine(x1,y,x2,y);
		a+=amountLine;
	}
	a=-amountLine;
	while (a>a_min) {
		y = amount2pixel(a);
		p.drawLine(x1,y,x2,y);
		a-=amountLine;
	}
	// Dotted
	p.setPen(QPen(Qt::gray, 2, Qt::DotLine));
	a=amountDotted;
	while (a<a_max) {
		y = amount2pixel(a);
		p.drawLine(x1,y,x2,y);
		a+=amountLine;
	}
	a=-amountDotted;
	while (a>a_min) {
		y = amount2pixel(a);
		p.drawLine(x1,y,x2,y);
		a-=amountLine;
	}
	// text
	p.setPen(QPen(Qt::black, 2, Qt::SolidLine));
	a=0;
	while (a<a_max) {
		y = amount2pixel(a);
		p.drawText(HSPACE,y-h/2,textw,h,Qt::AlignRight|Qt::AlignVCenter,QString("%1").arg(int(a/100)) );
		a+=amountLine;
	}
	a=-amountLine;
	while (a>a_min) {
		y = amount2pixel(a);
		p.drawText(HSPACE,y-h/2,textw,h,Qt::AlignRight|Qt::AlignVCenter,QString("%1").arg(int(a/100)) );
		a-=amountLine;
	}
	// amount zero
	p.setPen(QPen(Qt::magenta, 2, Qt::SolidLine));
	y = amount2pixel(0); p.drawLine(x1,y,x2,y);

	//
	// draw time grid
	//
	QDate now = QDate::currentDate();
	if ( start<=now && now<=end ) {
		x = date2pixel(now);
		p.drawLine(x,y1,x,y2); // As of now...
		}
	switch (dateMode) {
		case dm_shortyear:
			displayTime(p,QPen(Qt::gray, 1, Qt::SolidLine),1, y2+VSPACE, texth);
			break;
		case dm_year:
			displayTime(p,QPen(Qt::gray, 2, Qt::SolidLine),1, y2+VSPACE, texth);
			break;
		case dm_yearmonth:
			displayTime(p,QPen(Qt::gray, 1, Qt::SolidLine),2, y2+VSPACE, texth); // month 
			displayTime(p,QPen(Qt::black, 2, Qt::SolidLine),1, y2+VSPACE+VSPACE+texth, texth); // year
			break;
		case dm_month:
			displayTime(p,QPen(Qt::gray, 1, Qt::SolidLine),2, y2+VSPACE, texth);
			break;
		case dm_monthdays:
			displayTime(p,QPen(Qt::gray, 1, Qt::SolidLine),3, y2+VSPACE, texth); // days
			displayTime(p,QPen(Qt::gray, 2, Qt::SolidLine),2, y2+VSPACE+VSPACE+texth, texth); // months
			break;
	};

// debug p.setPen(Qt::red); p.drawRect(x1,y1,x2-x1,y2-y1);
	// draw actual data
	p.setClipRect(x1-1,y1-1,x2-x1+2,y2-y1+2);// protect
	QValueList<GraphData>::iterator it;

	//	worst
	p.setPen(QPen(Qt::blue, 2, Qt::SolidLine));
        it = chartData.begin();
	x = date2pixel( (*it).qdt.date() );
	y = amount2pixel( (*it).worst );
        for ( ++it; it != chartData.end(); ++it ) {
		z = date2pixel( (*it).qdt.date() );
		p.drawLine(x,y,z,y); x=z;
		z = amount2pixel( (*it).worst );
		p.drawLine(x,y,x,z); y=z;
		}
	z = date2pixel( end );
	p.drawLine(x,y,z,y);
	
	//	worst2
	p.setPen(QPen(Qt::blue, 1, Qt::SolidLine));
        it = chartData.begin();
	x = date2pixel( (*it).qdt.date() );
	y = amount2pixel( (*it).worst2 );
        for ( ++it; it != chartData.end(); ++it ) {
		z = date2pixel( (*it).qdt.date() );
		p.drawLine(x,y,z,y); x=z;
		z = amount2pixel( (*it).worst2 );
		p.drawLine(x,y,x,z); y=z;
		}
	z = date2pixel( end );
	p.drawLine(x,y,z,y);
	
#ifdef DISPLAY_BEST_CASE // best case are useless anyway...
	//	best
	p.setPen(QPen(Qt::green, 2, Qt::SolidLine));
        it = chartData.begin();
	x = date2pixel( (*it).qdt.date() );
	y = amount2pixel( (*it).best );
        for ( ++it; it != chartData.end(); ++it ) {
		z = date2pixel( (*it).qdt.date() );
		p.drawLine(x,y,z,y); x=z;
		z = amount2pixel( (*it).best );
		p.drawLine(x,y,x,z); y=z;
		}
	
	//	best2
	p.setPen(QPen(Qt::green, 1, Qt::SolidLine));
        it = chartData.begin();
	x = date2pixel( (*it).qdt.date() );
	y = amount2pixel( (*it).best2 );
        for ( ++it; it != chartData.end(); ++it ) {
		z = date2pixel( (*it).qdt.date() );
		p.drawLine(x,y,z,y); x=z;
		z = amount2pixel( (*it).best2 );
		p.drawLine(x,y,x,z); y=z;
		}
#endif
	
	//	Balance, red
	p.setPen(QPen(Qt::red, 2, Qt::SolidLine));
        it = chartData.begin();
	x = date2pixel( (*it).qdt.date() );
	y = amount2pixel( (*it).balance );
        for ( ++it; it != chartData.end(); ++it ) {
		z = date2pixel( (*it).qdt.date() );
		p.drawLine(x,y,z,y); x=z;
		z = amount2pixel( (*it).balance );
		p.drawLine(x,y,x,z); y=z;
		}
	z = date2pixel( end );
	p.drawLine(x,y,z,y);

	p.end();
	update(); // QWidget:: ...
}

void GraphWidget::displayTime(QPainter &p, QPen pen, int what, int y, int h)
{
	// 0 -> year
	// 1 -> month
	// 2 -> days

	QDate t;
	QPen textpen(Qt::black, 2, Qt::SolidLine);
	int x,w;
	switch ( what ) {
		case 1:
			w = (365*(x2-x1)/datelen); w+=2;
			t.setYMD(start.year(), 1, 1); // previous year start
			t = t.addYears(1);
			p.setPen(pen);
			while (t<end) {
				x = date2pixel(t);
				p.drawLine(x,y1,x,y2);
				t = t.addYears(1);
				}
			t.setYMD(start.year(), 1, 1); // previous year start
			p.setPen(textpen);
			while (t<end) {
				x = date2pixel(t);
				p.drawRect(x,y,w,h);
				p.drawText(x,y,w,h,Qt::AlignCenter,QString("%1").arg(t.year()));
				t = t.addYears(1);
				}
			break;
		case 2:
			w = (31*(x2-x1)/datelen); w+=2;
			t.setYMD(start.year(), start.month(),1); // previous month start
			t = t.addMonths(1);
			p.setPen(pen);
			while (t<end) {
				x = date2pixel(t);
				p.drawLine(x,y1,x,y2);
				t = t.addMonths(1);
				}
			t.setYMD(start.year(), start.month(),1); // previous month start
			p.setPen(textpen);
			while (t<end) {
				x = date2pixel(t);
//				p.drawRect(x,y,w,h);
				p.drawText(x,y,w,h,Qt::AlignCenter,QString("%1").arg(t.month()));
				t = t.addMonths(1);
				}
			break;
		case 3:
			w = ((x2-x1)/datelen); w+=2;
			t = start; // previous day
			t = t.addDays(1);
			p.setPen(pen);
			while (t<end) {
				x = date2pixel(t);
				p.drawLine(x,y1,x,y2);
				t = t.addDays(1);
				}
			p.setPen(textpen);
			t = start; // previous day
			while (t<end) {
				x = date2pixel(t);
				p.drawRect(x,y,w,h);
				p.drawText(x,y,w,h,Qt::AlignCenter,QString("%1").arg(t.day()));
				t = t.addDays(1);
				}
			break;
		default:
			qFatal("displayTime() panic!!!! unknown time kind to display");
	}

}

#include "graph.moc"


