//---------------------------------------------------------------------------
#include "amatrix.h"

//---------------------------------------------------------------------------
CMatrix::CMatrix( void )
{
   Rows = 0;
   Cols = 0;
   Cells = NULL;
}
//---------------------------------------------------------------------------
CMatrix::CMatrix( int ARows, int ACols, long double* Values )
{
   int j;

   Rows = ARows;
   Cols = ACols;
   Cells = new CComplex[ Rows * Cols ];
   if( Values )
   {
      for( j = 0; j < Rows * Cols; j++ )
      {
         Cells[ j ] = CComplex( Values[ j ], 0.0 );
      }
   }
   else
   {
      for( j = 0; j < Rows * Cols; j++ )
      {
         Cells[ j ] = CComplex( 0.0, 0.0 );
      }
   }
}
//---------------------------------------------------------------------------
CMatrix::CMatrix( int ARows, int ACols, CComplex* Values )
{
   int j;

   Rows = ARows;
   Cols = ACols;
   Cells = new CComplex[ Rows * Cols ];
   if( Values )
   {
      for( j = 0; j < Rows * Cols; j++ )
      {
         Cells[ j ] = Values[ j ];
      }
   }
   else
   {
      for( j = 0; j < Rows * Cols; j++ )
      {
         Cells[ j ] = CComplex( 0.0, 0.0 );
      }
   }
}
//---------------------------------------------------------------------------
CMatrix::CMatrix( const CVector& v )
{
   Rows = 1;
   Cols = v.GetSize();
   for( int j = 0; j < Cols; j++ )
   {
      Cells[ j ] = v.GetValue( j );
   }
}
//---------------------------------------------------------------------------
CMatrix::CMatrix( const CVector* v )
{
   Rows = 1;
   Cols = v->GetSize();
   for( int j = 0; j < Cols; j++ )
   {
      Cells[ j ] = v->GetValue( j );
   }
}
//---------------------------------------------------------------------------
CMatrix::CMatrix( CMatrix* M )
{
   if( M )
   {
      int i, j;

      Rows = M->RowCount();
      Cols = M->ColCount();
      Cells = new CComplex[ Rows * Cols ];
      for( i = 0; i < Rows; i++ )
      {
         for( j = 0; j < Cols; j++ )
         {
            Cells[ i * Cols + j ] = M->GetCell( i, j );
         }
      }
   }
   else
   {
      Rows = 0;
      Cols = 0;
      Cells = NULL;
   }
}
//---------------------------------------------------------------------------
int CMatrix::ColCount( void ) const
{
   return Cols;
}
//---------------------------------------------------------------------------
int CMatrix::RowCount( void ) const
{
   return Rows;
}
//---------------------------------------------------------------------------
bool CMatrix::IsEmpty( void ) const
{
   return ( !Cols && !Rows );
}
//---------------------------------------------------------------------------
void CMatrix::SetSize( int ARow, int ACol )
{
   if( ACol > 0 && ARow > 0 )
   {
      if( Cells )
      {
         delete Cells;
         Cells = NULL;
      }
      Cells = new CComplex[ ACol * ARow ];
      Cols = ACol;
      Rows = ARow;
   }
   else
   {
      if( Cells )
      {
         delete Cells;
      }
      Cells = NULL;
      Cols = 0;
      Rows = 0;
   }
}
//---------------------------------------------------------------------------
void CMatrix::AddRow( void )
{
   CComplex* cpom;
   int x, y;

   if( Rows && Cols )
   {
      cpom = new CComplex[ ( Rows + 1 ) * Cols ];
      for( y = 0; y < Rows + 1; y++ )
      {
         for( x = 0; x < Cols; x++ )
         {
            if( y < Rows )
            {
               cpom[ y * Cols + x ] = GetCell( y, x );
            }
            else
            {
               cpom[ y * Cols + x ] = CComplex( 0.0, 0.0 );
            }
         }
      }
      delete Cells;
      Rows++;
      Cells = cpom;
   }
}
//---------------------------------------------------------------------------
void CMatrix::InsertRow( int Index )
{
   CComplex* cpom;
   int x, y, i;

   if( Rows && Cols & Index >= 0 && Index < Rows )
   {
      cpom = new CComplex[ ( Rows + 1 ) * Cols ];
      for( y = 0, i = 0; y < Rows; y++ )
      {
         if( y == Index )
         {
            i = 1;
            for( x = 0; x < Cols; x++ )
            {
               cpom[ y * Cols + x ] = CComplex( 0.0, 0.0 );
               cpom[ ( y + i ) * Cols + x ] = Cells[ y * Cols + x ];
            }
         }
         else
         {
            for( x = 0; x < Cols; x++ )
            {
               cpom[ ( y + i ) * Cols + x ] = Cells[ y * Cols + x ];
            }
         }
      }
      delete Cells;
      Rows++;
      Cells = cpom;
   }
}
//---------------------------------------------------------------------------
void CMatrix::DeleteRow( int Index )
{
   CComplex* cpom;
   int x, y, i;

   if( Rows > 1 && Cols && Index >= 0 && Index < Rows )
   {
      cpom = new CComplex[ ( Rows - 1 ) * Cols ];
      for( y = 0, i = 0; y < Rows; y++ )
      {
         if( y == Index )
         {
         }
         else
         {
            for( x = 0; x < Cols; x++ )
            {
               cpom[ i * Cols + x ] = Cells[ y * Cols + x ];
            }
            i++;
         }
      }
      delete Cells;
      Rows--;
      Cells = cpom;
   }
   else
   {
      //throw Exception( "Row deleting error!\nRows=" + AnsiString( Rows ) +
      //                 ";Cols=" + AnsiString( Cols ) + ";Index=" + AnsiString( Index ) );
   }
}
//---------------------------------------------------------------------------
void CMatrix::AddCol( void )
{
   CComplex* cpom;
   int x, y;

   if( Rows && Cols )
   {
      cpom = new CComplex[ Rows * ( Cols + 1 ) ];
      for( y = 0; y < Rows; y++ )
      {
         for( x = 0; x < Cols + 1; x++ )
         {
            if( x < Cols )
            {
               cpom[ y * ( Cols + 1 ) + x ] = GetCell( y, x );
            }
            else
            {
               cpom[ y * ( Cols + 1 ) + x ] = CComplex( 0.0, 0.0 );
            }
         }
      }
      delete Cells;
      Cols++;
      Cells = cpom;
   }
}
//---------------------------------------------------------------------------
void CMatrix::InsertCol( int Index )
{
   CComplex* cpom;
   int x, y, i;

   if( Rows && Cols && Index >= 0 && Index < Cols )
   {
      cpom = new CComplex[ Rows * ( Cols + 1 ) ];
      for( y = 0; y < Rows; y++ )
      {
         for( x = 0, i = 0; x < Cols; x++ )
         {
            if( x == Index )
            {
               cpom[ y * ( Cols + 1 ) + x ] = CComplex( 0.0, 0.0 );
               cpom[ y * ( Cols + 1 ) + x + 1 ] = Cells[ y * Cols + x ];
               i = 1;
            }
            else
            {
               cpom[ y * ( Cols + 1 ) + x + i ] = Cells[ y * Cols + x ];
            }
         }
      }
      delete Cells;
      Cols++;
      Cells = cpom;
   }
}
//---------------------------------------------------------------------------
void CMatrix::DeleteCol( int Index )
{
   CComplex* cpom;
   int x, y, i;

   if( Rows && Cols > 1 && Index >= 0 && Index < Cols )
   {
      cpom = new CComplex[ Rows * ( Cols - 1 ) ];
      for( y = 0; y < Rows; y++ )
      {
         for( x = 0, i = 0; x < Cols; x++ )
         {
            if( x == Index )
            {
            }
            else
            {
               cpom[ y * ( Cols - 1 ) + i ] = Cells[ y * Cols + x ];
               i++;
            }
         }
      }
      delete Cells;
      Cols--;
      Cells = cpom;
   }
}
//---------------------------------------------------------------------------
CComplex CMatrix::GetCell( int ARow, int ACol ) const
{
   CComplex v;

   if( ARow >= 0 && ARow < Rows && ACol >= 0 && ACol < Cols )
   {
      v = Cells[ ARow * Cols + ACol ];
   }
   else
   {
      v = CComplex( 0.0, 0.0 );
      //throw Exception( "Invalid cell: [" + AnsiString( ARow ) + "][" + AnsiString( ACol ) + "]!" );
   }
   return v;
}
//---------------------------------------------------------------------------
CComplex CMatrix::GetCell( int ANumber ) const
{
   CComplex v;

   if( ANumber >= 0 && ANumber < Rows * Cols )
   {
      v = Cells[ ANumber ];
   }
   else
   {
      v = CComplex( 0.0, 0.0 );
      //throw Exception( "Invalid cell number: [" + AnsiString( ANumber ) + "]!" );
   }
   return v;
}
//---------------------------------------------------------------------------
bool CMatrix::SetCell( int ARow, int ACol, CComplex v )
{
   bool pv;

   if( ARow >= 0 && ARow < Rows && ACol >= 0 && ACol < Cols )
   {
      Cells[ ARow * Cols + ACol ] = v;
      pv = true;
   }
   else
   {
      pv = false;
   }
   return pv;
}
//---------------------------------------------------------------------------
bool CMatrix::SetCell( int ANumber, CComplex v )
{
   bool pv;

   if( ANumber >= 0 && ANumber < Rows * Cols )
   {
      Cells[ ANumber ] = v;
      pv = true;
   }
   else
   {
      pv = false;
   }
   return pv;
}
//---------------------------------------------------------------------------
CMatrix CMatrix::GetRange( int Left, int Top, int Right, int Bottom )
{
   CMatrix v;
   int y, x;

   if( Left >= 0 && Left <= Right && Top >= 0 && Top <= Bottom &&
       Right < Cols && Bottom < Rows )
   {
      v.SetSize( Bottom - Top + 1, Right - Left + 1 );
      for( y = Top; y <= Bottom; y++ )
      {
         for( x = Left; x <= Right; x++ )
         {
            v.SetCell( y - Top, x - Left, Cells[ y * Cols + x ] );
         }
      }
   }
   else
   {
      //throw Exception( "Invalid range!" );
   }
   return v;
}
//---------------------------------------------------------------------------
AnsiString CMatrix::ToString( void )
{
   AnsiString p;
   int y, x;

   p = "[";
   for( y = 0; y < Rows; y++ )
   {
      p += AnsiString( "[" );
      for( x = 0; x < Cols; x++ )
      {
         p += ( ComplexToStr( GetCell( y, x ) ) );
         if( x < Cols - 1 )
         {
            p += AnsiString( "," );
         }
      }
      p += AnsiString( "]" );
      if( y < Rows - 1 )
      {
         p += AnsiString( "," );
      }
   }
   p += AnsiString( "]" );

   return p;
}
//---------------------------------------------------------------------------
CComplex CMatrix::Det( void ) const
{
   CMatrix v;
   CComplex s, t, c, ctmp1, ctmp2;
   int x, y;
   int i, j, k, size;

   if( Rows == Cols )
   {
      v.SetSize( Rows, Cols );
      size = Rows;
      for( y = 0; y < Rows; y++ )
      {
         for( x = 0; x < Cols; x++ )
         {
            v.SetCell( y, x, Cells[ y * Cols + x ] );
         }
      }
      for( i = 0; i < size - 1; i++ )
      {
         s = v.GetCell( i, i );
         for( k = i + 1; k < size; k++ )
         {
            t = v.GetCell( k, i );
            for( j = i; j < size; j++ )
            {
               v.SetCell( k, j, v.GetCell( k, j ) - v.GetCell( i,j ) * t / s );
            }
         }
      }
      c = CComplex( 1.0, 0.0 );
      for( i = 0; i < size; i++ )
      {
         c *= v.GetCell( i, i );
      }
/*
      if( Rows == 2 )
      {
         c = Cells[ 0 ] * Cells[ 3 ] - Cells[ 1 ] * Cells[ 2 ];
      }
      else
      {
         v.SetSize( Rows, Cols * 2 - 1 );
         for( y = 0; y < Rows; y++ )
         {
            for( x = 0; x < Cols; x++ )
            {
               v.SetCell( y, x, GetCell( y, x ) );
               if( x < Cols -  1 )
               {
                  v.SetCell( y, x + Cols, GetCell( y, x ) );
               }
            }
         }

         c = 0.0;
         for( x = 0; x < Cols; x++ )
         {
            ctmp1 = 1.0;
            for( y = 0; y < Rows; y++ )
            {
               ctmp1 *= v.GetCell( y, x + y );
            }
            ctmp2 = 1.0;
            for( y = Rows - 1; y >= 0; y-- )
            {
               ctmp2 *= v.GetCell( y, x + ( Rows - y - 1 ) );
            }
            c += ( ctmp1 - ctmp2 );
         }
      }
*/
   }
   else
   {
      //throw Exception( "Not square matrix!" );
   }
   return c;
}
//---------------------------------------------------------------------------
CComplex CMatrix::Minor( int Row, int Col ) const
{
   CComplex c;
   CMatrix v;

   if( Row >= 0 && Row < Rows && Col >= 0 && Col < Cols )
   {
      if( Rows == Cols )
      {
         v = *this;
         v.DeleteRow( Row );
         v.DeleteCol( Col );
         c = v.Det();
      }
      else
      {
         //throw Exception( "Not square matrix!" );
      }
   }
   else
   {
      //throw Exception( "Invalid row or column: [" + AnsiString( Row ) +
      //                 "][" + AnsiString( Col ) + "]!" );
   }
   return c;
}
//---------------------------------------------------------------------------
void CMatrix::Transpose( void )
{
   CComplex* cpom;
   int y, x;
   int NewCols, NewRows;

   if( Rows && Cols )
   {
      NewRows = Cols;
      NewCols = Rows;
      cpom = new CComplex[ NewRows * NewCols ];
      for( y = 0; y < NewRows; y++ )
      {
         for( x = 0; x < NewCols; x++ )
         {
            cpom[ y * NewCols + x ] = Cells[ x * Cols + y ];
         }
      }
      Cols = NewCols;
      Rows = NewRows;
      delete Cells;
      Cells = cpom;
   }
}
//---------------------------------------------------------------------------
int CMatrix::GetSize( void )
{
   return 0;
}
//---------------------------------------------------------------------------
int CMatrix::Save( FILE* f )
{
   int written = 0;
   if( f )
   {
      int x, y;
      CComplex c;
      
      written += fwrite( &Rows, 1, sizeof( int ), f );
      written += fwrite( &Cols, 1, sizeof( int ), f );
      for( y = 0; y < Rows; y++ )
      {
         for( x = 0; x < Cols; x++ )
         {
            c = GetCell( y, x ); // y = row, x = col
            written += c.Save( f );
         }
      }
   }
   return written;
}
//---------------------------------------------------------------------------
int CMatrix::Load( FILE* f )
{
   int readed = 0;
   if( f )
   {
      int r, x, y;
      CComplex c;
      
      readed += fread( &y, 1, sizeof( int ), f );
      readed += fread( &x, 1, sizeof( int ), f );
      SetSize( y, x );
      for( y = 0; y < Rows; y++ ) // Rows
      {
         for( x = 0; x < Cols; x++ ) // Cols
         {
            r = c.Load( f );//c = GetCell( y, x );
            if( r )
            {
               SetCell( y, x, c ); // y = row, x = col
               readed += r;
            }
         }
      }
   }
   return readed;
}
//---------------------------------------------------------------------------
//
// Friend CMatrix class functions
//
//---------------------------------------------------------------------------
const CMatrix& CMatrix::operator=( const CMatrix& val )
{
   if( Cells )
   {
      delete Cells;
   }

   Cols = val.ColCount();
   Rows = val.RowCount();
   Cells = new CComplex[ Cols * Rows ];
   for( int y = 0; y < Rows; y++ )
   {
      for( int x = 0; x < Cols; x++ )
      {
         Cells[ y * Cols + x ] = val.GetCell( y, x );
      }
   }
   return *this;
}
//---------------------------------------------------------------------------
CMatrix operator+( const CMatrix& a, const CMatrix& b )
{
   CMatrix v;
   int y, x;

   if( a.Cols == b.Cols && a.Rows == b.Rows )
   {
      v.SetSize( a.Rows, a.Cols );
      for( y = 0; y < v.Rows; y++ )
      {
         for( x = 0; x < v.Cols; x++ )
         {
            v.Cells[ y * v.Cols + x ] = a.Cells[ y * a.Cols + x ] +
                                        b.Cells[ y * b.Cols + x ];
         }
      }

   }
   return v;
}
//---------------------------------------------------------------------------
CMatrix operator-( const CMatrix& a, const CMatrix& b )
{
   CMatrix v;
   int y, x;

   if( a.Cols == b.Cols && a.Rows == b.Rows )
   {
      v.SetSize( a.Rows, a.Cols );
      for( y = 0; y < v.Rows; y++ )
      {
         for( x = 0; x < v.Cols; x++ )
         {
            v.Cells[ y * v.Cols + x ] = a.Cells[ y * a.Cols + x ] -
                                        b.Cells[ y * b.Cols + x ];
         }
      }

   }
   return v;
}
//---------------------------------------------------------------------------
CMatrix operator*( const CMatrix& a, const CMatrix& b )
{
   CMatrix v;
   CComplex c;
   int y, x, j;
   bool lError;

   if( a.Cols == b.Rows )
   {
      v.SetSize( a.Rows, b.Cols );
      for( y = 0; y < v.Rows; y++ )
      {
         for( x = 0; x < v.Cols; x++ )
         {
            c = CComplex( 0.0, 0.0 );
            for( j = 0; j < a.Cols; j++ )
            {
               c += ( a.GetCell( y, j ) * b.GetCell( j, x ) );
            }
            if( v.SetCell( y, x, c ) )
            {
            }
            else
            {
               //throw Exception( "Invalid matrix cell [" + AnsiString( y ) + "][ " + AnsiString( x ) + "]!" );
            }
         }
      }
   }
   else
   {
      //throw Exception( "Dimensions of arguments not passed!" );
   }
   return v;
}
//---------------------------------------------------------------------------
CMatrix operator*( const CMatrix& a, const CComplex& b )
{
   CMatrix v;
   int y, x;

   v.SetSize( a.Rows, a.Cols );
   for( y = 0; y < v.Rows; y++ )
   {
      for( x = 0; x < v.Cols; x++ )
      {
         v.Cells[ y * v.Cols + x ] = a.Cells[ y * a.Cols + x ] * b;
      }
   }
   return v;
}
//---------------------------------------------------------------------------
CMatrix operator*( const CMatrix& a, const CVector& b )
{
   CMatrix m = CMatrix( b );

   return a * m;
}
//---------------------------------------------------------------------------
CMatrix operator*( const CVector& a, const CMatrix& b )
{
   CMatrix m = CMatrix( a );

   return m * b;
}
//---------------------------------------------------------------------------
CMatrix operator*( const CComplex& a, const CMatrix& b )
{
   return b * a;
}
//---------------------------------------------------------------------------
CMatrix operator/( const CMatrix& a, const CComplex& b )
{
   CMatrix v;
   int y, x;

   if( b != CComplex( 0.0, 0.0 ) )
   {
      v.SetSize( a.Rows, a.Cols );
      for( y = 0; y < v.Rows; y++ )
      {
         for( x = 0; x < v.Cols; x++ )
         {
            v.Cells[ y * v.Cols + x ] = a.Cells[ y * a.Cols + x ] / b;
         }
      }
   }
   else
   {
      //throw Exception( "Divide by zero!" );
   }
   return v;
}
//---------------------------------------------------------------------------
CMatrix operator^( const CMatrix& a, const int& b )
{
   CMatrix v;
   int j, i;

   if( a.Cols == a.Rows )
   {
      if( b == -1 )
      {
         v = Inverse( a );
      }
      else if( b == 0 )
      {
         v = SingleMatrix( a );
      }
      else
      {
         v = a;
         if( b > 1 )
         {
            for( j = 1; j < b; j++ )
            {
               v = v * a;
            }
         }
      }
   }
   else
   {
      //throw Exception( "Not square matrix!" );
   }
   return v;
}
//---------------------------------------------------------------------------
bool operator==( const CMatrix& a, const CMatrix& b )
{
   bool lEqual;
   int x, y;

   if( a.Cols == b.Cols && a.Rows == b.Rows )
   {
      lEqual = true;
      for( y = 0; y < a.Rows && lEqual; y++ )
      {
         for( x = 0; x < a.Cols && lEqual; x++ )
         {
            if( a.Cells[ y * a.Cols + x ] != b.Cells[ y * b.Cols + x ] )
            {
               lEqual = false;
            }
         }
      }
   }
   else
   {
      lEqual = false;
   }
   return lEqual;
}
//---------------------------------------------------------------------------
bool operator!=( const CMatrix& a, const CMatrix& b )
{
   return !( a == b );
}
//---------------------------------------------------------------------------
CMatrix SingleMatrix( const CMatrix& a )
{
   CMatrix v;
   int y, x;

   if( a.Cols == a.Rows )
   {
      v.SetSize( a.Rows, a.Cols );
      for( y = 0; y < v.Rows; y++ )
      {
         for( x = 0; x < v.Cols; x++ )
         {
            if( y == x )
            {
               v.Cells[ y * v.Cols + x ] = CComplex( 1.0, 0.0 );
            }
            else
            {
               v.Cells[ y * v.Cols + x ] = CComplex( 0.0, 0.0 );
            }
         }
      }
   }
   else
   {
      //throw Exception( "Not square matrix!" );
   }
   return v;
}
//---------------------------------------------------------------------------
CMatrix Transpose( const CMatrix& a )
{
   CMatrix v;
   int y, x;
   int NewCols, NewRows;

   if( a.Rows && a.Cols )
   {
      NewRows = a.Cols;
      NewCols = a.Rows;
      v.SetSize( NewRows, NewCols );

      for( y = 0; y < NewRows; y++ )
      {
         for( x = 0; x < NewCols; x++ )
         {
            v.Cells[ y * NewCols + x ] = a.Cells[ x * a.Cols + y ];
         }
      }
   }
   else
   {
      //throw Exception( "Zero matrix!" );
   }
   return v;
}
//---------------------------------------------------------------------------
CMatrix Inverse( const CMatrix& a )
{
   CMatrix v;
   CComplex s, t, c, c1, c2;
   int size;
   int x, y;
   int j, i, k;

   if( a.Rows != a.Cols )
   {
      //throw Exception( "Not square matrix!" );
   }
   else if( a.Det() == CComplex( 0.0, 0.0 ) )
   {
      //throw Exception( "Cannot inverse!" );
   }
   else
   {
      size = a.Cols;
      v.SetSize( size, size * 2 );
      for( i = 0; i < a.Rows; i++ )
      {
         for( j = 0; j < a.Cols; j++ )
         {
            v.SetCell( i, j, a.Cells[ i * a.Cols + j ] );
         }
      }
      for( i = 0; i < size; i++ )
      {
         for( j = size; j < 2 * size; j++ )
         {
            if( i == j - size )
            {
               v.SetCell( i, j, CComplex( 1.0, 0.0 ) );
            }
            else
            {
               v.SetCell( i, j, CComplex( 0.0, 0.0 ) );
            }
         }
      }

      for( i = 0; i < size; i++ )
      {
         s = v.GetCell( i, i );
         for( j = i; j < 2 * size; j++ )
         {
            v.SetCell( i, j, v.GetCell( i, j ) / s );
         }
         for( j = i + 1; j < size; j++ )
         {
            t = v.GetCell( j, i );
            for( k = i; k < 2 * size; k++ )
            {
               v.SetCell( j, k, ( -1.0 ) * v.GetCell( i, k ) * t
                + v.GetCell( j, k ) );
            }
         }
      }

      for( k = size; k < 2 * size; k++ )
      {
         for( i = size - 1; i >= 0; i-- )
         {
            s = v.GetCell( i, k );
            for( j = i + 1; j < size; j++ )
            {
               s -= ( v.GetCell( i, j ) * v.GetCell( j, k ) );
            }
            v.SetCell( i, k, s );
         }
      }
      return v.GetRange( size, 0, 2 * size - 1, size - 1 );
   }
   return v;
}
//---------------------------------------------------------------------------
CMatrix round( const CMatrix& a, int ARound )
{
   CMatrix r;
   int j;

   if( !a.IsEmpty() )
   {
      r.SetSize( a.Rows, a.Cols );
      for( j = 0; j < a.Rows * a.Cols; j++ )
      {
         r.Cells[ j ] = round( a.Cells[ j ], ARound );
      }
   }
   return r;
}
//---------------------------------------------------------------------------
//
//
//
//---------------------------------------------------------------------------
AnsiString MatrixToStr( CMatrix m )
{
   return m.ToString();
}
//---------------------------------------------------------------------------
AnsiString MatrixToStrF( CMatrix m, int prec )
{
   AnsiString p;
   int x, y;
   
   if( prec < 0 )
   {
      prec = 6;
   }
   p = "[";
   for( y = 0; y < m.RowCount(); y++ )
   {
      p += AnsiString( "[" );
      for( x = 0; x < m.ColCount(); x++ )
      {
         p += ComplexToStrF( m.GetCell( y, x ), prec );
      }
      p += AnsiString( "]" );
   }
   p += AnsiString( "]" );
   return p;
}
//---------------------------------------------------------------------------
CMatrix StrToMatrix( AnsiString Text )
{
   CMatrix m;
   
   return m;
}
//---------------------------------------------------------------------------
