#include <astack.h>
#include <atomutil.h>
#include <aconst.h>

#ifdef __WIN32__
#include <windows.h>
#endif

#ifdef __YACAS__
#include "stdcommandline.h"
#include "standard.h"
#include "numbers.h"
#include "arggetter.h"
#include "archiver.h"

#include <scripts.cc>
#endif

//------------------------------------------------------------------------------------------------
typedef struct afunc
{
   TStackFunc func;
   char name[ 20 ];
   int argc;
} __AFunc__;

afunc _funcs[] = {
                 { sfAdd, "+", 2 },
                 { sfSub, "-", 2 },
                 { sfMul, "*", 2 },
                 { sfDiv, "/", 2 },
                 { sfPow, "^", 2 },
                 { sfFactorial, "!", 1 },
                 { sfPercent, "%", 2 },
                 { sfPercentT, "%T", 2 },
                 { sfPercentCH, "%CH", 2 },
                 { sfPow2, "^2", 1 },
                 { sfNeg, "Neg", 1 },
                 //{ sfNeg, "+/-", 1 },
                 { sfSign, "Sign", 1 },
                 { sfAbs, "Abs", 1 },
                 { sf1perX, "1/", 1 },
                 { sfSqrt,  "Sqrt", 1 },
                 { sfRootN, "RootN", 2 },
                 { sfExp, "Exp", 1 },
                 { sfLn, "Ln", 1 },
                 { sfLog10, "Log10", 1 },
                 { sfALog, "ALog", 1 },
                 { sfLnp1, "Lnp1", 1 },
                 { sfExpm, "Expm", 1 },
                 { sfReal2Exponent, "R2E", 1 },
                 { sfExponent2Real, "E2R", 1 },

                 { sfSin, "Sin", 1 },
                 { sfSinH, "SinH", 1 },
                 { sfASin, "ASin", 1 },
                 { sfASinH, "ASinH", 1 },
                 
                 { sfCos, "Cos", 1 },
                 { sfCosH, "CosH", 1 },
                 { sfACos, "ACos", 1 },
                 { sfACosH, "ACosH", 1 },
                 
                 { sfTan, "Tan", 1 },
                 { sfTanH, "TanH", 1 },
                 { sfATan, "ATan", 1 },
                 { sfATanH, "ATanH", 1 },
                 
                 { sfCtg, "Ctg", 1 },
                 { sfCtgH, "CtgH", 1 },
                 { sfACtg, "ACtg", 1 },
                 { sfACtgH, "ACtgH", 1 },
                 
                 { sfSec, "Sec", 1 },
                 { sfSecH, "SecH", 1 },
                 { sfASec, "ASec", 1 },
                 { sfASecH, "ASecH", 1 },
                 
                 { sfCsc, "Csc", 1 },
                 { sfCscH, "CscH", 1 },
                 { sfACsc, "ACsc", 1 },
                 { sfACscH, "ACscH", 1 },
                 
                 { sfSetRad, "SetRad", 0 },
                 { sfSetDeg, "SetDeg", 0 },
                 { sfDeg2Rad, "Deg2Rad", 1 },
                 { sfRad2Deg, "Rad2Deg", 1 },

                 { sfReal2Complex, "R2C", 2 },
                 { sfReal, "Re", 1 },
                 { sfImag, "Im", 1 },
                 { sfArg, "Arg", 1 },
                 { sfConj, "Conj", 1 },
                 { sfPolar, "Polar", 2 },
                 { sfNorm, "Norm", 1 },
                 
                 { sfIp, "Ip", 1 },
                 { sfFp, "Fp", 1 },
                 
                 { sfSum, "Sum", -1 },
                 { sfAvg, "Avg", -1 },
                 { sfMin, "Min", -1 },
                 { sfMax, "Max", -1 },
                 
                 { sfEq, "==", 2 },
                 { sfNEq, "<>", 2 },
                 { sfGreat, ">", 2 },
                 { sfGreatEq, ">=", 2 },
                 { sfLess, "<", 2 },
                 { sfLessEq, "<=", 2 },
                 
                 { sfConvert, "Conv", 2 },
                 
                 { sfCross, "Cross", 2 },
                 { sfDot, "Dot", 2 },
                 { sfReal2Vector2, "R2V2", 2 },
                 { sfReal2Vector3, "R2V3", 3 },
                 
                 { sfLength, "Length", 1 },
                 
                 { sfBin, "Bin", 1 },
                 { sfDec, "Dec", 1 },
                 { sfHex, "Hex", 1 },
                 { sfOct, "Oct", 1 },
                 
                 { sfAnd, "And", 2 },
                 { sfOr, "Or", 2 },
                 { sfXor, "Xor", 2 },
                 { sfNot, "Not", 1 },
                 { sfIntDiv, "Div", 2 },
                 { sfIntMod, "Mod", 2 },
                 { sfSlb, "Slb", 1 },
                 { sfSrb, "Srb", 1 },
                 { sfRlb, "Rlb", 1 },
                 { sfRrb, "Rrb", 1 },

                 { sfIsPrime, "IsPrime", 1 },
                 { sfGamma, "Gamma", 1 },
                 { sfPsi, "Psi", 1 },
                 
                 { sfEvalExp, "Eval", 1 },
                 { sfNum, "Num", 1 },
                 { sfDeriv, "Deriv", 2 },
                 { sfDervx, "Dervx", 1 },
                 { sfInteg, "Integ", 2 },
                 { sfIntvx, "Intvx", 1 },
                 { sfIntegN, "Intgn", 4 },
                 { sfSolve, "Solve", 2 },
                 { sfSolvx, "Solvx", 1 },
                 { sfLimit, "Limit", 3 },
                 { sfLimvx, "Limvx", 2 },

                 { sfUnknown, "?", 0 }
                 };
//------------------------------------------------------------------------------------------------
const char* _errmsgs[] = {
                   "OK!", // 0
                   "Unknown stack function!", // 1
                   "Not enough arguments on stack!", // 2
                   "NULL stack argument!", // 3
                   "Division by zero!", // 4
                   "NULL result!", // 5
                   "Empty stack!", // 6
                   "Index out of bounds!", // 7
                   "Inconsistent arguments!", // 8
                   "Operation not allowed!", // 9
                   "Invalid result!", // 10
                   "Invalid argument(s)!", // 11
                   "YACAS not initialized!", // 12
                   "Unknown error!"
                   };

//------------------------------------------------------------------------------------------------
int split2strings( char* cmdline, TStringList* Strings, bool caseSens = true )
{
   AnsiString p, s;
   int len, count;

   if( !Strings )
   {
      return 0;
   }
   
   s = AnsiString( cmdline ).Trim();
   len = s.Length();
   
   if( len > 0 )
   {
      int j, rndbr, sqrbr, quota, apost;
      char z;
      
      rndbr = 0;
      sqrbr = 0;
      quota = 0;
      apost = 0;
      p = "";
      
      for( j = 1; j <= len; j++ )
      {
         z = s[ j ];
         if( ( z == ' ' ) && p.Length() > 0 && !sqrbr && !quota && !apost )
         {
            Strings->Add( p );
            count++;
            p = "";
         }
         else
         {
            if( z == '[' )
            {
               sqrbr++;
            }
            else if( z == ']' )
            {
               sqrbr--;
            }
            else if( z == '\"' )
            {
               if( quota )
               {
                  quota = 0;
               }
               else
               {
                  quota = 1;
               }
            }
            else if( z == '\'' )
            {
               if( apost )
               {
                  apost = 0;
               }
               else
               {
                  apost = 1;
               }
            }
            p += AnsiString( z );
         }
      }
      if( p.Length() > 0 )
      {
         Strings->Add( p );
         count++;
      }
   }
   return count;
}
//------------------------------------------------------------------------------------------------
//
//
//
//------------------------------------------------------------------------------------------------
atomstack::atomstack( int size )
{
   #ifdef __YACAS__
   stringout = NEW LispString();
   output = NEW StringOutput(*stringout);
   _yacas = CYacas::NewL(output);
   _yacas_valid = false;

   init_yacas();
   #endif
   
   _size = size;
   _atoms = new atom*[ _size ];
   _count = 0;
   _ptr = -1;
   _prec = 6;
   _anglemeasure = samRadian;
   _errcode = 0;
   _errmsg[ 0 ] = '\0';

   //create_const_list();
   _const = NULL;
   _vars = NULL;
}
//------------------------------------------------------------------------------------------------
atomstack::~atomstack()
{
   #ifdef __YACAS__
   delete _yacas;
   //delete _yacas_out;
   #endif

   delete [] _atoms;
   //delete _const;
   delete _vars;
}
//------------------------------------------------------------------------------------------------
#ifdef __YACAS__
//------------------------------------------------------------------------------------------------
//
//
//
//------------------------------------------------------------------------------------------------
bool atomstack::init_yacas( void )
{
   bool res = false;

   if( _yacas )
   {
      CCompressedArchive *a = NEW CCompressedArchive( __scripts_data, __scripts_size, 1);
      (*_yacas)()().iArchive = a;
       _yacas->Evaluate("Load(\"yacasinit.ys\");");
      if( !_yacas->IsError() )
      {
         _yacas->Evaluate("N(Exp(1))");
         if( !_yacas->IsError() )
         {
            res = true;
            _yacas_valid = true;
         }
      }
      else
      {
         MsgBox( "Error initializing YACAS!!!" );
      }
   }
   return res;
}
//------------------------------------------------------------------------------------------------
bool atomstack::init_yacas( char* archive )
{
   char* init_script = "yacasinit.ys";
   bool res = false;

   if( _yacas )
   {
     if( archive )
     {
        FILE* fin = fopen(archive,"rb");
        if (!fin)
        {
            MsgBox( AnsiString( "Error, could not open archive file [" ) + AnsiString( archive ) + "]" );
        }
        else
        {
            fseek(fin,0,SEEK_END);
            int fullsize = ftell(fin);
            fseek(fin,0,SEEK_SET);
            unsigned char* fullbuf = (unsigned char*)PlatAlloc(fullsize);
            if (fullbuf)
            {
                fread(fullbuf,1,fullsize,fin);
                CCompressedArchive *a =
                    NEW CCompressedArchive(fullbuf, fullsize, 1 ); //compressed_archive);
                if (a->iFiles.IsValid())
                {
                    char buf[ 100 ];
                    
                    (*_yacas)()().iArchive = a;
                    sprintf(buf,"Load(\"%s\");",init_script);
                    _yacas->Evaluate( buf );
                    if( !_yacas->IsError() )
                    {
                       res = true;
                    }
                    else
                    {
                    }
                }
                else
                {
                    MsgBox( AnsiString( "Error " ) + AnsiString( archive ) + " is not a valid archive file." );
                    delete a;
                }
            }
            else
            {
                //lprintf("Archive file %s too large, perhaps it is time we\nimplement disk-accessed compressed files.\n",archive);
            }
            fclose(fin);
        }
    }
  }
    _yacas_valid = res;
    return res;
}
//------------------------------------------------------------------------------------------------
#endif // __YACAS__
//------------------------------------------------------------------------------------------------
//void atomstack::create_const_list( void )
//{
   //_const = create_std_const_list();
//}
//------------------------------------------------------------------------------------------------
void atomstack::assignConstants( atomlist* AList )
{
   _const = AList;
}
//------------------------------------------------------------------------------------------------
void atomstack::assignVariables( atomlist* AList )
{
   _vars = AList;
}
//------------------------------------------------------------------------------------------------
void atomstack::set_precision( int aprec )
{
   _prec = aprec;
}
//------------------------------------------------------------------------------------------------
void atomstack::set_anglemeasure( TStackAngleMeasure aanglemeasure )
{
   _anglemeasure = aanglemeasure;
}
//------------------------------------------------------------------------------------------------
bool atomstack::isempty( void )
{
   return ( _ptr == -1 );
}
//------------------------------------------------------------------------------------------------
void atomstack::clear( void )
{
   if( _count && _ptr >= 0 )
   {
      int j; 
   
      for( j = _ptr; j >= 0; j-- )
      {
         delete _atoms[ j ];
      }
      _ptr = -1;
      _count = 0;
   }
   _errcode = 0;
   _errmsg[ 0 ] = '\0';
}
//------------------------------------------------------------------------------------------------
void atomstack::push( atom& a )
{
   if( _ptr < _size - 1 )
   {
      //fprintf( __PGMathLog, "Pushing... (Before:Count=%d;Ptr=%d)\n", _count, _ptr );
      _ptr++;
      _count++;
      _atoms[ _ptr ] = new atom( a );
      //_errcode = 0;
      //fprintf( __PGMathLog, "New atom created...\n" );
   }
}
//------------------------------------------------------------------------------------------------
atom atomstack::pop( void )
{
   atom a;
   if( _ptr >= 0 )
   {
      a = *( _atoms[ _ptr ] );
      delete _atoms[ _ptr ];
      _atoms[ _ptr ] = NULL;
      _ptr--;
      _count--;
   }
   else
   {
      a = nullatom;
   }
   return a;
}
//------------------------------------------------------------------------------------------------
atom atomstack::get( int index )
{
   atom a;
   if( _ptr >= 0 && index >= 0 && index <= _ptr )
   {
      a = *( _atoms[ _ptr - index ] );
   }
   else
   {
      ////fprintf( __PGMathLog, "_Ptr is NULL!" );
      a = nullatom;
   }
   ////fprintf( __PGMathLog, "Returning!" );
   return a;
}
//------------------------------------------------------------------------------------------------
bool atomstack::drop( int index )
{
   bool lResult = false;
   if( _ptr >= 0 && index >= 0 && index <= _ptr )
   {
      int i, j;
      
      for( j = _ptr - index; j < _ptr; j++ )
      {
         *( _atoms[ j ] ) = *( _atoms[ j + 1 ] );
      }
      delete _atoms[ _ptr ];
      _atoms[ _ptr ] = NULL;
      
      _ptr--;
      _count--;
      _errmsg[ 0 ] = '\0';
      _errcode = 0;
      lResult = true;
   }
   else
   {
      if( _count <= 0 )
      {
         _errcode = 6;
         create_errmsg( "DROP" );
      }
      else if( index < 0 || index >= _count )
      {
         _errcode = 7;
         create_errmsg( "DROP" );
      }
   }
   return lResult;
}
//------------------------------------------------------------------------------------------------
bool atomstack::totop( int index )
{
   bool lResult = false;
   if( _ptr >= 0 && index > 0 && index <= _ptr )
   {
      atom a = *( _atoms[ _ptr - index ] );
      int i, j;
      
      for( j = _ptr - index; j < _ptr; j++ )
      {
         *( _atoms[ j ] ) = *( _atoms[ j + 1 ] );
      }

      *( _atoms[ _ptr ] ) = a;
      _errmsg[ 0 ] = '\0';
      _errcode = 0;
      lResult = true;
   }
   else
   {
      if( _count <= 0 )
      {
         _errcode = 6;
         create_errmsg( "2TOP" );
      }
      else if( index < 0 || index >= _count )
      {
         _errcode = 7;
         create_errmsg( "2TOP" );
      }
   }
   return lResult;
}
//------------------------------------------------------------------------------------------------
bool atomstack::swap( int i1, int i2 )
{
   bool lResult = false;
   if( i1 >= 0 && i1 < _count && i2 >= 0 && i2 <= _count && i1 != i2 )
   {
      atom a = *( _atoms[ i1 ] );
      *( _atoms[ i1 ] ) = *( _atoms[ i2 ] );
      *( _atoms[ i2 ] ) = a;
      _errmsg[ 0 ] = '\0';
      _errcode = 0;
      lResult = true;
   }
   else
   {
      if( _count <= 0 )
      {
         _errcode = 6;
         create_errmsg( "SWAP" );
      }
      else if( i1 < 0 || i1 >= _count || i2 < 0 || i2 >= _count )
      {
         _errcode = 7;
         create_errmsg( "SWAP" );
      }
   }  
   return lResult;
}
//------------------------------------------------------------------------------------------------
void atomstack::create_errmsg( TStackFunc func )
{
   bool lFound = false;
   int j = 0;
   int index = -1;
   
   _errmsg[ 0 ] = '\0';
   
   while( _funcs[ j ].func != sfUnknown && !lFound )
   {
      if( _funcs[ j ].func == func )
      {
         index = j;
         lFound = true;
      }   
      else
      {
         j++;
      }
   }
   if( index >= 0 )
   {
      create_errmsg( _funcs[ index ].name );
   }
   else
   {
      create_errmsg( "STACK" );
   }
}
//------------------------------------------------------------------------------------------------
void atomstack::create_errmsg( const char* lpszErr )
{
   sprintf( _errmsg, "(%s): %s", lpszErr, _errmsgs[ _errcode ] );
}
//------------------------------------------------------------------------------------------------
const char* atomstack::errmsg( void )
{
   return _errmsg;//s[ _errcode ];
}
//------------------------------------------------------------------------------------------------
atom atomstack::Deg2Rad( atom& a )
{
   return ( _anglemeasure == samRadian || a.type() == aSymbol || a.type() == aExpression ? a : deg2rad( a ) );
}
//------------------------------------------------------------------------------------------------
atom atomstack::Rad2Deg( atom& a )
{
   return ( _anglemeasure == samDegree && a.type() != aSymbol && a.type() != aExpression ? rad2deg( a ) : a );
}
//------------------------------------------------------------------------------------------------
int atomstack::getargc( TStackFunc func )
{
   int j; 
   int argc = 0;
   bool lFound = false;
   
   j = 0;
   while( _funcs[ j ].func != sfUnknown && !lFound )
   {
      if( _funcs[ j ].func == func )
      {
         argc = _funcs[ j ].argc;
         lFound = true;
      }
      j++;
   }
   return argc;
}
//------------------------------------------------------------------------------------------------
bool atomstack::eval( TStackFunc func )
{
   atom* argv;
   atom res;
   atom tmp;
   int argc;
   int j, i;
   bool lResult = false;
   
   _errcode = 0;
   
   //if( !_count )
   //{
   //   _errcode = 6;
   //   create_errmsg( func );
   //   return false;
   //}
   
   argc = getargc( func );
   if( argc < 0 )
   {
      argc = _count;
   }
   else if( argc == 0 )
   {
   }
   else if( /*argc == 0 ||*/ argc > _count )
   {
      _errcode = ( argc == 0 ? 1 : 2 );
      create_errmsg( func );
      return false;
   }
   
   argv = new atom[ argc ];
   for( j = argc - 1; j >= 0 && !_errcode; j-- )
   {
      tmp = pop();
      if( tmp.isnull() )
      {
         for( i = j + 1; i < argc; i++ )
         {
            push( argv[ i ] );
         }
         _errcode = 3;
      }
      else
      {
         argv[ j ] = tmp;
      }
   }
   if( !_errcode )
   {
      switch( func )
      {
         case sfAdd:
         res = argv[ 0 ] + argv[ 1 ];
         break;
      
         case sfSub:
         res = argv[ 0 ] - argv[ 1 ];
         break;
      
         case sfMul:
         res = argv[ 0 ] * argv[ 1 ];
         break;
      
         case sfDiv:
         res = argv[ 0 ] / argv[ 1 ];
         if( res.isnull() )
         {
            _errcode = 4; // Division by zero!
         }
         else if( !res.valid() )
         {
            _errcode = 9;
         }
         break;
         
         case sfFactorial:
         res = factorial( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11; // Invalid argument(s)
         }
         break;
         
         case sfNeg:
         tmp = atom( -1.0 );
         res = tmp * argv[ 0 ];
         break;
         
         case sfPow:
         res = pow( argv[ 0 ], argv[ 1 ] );
         break;
         
         case sfPow2:
         tmp = atom( 2.0 );
         res = pow( argv[ 0 ], tmp );
         break;
         
         case sfPercent:
         res = percent( argv[ 0 ], argv[ 1 ] );
         break;
         
         case sfPercentT:
         res = percentt( argv[ 0 ], argv[ 1 ] );
         if( res.isnull() )
         {
            _errcode = 4; // Division by zero!
         }
         break;
         
         case sfPercentCH:
         res = percentch( argv[ 0 ], argv[ 1 ] );
         if( res.isnull() )
         {
            _errcode = 4; // Division by zero!
         }
         break;

         case sf1perX:
         tmp = atom( 1.0 );
         res = tmp / argv[ 0 ];
         if( res.isnull() )
         {
            _errcode = 4;
         }
         break;
      
         case sfSqrt:
         res = sqrt( argv[ 0 ] );
         break;
         
         case sfRootN:
         res = rootn( argv[ 0 ], argv[ 1 ] );
         break;
         
         case sfSign:
         res = sign( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11; // Invalid argument(s)
         }
         break;
         
         case sfAbs:
         res = abs( argv[ 0 ] );
         break;
         
         case sfExp:
         res = exp( argv[ 0 ] );
         break;
         
         case sfLn:
         res = ln( argv[ 0 ] );
         break;
         
         case sfLog10:
         res = log10( argv[ 0 ] );
         break;
         
         case sfALog:
         tmp = atom( 10.0 );
         res = pow( tmp, argv[ 0 ] );
         break;
         
         case sfLnp1:
         res = lnp1( argv[ 0 ] );
         break;
         
         case sfExpm:
         res = expm( argv[ 0 ] );
         break;
         
         case sfReal2Exponent:
         res = real2exponent( argv[ 0 ] );
         break;
         
         case sfExponent2Real:
         res = exponent2real( argv[ 0 ] );
         break;
         
         case sfSin:
         tmp = Deg2Rad( argv[ 0 ] );
         res = sin( tmp );
         break;
         
         case sfSinH:
         tmp = Deg2Rad( argv[ 0 ] );
         res = sinh( tmp );
         break;
         
         case sfASin:
         tmp = asin( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfASinH:
         tmp = asinh( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfCos:
         tmp = Deg2Rad( argv[ 0 ] );
         res = cos( tmp );
         break;
         
         case sfCosH:
         tmp = Deg2Rad( argv[ 0 ] );
         res = cosh( tmp );
         break;
         
         case sfACos:
         tmp = acos( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfACosH:
         tmp = acosh( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfTan:
         tmp = Deg2Rad( argv[ 0 ] );
         res = tan( tmp );
         break;
         
         case sfTanH:
         tmp = Deg2Rad( argv[ 0 ] );
         res = tanh( tmp );
         break;
         
         case sfATan:
         tmp = atan( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfATanH:
         tmp = atanh( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfCtg:
         tmp = Deg2Rad( argv[ 0 ] );
         res = cot( tmp );
         break;
         
         case sfCtgH:
         tmp = Deg2Rad( argv[ 0 ] );
         res = coth( tmp );
         break;
         
         case sfACtg:
         tmp = acot( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfACtgH:
         tmp = acoth( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfSec:
         tmp = Deg2Rad( argv[ 0 ] );
         res = sec( tmp );
         break;
         
         case sfSecH:
         tmp = Deg2Rad( argv[ 0 ] );
         res = sech( tmp );
         break;
         
         case sfASec:
         tmp = asec( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfASecH:
         tmp = asech( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfCsc:
         tmp = Deg2Rad( argv[ 0 ] );
         res = csc( tmp );
         break;
         
         case sfCscH:
         tmp = Deg2Rad( argv[ 0 ] );
         res = csch( tmp );
         break;
         
         case sfACsc:
         tmp = acsc( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfACscH:
         tmp = acsch( argv[ 0 ] );
         res = Rad2Deg( tmp );
         break;
         
         case sfSetRad:
         _anglemeasure = samRadian;
         break;
         
         case sfSetDeg:
         _anglemeasure = samDegree;
         break;
         
         case sfDeg2Rad:
         res = deg2rad( argv[ 0 ] );
         break;
         
         case sfRad2Deg:
         res = rad2deg( argv[ 0 ] );
         break;
         
         case sfReal2Complex:
         res = real2complex( argv[ 0 ], argv[ 1 ] );
         if( res.isnull() )
         {
            _errcode = 11; // Invalid argument(s)
         }
         break;
         
         case sfReal:
         res = real( argv[ 0 ] );
         break;
         
         case sfImag:
         res = imag( argv[ 0 ] );
         break;
         
         case sfArg:
         res = arg( argv[ 0 ] );
         break;
         
         case sfConj:
         res = conj( argv[ 0 ] );
         break;
         
         case sfPolar:
         res = polar( argv[ 0 ], argv[ 1 ] );
         break;
         
         case sfNorm:
         res = norm( argv[ 0 ] );
         break;
         
         case sfIp:
         res = atom_ip( argv[ 0 ] );
         break;
         
         case sfFp:
         res = atom_fp( argv[ 0 ] );
         break;
         
         case sfEq:
         res = equal( argv[ 0 ], argv[ 1 ] );
         break;
         
         case sfNEq:
         res = !equal( argv[ 0 ], argv[ 1 ] );
         break;
         
         case sfGreat:
         res = ( argv[ 0 ] > argv[ 1 ] );
         break;
         
         case sfGreatEq:
         res = ( argv[ 0 ] >= argv[ 1 ] );
         break;
         
         case sfLess:
         res = ( argv[ 0 ] < argv[ 1 ] );
         break;
         
         case sfLessEq:
         res = ( argv[ 0 ] <= argv[ 1 ] );
         break;
         
         case sfSum:
         res = atom( 0.0 );
         for( j = 0; j < argc && !res.isnull(); j++ )
         {
            res = res + argv[ j ];
            if( res.isnull() )
            {
               _errcode = 8;
               //res = nullatom;
            }
         }
         break;
         
         case sfMin:
         res = argv[ 0 ];
         for( j = 1; j < argc && !_errcode; j++ )
         {
            if( res.iscompatible( argv[ j ] ) )
            {
               if( res > argv[ j ] )
               {
                  res = argv[ j ];
               }
            }   
            else
            {       
               _errcode = 8;
               //res = nullatom;
            }
         }
         break;
         
         case sfMax:
         res = argv[ 0 ];
         for( j = 1; j < argc && !_errcode; j++ )
         {
            if( res.iscompatible( argv[ j ] ) )
            {
               if( res < argv[ j ] )
               {
                  res = argv[ j ];
               }
            }   
            else
            {       
               _errcode = 8;
               //res = nullatom;
            }
         }
         break;
         
         case sfAvg:
         res = atom( 0.0 );
         tmp = atom( argc );
         for( j = 0; j < argc && !res.isnull(); j++ )
         {
            res = res + argv[ j ];
            if( res.isnull() )
            {
               _errcode = 8;
               //res = nullatom;
            }
         }
         if( !_errcode )
         {
            res = res / tmp;
         }
         break;
         
         case sfConvert:
         res = uomconvert( argv[ 0 ], argv[ 1 ] );
         if( res.isnull() )
         {
            _errcode = 8;
         }
         break;
         
         case sfCross:
         res = cross( argv[ 0 ], argv[ 1 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;
         
         case sfDot:
         res = dot( argv[ 0 ], argv[ 1 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;
         
         case sfReal2Vector2:
         res = real2vector2( argv[ 0 ], argv[ 1 ] );
         break;

         case sfReal2Vector3:
         res = real2vector3( argv[ 0 ], argv[ 1 ], argv[ 2 ] );
         break;

         case sfLength:
         res = length( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 5;
         }
         break;
         
         case sfBin:
         res = int2bin( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;
         
         case sfDec:
         res = int2dec( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfHex:
         res = int2hex( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfOct:
         res = int2oct( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfAnd:
         res = aand( argv[ 0 ], argv[ 1 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;
         
         case sfOr:
         res = aor( argv[ 0 ], argv[ 1 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfXor:
         res = axor( argv[ 0 ], argv[ 1 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfNot:
         res = anot( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfIntDiv:
         if( (int)( argv[ 1 ] ) != 0 )
         {
            res = adiv( argv[ 0 ], argv[ 1 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 4;
         }
         break;

         case sfIntMod:
         if( (int)( argv[ 1 ] ) != 0 )
         {
            res = amod( argv[ 0 ], argv[ 1 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 4;
         }
         break;
         
         case sfSlb:
         res = aslb( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfSrb:
         res = asrb( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfRlb:
         res = arlb( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfRrb:
         res = arrb( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfIsPrime:
         res = isprime( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11; // Invalid argument(s)
         }
         break;
         
         case sfGamma:
         res = gamma( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfPsi:
         res = psi( argv[ 0 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;

         case sfPV:
         res = pv( argv[ 0 ], argv[ 1 ], argv[ 2 ], argv[ 3 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;
         
         case sfPMT:
         res = pmt( argv[ 0 ], argv[ 1 ], argv[ 2 ], argv[ 3 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;
         
         case sfFV:
         res = fv( argv[ 0 ], argv[ 1 ], argv[ 2 ], argv[ 3 ] );
         if( res.isnull() )
         {
            _errcode = 11;
         }
         break;
         
         #ifdef __YACAS__
         case sfEvalExp:
         if( _yacas_valid )
         {
            res = evalexp( argv[ 0 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;
         
         case sfNum:
         if( _yacas_valid )
         {
            res = evalnum( argv[ 0 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;

         case sfDeriv:
         if( _yacas_valid )
         {
            res = deriv( argv[ 0 ], argv[ 1 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;

         case sfDervx:
         if( _yacas_valid )
         {
            res = dervx( argv[ 0 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;

         case sfInteg:
         if( _yacas_valid )
         {
            res = integ( argv[ 0 ], argv[ 1 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;

         case sfIntvx:
         if( _yacas_valid )
         {
            res = intvx( argv[ 0 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;

         case sfIntegN:
         if( _yacas_valid )
         {
            res = integn( argv[ 0 ], argv[ 1 ], argv[ 2 ], argv[ 3 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;

         case sfSolve:
         if( _yacas_valid )
         {
            res = solve( argv[ 0 ], argv[ 1 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;

         case sfSolvx:
         if( _yacas_valid )
         {
            res = solvx( argv[ 0 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;

         case sfLimit:
         if( _yacas_valid )
         {
            res = limit( argv[ 0 ], argv[ 1 ], argv[ 2 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;

         case sfLimvx:
         if( _yacas_valid )
         {
            res = limvx( argv[ 0 ], argv[ 1 ] );
            if( res.isnull() )
            {
               _errcode = 11;
            }
         }
         else
         {
            _errcode = 12; // YACAS not initialized;
         }
         break;
         #endif // __YACAS__
         
         default:
         _errcode = 1;
         break;
      }
      if( !res.isnull() && res.valid() && argc > 0 )
      {
         push( res );
         _errcode = 0;
         lResult = true;
      }
      else
      {
         for( j = 0; j < argc; j++ )
         {
            push( argv[ j ] );
         }
         if( !_errcode && res.isnull() && argc > 0 )
         {
            _errcode = 5; // NULL Result
         }
         else if( !_errcode && !res.valid() && argc > 0 )
         {
            _errcode = 10; // NULL Result
         }
      }
   }
   delete [] argv;
   
   if( _errcode )
   {
      create_errmsg( func );
   }
   
   return lResult;
}
//------------------------------------------------------------------------------------------------
int atomstack::isfunc( const char* funcname, bool caseSens )
{
   AnsiString p;
   int index = -1;
   int j = 0;

   p = AnsiString( funcname );
   if( !caseSens )
   {
      p = p.UpperCase();
   }
   while( _funcs[ j ].func != sfUnknown && index == -1 )
   {
      if( ( strcmp( _funcs[ j ].name, p.c_str() ) == 0 && caseSens ) ||
          ( AnsiString( _funcs[ j ].name ).UpperCase() == p.c_str() )
        )
      {
         index = j;
      }
      j++;
   }
   return index;
}
//------------------------------------------------------------------------------------------------
int atomstack::isconst( const char* constname, bool caseSens )
{
   AnsiString p;
   int index = -1;
   int j = 0;

   if( _const )
   {
      p = AnsiString( constname );
      if( !caseSens )
      {
         p = p.UpperCase();
      }
      for( j = 0; j < _const->count() && index == -1; j++ )
      {
         if( ( strcmp( _const->getAtom( j ).name(), p.c_str() ) == 0 && caseSens ) ||
             ( AnsiString( _const->getAtom( j ).name() ).UpperCase() == p.c_str() )
           )
         {
            index = j;
         }
      }
   }
   return index;
}
//------------------------------------------------------------------------------------------------
bool atomstack::eval( char* funcname, bool caseSens )
{
   bool lResult = false;
   int index = isfunc( funcname, caseSens );
   
   if( index >= 0 )
   {
      lResult = eval( _funcs[ index ].func );
   }
   return lResult;
}
//------------------------------------------------------------------------------------------------
bool atomstack::evalcmd( char* cmdline, bool caseSens )
{
   TStringList* s;
   AnsiString p;
   int j;
   bool lResult;
   atom a;
   
   s = new TStringList();
   if( split2strings( cmdline, s, caseSens ) > 0 )
   {
      lResult = true;
      #ifdef __BORLANDC__
      for( j = 0; j < s->Count && lResult; j++ )
      {
         p = s->Strings[ j ];
      #else
      for( j = 0; j < s->Count() && lResult; j++ )
      {
         p = s->String( j );
      #endif
         //fprintf( __PGMathLog, "%02d. [%s]\n\r", j + 1, p.c_str() );
         if( isfunc( p.c_str(), caseSens ) >= 0 )
         {
            lResult = eval( p.c_str(), caseSens );
         }
         else
         {
            a = str2atom( p.c_str() );
            if( !a.isnull() )
            {
               push( a );
            }
            else
            {
               lResult = false;
            }
         }
      }
   }
   else
   {
      lResult = false;
   }
   delete s;
   
   return lResult;
}
//------------------------------------------------------------------------------------------------
#ifdef __YACAS__
//------------------------------------------------------------------------------------------------
//
//
//
//------------------------------------------------------------------------------------------------
atom atomstack::evalexp( atom e )
{
   atom x;
   char* buf;
   //FILE* f = fopen( ".\\yacaslib.log", "wt" );

   //fprintf( f, "e.type() == aExpression : %s\n", e.type() == aExpression ? "True" : "False" );
   if( e.type() == aExpression )
   {
      //fprintf( f, "e.value() = [%s]\n", e.value() ? (char*)( e.value() ) : "NULL" );
      if( e.value() )
      {
         AnsiString p;
         int len = strlen( (char*)( e.value() ) );
         buf = new char[ len + 1 ];
         strcpy( buf, (char*)( e.value() ) );
         //fprintf( f, "Before Yacas evaluate...\n" );
         _yacas->Evaluate( buf );
         if( !_yacas->IsError() )
         {
            //fprintf( f, "After Yacas evaluate...(%s)\n", _yacas->Result() );
            
            AnsiString s = AnsiString( _yacas->Result() );
            int pos = s.Pos( ";" );
            if( pos )
            {
               s = s.Delete( pos, 1 );
            }
            
            if( isrealstr( s.c_str() ) ||
                isintstr( s.c_str() ) ||
                iscomplexstr( s.c_str() ) ||
                isvectorstr( s.c_str() ) )
            {
               //printf( "N(%s)\n", s.c_str() );
               x = str2atom( s.c_str() );
            }
            else
            {
               p = AnsiString( "'" ) + s + AnsiString( "'" );
               x = atom( p.c_str() );
            }
            //fprintf( f, "Creating atom from s=(%s)...\n", s.c_str() );
            //fprintf( f, "Created atom x = [%s]\n", x.getstring() );
         }
         else
         {
            //fprintf( f, "Error: %s\n", _yacas->Error() );
            //sprintf( _errmsg, "%s", _yacas->Error() );
         }
         delete [] buf;
      }
   }
   //fprintf( f, "End Yacas\n" );
   //fclose( f );
   
   return x;
}
//------------------------------------------------------------------------------------------------
atom atomstack::evalnum( atom e )
{
   atom x;

   if( e.type() == aExpression )
   {
      if( e.value() )
      {
         AnsiString p = AnsiString( "N(" ) +
                        AnsiString( (char*)( e.value() ) ) +
                        AnsiString( ")" );

         _yacas->Evaluate( p.c_str() );
         if( !_yacas->IsError() )
         {
            AnsiString s = AnsiString( _yacas->Result() );
            int pos = s.Pos( ";" );
            if( pos )
            {
               s = s.Delete( pos, 1 );
            }

            if( isrealstr( s.c_str() ) ||
                isintstr( s.c_str() ) ||
                iscomplexstr( s.c_str() ) ||
                isvectorstr( s.c_str() ) )
            {
               x = str2atom( s.c_str() );
            }
            else
            {
               p = AnsiString( "'" ) + s + AnsiString( "'" );
               x = atom( p.c_str() );
            }
         }
         else
         {
         }
      }
   }
   return x;
}
//------------------------------------------------------------------------------------------------
atom atomstack::deriv( atom e, atom v )
{
   atom x;

   if( e.type() == aExpression && ( v.type() == aExpression || v.type() == aSymbol ) )
   {
      if( e.value() && v.value() )
      {
         char buf[ 1000 ];
         AnsiString p;
         sprintf( buf, "D(%s)%s", (char*)( v.value() ), (char*)( e.value() ) );

         _yacas->Evaluate( buf );
         if( !_yacas->IsError() )
         {
            AnsiString s = AnsiString( _yacas->Result() );
            int pos = s.Pos( ";" );
            if( pos )
            {
               s = s.Delete( pos, 1 );
            }

            p = AnsiString( "'" ) + s + AnsiString( "'" );
            x = atom( p.c_str() );
         }
         else
         {
         }
      }
   }
   return x;
}
//------------------------------------------------------------------------------------------------
atom atomstack::dervx( atom e )
{
   atom v = atom( "X" );
   atom x = deriv( e, v );
   
   return x;
}
//------------------------------------------------------------------------------------------------
atom atomstack::integ( atom e, atom v )
{
   atom x;

   if( e.type() == aExpression && ( v.type() == aExpression || v.type() == aSymbol ) )
   {
      if( e.value() && v.value() )
      {
         char buf[ 1000 ];
         AnsiString p;
         sprintf( buf, "Integrate(%s)%s", (char*)( v.value() ), (char*)( e.value() ) );

         _yacas->Evaluate( buf );
         if( !_yacas->IsError() )
         {
            AnsiString s = AnsiString( _yacas->Result() );
            int pos = s.Pos( ";" );
            if( pos )
            {
               s = s.Delete( pos, 1 );
            }

            p = AnsiString( "'" ) + s + AnsiString( "'" );
            x = atom( p.c_str() );
         }
         else
         {
         }
      }
   }
   return x;
}
//------------------------------------------------------------------------------------------------
atom atomstack::intvx( atom e )
{
   atom v = atom( "X" );
   atom x = integ( e, v );
   
   return x;
}
//------------------------------------------------------------------------------------------------
atom atomstack::integn( atom e, atom v, atom b1, atom b2 )
{
   atom x;

   if( e.type() == aExpression &&
      ( v.type() == aExpression || v.type() == aSymbol ) &&
      ( b1.type() == aInteger || b1.type() == aDouble ) &&
      ( b2.type() == aInteger || b2.type() == aDouble ) )
   {
      if( e.value() && v.value() )
      {
         char buf[ 1000 ];
         AnsiString p;
         sprintf( buf, "Integrate(%s,%s,%s)%s",
                       (char*)( v.value() ),
                       DoubleToStr( (double)( b1 ) ).c_str(),
                       DoubleToStr( (double)( b2 ) ).c_str(),
                       (char*)( e.value() ) );

         _yacas->Evaluate( buf );
         if( !_yacas->IsError() )
         {
            AnsiString s = AnsiString( _yacas->Result() );
            int pos = s.Pos( ";" );
            if( pos )
            {
               s = s.Delete( pos, 1 );
            }

            p = AnsiString( "'" ) + s + AnsiString( "'" );
            x = atom( p.c_str() );
         }
         else
         {
         }
      }
   }
   return x;
}
//------------------------------------------------------------------------------------------------
atom atomstack::solve( atom e, atom v )
{
   atom x;

   if( e.type() == aExpression && ( v.type() == aExpression || v.type() == aSymbol ) )
   {
      if( e.value() && v.value() )
      {
         char buf[ 1000 ];
         AnsiString p;
         sprintf( buf, "Solve(%s,%s)", (char*)( e.value() ), (char*)( v.value() ) );

         _yacas->Evaluate( buf );
         if( !_yacas->IsError() )
         {
            AnsiString s = AnsiString( _yacas->Result() );
            int pos = s.Pos( ";" );
            if( pos )
            {
               s = s.Delete( pos, 1 );
            }

            p = AnsiString( "'" ) + s + AnsiString( "'" );
            x = atom( p.c_str() );
         }
         else
         {
         }
      }
   }
   return x;
}
//------------------------------------------------------------------------------------------------
atom atomstack::solvx( atom e )
{
   atom v = atom( "X" );
   atom x = solve( e, v );

   return x;
}
//------------------------------------------------------------------------------------------------
atom atomstack::limit( atom e, atom lim, atom v )
{
   atom x;

   if( e.type() == aExpression &&
       ( lim.type() == aDouble || lim.type() == aInteger ) &&
       ( v.type() == aExpression || v.type() == aSymbol ) )
   {
      if( e.value() && v.value() )
      {
         char buf[ 1000 ];
         AnsiString p;
         sprintf( buf, "Limit(%s,%s)(%s)",
                  (char*)( v.value() ),
                  DoubleToStr( (double)( lim ) ).c_str(),
                  (char*)( e.value() ) );

         _yacas->Evaluate( buf );
         if( !_yacas->IsError() )
         {
            AnsiString s = AnsiString( _yacas->Result() );
            int pos = s.Pos( ";" );
            if( pos )
            {
               s = s.Delete( pos, 1 );
            }

            p = AnsiString( "'" ) + s + AnsiString( "'" );
            x = atom( p.c_str() );
         }
         else
         {
         }
      }
   }
   return x;
}
//------------------------------------------------------------------------------------------------
atom atomstack::limvx( atom e, atom lim )
{
   atom v = atom( "X" );
   atom x = limit( e, lim, v );

   return x;
}
//------------------------------------------------------------------------------------------------
#endif // __YACAS__
//------------------------------------------------------------------------------------------------
bool atomstack::save( FILE* f )
{
   bool res = false;
   
   if( f )
   {
      bool tmpres;
      int written = 0;
      int j, cnt;
      atom a;
      
      written += fwrite( "STACK", 1, 6, f );
      written += fwrite( &_count, 1, sizeof( int ), f );
      for( j = _count - 1, tmpres = true; j >= 0 && tmpres; j-- )
      {
         a = get( j );
         tmpres = a.save( f );
         if( tmpres )
         {
         }
         else
         {
         }
      }
      if( tmpres )
      {
         res = true;
      }
   }
   return res;
}
//------------------------------------------------------------------------------------------------
bool atomstack::load( FILE* f )
{
   bool res = false;
   
   if( f )
   {
      char buf[ 10 ];
      bool tmpres;
      int readed = 0;
      int r, j, cnt;
      atom a;
      
      readed += fread( buf, 1, 6, f );
      readed += fread( &cnt, 1, sizeof( int ), f );
      for( j = cnt - 1, tmpres = true; j >= 0 && tmpres; j-- )
      {
         tmpres = a.load( f ); // = get( j );
         if( tmpres )
         {
            push( a );
         }
         else
         {
         }
      }
      if( tmpres && strcmp( buf, "STACK" ) == 0 )
      {
         res = true;
      }
   }
   return res;
}
//------------------------------------------------------------------------------------------------

