//---------------------------------------------------------------------------
#include <math.h>

#include "avector.h"
#include "atomutil.h"

//---------------------------------------------------------------------------
CVector::CVector( void )
{
   FSize = 0;
   Values = NULL;
}
//---------------------------------------------------------------------------
CVector::CVector( double* values, int size )
{
   if( size > 0 )
   {
      FSize = size;
      Values = new CComplex[ FSize ];
      for( int j = 0; j < FSize; j++ )
      {
         if( values[ j ] )
         {
            Values[ j ] = CComplex( values[ j ], 0.0 );
         }
         else
         {
            Values[ j ] = CComplex( 0.0, 0.0 );
         }
      }
   }
   else
   {
      FSize = 0;
      Values = NULL;
   }
}
//---------------------------------------------------------------------------
CVector::CVector( CComplex* values, int size )
{
   if( size > 0 && values )
   {
      FSize = size;
      Values = new CComplex[ FSize ];
      for( int j = 0; j < FSize; j++ )
      {
         Values[ j ] = values[ j ];
      }
   }
   else
   {
      FSize = 0;
      Values = NULL;
   }
}
//---------------------------------------------------------------------------
CVector::CVector( const CVector& Var )
{
   if( Var.GetSize() )
   {
      FSize = Var.GetSize();
      Values = new CComplex[ FSize ];
      for( int j = 0; j < FSize; j++ )
      {
         Values[ j ] = Var.GetValue( j );
      }
   }
   else
   {
      FSize = 0;
      Values = NULL;
   }
}
//---------------------------------------------------------------------------
CVector::CVector( CVector* Var )
{
   if( Var )
   {
      if( Var->GetSize() )
      {
         FSize = Var->GetSize();
         Values = new CComplex[ FSize ];
         for( int j = 0; j < FSize; j++ )
         {
            Values[ j ] = Var->GetValue( j );
         }
      }
      else
      {
         FSize = 0;
         Values = NULL;
      }
   }
   else
   {
      FSize = 0;
      Values = NULL;
   }
}
//---------------------------------------------------------------------------
CVector::~CVector( )
{
   if( Values )
   {
      delete Values;
      FSize = 0;
      Values = NULL;
   }
}
//---------------------------------------------------------------------------
int CVector::GetSize( void ) const
{
   return FSize;
}
//---------------------------------------------------------------------------
int CVector::SetSize( int ASize )
{
   if( ASize > 0 )
   {
      if( Values )
      {
         delete Values;
      }
      FSize = ASize;
      Values = new CComplex[ FSize ];
      for( int j = 0; j < FSize; j++ )
      {
         Values[ j ] = CComplex( 0.0, 0.0 ); // long double( 0.0, 0.0 );
      }
   }
   else
   {
      if( Values )
      {
         delete Values;
      }
      Values = NULL;
      FSize = 0;
   }
   return FSize;
}
//---------------------------------------------------------------------------
CComplex CVector::GetValue( int Index ) const
{
   CComplex v;

   if( Index >= 0 && Index < FSize )
   {
      v = Values[ Index ];
   }
   else
   {
      v = CComplex( 0.0, 0.0 );
   }
   return v;
}
//---------------------------------------------------------------------------
double CVector::Length( void ) const
{
   CComplex v = CComplex( 0.0, 0.0 );

   if( FSize && Values )
   {
      for( int j = 0; j < FSize; j++ )
      {
         v += ( Values[ j ] * Values[ j ] );
      }
      v = sqrt( v );
   }
   return v.imag() ? 0.0 : v.real();
}
//---------------------------------------------------------------------------
AnsiString CVector::ToString( int prec ) const
{
   AnsiString p = "[";

   for( int j = 0; j < FSize; j++ )
   {
      if( prec < 0 )
      {
         p += ComplexToStr( Values[ j ] );
      }
      else
      {
         p += ComplexToStrF( Values[ j ], prec );
      }
      if( j < FSize - 1 )
      {
         p += AnsiString( " " );
      }
   }
   p += AnsiString( "]" );

   return p;
}
//---------------------------------------------------------------------------
int CVector::Save( FILE* f )
{
   int written = 0;
   if( f )
   {
      int j;
      written += fwrite( &FSize, 1, sizeof( int ), f );
      for( j = 0; j < FSize; j++ )
      {
         written += Values[ j ].Save( f );
      }
   }
   return written;
}
//---------------------------------------------------------------------------
int CVector::Load( FILE* f )
{
   int readed = 0;
   if( f )
   {
      CComplex c;
      int j, size, r;
      readed += fread( &size, 1, sizeof( int ), f );
      SetSize( size );
      for( j = 0; j < FSize; j++ )
      {
         r = c.Load( f );
         if( r == 2 * sizeof( double ) )
         {
            Values[ j ] = c;
            readed += r;
         }
      }
   }
   return readed;
}
//---------------------------------------------------------------------------
void CVector::SetValue( int Index, CComplex Value )
{
   if( Index >= 0 && Index < FSize )
   {
      Values[ Index ] = Value;
   }
}
//---------------------------------------------------------------------------
CComplex& CVector::operator[]( const int& index )
{
   if( index >= 0 && index < FSize )
   {
      //v = Values[ index ];
      //return Values[ index ];
      return *( Values + index );
   }
   else
   {
      AnsiString msg = "CVector: Index [" + AnsiString( index ) +
                       "] is out of range!\n(Vector Size=" + AnsiString( FSize ) +
                       ")";
      //throw Exception( msg );
   }
   return CCNull;
}
//---------------------------------------------------------------------------
const CVector& CVector::operator=( const CVector& val )
{
   if( Values )
   {
      delete Values;
   }

   FSize = val.GetSize();
   Values = new CComplex[ FSize ];
   for( int j = 0; j < val.GetSize(); j++ )
   {
      Values[ j ] = val.GetValue( j );
   }
   return *this;
}
//---------------------------------------------------------------------------
const CVector& CVector::operator-=( const CVector& val )
{
   if( FSize == val.GetSize() && Values )
   {
      for( int j = 0; j < FSize; j++ )
      {
         Values[ j ] -= val.GetValue( j );
      }
   }
   return *this;
}
//---------------------------------------------------------------------------
const CVector& CVector::operator+=( const CVector& val )
{
   if( FSize == val.GetSize() && Values )
   {
      for( int j = 0; j < FSize; j++ )
      {
         Values[ j ] += val.GetValue( j );
      }
   }
   return *this;
}
//---------------------------------------------------------------------------
/*
CComplex CVector::operator*=( const CVector& val )
{
   CComplex r = 0.0; //( 0.0, 0.0 );

   if( FSize == val.GetSize() && Values )
   {
      for( int j = 0; j < FSize; j++ )
      {
         r += ( Values[ j ] * val.GetValue( j ) );
      }
   }
   return r;
}
*/
//---------------------------------------------------------------------------
const CVector& CVector::operator*=( const CComplex& val )
{
   if( FSize && Values )
   {
      for( int j = 0; j < FSize; j++ )
      {
         Values[ j ] *= val;
      }
   }
   return *this;
}
//---------------------------------------------------------------------------
/*
const CVector& CVector::operator*=( const CComplex& val )
{
   if( FSize && Values )
   {
      for( int j = 0; j < FSize; j++ )
      {
         Values[ j ] *= val;
      }
   }
   return *this;
}
*/
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
CVector operator+( const CVector& a, const CVector& b )
{
   CVector v;

   if( a.GetSize() == b.GetSize() )
   {
      v.SetSize( a.GetSize() );
      for( int j = 0; j < v.GetSize(); j++ )
      {
         v.Values[ j ] = a.GetValue( j ) + b.GetValue( j );
         //v.Values[ j ] = a[ j ] + b[ j ];
      }
   }
   return v;
}
//---------------------------------------------------------------------------
CVector operator-( const CVector& a, const CVector& b )
{
   CVector v;

   if( a.GetSize() == b.GetSize() )
   {
      v.SetSize( a.GetSize() );
      for( int j = 0; j < v.GetSize(); j++ )
      {
         v.Values[ j ] = a.GetValue( j ) - b.GetValue( j );
         //v.Values[ j ] = a[ j ] - b[ j ];
      }
   }
   return v;
}
//---------------------------------------------------------------------------
//
// Multiplication vector operators
//
//---------------------------------------------------------------------------
CVector operator*( const CVector& a, const CComplex& b )
{
   CVector v;

   if( a.GetSize() )
   {
      v.SetSize( a.GetSize() );
      for( int j = 0; j < v.GetSize(); j++ )
      {
         v.Values[ j ] = a.GetValue( j ) * b;
         //v.Values[ j ] = a[ j ] * b;
      }
   }
   return v;
}
//---------------------------------------------------------------------------
CVector operator*( const CComplex& a, const CVector& b )
{
   CVector v;

   if( b.GetSize() )
   {
      v.SetSize( b.GetSize() );
      for( int j = 0; j < v.GetSize(); j++ )
      {
         v.Values[ j ] = a * b.GetValue( j );
         //v.Values[ j ] = a[ j ] * b;
      }
   }
   return v;
}
//---------------------------------------------------------------------------
//
// Division vector operators
//
//---------------------------------------------------------------------------
CVector operator/( const CVector& a, const CComplex& b )
{
   CVector v;

   if( a.GetSize() )
   {
      if( b != CComplex( 0.0, 0.0 ) )
      {
         v.SetSize( a.GetSize() );
         for( int j = 0; j < v.GetSize(); j++ )
         {
            v.Values[ j ] = a.GetValue( j ) / b;
            //v.Values[ j ] = a[ j ] * b;
         }
      }
      else
      {
         AnsiString msg = "Vector div by 0!";
         v.SetSize( 0 );
         //throw Exception( msg );
      }
   }
   else
   {
      v.SetSize( 0 );
   }
   return v;
}
//---------------------------------------------------------------------------
//
// Boolean vector operators;
//
//---------------------------------------------------------------------------
bool operator==( const CVector& a, const CVector& b )
{
   return a.Length() == b.Length();
}
//---------------------------------------------------------------------------
bool operator!=( const CVector& a, const CVector& b )
{
   return a.Length() != b.Length();
}
//---------------------------------------------------------------------------
bool operator>=( const CVector& a, const CVector& b )
{
   return a.Length() >= b.Length();
}
//---------------------------------------------------------------------------
bool operator<=( const CVector& a, const CVector& b )
{
   return a.Length() <= b.Length();
}
//---------------------------------------------------------------------------
bool operator>( const CVector& a, const CVector& b )
{
   return a.Length() > b.Length();
}
//---------------------------------------------------------------------------
bool operator<( const CVector& a, const CVector& b )
{
   return a.Length() < b.Length();
}
//---------------------------------------------------------------------------
//
// Vector manipulations;
//
//---------------------------------------------------------------------------
CVector cross( const CVector& a, const CVector& b )
{
   CVector r;
   int j;

   if( a.FSize == b.FSize && a.FSize >= 2 && a.FSize <= 3 )
   {
      r.SetSize( 3 );
      if( a.FSize == 3 )
      {
         r.Values[ 0 ] = a.Values[ 1 ] * b.Values[ 2 ] - a.Values[ 2 ] * b.Values[ 1 ];
         r.Values[ 1 ] = a.Values[ 2 ] * b.Values[ 0 ] - a.Values[ 0 ] * b.Values[ 2 ];
         r.Values[ 2 ] = a.Values[ 0 ] * b.Values[ 1 ] - a.Values[ 1 ] * b.Values[ 0 ];
      }
      else if( a.FSize == 2 )
      {
         r.Values[ 0 ] = CComplex( 0.0, 0.0 );
         r.Values[ 1 ] = CComplex( 0.0, 0.0 );
         r.Values[ 2 ] = a.Values[ 0 ] * b.Values[ 1 ] - a.Values[ 1 ] * b.Values[ 0 ];
      }
   }
   else
   {
      r.SetSize( 0 );
   }
   return r;
}
//---------------------------------------------------------------------------
CComplex dot( const CVector& a, const CVector& b )
{
   CComplex v;

   v = CComplex( 0.0, 0.0 );
   if( a.FSize == b.FSize && a.Values && b.Values )
   {
      for( int j = 0; j < a.FSize; j++ )
      {
         v += ( a.Values[ j ] * b.Values[ j ] );
      }
   }
   return v;
}
//---------------------------------------------------------------------------
CVector round( const CVector& a, int ARound )
{
   CVector r;
   int j;

   if( a.FSize > 0 )
   {
      r.SetSize( a.FSize );
      for( j = 0; j < a.FSize; j++ )
      {
         r.Values[ j ] = round( a.Values[ j ], ARound );
      }
   }
   return r;
}
//---------------------------------------------------------------------------
//
// String vector operations;
//
//---------------------------------------------------------------------------
AnsiString VectorToStr( const CVector& v )
{
   /*
   AnsiString p = "[";

   for( int j = 0; j < v.FSize; j++ )
   {
      //p += ComplexToStr( v.Values[ j ] );
      p += ComplexToStr( v.Values[ j ] );
      if( j < v.FSize - 1 )
      {
         p += " ";
      }
   }
   p += "]";
   */
   AnsiString p = v.ToString();
   return p;
}
//---------------------------------------------------------------------------
AnsiString VectorToStrF( const CVector& v, int prec )
{
   /*
   AnsiString p = "[";

   if( prec < 0 )
   {
      prec = 6;
   }
   for( int j = 0; j < v.FSize; j++ )
   {
      //p += ComplexToStr( v.Values[ j ] );
      p += ComplexToStrF( v.Values[ j ], prec );
      if( j < v.FSize - 1 )
      {
         p += " ";
      }
   }
   p += "]";
   */
   AnsiString p = v.ToString( prec );
   return p;
}
//---------------------------------------------------------------------------
CVector StrToVector( AnsiString t )
{
   CVector v;
   CComplex c;
   AnsiString s, p;
   int j, pos, len, count;
   TStringList* vals;

   p = t.Trim();
   /*
   j = -1;
   while( j )
   {
      j = p.Pos( " " );
      if( j )
      {
         p = p.Delete( j, 1 );
      }
   }
   */

   len = p.Length();
   if( p[ 1 ] == '[' && p[ len ] == ']' )
   {
      vals = new TStringList();

      p = p.SubString( 2, len - 2 ) + ",";
      j = -1;
      pos = 1;
      while( j )
      {
         #ifdef __BORLANDC__
         j = p.SubString( pos, p.Length() ).Pos( "," );
         #else
         j = p.Pos( " ", pos );
         if( !j )
         {
            j = p.Pos( ",", pos );
         }
         #endif
         if( j )
         {
            s = p.SubString( 1, j - 1 );
            if( isdoublestr( s.c_str() ) || iscomplexstr( s.c_str() ) )
            {
               vals->Add( p.SubString( 1, j - 1 ) );
               p = p.SubString( j + 1, p.Length() );
               pos = 1;
            }
            else
            {
               pos = j + 1;
            }
         }
      }

      #ifdef __BORLANDC__
      count = vals->Count;
      #else
      count = vals->Count();
      #endif
      if( count )
      {
         v.SetSize( count );
         for( j = 0; j < count; j++ )
         {
            //v.Values[ j ] = StrToComplex( vals->Strings[ j ] );
            #ifdef __BORLANDC__
            p = vals->Strings[ j ];
            #else
            p = vals->String( j );
            #endif
            v.Values[ j ] = StrToComplex( p.c_str() );
         }
      }

      delete vals;
   }
   return v;
}
//---------------------------------------------------------------------------
