#include <stdlib.h>
#include "qcursor.h"
#include "qcanvas.h"
#include "qpainter.h"
#include "qdir.h"
#include <kstandarddirs.h>
#include <kglobal.h>
#include <kiconloader.h>

#include "atomutil.h"
#include "pgcalc.h"
#include "uCalcMnu.h"
#include "uCalcWin.h"
#include "TInputBox.h"

#include "calctypes.h"

KApplication* App;
TCalc* Calc;
QString MainPath;
QString SkinPath;
QString SkinName;

const TCmdMenuItem CmdMenuItems[] = {
                              { baFactorial, "!" },

                              { baPercent, "%" },
                              { baPercentCH, "%CH" },
                              { baPercentT, "%T" },

                              { baHash, "#" },
                              { baAdd, "+" },
                              { baSub, "-" },
                              { baMul, "*" },
                              { baDiv, "/" },
                              { baYpowX, "^" },
                              { baNeg, "+/-" },

                              { baEq, "==" },
                              { baNEq, "<>" },
                              { baLess, "<" },
                              { baLessEq, "<=" },
                              { baGreat, ">" },
                              { baGreatEq, ">=" },

                              { ba1perX, "1/X" },

                              { baAbs, "ABS()" },
                              { baACos, "ACOS()" },
                              { baArgCH, "ACOSH()" },
                              { baACsc, "ACSC()" },
                              { baACscH, "ACSCH()" },
                              { baACtg, "ACTG()" },
                              { baArgCTH, "ACTGH()" },
                              { baALog, "ALOG()" },
                              { baAnd, "AND" },
                              { baArg, "ARG()" },
                              { baASec, "ASEC()", },
                              { baASecH, "ASECH()" },
                              { baASin, "ASIN()" },
                              { baArgSH, "ASINH()" },
                              { baATan, "ATAN()" },
                              { baArgTH, "ATANH()" },
                              { baAvg, "AVG()" },

                              { ba2Bin, "BIN()" },

                              { baCos, "COS()" },
                              { baCosH, "COSH()" },
                              { baCrossVector, "CROSS()" },
                              { baCsc, "CSC()" },
                              { baCscH, "CSCH()" },
                              { baCtg, "CTG()" },
                              { baCtgH, "CTGH()" },

                              { baDeg2Rad, "D->R" },
                              { ba2Dec, "DEC()" },
                              //{ baSetDeg, "DEG" },
                              //{ baDeriv, "DERIV" },
                              //{ baDervx, "DERVX" },
                              { baIntDiv, "DIV" },
                              { baDotVector, "DOT()" },

                              { baExp, "EXP()" },
                              { baExpm, "EXPM()" },

                              { baFactorial, "FACT()" },
                              { baFp, "FP()" },

                              { baGamma, "GAMMA()" },

                              { ba2Hex, "HEX()" },

                              { baIm, "IM()" },
                              //{ baInteg, "INTEG" },
                              //{ baIntegN, "INTEGN" },
                              //{ baIntvx, "INTVX" },
                              { baIp, "IP()" },
                              { baIsPrime, "ISPRIME()" },

                              { baLength, "LENGTH()" },
                              { baLn, "LN()" },
                              { baLnp1, "LNP1()" },
                              { baLog, "LOG()" },

                              { baMax, "MAX()" },
                              { baMin, "MIN()" },
                              { baIntMod, "MOD" },

                              { baNot, "NOT" },

                              { ba2Oct, "OCT()" },
                              { baOr, "OR" },

                              { baPsi, "PSI()" },

                              { baReal2Complex, "R2C()" },
                              { baRad2Deg, "R2D()" },
                              { baReal2Vector2, "R2V2()" },
                              { baReal2Vector3, "R2V3()" },
                              //{ baSetRad, "RAD" },
                              { baRe, "RE()" },
                              { baRlb, "RLB()" },
                              { baRrb, "RRB()" },

                              { baSec, "SEC()" },
                              { baSecH, "SECH()" },
                              { baSign, "SIGN()" },
                              { baSin, "SIN()" },
                              { baSinH, "SINH()" },
                              { baSlb, "SLB()" },
                              //{ baSolve, "SOLVE" },
                              //{ baSolvx, "SOLVX" },
                              { baSqrt, "SQRT()" },
                              { baSrb, "SRB()" },
                              { baSum, "SUM()" },

                              { baTan, "TAN()" },
                              { baTanH, "TANH()" },

                              { baXor, "XOR" },
                              { baXRoot, "XROOT()" },

                              { baUnknown, "?" }
                              };


TCalc::TCalc(QWidget *parent, const char *name) : QWidget(parent, name)
{  /*
   WFlags f = getWFlags();
   f += Qt::WRepaintNoErase;
   setWFlags( f );
   */
   LCD = NULL;
   Buttons = new TCalcButtonList();
   ButtonMode = cbmNone;
   CurrentButton = NULL;
   ActiveObject = NULL;
   FTerminated = false;
   FMenuAllowed = true;
   FLastCommand = -1;
	FPrec = 5;

   LoadDefaults();

   SysTray = new TPGSystemTray( this, "TPGSystemTray::SysTray" );
   
	QString icopath;
	KIconLoader *loader = KGlobal::iconLoader();
   QPixmap ico = loader->loadIcon("pgcalc2", KIcon::Desktop, 24, KIcon::DefaultState, &icopath ); //QPixmap( MainPath + QString( "/bin/pgcalc2-24.xpm" ) );
   setIcon( ico );
   
   #ifdef __DEBUG__
   fprintf( stderr, "Icon path: [%s]\n", (const char*)( icopath ) );
   #endif
    
   mpressed = false;
   mx = 0;
   my = 0;
   mbutton = NoButton;
      
   setCaption( tr( "PG Calculator " VERSION ) );
   LoadSkin( SkinName ); //"default" );

	History = new TStringList();
	
	CreateConstants();
	CreateVariables();
	
   Stack = new atomstack( 256 );
	Stack->set_precision( FPrec );
	Stack->assignConstants( Constants );
	Stack->assignVariables( Variables );
	
	Inter = new atominter();
	Inter->assignStack( Stack );
	Inter->assignConstants( Constants );
	Inter->assignVariables( Variables );
	
   RPN = new TCalcRPN( this );
	Alg = new TCalcAlg( this );
	
   ConstLib = new TCalcConstLib( this );
   ConstLib->assignConstants( Constants );

	VarList = new TCalcVarList( this );
	VarList->assignVariables( Variables );
	VarList->assignConstants( Constants );
	
       TVM = new TCalcTVM( this );
       TVM->assignStack( Stack );
       TVM->assignInter( Inter );
           
       UnitConv = new TCalcUnitConv( this );
       UnitConv->assignStack( Stack );
       UnitConv->assignInter( Inter );
       
   ActiveObject = RPN;
   ActiveObject->Activate();
}

TCalc::~TCalc()
{
	delete RPN;
	delete Alg;
	delete ConstLib;
	
   delete SysTray;
	delete Inter;
   delete Stack;
	delete History;
	delete Variables;
	delete Constants;
   delete Buttons;
}

atomlist* TCalc::CreateConstants( void )
{
   Constants = new atomlist();
   Constants->addatom( new atom( M_PI ), "Pi", "Value of PI" );
   Constants->addatom( new atom( 2.71828182845904524 ), "e", "Natural log base" );
   Constants->addatom( new atom( CComplex( 0.0, 1.0 ) ), "i", "Complex (0,1i)" );
   Constants->addatom( new atom( 0.577215664901532 ), "Eu", "Euler's constant" );
   Constants->addatom( new atom( CMeasure( 9.80665, "kg*m/s^2" ) ), "g", "Accel of gravity" );
   Constants->addatom( new atom( CMeasure( 6.67259E-11, "m^3/kg*s^2" ), aifExp ), "G", "Gravitational const." );
   Constants->addatom( new atom( CMeasure( 6.6260755E-34, "J*s" ), aifExp ), "h", "Planck's constant" );
   Constants->addatom( new atom( CMeasure( 6.0221367E23, "1/mol" ), aifExp ), "NA", "Avogadro's Number" );
   Constants->addatom( new atom( CMeasure( 1.380658E-23, "J/K" ), aifExp ), "k", "Boltzmann" );
   Constants->addatom( new atom( CMeasure( 22.4141, "m^3/mol" ) ), "Vm", "Molar Volume" );
   Constants->addatom( new atom( CMeasure( 8.31451, "J/mol*K" ) ), "R", "Universal gas" );
   Constants->addatom( new atom( CMeasure( 273.15, "K" ) ), "StdT", "Std. temperature" );
   Constants->addatom( new atom( CMeasure( 101.325, "kPa" ) ), "StdP", "Std. pressure" );
   Constants->addatom( new atom( CMeasure( 299792458.0, "m/s" ) ), "c", "Speed of light" );
   Constants->addatom( new atom( CMeasure( 96485.309, "C/mol" ) ), "F", "Faraday constant" );  
   Constants->addatom( new atom( CMeasure( 0.0529177249, "nm" ) ), "a0", "Bohr Radius" );
   Constants->addatom( new atom( CMeasure( 1.60217733E-19, "C" ), aifExp ), "q", "Elementary charge" );
   Constants->addatom( new atom( CMeasure( 9.1093897E-31, "kg" ), aifExp ), "me", "Electron rest mass" );
   Constants->addatom( new atom( CMeasure( 175881962000.0, "C/kg" ) ), "qme", "q/me ratio" );
   Constants->addatom( new atom( CMeasure( 1.6726231E-27, "kg" ), aifExp ), "mp", "Proton rest mass" );
   Constants->addatom( new atom( 1836.152701 ), "mpme", "mp/me ration" );
   Constants->addatom( new atom( CMeasure( 1.602176E-19, "J" ), aifExp ), "eV", "Electron Volt" );
   Constants->addatom( new atom( CMeasure( 8.85418781761E-12, "F/m" ), aifExp ), "e0", "Permittivity of vacuum" );
   Constants->addatom( new atom( CMeasure( 1.25663706144E-6, "H/m" ), aifExp ), "m0", "Permeability of vacuum" );
   
   return Constants;
}

atomlist* TCalc::CreateVariables( void )
{
   Variables = new atomlist();
   Variables->addatom( new atom( 0.0 ), "X" );
   
   return Variables;
}

void TCalc::setActiveObject( TCalcObject* obj )
{
   ActiveObject = obj;
   if( ActiveObject )
   {
      ActiveObject->Activate();
      ActiveObject->Refresh();
   }
}

QRgb TCalc::str2rgb( const char* str )
{
   int r, g, b;
   
   sscanf( str, "%d,%d,%d", &r, &g, &b );
   return qRgb( r, g, b );
}

void TCalc::LoadSkin( QString ASkinName )
{
   int j, x, y, w, h, l, t;
   const char* btn = "Button ";
   QRgb transpColor;
   QString s;
   QString bkgrbmp;
   QString fontbmp, smfontbmp;
   QString path = SkinPath + ASkinName + "/";
   QString file = path + "skin.csk";
   QString Chr;
   QRgb fColor, bColor;
   TButtonAction act, m1a, m2a, ala;
   int key;
	int kind;
   TCalcButton* b;
	
	#ifdef __DEBUG__
   char strbuf[ 200 ];
   fprintf( stderr, "Start loading skin from [%s]...\n", (const char*)( file ) );
   //sprintf( strbuf, "Start loading skin from [%s]...\n", (const char*)( file ) );
   //QMessageBox::information( this, strbuf, file );
	#endif
	
   KConfig* ini = new KConfig( file );
   //ini->setGroup( "Main" );
   //QMessageBox::information( this, "Author", ini->readEntry( "Author", "***" ) );
   if( ini->hasGroup( "Main" ) )
   {
      ini->setGroup( "Main" );
      bkgrbmp = ini->readEntry( "Background", "?" );
      //QMessageBox::information( this, "Loading...", bkgrbmp );
      if( bkgrbmp != "?" )
      {
         img = QImage( path + bkgrbmp );
         if( !img.isNull() )
         {
            w = img.width();
            h = img.height();
            bkgr = QPixmap( img );
            setFixedSize( w, h );
            setBackgroundPixmap( bkgr );
            /*
            lbl = new QLabel( this, "Label" );
            lbl->setGeometry( 0, 0, w, h );
            lbl->setPixmap( bkgr );
            */
            QRegion mask( 0, 0, w, h );
            transpColor = img.pixel( 0, 0 );

            for( y = 0; y < h; y++ )
            {
               for( x = 0; x < w; x++ )
               {
                  if( img.pixel( x, y ) == transpColor )
                  {
                     mask -= QRegion( x, y, 1, 1 );
                  }
               }
            }
            //mask -= QRegion( 0, 0, 5, 1 );
            //mask -= QRegion( 0, 1, 3, 1 );
            //mask -= QRegion( 0, 2, 2, 1 );
            //mask -= QRegion( 0, 3, 1, 2 );

            setMask( mask );

            ini->setGroup( "Display" );
            fontbmp = path + ini->readEntry( "NormalFont", "?" );
            smfontbmp = path + ini->readEntry( "SmallFont", "?" );

            fColor = str2rgb( ini->readEntry( "FontColor" ) );
            bColor = str2rgb( ini->readEntry( "BackColor" ) );
            l = ini->readNumEntry( "Left", 0 );
            t = ini->readNumEntry( "Top", 0 );
            w = ini->readNumEntry( "Width", 100 );
            h = ini->readNumEntry( "Height", 50 );
            
            LCD = new TLCDDisplay( this );
            LCD->setBounds( l, t, w, h );
            LCD->setTRHeight( ini->readNumEntry( "UpRowSize", 10 ) );
            LCD->setBRHeight( ini->readNumEntry( "DownRowSize", 10 ) );
            
            if( LCD->LoadFonts( fontbmp, smfontbmp ) )
            {
               LCD->setFontColor( fColor );
               LCD->setBackColor( bColor );

               j = 0;
               while( ini->hasGroup( btn + QString::number( j + 1 ) ) )
               {
                  ini->setGroup( btn + QString::number( j + 1 ) );
                  s = ini->readEntry( "Name", "?" );
                  l = ini->readNumEntry( "Left", 0 );
                  t = ini->readNumEntry( "Top", 0 );
                  w = ini->readNumEntry( "Width", 100 );
                  h = ini->readNumEntry( "Height", 50 );
						kind = ini->readNumEntry( "Kind", 0 );
                  Chr = ini->readEntry( "Char", "\0" );
                  
                  key = TranslateKey( ini->readEntry( "Key", "000" ) );
                  act = TranslateAction( ini->readEntry( "Action1", "?" ) );
                  m1a = TranslateAction( ini->readEntry( "Action2", "?" ) );
                  m2a = TranslateAction( ini->readEntry( "Action3", "?" ) );
                  ala = baUnknown;

                  b = new TCalcButton( this );
                  b->setName( s );
                  b->setBounds( l, t, w, h );
                  b->setKey( key );
						b->setKind( kind );
                  b->setChr( Chr.length() > 0 ? ((const char*)( Chr ))[ 0 ] : (char)( '\0' ) );
                  b->setActions( act, m1a, m2a, ala );
                  
                  Buttons->Add( b );

                  j++;
               }
               LCD->CreateHeaderSections();
               LCD->setHeaderText( 1, "RPN" );
               LCD->setHeaderText( 2, "Rad" );
               LCD->setFuncKey( 5, "CANCL" );
               LCD->setFuncKey( 6, "OK" );
               //QMessageBox::information( this, "Info", "Skin has " +
               //                          QString::number( Buttons->count() ) +
               //                          " button(s)!" );
            }
            else
            {
               QMessageBox::critical( this, "Error", "Error loading fonts!" );
					exit( 2 );
            }
            LCD->paint();
         }
         else
         {
            QMessageBox::critical( this, "Error!", "Error loading background bitmap:\nFileName = [" + bkgrbmp + "]!" );
				exit( 1 );
         }
      }
      else
      {
         QMessageBox::critical( this, "Error!", "No background bitmap!" );
			exit( 3 );
      }
   }
   else
   {
      QMessageBox::critical( this, "Error in INI!", "Error in file [" + file + "]\nor file doesn't exist!" );
		exit( 4 );
   }
   delete ini;
}

TCalcButton* TCalc::GetButtonByAction( TButtonAction Action )
{
	TCalcButton* b = NULL;
	if( Buttons )
	{
		b = Buttons->GetButtonByAction( Action );
	}
	return b;
}

TButtonAction TCalc::getButtonAction( TCalcButton* Button )
{
   TButtonAction a;

   if( Button )
   {
      if( ButtonMode == cbmNone )
      {
         a = Button->action();
         switch( a )
         {
            case baMode1:
            ButtonMode = cbmMode1;
            a = baUnknown;
            break;

            case baMode2:
            ButtonMode = cbmMode2;
            a = baUnknown;
            break;

            case baAlpha:
            ButtonMode = cbmAlpha;
            a = baUnknown;
            break;

            default:
            break;
         }            
      }
      else if( ButtonMode == cbmMode1 )
      {
         a = Button->mode1action();
         ButtonMode = cbmNone;
      }
      else if( ButtonMode == cbmMode2 )
      {
         a = Button->mode2action();
         ButtonMode = cbmNone;
      }
      else if( ButtonMode == cbmAlpha )
      {
         unsigned char chr = Button->chr();
         if( chr >= '0' && chr <= '9' )
         {
            a = (TButtonAction)( chr - 48 + (int)ba0 ); // Button->AlphaAction;
         }
         else if( chr >= 'A' && chr <= 'Z' )
         {
            a = (TButtonAction)( chr - 65 + (int)baA ); // Button->AlphaAction;
         }
         //a = Button->alphaaction();
         ButtonMode = cbmNone;
      }
   }
   return a;
}

void TCalc::showButtonModes( void )
{
   LCD->setHeaderText( 3,
                       ( ButtonMode & cbmMode2 ?
                       AnsiString( (char)( 12 ) ) + AnsiString( (char)( 13 ) ) :
                       AnsiString( "  " ) ).c_str() );
   LCD->setHeaderText( 4,
                       ( ButtonMode & cbmMode1 ?
                       AnsiString( (char)( 14 ) ) + AnsiString( (char)( 15 ) ) :
                       AnsiString( "  " ) ).c_str() );
   LCD->setHeaderText( 5,
                       ( ButtonMode & cbmAlpha ?
                       AnsiString( "ALPHA" ) :
                       AnsiString( "     " ) ).c_str() );
   LCD->begin();
   LCD->drawHeaders();
   LCD->end();
}

void TCalc::hideToSysTray( void )
{
   hide();
   SysTray->show();
}

void TCalc::ExecAction( TButtonAction& Action )
{
	bool lProcessed = true;
	
   showButtonModes();

   if( Action == baClose )
   {
      FTerminated = true;
      SaveState();
      close();
		exit(0);
   }
   else if( Action == baMinimize )
   {
      //showMinimized();
      hideToSysTray();
   }
   
	if( ActiveObject )
   {
      ActiveObject->ExecAction( Action );
      if( Action != baUnknown )
      {
         lProcessed = false;
      }
   }   
  
   if( ( ActiveObject ? !ActiveObject->IsModalEx() : true ) && !lProcessed )
   {
      lProcessed = true;
	   
		if( Action == baAlgebraMode )
   	{
	      SetMode( baAlgebraMode );
   	}
   	else if( Action == baRPNMode )
   	{
      	SetMode( baRPNMode );
   	}
   	else if( Action == baConstants )
   	{
      	ConstLibDialog(); 
	   }
   	else if( Action == baVar )
   	{
      	VarListDialog(); 
   	}
   	else if( Action == baFinance )
   	{
	      TVMDialog();
   	}
   	else if( Action == baUnitConverter )
   	{
      	UnitConverter();
	   }
   	else if( Action == baSetPrecision )
   	{
      	SetPrecision();
   	}

      else if( Action == baCat && FMenuAllowed )
      {
         CmdCatalog();
      }
      else if( Action == baApps && FMenuAllowed )
      {
         AppsMenu();
      }
      else if( ( Action == baMode || Action == baConfigMenu ) && FMenuAllowed )
      {
         ConfigMenu();
      }
      else if( Action == baModeMenu && FMenuAllowed )
      {
         ModeMenu();
      }
      else if( Action == baAngleMeasure && FMenuAllowed )
      {
         AngleMeasureMenu();
      }
      else if( Action == baMathMenu && FMenuAllowed )
      {
         MathMenu();
      }
      else if( Action == baComplexMenu && FMenuAllowed )
      {
         ComplexFuncMenu();
      }
      else if( Action == baUnitsMenu && FMenuAllowed )
      {
         UOMCategory();
      }
      else if( Action == baTrigMenu && FMenuAllowed )
      {
         TrigFuncMenu();
      }
      else if( Action == baHyperMenu && FMenuAllowed )
      {
         HyperFuncMenu();
      }
      else if( Action == baRealMenu && FMenuAllowed )
      {
         RealFuncMenu();
      }
      else if( Action == baBaseMenu && FMenuAllowed )
      {
         BaseFuncMenu();
      }
      else if( Action == baBitsMenu && FMenuAllowed )
      {
         BitsFuncMenu();
      }
      else if( Action == baLogicMenu && FMenuAllowed )
      {
         LogicFuncMenu();
      }
      else if( Action == baVectorMenu && FMenuAllowed )
      {
         VectorFuncMenu();
      }
      else if( Action == baCalcMenu && FMenuAllowed )
      {
         CalcFuncMenu();
      }
      else if( Action == baDerivIntegMenu && FMenuAllowed )
      {
         DerivIntegFuncMenu();
      }
      else if( Action == baAlgMenu && FMenuAllowed )
      {
         AlgFuncMenu();
      }
      else if( Action == baSolveMenu && FMenuAllowed )
      {
         SolveFuncMenu();
      }
	   else if( Action == baSetRad )
   	{
	      LCD->setHeaderText( 2, "Rad" );
   	   Stack->set_anglemeasure( samRadian );
      	showButtonModes();
	   }
   	else if( Action == baSetDeg )
   	{
      	LCD->setHeaderText( 2, "Deg" );
      	Stack->set_anglemeasure( samDegree );
      	showButtonModes();
   	}
      else
      {
         lProcessed = false;
      }   
	}
	/*
   else
   {
      if( ActiveObject )
      {
         //fprintf( stderr, "ActiveObject(%s)->ExecAction(%d)\n", ActiveObject->GetName().c_str(), (int)( Action ) );
         ActiveObject->ExecAction( Action );
      }
   }
	*/
   //update( LCD->left(), LCD->top(), LCD->width(), LCD->height() );
}

void TCalc::SetMode( TButtonAction Action )
{  
   if( Action == baAlgebraMode && Alg )
   {
      ActiveObject = Alg;
   }
   else if( Action == baRPNMode && RPN )
   {
      ActiveObject = RPN;
   }   
   ActiveObject->Activate();
}

void TCalc::CmdCatalog( void )
{
   AnsiString p;
   TCalcMenu* mnu;
   TCalcObject* oldObj;
   int result;
   int j = 0;

   if( !ActiveObject )
   {
      return;
   }

   oldObj = ActiveObject;
   mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "---" );
   while( CmdMenuItems[ j ].command != baUnknown )
   {
      mnu->AddItem( CmdMenuItems[ j ].caption, CmdMenuItems[ j ].command );
      j++;
   }
   p = AnsiString( "CATALOG (" ) + IntToStr( j ) + AnsiString( " COMMANDS)" );
   mnu->SetCaption( p );

   if( FLastCommand >= 0 && FLastCommand < j )
   {
      mnu->SetDefault( FLastCommand );
   }
   result = mnu->ShowModal();
   FLastCommand = mnu->GetPosition();
   delete mnu;
   
   //fprintf( stderr, "After CmdCatalog ShowModal: Result=%d\n", result );

   setActiveObject( oldObj ); // = oldObj;
   if( ActiveObject )
   {
      //ActiveObject->Refresh();
      if( result != baUnknown && result ) //&& ActiveObject ) //&& !FTerminate )
      {
         TButtonAction Action = (TButtonAction)( result );
         ExecAction( Action );
         //fprintf( stderr, "After CmdCatalog execution: Action=%d...\n", (int)( Action ) );
      }
   }
}

void TCalc::ConstLibDialog( void )
{
   int res;
   TCalcObject* oldObj = ActiveObject;
      
   ActiveObject = ConstLib;
   ConstLib->setParentName( oldObj ? oldObj->GetName() : "?" );
   ConstLib->Activate();
   res = ConstLib->ShowModal();

   setActiveObject( oldObj );
   
   if( Constants )
   {
      if( res >= 0 && res < Constants->count() )
      {
         if( ActiveObject == RPN )
         {
            atom a = Constants->getAtom( res );
            Stack->push( a );
         }
         else if( ActiveObject == Alg )
         {        
            LCD->InpLine->InsertText( Constants->getAtom( res ).name() );
         }
      }
   }
   ActiveObject->Activate();
}

void TCalc::VarListDialog( void )
{
   int res;
   TCalcObject* oldObj = ActiveObject;
      
   ActiveObject = VarList;
   VarList->setParentName( oldObj ? oldObj->GetName() : "?" );
   VarList->Activate();
   res = VarList->ShowModal();

   setActiveObject( oldObj );
   //ActiveObject->Activate();
   
   if( Variables )
   {
      if( res >= 0 && res < Variables->count() )
      {
         if( ActiveObject == RPN )
         {
            atom a = Variables->getAtom( res );
            Stack->push( a );
         }
         else if( ActiveObject == Alg )
         {        
            LCD->InpLine->InsertText( Variables->getAtom( res ).name() );
         }
      }
   }
	//ActiveObject->Activate();
}

void TCalc::TVMDialog( void )
{
   int res;
   TCalcObject* oldObj = ActiveObject;

   ActiveObject = TVM;
   TVM->setParentName( oldObj ? oldObj->GetName() : "?" );
   TVM->Activate();
   res = TVM->ShowModal();

   setActiveObject( oldObj );
}

void TCalc::UnitConverter( void )
{
   int res;
   TCalcObject* oldObj = ActiveObject;

   ActiveObject = UnitConv;
   UnitConv->setParentName( oldObj ? oldObj->GetName() : "?" );
   UnitConv->Activate();
   res = UnitConv->ShowModal();

   setActiveObject( oldObj );
}

void TCalc::ConfigMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** CONFIGURATION ***" );
   mnu->AddItem( "1.WORKING MODE", baModeMenu );
   mnu->AddItem( "2.DECIMAL PRECISION", baSetPrecision );
   mnu->AddItem( "3.ANGLE MEASURE", baAngleMeasure );
   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj );
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::ModeMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** CONFIGURATION ***" );
   mnu->AddItem( "ALGEBRAIC", baAlgebraMode );
   mnu->AddItem( "RPN", baRPNMode );
	if( oldObj == RPN )
	{
		mnu->SetDefault( 1 );
	}
   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::SetPrecision( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;
   AnsiString strPrec;
   
   TInputBox* ib = new TInputBox( this, ActiveObject );
   ib->SetPrompt( "Decimal Precision" );
   ib->SetStyle( ilsAllowNumber );
   ib->SetText( IntToStr( FPrec ) );
   ib->SetButtons( btOk | btCancel );
   result = ib->ShowModal();
   strPrec = ib->GetText();
   delete ib;
   
   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result == mrOk )
   {
      FPrec = strPrec.ToInt();
      Stack->set_precision( FPrec );
   }
	ActiveObject->Activate();
}

void TCalc::AngleMeasureMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** ANGLE MEASURE ***" );
   mnu->AddItem( "RADIAN", baSetRad );
   mnu->AddItem( "DEGREE", baSetDeg );
   if( Stack->angleMeasure() == samDegree )
   {
      mnu->SetDefault( 1 );
   }
   else
   {
   }
   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::AppsMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** APPLICATIONS ***" );
   mnu->AddItem( "1.CONSTANTS LIBRARY", baConstants );
   mnu->AddItem( "2.VARIABLES LIST", baVar );
   mnu->AddItem( "3.TIME VALUE OF MONEY", baFinance );
   mnu->AddItem( "4.UNIT CONVERTER", baUnitConverter );
   mnu->AddItem( "  MATHEMATICS...", baMathMenu );
   
   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::MathMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** MATH MENU ***" );
   mnu->AddItem( "1.REAL...", baRealMenu );
   mnu->AddItem( "2.COMPLEX...", baComplexMenu );
   mnu->AddItem( "3.UNITS...", baUnitsMenu );
   mnu->AddItem( "4.BASE...", baBaseMenu );
   mnu->AddItem( "5.BITS...", baBitsMenu );
   mnu->AddItem( "6.LOGICAL...", baLogicMenu );
   mnu->AddItem( "7.VECTOR...", baVectorMenu );
   mnu->AddItem( "8.TRIGONOMETRIC...", baTrigMenu );
   mnu->AddItem( "9.HYPERBOLIC...", baHyperMenu );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::RealFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** REAL MENU ***" );
   mnu->AddItem( "0.MATH...", baMathMenu );
   mnu->AddItem( "1.%", baPercent );
   mnu->AddItem( "2.%CH", baPercentCH );
   mnu->AddItem( "3.%T", baPercentT );
   mnu->AddItem( "4.MIN", baMin );
   mnu->AddItem( "5.MAX", baMax );
   mnu->AddItem( "6.ABS", baAbs );
   mnu->AddItem( "7.SIGN", baSign );
   mnu->AddItem( "8.IP", baIp );
   mnu->AddItem( "9.FP", baFp );
   mnu->AddItem( "RND", baUnknown );
   mnu->AddItem( "D->R", baDeg2Rad );
   mnu->AddItem( "R->D", baRad2Deg );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::BaseFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** BASE MENU ***" );
   mnu->AddItem( "MATH...", baMathMenu );
   mnu->AddItem( "1.#", baHash );
   mnu->AddItem( "2.BIN()", ba2Bin );
   mnu->AddItem( "3.DEC()", ba2Dec );
   mnu->AddItem( "4.HEX()", ba2Hex );
   mnu->AddItem( "5.OCT()", ba2Oct );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::BitsFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** BITS MENU ***" );
   mnu->AddItem( "MATH...", baMathMenu );
   mnu->AddItem( "1.AND", baAnd );
   mnu->AddItem( "2.OR", baOr );
   mnu->AddItem( "3.XOR", baXor );
   mnu->AddItem( "4.NOT", baNot );
   mnu->AddItem( "5.RLB", baRlb );
   mnu->AddItem( "6.RRB", baRrb );
   mnu->AddItem( "7.SLB", baSlb );
   mnu->AddItem( "8.SRB", baSrb );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::LogicFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** LOGIC MENU ***" );
   mnu->AddItem( "MATH...", baMathMenu );
   mnu->AddItem( "1. ==", baEq );
   mnu->AddItem( "2. <>", baNEq );
   mnu->AddItem( "3. >", baGreat );
   mnu->AddItem( "4. >=", baGreatEq );
   mnu->AddItem( "5. <", baLess );
   mnu->AddItem( "6. <=", baLessEq );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::ComplexFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** COMPLEX MENU ***" );
   mnu->AddItem( "MATH...", baMathMenu );
   mnu->AddItem( "1.i", baComplexI );
   mnu->AddItem( "2.ARG()", baArg );
   mnu->AddItem( "3.CONJ()", baConj );
   mnu->AddItem( "4.IM()", baIm );
   mnu->AddItem( "5.NORM()", baNorm );
   mnu->AddItem( "6.POLAR()", baPolar );
   mnu->AddItem( "7.RE()", baRe );
   mnu->AddItem( "8.R->C", baReal2Complex );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::VectorFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** VECTOR MENU ***" );
   mnu->AddItem( "0.MATH...", baMathMenu );
   mnu->AddItem( "1.ABS", baAbs );
   mnu->AddItem( "2.DOT", baDotVector );
   mnu->AddItem( "3.CROSS", baCrossVector );
   mnu->AddItem( "4.R->V2", baReal2Vector2 );
   mnu->AddItem( "5.R->V3", baReal2Vector3 );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::CalcFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "CALCULUS MENU" );
   mnu->AddItem( "MATH...", baMathMenu );
   mnu->AddItem( "DERIV & INTEG ...", baDerivIntegMenu );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::DerivIntegFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "DERIV & INTEG MENU" );
   mnu->AddItem( "CALCULUS...", baCalcMenu );
   mnu->AddItem( "DERIV", baDeriv );
   mnu->AddItem( "DERVX", baDervx );
   mnu->AddItem( "INTEG", baInteg );
   mnu->AddItem( "INTVX", baIntvx );
   mnu->AddItem( "INTEGN", baIntegN );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::AlgFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "ALG MENU" );
   mnu->AddItem( "MATH...", baMathMenu );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::SolveFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "SOLVE MENU" );
   mnu->AddItem( "SOLVE", baSolve );
   mnu->AddItem( "SOLVX", baSolvx );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::TrigFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** TRIGONOMETRIC ***" );
   mnu->AddItem( "MATH...", baMathMenu );
   mnu->AddItem( "HYPERBOLIC...", baHyperMenu );
   mnu->AddItem( "ACOS()", baACos );
   mnu->AddItem( "ACSC()", baACsc );
   mnu->AddItem( "ACTG()", baACtg );
   mnu->AddItem( "ASEC()", baASec );
   mnu->AddItem( "ASIN()", baASin );
   mnu->AddItem( "ATAN()", baATan );

   mnu->AddItem( "COS()", baCos );
   mnu->AddItem( "CSC()", baCsc );
   mnu->AddItem( "CTG()", baCtg );
   mnu->AddItem( "SEC()", baSec );
   mnu->AddItem( "SIN()", baSin );
   mnu->AddItem( "TAN()", baTan );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

void TCalc::HyperFuncMenu( void )
{
   TCalcObject* oldObj = ActiveObject;
   int result;

   TCalcMenu* mnu = new TCalcMenu( this, ActiveObject );
   mnu->SetCaption( "*** HYPERBOLIC ***" );
   mnu->AddItem( "MATH...", baMathMenu );
   mnu->AddItem( "TRIGONOMETRIC...", baTrigMenu );
   mnu->AddItem( "ACOSH()", baArgCH );
   mnu->AddItem( "ACSCH()", baACscH );
   mnu->AddItem( "ACTGH()", baArgCTH );
   mnu->AddItem( "ASECH()", baASecH );
   mnu->AddItem( "ASINH()", baArgSH );
   mnu->AddItem( "ATANH()", baArgTH );

   mnu->AddItem( "COSH()", baCosH );
   mnu->AddItem( "CSCH()", baCscH );
   mnu->AddItem( "CTGH()", baCtgH );
   mnu->AddItem( "SECH()", baSecH );
   mnu->AddItem( "SINH()", baSinH );
   mnu->AddItem( "TANH()", baTanH );

   result = mnu->ShowModal();
   delete mnu;

   setActiveObject( oldObj ); //ActiveObject = oldObj;
   if( result && result != baUnknown )
   {
      TButtonAction Action = (TButtonAction)( result );
      ExecAction( Action );
   }
}

int TCalc::UOMCategory( void )
{
   TCalcObject* oldObj;
   int j, res, cat;
   TStrings* catStr = new TStrings();
   TStrings* catDsc = new TStrings();
   TCalcMenu* mnu;
   bool lExit = false;

   while( !lExit )
   {
      catStr->Clear();
      catDsc->Clear();
      GetMeasureCategories( catStr, catDsc );
      oldObj = ActiveObject;
      mnu = new TCalcMenu( this, ActiveObject );
      mnu->SetCaption( "*** MEASURE CATEGORIES ***" );
      for( j = 0; j < catStr->Count(); j++ )
      {
         mnu->AddItem( IntToStr( j + 1 ) + "." + catDsc->String( j ).UpperCase(),
                       (TButtonAction)( j + 1 ) );
      }
      res = mnu->ShowModal();
      delete mnu;
      setActiveObject( oldObj ); //ActiveObject = oldObj;
      //ActiveObject->closeChild();

      if( res == baUnknown || FTerminated )
      {
         lExit = true;
      }
      else
      {
         if( res > 0 )
         {
            cat = catStr->String( res - 1 ).ToInt(); //(int)( res );
            res = UOMUnits( cat, catDsc->String( res - 1 ).UpperCase() );
            if( res > 1 )
            {
               lExit = true;
            }
            else if( res == baUnknown )
            {
               lExit = true;
            }
         }
      }
   }
   delete catStr;
   delete catDsc;

   return 0;
}

int TCalc::UOMUnits( int Category, AnsiString Description )
{
   int j, res, cat;
   TStrings* uomStr = new TStrings();
   TStrings* uomDsc = new TStrings();
   TCalcMenu* mnu;
   AnsiString p;
   TCalcObject* oldObj;

   res = 0;
   if( GetMeasuresInCategory( Category, uomStr, uomDsc ) )
   {
      oldObj = ActiveObject;
      mnu = new TCalcMenu( this, ActiveObject );
      mnu->SetCaption( AnsiString( "*** " ) + Description + " ***" );
      mnu->AddItem( "BACK TO CATEGORIES...", 1 );
      for( j = 0; j < uomStr->Count(); j++ )
      {
         mnu->AddItem( uomDsc->String( j ).UpperCase() +
                       AnsiString( " [" ) + uomStr->String( j ) + AnsiString( "]" ),
                       (TButtonAction)( j + 2 ) );
      }
      res = mnu->ShowModal();
      delete mnu;
      setActiveObject( oldObj ); //ActiveObject = oldObj;
      //ActiveObject->closeChild();

      if( res != baUnknown && res > 1 )
      {
         p = uomStr->String( res - 2 );
         //fprintf( stderr, "Selected unit [%s]...\n", p.c_str() );

         if( LCD->InpLine->Buffer.Pos( "_" ) )
         {
         }
         else
         {
            p = "_" + p;
         }
         LCD->InpLine->InsertText( p );
         ActiveObject->Refresh();
      }
   }
   delete uomStr;
   delete uomDsc;

   return res;
}

bool TCalc::LoadDefaults( void )
{
   bool lResult = true;
	
	MainPath = locate( "data", "pgcalc2/" );
	SkinPath = locate( "data", "pgcalc2/skins/" ); //pgcalc2ui.rc" );
	
	#ifdef __DEBUG__
	fprintf( stderr, "MainPath=[%s], SkinPath=[%s]\n", (const char*)( MainPath ), (const char*)( SkinPath ) );
	#endif
	   
	QDir qdir = QDir( SkinPath );
	if( SkinPath.length() == 0 || !qdir.exists() )
	{
      QMessageBox::critical( this, "Skin Error!", "Directory \"pgcalc2/skins/\" cannot be located. Directory [" + SkinPath + "] doesn't exist!" );
		exit( 5 );
	}
	
   return lResult;
}

bool TCalc::LoadState( void )
{
   bool lResult = true;
   QDir qdir;
   QString file = qdir.homeDirPath() + QString( "/.pgcalcrc" );
   QString filexml = qdir.homeDirPath() + QString( "/.pgcalcxmlrc" );
   FILE* f;

      xmlnode* xmain;
      xmlnode* xvar;
      xmlnode* xpos;
      xmlnode* xrpn;
      xmlnode* xalg;
		xmlnode* xtvm;
		xmlnode* xconv;
      
      xmltree* xtree = new xmltree();
      
      xtree->loadFromFile( AnsiString( (const char*)( filexml ) ) );
      if( !xtree->lastError() && xtree->getRootName() == "PGCalc" )
      {
         AnsiString workMode;
         
         xmain = xtree->getRoot();
         xpos = xmain->getNodeByName( "Position" );
         
         if( xpos )
         {
            int x, y;     
            
            x = xpos->readInteger( "Left" );
            y = xpos->readInteger( "Top" );
            FStayOnTop = xpos->readBoolean( "StayOnTop" );
            /*
            GetWindowRect( GetDesktopWindow(), &r );
            if( x < 0 || x > r.right )
            {
               x = 0;
            }
            if( y < 0 || y > r.bottom )
            {
               y = 0;
            }
				*/
				move( x, y );
         }
         
         workMode = xmain->readString( "Mode" ).UpperCase();
         if( workMode == "ALG" )
         {
            ActiveObject = Alg;
         }
         else
         {
            ActiveObject = RPN;
         }
         ActiveObject->Activate();
         
         FPrec = xmain->readInteger( "Precision" );
         Stack->set_precision( FPrec );

         xvar = xmain->getNodeByName( "Variables" );
         if( xvar && Variables )
         {
            atom a;
            int type, format;
            xmlnode* n;
            AnsiString v, name;           
            
            n = xvar->getFirstCh();
            while( n )
            {
               v = n->getValue();
               name = n->getAttrValue( "name" );
               type = n->getAttrValue( "type" ).ToInt();
               format = n->getAttrValue( "format" ).ToInt();
               if( name.UpperCase() == "X" )
               {
                  int index = Variables->findAtom( "X" );
                  if( index >= 0 && index < Variables->count() )
                  {
                     Variables->replaceatom( index, new atom( StrToDouble( v ) ), "X" );
                  }
               }
               else
               {
                  switch( type )
                  {
                     case aInteger:
                     switch( format )
                     {
                        case aifBin:
                        a = atom( BinStrToInt( v ), aifBin );
                        break;
                        
                        case aifOct:
                        a = atom( OctStrToInt( v ), aifOct );
                        break;
                        
                        case aifHex:
                        a = atom( HexStrToInt( v ), aifHex );
                        //MessageBox( Handle, a.getstring(), "HEX", MB_OK );
                        break;
                        
                        default:
                        a = atom( v.ToInt(), format );
                        break;
                     }
                     break;
               
                     case aDouble:
                     switch( format )
                     {
                        case aifExp:
                        a = atom( StrEToDouble( v ), aifExp );
                        break;
                        
                        default:
                        a = atom( StrToDouble( v ), aifDec );
                        break;
                     }
                     break;
               
                     case aComplex:
                     a = atom( StrToComplex( v ), aifDec );
                     break;
               
                     case aMeasure:
                     switch( format )
                     {
                        case aifExp:
                        a = atom( StrToMeasure( v ), aifExp );
                        break;
                        
                        default:
                        a = atom( StrToMeasure( v ), aifDec );
                        break;
                     }
                     break;
                
                     default:
                     a = str2atom( v.c_str() );
                     break;
                  }
                  //a.setname( name.c_str() );
                  Variables->addatom( new atom( a ), name.c_str() );
               }
               
               n = n->getNext();
            }
         }
                  
         xrpn = xmain->getNodeByName( "RPN" );
         if( xrpn && RPN )
         {
            RPN->LoadState( xrpn );
         }
         
         xalg = xmain->getNodeByName( "Algebra" );
         if( xalg && Alg )
         {
            Alg->LoadState( xalg );
         }
			
			xtvm = xmain->getNodeByName( "TVM" );
			if( xtvm && TVM )
			{
				TVM->LoadState( xtvm );
			}
			
			xconv = xmain->getNodeByName( "UnitConverter" );
			if( xconv && UnitConv )
			{
				UnitConv->LoadState( xconv );
			}
      }

   return lResult;
}

bool TCalc::SaveState( void )
{
   bool lResult = true;
   int res;
   //AnsiString path = GetStatePath() + "\\PG Calculator";
   QDir qdir;
   QString path = qdir.homeDirPath();
   QString file = qdir.homeDirPath() + QString( "/.pgcalcrc" ); //path + "\\pgcalc.dat";
   QString filexml = qdir.homeDirPath() + QString( "/.pgcalcxmlrc" ); //path + "\\pgcalc.dat";
   FILE* f;

   //fprintf( stderr, "Writing PG Calculator state to %s...\n", (const char*)( file ) );
   if( lResult )
   {
         RECT r;
         int j, x, y;
         char state;
         int written = 0;
         char descr[] = "PG";
         xmlnode* xmain;
         xmlnode* xpos;
         xmlnode* xrpn;    
         xmlnode* xalg;
         xmlnode* xvar;
			xmlnode* xtvm;
			xmlnode* xconv;
         xmlnode* n;
         AnsiString workMode;

         x = QWidget::x(); //r.left;
         y = QWidget::y(); //r.top;
         state = FStayOnTop ? char(1) : char(0);

         xmltree* xtree = new xmltree( "PGCalc" );
         xmain = xtree->getRoot();
         xpos = xtree->addNode( "Position", "", xmain );
         xtree->addNode( "Left", IntToStr( x ), xpos );
         xtree->addNode( "Top", IntToStr( y ), xpos );
         xtree->addNode( "StayOnTop", AnsiString( FStayOnTop ? "True" : "False" ), xpos );
         if( ActiveObject == Alg )
         {
            workMode = "ALG";
         }
         else if( ActiveObject == RPN )
         {
            workMode = "RPN";
         }
         else
         {
            workMode = "";
         }
         xtree->addNode( "Mode", workMode, xmain );
         xtree->addNode( "Precision", IntToStr( FPrec ), xmain );

         xvar = xtree->addNode( "Variables", "", xmain );
         if( Variables )
         {
            for( j = 0; j < Variables->count(); j++ )
            {
               n = xtree->addNode( "Variable", AnsiString( Variables->getAtom( j ).getstring( 12 ) ), xvar );
               n->setAttrValue( "name", AnsiString( Variables->getAtom( j ).name() ) );
               n->setAttrValue( "type", IntToStr( Variables->getAtom( j ).type() ) );
               n->setAttrValue( "format", IntToStr( Variables->getAtom( j ).format() ) );
            }
         }
         
         xrpn = xtree->addNode( "RPN", "", xmain );
         xalg = xtree->addNode( "Algebra", "", xmain );
         xtvm = xtree->addNode( "TVM", "", xmain );
			xconv = xtree->addNode( "UnitConverter", "", xmain );
			
         if( RPN )
         {
            RPN->SaveState( xrpn );
         }
         if( Alg )
         {
            Alg->SaveState( xalg );
         }
         if( TVM )
			{
				TVM->SaveState( xtvm );
			}
			if( UnitConv )
			{
				UnitConv->SaveState( xconv );
			}
			
			//InfoMsg( "Write state OK!" );
         xtree->saveToFile( ( const char* )( filexml ) );
         delete xtree;
   }
   return lResult;
}

void TCalc::paintEvent( QPaintEvent* e )
{
   if( LCD )
   {
      LCD->clear();
      if( ActiveObject )
      {
         //QMessageBox::information( this, "Active", "Refreshing ActiveObject...\n" );
         //fprintf( stderr, "Refreshing ActiveObject...\n" );
         ActiveObject->Refresh();
      }
      LCD->paint();
   }
}

void TCalc::mousePressEvent( QMouseEvent* e )
{
   TCalcButton* b;
   
   b = Buttons->itemAt( e->x(), e->y() );
   if( b )
   {
      CurrentButton = b;
      CurrentButton->drawPressed();
      mpressed = false;
   }
   else
   {
      CurrentButton = NULL;
      if( e->button() == LeftButton )
      {
         mbutton = LeftButton;
         mpressed = true;
         QPoint p = mapToGlobal( e->pos() );
         mx = p.x();
         my = p.y();
         ml = x();
         mt = y();
      }
   }
}

void TCalc::mouseReleaseEvent( QMouseEvent* e )
{
   TButtonAction Action;
   TCalcButton* b;
   
   if( CurrentButton )
   {
 		b = Buttons->itemAt( e->x(), e->y() );
		if( b == CurrentButton )
		{
	      mpressed = false;
   	   Action = getButtonAction( CurrentButton ); //->action();
      	CurrentButton->drawNormal();
      	CurrentButton = NULL;
      	ExecAction( Action );
		}
   }
   else if( mpressed )
   {
      mpressed = false;
      mbutton = NoButton;
      mx = 0;
      my = 0;
   }
}

void TCalc::mouseMoveEvent( QMouseEvent* e )
{
   TCalcButton* b;
   
	if( mpressed && !CurrentButton ) //&& e->button() == mbutton )
   {
   	QPoint p = mapToGlobal( e->pos() );
   	int dx = p.x() - mx;
   	int dy = p.y() - my;
      ml += dx;
      mt += dy;
      move( ml, mt );
      mx = p.x();
      my = p.y();
   }
	else
	{
		if( CurrentButton )
		{
   		b = Buttons->itemAt( e->x(), e->y() );
			if( !b || ( b != CurrentButton ) )
			{
				CurrentButton->drawNormal();
			}
			else if( b == CurrentButton )
			{
				CurrentButton->drawPressed();
			}
		}
	}	
}

void TCalc::keyPressEvent( QKeyEvent* e )
{
   TCalcButton* b;
   int code = e->key();

   if( code == 46 )
   {
      code = 4103;
   }
   else if( code == 4101 )
   {
      code = 4100;
   }
   
   b = Buttons->getByKeyCode( code );
   if( b )
   {
      CurrentButton = b;
      CurrentButton->drawPressed();
   }
   else
   {
      //QMessageBox::information( this, "Key code", "Key code=[" + QString::number( e->key() ) + "]" );
   }
}

void TCalc::keyReleaseEvent( QKeyEvent* e )
{
   TButtonAction Action;
   
   if( CurrentButton )
   {
      Action = getButtonAction( CurrentButton );//->action();
      CurrentButton->drawNormal();
      CurrentButton = NULL;
      //fprintf( stderr, "Key release event (KeyCode=%d)\n", e->key() );
      ExecAction( Action );
   }
}

void TCalc::showEvent( QShowEvent* e )
{
   SysTray->hide();
}

