//---------------------------------------------------------------------------
#ifdef __WIN32__
#include <dir.h>
#include <windows.h>
#endif
#include <astring.h>

#include "ptree.h"
//---------------------------------------------------------------------------
//
//
//
//---------------------------------------------------------------------------
pnode::pnode( const char* atext )
{
   level = -1;
   tag = 0;
   next = NULL;
   prev = NULL;
   nodes = NULL;
   parent = NULL;
   owner = NULL;
   if( atext )
   {
      int len = strlen( atext );
      text = new char[ len + 1 ];
      memcpy( text, atext, len );
      text[ len ] = '\0';
   }
   else
   {
      text = NULL;
   }
}
//---------------------------------------------------------------------------
pnode::~pnode()
{
   if( text )
   {
      delete [] text;
   }
}
//---------------------------------------------------------------------------
void pnode::settag( int atag )
{
   tag = atag;
}
//---------------------------------------------------------------------------
int pnode::gettag( void )
{
   return tag;
}
//---------------------------------------------------------------------------
void pnode::setText( const char* atext )
{
   if( text )
   {
      delete [] text;
   }
   int len = strlen( atext );
   text = new char[ len + 1 ];
   memcpy( text, atext, len );
   text[ len ] = '\0';
}
//---------------------------------------------------------------------------
char* pnode::getText( void ) const
{
   return text;
}
//---------------------------------------------------------------------------
void pnode::setdata( int asize, void* adata )
{
   data.size = asize;
   data.ptr = adata;
}
//---------------------------------------------------------------------------
void* pnode::getdata( void )
{
   return data.ptr;
}
//---------------------------------------------------------------------------
int pnode::getdatasize( void )
{
   return data.size;
}
//---------------------------------------------------------------------------
void pnode::cleardata( void )
{
   data.size = 0;
   data.ptr = NULL;
}
//---------------------------------------------------------------------------
void pnode::clear( void )
{
   pnode* p = nodes;
   pnode* n;

   while( p )
   {
      n = p->next;
      p->clear();
      p->cleardata();
      delete p;
      p = n;
   }
   nodes = NULL;
}
//---------------------------------------------------------------------------
pnode* pnode::getparent( void )
{
   return parent;
}
//---------------------------------------------------------------------------
ptree* pnode::getowner( void )
{
   return owner;
}
//---------------------------------------------------------------------------
void pnode::setlevel( int alevel )
{
   level = alevel;
   pnode* n = getfirstch();
   while( n )
   {
      n->setlevel( alevel + 1 );
      n = n->getnext();
   }
}
//---------------------------------------------------------------------------
void pnode::addchild( pnode* anode )
{
   if( !anode )
   {
      return;
   }

   if( !nodes )
   {
      nodes = anode;
      anode->prev = NULL;
      anode->next = NULL;
   }
   else
   {
      pnode* p = nodes;
      while( p->next )
      {
         p = p->next;
      }
      anode->prev = p;
      anode->next = NULL;
      p->next = anode;
   }
   anode->setlevel( level + 1 ); 
   anode->owner = owner;
   anode->parent = this;
}
//---------------------------------------------------------------------------
void pnode::delchild( pnode* anode )
{
   if( anode )
   {
      if( anode->parent == this )
      {
         pnode* next = anode->next;
         pnode* prev = anode->prev;

         if( next )
         {
            next->prev = prev;
         }
         if( prev )
         {
            prev->next = next;
         }
         if( nodes == anode )
         {
            nodes = next;
         }
         delete anode;
      }
   }
}
//---------------------------------------------------------------------------
pnode* pnode::getfirst( void )
{
   pnode* p = this;
   while( p->prev )
   {
      p = p->prev;
   }
   return p;
}
//---------------------------------------------------------------------------
pnode* pnode::getlast( void )
{
   pnode* p = this;
   while( p->next )
   {
      p = p->next;
   }
   return p;
}
//---------------------------------------------------------------------------
pnode* pnode::getnext( void )
{
   return next;
}
//---------------------------------------------------------------------------
pnode* pnode::getprev( void )
{
   return prev;
}
//---------------------------------------------------------------------------
pnode* pnode::getfirstch( void )
{
   return nodes;
}
//---------------------------------------------------------------------------
pnode* pnode::getlastch( void )
{
   pnode* p = nodes;
   if( p )
   {
      while( p->next )
      {
         p = p->next;
      }
   }
   return p;
}
//---------------------------------------------------------------------------
pnode* pnode::getnextch( pnode* node )
{
   pnode* p = NULL;
   if( node->parent == this )
   {
      p = node->next;
   }
   return p;
}
//---------------------------------------------------------------------------
pnode* pnode::getprevch( pnode* node )
{
   pnode* p = NULL;
   if( node->parent == this )
   {
      p = node->prev;
   }
   return p;
}
//---------------------------------------------------------------------------
void pnode::moveto( pnode* node )
{
   if( node )
   {
      pnode* anext = next;
      pnode* aprev = prev;

      if( anext )
      {
         anext->prev = aprev;
      }
      if( aprev )
      {
         aprev->next = anext;
      }
      if( parent )
      {
         if( parent->nodes == this )
         {
            parent->nodes = anext;
         }
      }
      else
      {
         if( owner->nodes == this )
         {
            owner->nodes = anext;
         }
      }
      node->addchild( this );
   }
}
//---------------------------------------------------------------------------
void pnode::printf( int alevel )
{
   pnode* p;

   ::printf( "%*s[%s];Level=%d\n", alevel * 3 + 1, " ", text, level );
   p = getfirstch();
   while( p )
   {
      p->printf( alevel + 1 );
      p = p->next;
   }
}
//---------------------------------------------------------------------------
bool pnode::haschildren( void )
{
   return !( nodes == NULL );
}
//---------------------------------------------------------------------------
int pnode::count( void )
{
   pnode* p = nodes;
   int i = 0;

   while( p )
   {
      i++;
      p = p->next;
   }
   return i;
}
//---------------------------------------------------------------------------
int pnode::countall( void )
{
   pnode* p = nodes;
   int i = 0;

   while( p )
   {
      i++;
      i += p->countall();
      p = p->next;
   }
   return i;
}
//---------------------------------------------------------------------------
int pnode::savetofile( FILE* f )
{
   int written = 0;
   if( !f )
   {
   }
   else
   {
      pnode* n;
      int chcount = count();
      int textsize = strlen( text );
      int datasize = data.size;
      int nodesize = 6 * sizeof( int ) + textsize + datasize;

      written += fwrite( &nodesize, sizeof( int ), 1, f );
      written += fwrite( &tag, sizeof( int ), 1, f );
      written += fwrite( &level, sizeof( int ), 1, f );
      written += fwrite( &chcount, sizeof( int ), 1, f );
      written += fwrite( &textsize, sizeof( int ), 1, f );
      written += fwrite( &datasize, sizeof( int ), 1, f );

      written += fwrite( text, 1, strlen( text ), f );
      if( data.size && data.ptr )
      {
         written += fwrite( data.ptr, 1, data.size, f );
      }
      n = nodes;
      while( n )
      {
         written += n->savetofile( f );
         n = n->getnext();
      }
   }
   return written;
}
//---------------------------------------------------------------------------
int pnode::loadfromfile( FILE* f )
{
   int readed = 0;

   if( !f )
   {
   }
   else
   {
      int nodesize, textsize, datasize, chcount;
      //void* databuf;
      pnode* n;

      readed = fread( &nodesize, 1, sizeof( int ), f );
      if( feof( f ) )
      {
         //return 0;
      }
      else if( readed == sizeof( int ) )//&& !feof( f ) )
      {
         readed += fread( &tag, 1, sizeof( int ), f );
         readed += fread( &level, 1, sizeof( int ), f );
         readed += fread( &chcount, 1, sizeof( int ), f );
         readed += fread( &textsize, 1, sizeof( int ), f );
         readed += fread( &datasize, 1, sizeof( int ), f );
         if( readed == 6 * sizeof( int ) && !feof( f ) )
         {
            if( textsize )
            {
               text = new char[ textsize + 1 ];
               readed += fread( text, 1, textsize, f );
               text[ textsize ] = '\0';
            }
            if( datasize )
            {
               data.size = datasize;
               data.ptr = ( void* )( new char[ datasize ] );
               readed += fread( data.ptr, 1, datasize, f );
            }
            while( chcount )
            {
               n = new pnode();
               readed += n->loadfromfile( f );
               addchild( n );
               chcount--;
            }
         }
         else
         {
            ::printf( "Error reading node (Readed %d byte(s), Error 2 )\n", readed );
            return -2;
         }
      }
      else
      {
         ::printf( "Error reading node (Readed %d byte(s), Error 1)\n", readed );
         return -1;
      }
   }
   return readed;
}
//---------------------------------------------------------------------------
//
//
//
//---------------------------------------------------------------------------
ptree::ptree( void )
{
   nodes = NULL;
   root = NULL;
}
//---------------------------------------------------------------------------
ptree::~ptree( void )
{
   clear();
}
//---------------------------------------------------------------------------
void ptree::clear( void )
{
   pnode* p = nodes;
   pnode* n;

   while( p )
   {
      n = p->next;
      p->clear();
      delete p;
      p = n;
   }
   nodes = NULL;
}
//---------------------------------------------------------------------------
pnode* ptree::getfirst( void )
{
   return nodes;
}
//---------------------------------------------------------------------------
pnode* ptree::getlast( void )
{
   pnode* p;

   if( !nodes )
   {
      return NULL;
   }
   else
   {
      p = nodes;
      while( p->next )
      {
         p = p->next;
      }
   }
   return p;
}
//---------------------------------------------------------------------------
void ptree::addnode( pnode* anode, pnode* parent )
{
   if( !anode )
   {
      return;
   }

   if( !parent )
   {
      if( !nodes )
      {
         nodes = anode;
         anode->prev = NULL;
         anode->next = NULL;
         anode->parent = NULL;
         anode->level = 0;
      }
      else
      {
         pnode* p = getlast();
         p->next = anode;
         anode->prev = p;
         anode->next = NULL;
         anode->level = 0;
      }
      anode->owner = this;
   }
   else
   {
      if( parent->owner == this )
      {
         parent->addchild( anode );
      }
   }
}
//---------------------------------------------------------------------------
void ptree::addnode( char* atext, pnode* parent )
{
   pnode* n;
   if( !parent )
   {
      n = new pnode( atext );
      addnode( n, NULL );
   }
   else
   {
      if( parent->owner == this )
      {
         addnode( new pnode( atext ), parent );
      }
   }
}
//---------------------------------------------------------------------------
void ptree::insert_node_after( pnode* anode, pnode* node_above )
{
   if( anode && node_above )
   {
      pnode* node_below;

      node_below = node_above->next;

      anode->prev = node_above;
      anode->next = node_below;
      anode->parent = node_above->parent;
      anode->owner = this;

      node_above->next = anode;
      node_below->prev = anode;
   }
}
//---------------------------------------------------------------------------
void ptree::insert_node_before( pnode* anode, pnode* node_below )
{
   if( anode && node_below )
   {
      pnode* node_above;

      node_above = node_below->prev;

      anode->prev = node_above;
      anode->next = node_below;
      anode->parent = node_below->parent;
      anode->owner = this;

      node_below->prev = anode;
      if( node_above )
      {
         node_above->next = anode;
      }
      else
      {
         if( node_below->parent )
         {
            node_below->parent->nodes = anode;
         }
         else
         {
            nodes = anode;
         }
      }
   }
}
//---------------------------------------------------------------------------
void ptree::delnode( pnode* anode )
{
   if( anode )
   {
      if( anode->owner == this )
      {
         pnode* prev = anode->prev;
         pnode* next = anode->next;
         pnode* parent = anode->parent;

         if( prev )
         {
            prev->next = next;
         }
         if( next )
         {
            next->prev = prev;
         }

         if( parent )
         {
            if( parent->nodes == anode )
            {
               parent->nodes = next;
            }
         }
         else
         {
            if( nodes == anode )
            {
               nodes = next;
            }
         }
         anode->clear();
         delete anode;
      }
   }
}
//---------------------------------------------------------------------------
pnode* ptree::getnext( pnode* node )
{
   pnode* n = NULL;
   if( node )
   {
      n = n->next;
   }
   return n;
}
//---------------------------------------------------------------------------
pnode* ptree::getprev( pnode* node )
{
   pnode* n = NULL;
   if( node )
   {
      n = node->prev;
   }
   return n;
}
//---------------------------------------------------------------------------
pnode* ptree::getfirstch( pnode* node )
{
   pnode* p = NULL;

   if( node )
   {
      if( node->owner == this )
      {
         p = node->nodes;
      }
   }
   else
   {
      if( nodes )
      {
         p = nodes;
      }
   }
   return p;
}
//---------------------------------------------------------------------------
pnode* ptree::getlastch( pnode* node )
{
   pnode* p = NULL;

   if( node )
   {
      if( node->owner == this )
      {
         p = node->nodes;
         if( p )
         {
            while( p->next )
            {
               p = p->next;
            }
         }
      }
   }
   else
   {
      if( nodes )
      {
         p = nodes;
         while( p->next )
         {
            p = p->next;
         }
      }
   }
   return p;
}
//---------------------------------------------------------------------------
pnode* ptree::getnextch( pnode* parent, pnode* node )
{
   pnode* p = NULL;

   if( node )
   {
      if( parent == node->parent )
      {
         p = node->getnext();
      }
   }
   return p;
}
//---------------------------------------------------------------------------
pnode* ptree::getprevch( pnode* parent, pnode* node )
{
   pnode* p = NULL;

   if( node )
   {
      if( parent == node->parent )
      {
         p = node->getprev();
      }
   }
   return p;
}
//---------------------------------------------------------------------------
void ptree::moveto( pnode* src, pnode* dst )
{
   if( src && dst )
   {
      //printf( "Moving node [%s] under parent [%s]\n", src->text, dst->text );
      src->moveto( dst );
   }
}
//---------------------------------------------------------------------------
int ptree::count( void )
{
   pnode* p = nodes;
   int i = 0;

   while( p )
   {
      i++;
      p = p->next;
   }
   return i;
}
//---------------------------------------------------------------------------
int ptree::countall( void )
{
   pnode* p = nodes;
   int i = 0;

   while( p )
   {
      i++;
      i += p->countall();
      p = p->next;
   }
   return i;
}
//---------------------------------------------------------------------------
void ptree::printf( void )
{
   if( nodes )
   {
      pnode* p = nodes;
      ::printf( "Tree contents %d (All=%d) elements...\n", count(), countall() );
      while( p )
      {
         p->printf( 0 );
         p = p->next;
      }
   }
   else
   {
      ::printf( "There are no nodes in this tree...\n" );
   }
}
//---------------------------------------------------------------------------
int ptree::savetofile( FILE* f )
{
   int written = 0;
   if( !f )
   {
   }
   else
   {
      pnode* n = nodes;

      while( n )
      {
         written += n->savetofile( f );
         n = n->getnext();
      }
   }
   return written;
}
//---------------------------------------------------------------------------
int ptree::savetofile( const char* fname )
{
   int written = 0;

   if( fname )
   {
      FILE* f = fopen( fname, "wb" );
      if( f )
      {
         written = savetofile( f );
         fclose( f );
      }
   }
   return written;
}
//---------------------------------------------------------------------------
int ptree::loadfromfile( FILE* f )
{
   int i, readed = 0;
   pnode* n;

   while( !feof( f ) )
   {
      n = new pnode();
      i = n->loadfromfile( f );
      if( i > 0 )
      {
         readed += i;
         addnode( n );
      }
      else
      {
         if( i == 0 )
         {
            delete n;
         }
         else
         {
            ::printf( "Error reading tree\n" );
            return -3;
         }
      }
   }
   return readed;
}
//---------------------------------------------------------------------------
int ptree::loadfromfile( const char* fname )
{
   int readed = 0;

   if( fname )
   {
      #ifdef __WIN32__
		//struct ffblk ff;
      WIN32_FIND_DATA fd;
      HANDLE fh;
      
      fh = FindFirstFile( fname, &fd );
      if( fh != INVALID_HANDLE_VALUE )
      {
         FindClose( fh );
         
         FILE* f = fopen( fname, "rb" );
         if( f )
         {
            readed = loadfromfile( f );
            fclose( f );
         }
         else
         {
            ::printf( "Error opening file %s...\n", fname );
            return -4;
         }
      }
		#else
		#endif
   }
   return readed;
}
//---------------------------------------------------------------------------
pnode* ptree::search( const char* text, pnode* node )
{
   bool lFound = false;
   pnode* n;
   pnode* p;
   pnode* res = NULL;
   
   if( !node )
   {
      //MessageBox( 0, ( AnsiString( "Start searching... " ) + text ).c_str() , "", MB_OK );
      n = getfirst();
   }
   else
   {  
      //MessageBox( 0, ( AnsiString( "Searching in node...(" ) + AnsiString( node->text ) + ")" ).c_str(), "", MB_OK );
      n = getfirstch( node );
   }
   
   while( n && !lFound )
   {
      if( strcmp( text, n->text  ) == 0 )
      {
         lFound = true;
         res = n;
      }
      else
      {
         if( n->haschildren() )
         {
            p = search( text, n );
         }
         else
         {
            p = NULL;
         }
         if( p )
         {
            lFound = true;
            res = p;
         }
         else
         {
            n = n->getnext();
         }   
      }
   }
   return res;
}
//---------------------------------------------------------------------------

