/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "qhacctable.h"
#include "localfileplugin.h"
#include "qhaccutils.h"
#include "qhacc.h"

#include <qfile.h>
#include <qregexp.h>
#include <qvaluelist.h>
#include <qtextstream.h>

#include <map>
using namespace std;

#include <sys/stat.h>

// "factory" function calls
extern "C" {
	QHaccPlugin * create(){	return new LocalFileDBPlugin; }
	void destroy( LocalFileDBPlugin * p ){ delete p; }
}


const LocalFileDBInfo LocalFileDBPlugin::pinfo;

/* read and write local files */
LocalFileDBPlugin::LocalFileDBPlugin() : QHaccDBPlugin(){
	needsave=false;
	transactions=accounts=journals=splits=
		preferences=namedtrans=procs=0;
	dbs=0;
}

LocalFileDBPlugin::~LocalFileDBPlugin(){
	if( transactions!=0 ){
		for( int i=0; i<QC::NUMTABLES; i++ ) delete dbs[i];
		delete [] dbs;
	}
}
 
bool LocalFileDBPlugin::save( QString& err ){
  // return if the save was successful
  bool ret=true;
  
  for( int i=0; i<QC::NUMTABLES; i++ ) {
    if( !isavet( *dbs[i], home+"/"+QC::TABLENAMES[i], err ) ) ret=false;
  }
  
  needsave=!ret;
  return ret;
}

bool LocalFileDBPlugin::isavet( QHaccTable& t, const QString& f, QString& e ){
  return savet( t, f, !engine->getBP( "KEEPFILEPERMS" ), e );
}

bool LocalFileDBPlugin::savet( QHaccTable& t, const QString& f, bool perm,
															 QString& e ){
  std::ostream * str=0;
  bool ret=false;
  
  QFile file( f );
  if( file.open( IO_WriteOnly ) ){
    QTextStream out( &file );
    
    uint rows=t.rows();
		//int tpk=t.idcol();
		//QHaccTableIndex index( &t, tpk, t.coltype( tpk ) );
		//for( uint j=0; j<rows; j++ ) out<<t[index[j]].toString()<<endl;
		for( uint j=0; j<rows; j++ ) out<<t[j].toString()<<endl;
    file.close();
#ifdef Q_OS_WIN
		perm=false; // just do something to avoid a compiler warning on windows
#else
		if( perm ) chmod( f, S_IRUSR | S_IWUSR );
#endif
    if( Utils::debug( Utils::DBGMINOR, str ) )
      *str<<"wrote "<<rows<<" rows from "<<t.getName()<<" to "<<f<<endl;
    
    ret=true;
  }
  else{
    e="could not write "+t.getName()+" to "+f;
    if( Utils::error( Utils::ERROPER, str ) ) *str<<e<<endl;
    ret=false;
  }
  return ret;
}

bool LocalFileDBPlugin::loadt( QHaccTable& t, const QString& f, QString& e ){
	bool ret=false;
	std::ostream * str=0;

	QFile file( f );
	if( file.exists() && file.open( IO_ReadOnly ) ){
		QTextStream in( &file );
		int count=0;
		
		// first, figure out how many rows we're going to load
		while( !in.atEnd() ) {
			in.readLine();
			count++;
		}
		
		// get back to the beginning of the file and start loading
		file.at( 0 );
		t.startLoad( count );
		while ( !in.eof() ) t.loadRow( in.readLine() );
		t.stopLoad();
		file.close();

		if( Utils::debug( Utils::DBGMINOR, str ) )
			*str<<"loaded "<<count<<" row"<<( count>1 ? "s" : "" )<<" from "
					<<f<<" into "<<t.getName()<<endl;
		ret=true;
	}
	else{
		e="could not open file: "+f;
		if( Utils::error( Utils::ERROPER, str ) ) *str<<e<<endl;
	}
	return ret;
}

bool LocalFileDBPlugin::iconnect( QHacc *, const QString&, QString& ){
	return true;
}

bool LocalFileDBPlugin::connect( QHacc * e, const QString& h, QString& r ){
	transactions=new QHaccTable( QC::TCOLS, QC::TCOLTYPES,
															 QC::TABLENAMES[QC::TRANT] );
	splits      =new QHaccTable( QC::SCOLS, QC::SCOLTYPES,
															 QC::TABLENAMES[QC::SPLTT], 0, 10, 15 );
	accounts    =new QHaccTable( QC::ACOLS, QC::ACOLTYPES,
															 QC::TABLENAMES[QC::ACCTT], 0, 3, 5 );
	preferences =new QHaccTable( QC::PCOLS, QC::PCOLTYPES,
															 QC::TABLENAMES[QC::PREFT] );
	journals    =new QHaccTable( QC::LCOLS, QC::LCOLTYPES,
															 QC::TABLENAMES[QC::JRNLT] );
	namedtrans  =new QHaccTable( QC::NCOLS, QC::NCOLTYPES,
															 QC::TABLENAMES[QC::NAMET] );
	procs       =new QHaccTable( QC::JCOLS, QC::JCOLTYPES,
															 QC::TABLENAMES[QC::JOBST] );
	
	dbs=new QHaccTable * [QC::NUMTABLES];
	dbs[QC::PREFT]=preferences;
	dbs[QC::JRNLT]=journals;
	dbs[QC::ACCTT]=accounts;
	dbs[QC::TRANT]=transactions;
	dbs[QC::SPLTT]=splits;
	dbs[QC::NAMET]=namedtrans;
	dbs[QC::JOBST]=procs;
	
	for( int i=0; i<QC::NUMTABLES; i++ )
		dbs[i]->setPK( Utils::tpk( ( Table )i ) );
	
	accounts->addIndexOn( QC::APID, QC::AID );
	
	transactions->addIndexOn( QC::TDATE, QC::TID );
	transactions->addIndexOn( QC::TPAYEE, QC::TDATE );
	transactions->addIndexOn( QC::TTYPE );
	transactions->addIndexOn( QC::TLID );
	
	splits->addIndexOn( QC::SACCTID );
	splits->addIndexOn( QC::STID );
	
	namedtrans->addIndexOn( QC::NNAME );
	namedtrans->addIndexOn( QC::NACCTID );

	home=h;
	engine=e;

	return iconnect( e, h, r );
}

bool LocalFileDBPlugin::load( QString& err ){ return iload( err ); }

bool LocalFileDBPlugin::iload( QString& err ){
  for( int i=0; i<QC::NUMTABLES; i++ ) {
    dbs[i]->clear();
    iloadt( *dbs[i], home+"/"+QC::TABLENAMES[i], err );
  }
  
  return true; // no error should stop QHacc form proceeding
}

bool LocalFileDBPlugin::iloadt( QHaccTable& tbl, const QString& fn, 
																QString& error ){
	return loadt( tbl, fn, error );
}

auto_ptr<QHaccTable> LocalFileDBPlugin::xtrans( vector<TableSelect> mts ){
  auto_ptr<QHaccTable> ret( new QHaccTable( QC::XCOLS, QC::XCOLTYPES ) );
  ret->setName( "xtrans temp" );
  // we need to prune the splits and transactions as much as possible before
  // joining them, or we'll end up always getting a huge table to manipulate
  vector<TableSelect>splts;
  vector<TableSelect>trns;
  
  //cout<<"XTRANS: "<<endl;
  //for( uint i=0; i<mts.size(); i++ )	cout<<"  "<<mts[i].toString()<<endl;
	
	//cout<<"create xtrans"<<endl;
  
  for( uint i=0; i<mts.size(); i++ ){
    PosVal pv;
    int t=0;
    bool issplit=false;
    int newp=-1;
    
    mts[i].getAll( pv, t );
    const int P=pv.getp();
    const TableCol V=pv.getv();
    
    if( P==QC::XTID ){
      newp=QC::TID;
      // when searching on TID, we can prune both tables
      splts.push_back( TableSelect( PosVal( QC::STID, V ), t  ) );
    }
    else if( P==QC::XTPAYEE )  newp=QC::TPAYEE;
    else if( P==QC::XTDATE )   newp=QC::TDATE;
    else if( P==QC::XTLID )	   newp=QC::TLID;
    else if( P==QC::XTTYPE )   newp=QC::TTYPE;
    else if( P==QC::XTNUM )	   newp=QC::TNUM;
    else if( P==QC::XTMEMO )   newp=QC::TMEMO;
    else if( P==QC::XTMETA )   newp=QC::TMETA;
    else if( P==QC::XTVOID )   newp=QC::TVOID;
    else if( P==QC::XSID ){
      newp=QC::SID;
      issplit=true;
    }
    else if( P==QC::XSRECO ){
      newp=QC::SRECO;
      issplit=true;
    }
    else if( P==QC::XSACCTID ){
      newp=QC::SACCTID;
      issplit=true;
    }
    else if( P==QC::XSSUM ){
      newp=QC::SSUM;
      issplit=true;
    }
    else if( P==QC::XSRECODATE ){
      newp=QC::SRECODATE;
      issplit=true;
    }
    else if( P==QC::XSMETA ){
      newp=QC::SMETA;
      issplit=true;
    }
    else if( P==QC::XSTAXABLE ){
      newp=QC::STAXABLE;
      issplit=true;
    }
    
    TableSelect ts( PosVal( newp, V ), t );
    if( issplit ) splts.push_back( ts );
    else trns.push_back( ts );
  }
  
  // now fetch what we need to join
  //cout<<"XTRANS splits: "<<endl;
  //for( uint i=0; i<splts.size(); i++ ) cout<<"  "<<splts[i].toString()<<endl;
  
  uint srr=0;
  auto_ptr<QHaccResultSet> splitsgw=splits->getWhere( splts, srr );
  
  //cout<<"XTRANS trans: "<<endl;
  //for( uint i=0; i<trns.size(); i++ ) cout<<"  "<<trns[i].toString()<<endl;
  
  uint trr=0;
  auto_ptr<QHaccResultSet> transgw=transactions->getWhere( trns, trr );
  
  // create the combined table
  
  // first optimization: if one of our tables is empty, just return	
  if( srr==0 || trr==0 ) return ret;

	//cout<<"split rows="<<srr<<"; transaction rows="<<trr<<endl;

  // second optimization: we're going to hash-join these tables on stid/tid
	// we assume the transactions resultset is smaller, so we'll hash splits
	//
	// note that we probably have to hash the transactions because if we hashed
	// splits based on stid, we'd lose data if multiple splits from the same
	// transaction were returned in splitsgw
	map<uint, const TableRow *> mapper;
	
	for( uint i=0; i<trr; i++ )
		mapper[transgw->at( i )[QC::TID].getu()]=&( transgw->at( i ) );

	// now cycle through the splits, and look in the map for a transaction.
	// if we find a transaction, put the xtrans into the ret set
	map<uint, const TableRow *>::iterator it;
	for( uint i=0; i<srr; i++ ){
		const Split& spl=splitsgw->at( i );
		it=mapper.find( spl[QC::STID].getu() );

		if( it!=mapper.end() ){ // found a transaction, so add the xtrans
			ret->add( engine->makeXTrans( *( it->second ), spl ) );
		}
	}
	
  ret->stopLoad();
	//cout<<"xtrans returning "<<ret->rows()<<endl;
  return ret;
}

QHaccTable * LocalFileDBPlugin::table( Table t ) const {return dbs[( int )t];}
uint LocalFileDBPlugin::cnt( Table t ){ return table( t )->rows(); }
bool LocalFileDBPlugin::load( Table t, const QHaccResultSet * model ){
	if( t==XTRANS ) return false;
	needsave=true;
	return table( t )->load( model );
}

int LocalFileDBPlugin::add( Table t, const TableRow& r ){ 
	if( t==XTRANS ) return 0;
	needsave=true;
	return table( t )->add( r );
}

void LocalFileDBPlugin::updateWhere( Table t, const TableSelect& ts, 
																		 const TableUpdate& tu ){
	if( t==XTRANS ){
		std::ostream * str=0;
		if( Utils::error( Utils::ERROPER, str ) )
			*str<<"cannot update XTRANS"<<endl;
		return;
	}
	needsave=true;
	table( t )->updateWhere( ts, tu );
}

void LocalFileDBPlugin::updateWhere( Table t, const TableSelect& ts, 
																		const TableRow& tr ){
	if( t==XTRANS ){
		std::ostream * str=0;
		if( Utils::error( Utils::ERROPER, str ) )
			*str<<"cannot update XTRANS"<<endl;
		return;
	}
	needsave=true;
	table( t )->updateWhere( ts, tr );
}

auto_ptr<QHaccResultSet> LocalFileDBPlugin::getWhere( Table t,
																											const TableGet& tg,
																											vector<TableSelect> ts,
																											uint& retrows ){
	if( t==XTRANS ){
		// the call to xtrans() automatically does the TableSelect filtering,
		// so it's a bad idea to do it again on the getWhere()
		vector<TableSelect> v;
		return xtrans( ts )->getWhere( tg, v, retrows );
	}
	else return table( t )->getWhere( tg, ts, retrows );
}

auto_ptr<QHaccResultSet> LocalFileDBPlugin::getWhere( Table t,
																											vector<TableSelect> ts,
																											uint& retrows ){
	return getWhere( t, TableGet(), ts, retrows );
}

auto_ptr<QHaccResultSet> LocalFileDBPlugin::getWhere( Table t,
																											const TableSelect& s,
																											uint& retrows ){
	return getWhere( t, TableGet(), vector<TableSelect>( 1, s ), retrows );
}

void LocalFileDBPlugin::deleteWhere( Table t, const TableSelect& ts ){
	if( t==XTRANS ) return;
	needsave=true;
	table( t )->deleteWhere( ts );
}

TableCol LocalFileDBPlugin::max( Table t, int col ){
  TableCol ret;
  if( t==XTRANS ){
    vector<TableSelect> v;
    ret=xtrans( v )->max( col );
  }
  else ret=table( t )->max( col );
  return ret;
}

TableCol LocalFileDBPlugin::min( Table t, int col ){ 
  TableCol ret;
  if( t==XTRANS ){
    vector<TableSelect> v;
    ret=xtrans( v )->min( col );
  }
  else ret=table( t )->min( col );
  return ret;
}

void LocalFileDBPlugin::startLoad( Table t, uint i ){
  if( t==XTRANS ) return;
  table( t )->startLoad( i );
}
void LocalFileDBPlugin::stopLoad( Table t ){
	if( t==XTRANS ) return;
	table( t )->stopLoad();
}

bool LocalFileDBPlugin::dirty() const { return needsave; }
bool LocalFileDBPlugin::imprt( QHaccResultSet * tbls ){
	bool ret=true;
	for( int i=0; i<QC::NUMTABLES; i++ ) ret=dbs[i]->load( &tbls[i] );
	needsave=true;
	return ret;
}
bool LocalFileDBPlugin::exprt( QHaccResultSet * tbls ){
	for( int i=0; i<QC::NUMTABLES; i++ ) tbls[i]=*dbs[i];
	return true;
}

void LocalFileDBPlugin::setAtom( AtomicOp, QString ){}

QString LocalFileDBPlugin::create( const QString& extra ) const {
	return screate( extra );
}

QString LocalFileDBPlugin::screate( const QString& extra ) {
	QString ret="mkdir -p "+extra;

	for( int i=0; i<QC::NUMTABLES; i++ )
		ret.append( "\ntouch "+extra+"/" ).append( QC::TABLENAMES[i] );

	return ret;
}

const PluginInfo& LocalFileDBPlugin::info() const {	return pinfo; }



LocalFileDBInfo::LocalFileDBInfo(){
	description="Native";
	stubby="FILES";
	targ=DIRECTORY;
	atom=true;
	raw=false;

	piprefs.reset( new QHaccResultSet( QC::IPICOLS, QC::IPICOLTYPES ) );
	TableRow row( QC::IPICOLS );
	row.set( QC::IPIPREF, "KEEPFILEPERMS" );
	row.set( QC::IPITYPE, CTBOOL );
	row.set( QC::IPILABEL, "Save does not modify file permissions" );
	piprefs->add( row );
}
