/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "qhacctablerows.h"
#include "qhaccutils.h"

#include <qregexp.h>

/****************************************/
/* TABLEROW                             */
/*   A row of data in a qhacctable      */
/*                                      */
/****************************************/

TableRow::TableRow(){ init( 0 ); }
TableRow::TableRow( int sz ){ init( sz ); }

TableRow::TableRow( const QString& datastr, int sz ){
	init( sz );
	QString * ss=new QString[size];
	Utils::parser( datastr, QC::TCSEP, 0, ss, sz );
	for( int i=0; i<size; i++ ) data[i]=TableCol( ss[i] );
	delete [] ss;
}

TableRow::TableRow( const TableRow& model ){
	init( model.size );
	for( int i=0; i<size; i++ ) data[i]=model.data[i];
}

TableRow::TableRow( TableCol * acols, int ncols ){
	init( ncols );
	for( int i=0; i<ncols; i++ ) data[i]=acols[i];
}

TableRow::~TableRow(){ delete [] data; }

void TableRow::init( int sz ){ 
	size=sz;
	data=new TableCol[sz];
}

QString TableRow::toString() const {
	QString ret="";
	if( isNull() ) return ret;

	ret=data[0].gets();
	for( int j=1; j<size; j++ )	ret+=QC::TCSEP+data[j].gets();
	return ret;
}

const TableRow& TableRow::operator=( const TableRow& model ) {
	if( &model!=this ){
		delete [] data;
		init( model.size );
		for( int i=0; i<size; i++ ) data[i]=model.data[i];
	}
	
	return *this;
}

bool TableRow::isNull() const { return size==0; }
const TableCol& TableRow::operator[]( int i ) const { return data[i]; }
const TableCol& TableRow::get( int i ) const { return data[i]; }
int TableRow::cols() const { return size; }
void TableRow::set( int i, const TableCol& tc ){
	if( i>=size ) return;
	data[i]=tc;
}

void TableRow::set( const PosVal& pv ){
	int i;
	TableCol tc;
	pv.get( i, tc );
  set( i, tc );
}

int TableRow::geti( int i ) const { return get( i ).geti(); }
uint TableRow::getu( int i ) const { return get( i ).getu(); }
bool TableRow::getb( int i ) const { return get( i ).getb(); }
float TableRow::getf( int i ) const { return get( i ).getf(); }
QDate TableRow::getd( int i ) const { return get( i ).getd(); }
const QString& TableRow::gets( int i ) const { return get( i ).gets(); }


/****************************************/
/* TABLECOL                             */
/*   One piece of data in a tablerow    */
/*                                      */
/****************************************/

TableCol::TableCol(){ data=""; }

TableCol::TableCol( const char * s ){
	if( s ) data=QString( s );
	else data="";
}

TableCol::TableCol( const QString& s ){
	if( s.isNull() ) data="";
	else data=s;
}
TableCol::TableCol( int i ){ data.setNum( i ); }
TableCol::TableCol( uint i ){ data.setNum( i ); }
TableCol::TableCol( float f ){ data.setNum( f ); }
TableCol::TableCol( bool b ){	data=( b ? "Y" : "N" ); }
TableCol::TableCol( const QDate& date ){
	data=date.toString( Qt::ISODate );
}

int TableCol::geti() const { return data.toInt(); }
uint TableCol::getu() const { return data.toUInt(); }
bool TableCol::getb() const { return ( data=="N" ? false : true ); }
float TableCol::getf() const { return data.toFloat(); }
const QString& TableCol::gets() const { return data; }
QDate TableCol::getd() const {
	return QDate::fromString( data, Qt::ISODate );
}

const TableCol& TableCol::operator=( const TableCol& model ) {
	if( *this!=model ) data=model.data;
	return *this;
}

bool TableCol::operator==( const TableCol& v ) const { return data==v.gets(); }
bool TableCol::operator!=( const TableCol& v ) const { return data!=v.gets(); }

const compfunc TableCol::COMPARERS[]={
	&TableCol::compareNull,
	&TableCol::compareString,
	&TableCol::compareInt,
	&TableCol::compareUInt,
	&TableCol::compareFloat,
	&TableCol::compareBool,
	&TableCol::compareDate
};

int TableCol::compareTo( const TableCol& model, ColType t ) const {
	return ( this->*COMPARERS[( int )t] )( model );
}

int TableCol::compareString( const TableCol& model ) const {
	int val=data.compare( model.gets() );
	if( val==0 ) return val;
	if( val<0 ) return -1;
	return 1;
}
int TableCol::compareInt( const TableCol& model ) const {
	if( data==model.gets() ) return 0;
	if( geti()<model.geti() ) return -1;
	return 1;
}
int TableCol::compareUInt( const TableCol& model ) const {
	if( data==model.gets() ) return 0;
	if( getu()<model.getu() ) return -1;
	return 1;
}
int TableCol::compareFloat( const TableCol& model ) const {
	float i=model.getf();
	float me=getf();
	if( me==i ) return 0;
	if( me<i ) return -1;
	return 1;
}
int TableCol::compareDate( const TableCol& model ) const {
	//cout<<"comparing "<<data<<" to "<<model.gets()<<" ";
	//if( data==model.gets() ) cout<<0<<endl;
	//else if( getd()<model.getd() ) cout<<-1<<endl;
	//else cout<<1<<endl;

	if( data==model.gets() ) return 0;
	if( getd()<model.getd() ) return -1;
	return 1;
}
int TableCol::compareBool( const TableCol& model ) const {
	if( data==model.gets() ) return 0;
	if( !getb() && model.getb() ) return -1;
	return 1;
}
int TableCol::compareNull( const TableCol& ) const { return -1; }

/***********************************************************/
/* POSVAL                                                  */
/*   A structure to hold a column number and a value       */
/*                                                         */
/***********************************************************/
PosVal::PosVal(){ init( -1, TableCol() ); }
PosVal::PosVal( int p, const TableCol& c ){ init( p, c ); }
PosVal::PosVal( const PosVal& o ){
	if( &o==this ) return;
	init( o.getp(), o.getv() );
}
const PosVal& PosVal::operator=( const PosVal& o ){
	if( &o!=this ) init( o.getp(), o.getv() );
	return *this;
}

PosVal::~PosVal(){}

void PosVal::init( int p, const TableCol& c ){
	pos=p;
	val=c;
}

int PosVal::getp() const { return pos; }
const TableCol& PosVal::getv() const { return val; }
void PosVal::get( int& p, TableCol& c ) const {
	p=pos;
	c=val;
}
