/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************* * * * * * * * * * * * * ***************************/

#include "monthlygraph.h"
#include "qhacc.h"
#include "qhaccutils.h"
#include "qhacctable.h"
#include "qhaccsegmenter.h"
#include "qhaccconstants.h"

#include <stdlib.h>

#include <qpainter.h>

const uint MonthlyGraph::AID=      0;
const uint MonthlyGraph::NUMBER=   1;
const uint MonthlyGraph::INCREASES=2;
const uint MonthlyGraph::DECREASES=3;

MonthlyGraph::MonthlyGraph() : GraphBase(){ monthstrs=0; }

MonthlyGraph::~MonthlyGraph(){
	delete [] monthstrs; 
	if( data ) delete data;
}

void MonthlyGraph::setup( QHacc * e ) {
	if( data ) delete data;
	if( monthstrs ) delete [] monthstrs;

	GraphBase::setup( e );
	data=new QHaccTable( 4 );
	monthstrs=new QString[0];
}

void MonthlyGraph::isetData(){
	data->clear();
	if( monthstrs ) delete [] monthstrs;
	monthstrs=0;

	hiVal=loVal=0;
	bool totals=( !( ivse || engine->getBP( "GRAPHDELTAS" ) ) );

	for( uint i=0; i<accounts->rows(); i++ ){
		const Account& a=accounts->at( i );
		
		vector<TableSelect> criteria;
		criteria.push_back( TableSelect( QC::XTDATE, start, TableSelect::GE ) );
		criteria.push_back( TableSelect( QC::XTDATE, end, TableSelect::LT ) );
		criteria.push_back( TableSelect( QC::XTVOID, false ) );

		
		if( journalid!=0 )
			criteria.push_back( TableSelect( QC::XTLID, TableCol( journalid ) ) );

		uint rr=0;
		auto_ptr<QHaccResultSet> trans=engine->getXTForA( a, TableGet(),
																											criteria, rr );
		QHaccTableIndex idx( trans.get(), QC::XTDATE, CTDATE );

		uint * pos=0, sz=0;		
		QHaccSegmenter::segment( engine, &idx, start, end, pos, sz );

		
		//for( uint q=0; q<trans->rows(); q++ ) 
		//cout<<trans->at( idx[q] ).toString()<<endl;

		if( i==0 ){
			months=sz-1;
			monthstrs=new QString[months];
		}

		TableRow last;
		data->startLoad( months );

		QString sep=engine->getSP( "DATESEPARATOR" );
		int fmt=engine->getIP( "DATEFORMAT" );
		bool rollups=engine->getBP( "INCLUDESUBSONRECALC" );

		const MonCon& conv=engine->converter();

		for( uint month=0; month<months; month++ ){
			if( i==0 ){ // get the headings on the first run
				QDate ndate=start.addMonths( month );
				monthstrs[month]=Utils::shortStringFromDate( ndate, sep, fmt )+"- ";
				ndate=start.addMonths( month+1 );
				if( ndate>end ) ndate=end;
				monthstrs[month]+=Utils::shortStringFromDate( ndate.addDays( -1 ),
																											sep, fmt );
			}

			//cout<<"  month is ="<<month<<"; pos[month]="<<pos[month]<<"-"<<pos[month+1]<<endl;
			// figure out the values
			int incrs=0, decrs=0;
			if( totals ){
				if( last.isNull() ){
					incrs=engine->getABalOn( a, start, TableSelect() );

					// if we're calculating rollups, we need to
					// sum in the children accounts as well
					if( rollups ){
						uint rr=0;
						vector<TableSelect> vs( 1, TableSelect( QC::APID, a[QC::AID] ) );
						
						auto_ptr<QHaccResultSet> cs=engine->getWhere( ACCOUNTS,	vs,	rr );
						for( uint j=0; j<rr; j++ ){
							incrs+=engine->getABalOn( cs->at( j ), start,	TableSelect() );
						}
					}
				}
				else incrs=last.geti( INCREASES )+last.geti( DECREASES );
			}

			for( uint k=pos[month]; k<pos[month+1]; k++ ){
				const Transaction& t=trans->at( idx[k] );

				//cout<<"  k="<<k<<"  "<<t.toString()<<endl;
				int f=conv.converti( t.gets( QC::XSSUM ), EngVal, PrefVal );
				if( f<0 ) decrs+=f;
				else incrs+=f;
			}

			if( ivse ){
				if( incrs>hiVal ) hiVal=incrs;
				if( decrs<loVal ) loVal=decrs;
			}
			else{
				float val=incrs+decrs;
				if( val>hiVal ) hiVal=val;
				if( val<loVal ) loVal=val;
			}
			TableCol d[]={ a.get( QC::AID ), TableCol( month ),
										 TableCol( incrs ), TableCol( decrs ) };
			last=TableRow( d, 4 );
      data->add( last );
		}
		delete [] pos;
	}
	data->stopLoad();

	data->addIndexOn( AID );
	data->addIndexOn( NUMBER );
}

void MonthlyGraph::mouseDates( const QPoint& p, QDate& st,
																			QDate& en ) const {
	uint lim=p.x()/wpm;
	st=start.addMonths( lim );
	en=start.addMonths( lim+1 ).addDays( -1 );
}


TableSelect MonthlyGraph::mouseSel( const QPoint& p ) const {
	uint n=p.x()/wpa;
	uint acctn=n%accounts->rows();
	return TableSelect( QC::XSACCTID, accounts->at( acctn )[QC::AID] );
}

void MonthlyGraph::paintBase( QPainter * p, const QRect& size ){
	if( months==0 ) return;

	const MonCon& conv=engine->converter();

	wpm=size.width()/months;
	wpa=wpm/accounts->rows();
	//cout<<"monthlypainter base: wpam="<<wpm<<"; wpa="<<wpa<<endl;
	QFontMetrics fm=p->fontMetrics();
	const QColor GRAY=QColor( "darkgray" );
	int fmh=fm.height();
	int h=size.height(), w=size.width();


	int baseh=3*fmh;
	baseline=h-baseh;
	factor=( hiVal-loVal )/( baseline-2*fmh ); // fmh is wiggle room
	//cout<<"hiVal is "<<hiVal<<"; loVal is "<<loVal<<"; baseline is "<<baseline<<"; factor is "<<factor<<endl;
	if( factor==0 ) factor=1;
	zeroline=baseline-( int )( ( 0-loVal )/factor )-fmh; // more wiggling


  QPen pen=p->pen(); // save the original pen for later

	// draw the zero line in dark black
	p->setPen( QPen( "black" ) );	
	QString temp="0";
	p->drawText( 0, zeroline, temp );
	p->drawLine( 0, zeroline, w, zeroline );
	p->drawText( w-fm.width( temp ), zeroline, temp );

	// draw guidelines, if necessary
	if( engine->getBP( "GRAPHGUIDES" ) ){
		p->setPen( GRAY );
		const int NUMLINES=4;
		
		float spanpl=( hiVal-loVal )/( factor*NUMLINES );
		float liner=( zeroline-( hiVal/factor ) );
	
	  // the +1 puts a line at the lVal point
		for( int i=0; i<NUMLINES+1; i++ ){
			temp=conv.convert( ( int )( ( zeroline-liner )*factor ),
												 Engine, Engine );
			temp.setNum( temp.toFloat(), 'f', 0 );
			p->drawText( 0, ( int )liner, temp );
			p->drawLine( 0, ( int )liner, w, ( int )liner );
			p->drawText( w-fm.width( temp ), ( int )liner, temp );
			liner+=spanpl;
		}
	}

	// draw the month headers
	bool manyaccts=( accounts->rows()>1 );
	p->setPen( pen );
	for( uint i=0; i<months; i++ ){
		int wspot=wpm*i;
		QColor color=GRAY;
		
		if( manyaccts ){
			// alternate light, dark grays
		  if( i%2 ) color=color.light( 110 );
		}
		else color=colors[i%12];

		p->fillRect( wspot, baseline, wpm*( i+1 ), baseline+baseh, color );
		

		//if the column label is too long, just put a placeholder
		if ( fm.width( monthstrs[i] )>=( int )( wpm*2 ) ) temp="|";
		else temp=monthstrs[i];
			
		//column labels
		//			p->drawText( wspot-half, hspot, wpp, hespot,
		//AlignTop|AlignHCenter|WordBreak, temp );
		p->drawText( wspot, baseline, wpm, baseh,
		             p->AlignTop|p->AlignHCenter|p->WordBreak, temp );
	}
}

void MonthlyGraph::paintMain( QPainter * p, const QRect& size ){
	if( months==0 ) {
		p->drawText( size.left(), size.top(), size.width(), size.height(),
								 p->AlignCenter, EMPTYSTR );
		return;
	}
	
	bool drawSums=engine->getBP( "GRAPHSHOWTOTALS" );
	bool drawHeads=engine->getBP( "GRAPHHEADS" );
	bool manyaccts=( accounts->rows()>1 );
	uint arows=accounts->rows();
	
	// go through each account, and draw all its data before moving on
	// to the next account
	for( uint acctn=0; acctn<arows; acctn++ ){
		const Account& acct=accounts->at( acctn );
		
		uint rr=0;
		auto_ptr<QHaccResultSet> vals=data->getWhere( TableSelect( AID,
																															 acct[QC::AID] ),
																									rr );
		QHaccTableIndex idx( vals.get(), NUMBER, CTINT );

		TableRow last;
		for( uint month=0; month<months; month++ ){
			QString title;
			if( month==0 && drawHeads && manyaccts ) title=acct.gets( QC::ANAME );

			int wstart=wpm*month;
			int wspot=wstart+( acctn*wpa );
			

			const TableRow& row=vals->at( idx[month] );
			paintRow( p, last, row, wstart, wspot,
								colors[( manyaccts ? acctn : month )%12], drawSums, title );
			last=row;
		}
	}
}

void MonthlyGraph::drawTitle( QPainter * p, const QString& title,
															int wstart, int wspot, int hspot ) const {
	QFontMetrics fm=p->fontMetrics();
	int fmh=fm.height();
	if( line ) p->drawText( wspot+5, hspot-fmh/2, title );
	else{
		p->setPen( QPen( "black" ) );
		hspot=zeroline-( int )( hiVal/factor );
		for( uint j=0; j<title.length(); j++ ){
			hspot+=fmh;
			p->drawText( wstart, hspot, wpa, fmh,
									 p->AlignCenter, title.mid( j, 1 ) );
		}
	}
}

void MonthlyGraph::drawSum( QPainter * p, int sum,
														int x, int y, int w, int h ) const {
	p->setPen( QPen( "black" ) );
	QString temp;
	temp.setNum( ( float )( sum )/( float )QHacc::ROLLOVER, 'f', 0 );
	if( sum>0 ) y-=h; // put positive sums a little higher
	if( sum!=0 ) p->drawText( x, y, w, h, p->AlignCenter|p->DontClip, temp );
}

void MonthlyGraph::setLine( bool l ){ line=l; }
void MonthlyGraph::setIvsE( bool i ){ ivse=i; }
