/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "reporter.h"
#include "qhacc.h"
#include "resultset.h"
#include "qhaccutils.h"

#include <stdlib.h>

ReportBase::ReportBase(){ engine=0; }
ReportBase::~ReportBase(){}

void ReportBase::setup( QHacc * e ){ engine=e; }

QString ReportBase::writereport( const QString& title, 
																 const QHaccResultSet * data ){
	QString report=title;
	if( data->isEmpty() ) report.append( "\n\tNo Transactions" );
	else{
		uint cols      = data->columns();
		uint rows      = data->rows();
		QString format = "\n";
		
		// figure out each column's longest string
		vector<int> bigColLen;
		
		for( uint col=0; col!=cols; col++ ){
			format.append( " %" + QString::number( col+1 ) );
			int longest = 0;
			
			for( uint row=0; row!=rows; row++ ){
				const QString& str=data->at( row )[col].gets();
				
				if( str.length()>static_cast<uint>( longest ) ){
					longest=str.length();
				}
			}
			
			bigColLen.push_back( longest );
		}
		
		bigColLen[0]=0-bigColLen[0]; // left-justify the first col
		
		// create a string that contains the entire report
		for( uint row=0; row!=rows; row++ ){
			QString str=format;
			
			for( uint col=0; col!=cols; col++ ){
				str=str.arg( data->at( row ).gets( col ), bigColLen[col] );
			}
			
			report.append( str );
		}
	}
	
	return report;	
}

void ReportBase::selected( QDate&, QDate&, bool& enableacctsel,
													 bool& enablemultiselect ){
	enableacctsel=enablemultiselect=true;
}

auto_ptr<QHaccResultSet> ReportBase::gentrans( QHaccResultSet * accounts,
																							 vector<TableSelect> ss ){
	
	auto_ptr<QHaccResultSet> ret( new QHaccResultSet( QC::XCOLS,
																										QC::XCOLTYPES ) );
	// ignore void transactions
	ss.push_back( TableSelect( QC::XTVOID, false ) );

	uint arows=accounts->rows();
	for( uint i=0; i<arows; i++ ){
		const Account& acct=accounts->at( i );
		
		uint rr=0;
		auto_ptr<QHaccResultSet> temptrans=engine->getXTForA( acct, TableGet(),
																													ss, rr );
		ret->load( temptrans.get() );
	}
	
	return ret;
}


auto_ptr<QHaccResultSet> ReportBase::gentrans( QHaccResultSet * accounts,
																							 uint jid, const QDate& start,
																							 const QDate&  end ){
	// create the return table and row structure
	TableSelect ge( QC::XTDATE, TableCol( start ), TableSelect::GE );
	TableSelect le( QC::XTDATE, TableCol( end ),   TableSelect::LE ); 

	vector<TableSelect> crit;
	crit.push_back( ge );
	crit.push_back( le );

	if( jid!=0 ) crit.push_back( TableSelect( QC::XTLID, jid ) );
	auto_ptr<QHaccResultSet> t=gentrans( accounts, crit );
	return t;
}

QString ReportBase::titler( QHaccResultSet * accounts,
														uint jid, QDate start, QDate end ){
	QString title;
	if( jid!=0 ){
		const Journal& j=engine->getL( jid );
		title=j.gets( QC::LNAME )+"/";
	}

	uint arows=accounts->rows();
	if( arows==1 )
		title+=engine->getFNameOfA( accounts->at( 0 ).getu( QC::AID ) );
	else title+="<Multiple>";
	title+=": "+info().descr();
	
	if( start.isValid() ){
		QString sep=engine->getSP( "DATESEPARATOR" );
		int fmt=engine->getIP( "DATEFORMAT" );
		title+=" ("+Utils::stringFromDate( start, sep, fmt )+" -";
		title+=" "+Utils::stringFromDate( end, sep,  fmt )+")";
	}

	return title;
}

auto_ptr<QHaccResultSet> ReportBase::generate( const QString& home,
																							 QString& title, QString& err,
																							 bool& ok ){

	auto_ptr<QHaccResultSet> accts( new QHaccResultSet( 0 ) );
	uint jid=0;
	QDate start, stop;
	ok=homeok( home, jid, accts, start, stop, err );

	if( ok ) return generate( jid, accts.get(), start, stop, title );
	else return auto_ptr<QHaccResultSet>( new QHaccResultSet( 0 ) );
}

bool ReportBase::homeok( const QString& home, uint& jid,
												 auto_ptr<QHaccResultSet>& accts, QDate& start, 
												 QDate& stop, QString& err ){

	const Account& a=engine->getA( home );
	if( a.isNull() ){
		err="No account: "+home;
		return false;
	}

	accts.reset( new QHaccResultSet( QC::ACOLS, QC::ACOLTYPES, 0, 1 ) );
	accts->add( a );

	auto_ptr<QHaccResultSet> js=engine->getLs();
	const Journal& j=js->at( engine->getIP( "JOURNALINDEX" ) );
	jid=j[QC::LID].getu();

	start=engine->min( TRANSACTIONS, QC::TDATE ).getd();
	stop=QDate::currentDate();

	return true;
}


/********************/
/****Plugin Info*****/
/********************/

ReportInfo::ReportInfo(){
	raw=atom=false;
	piprefs.reset( new QHaccResultSet( QC::IPICOLS, QC::IPICOLTYPES ) );
	targ=NONE;
}

AccountsInfo::AccountsInfo(){
	description="Accounts";
	stubby="ACCT";
}

AvesInfo::AvesInfo(){
	description="Averages";
	stubby="AVES";
}

BudgetInfo::BudgetInfo(){
	description="Budget";
	stubby="BUDGET";
}

BalancesInfo::BalancesInfo(){
	description="Monthly Balances";
	stubby="BALS";
}

DeltasInfo::DeltasInfo(){
	description="Monthly Deltas";
	stubby="DELTAS";
}

JournalInfo::JournalInfo(){
	description="Journal";
	stubby="JOURNAL";
}

ProfitLossInfo::ProfitLossInfo(){
	description="Profit/Loss";
	stubby="PFLS";

	piprefs.reset( new QHaccResultSet( QC::IPICOLS, QC::IPICOLTYPES ) );
	TableRow row( QC::IPICOLS );
	row.set( QC::IPITYPE, CTBOOL );
	row.set( QC::IPIPREF, "ALLACCOUNTSINPROFITLOSS" );
	row.set( QC::IPILABEL, QT_TR_NOOP( "Allow all Account Types Profit/Loss Reports" ) );
	piprefs->add( row );
}

MonthlyBudgetInfo::MonthlyBudgetInfo(){
	description="Monthly Budget";
	stubby="MBUDGET";
}

PayeeInfo::PayeeInfo(){
	description="Total by Payee";
	stubby="PAYEE";
}

TransBalInfo::TransBalInfo(){
	description="Running Balance";
	stubby="RBAL";
}

TransInfo::TransInfo(){
	description="Transactions";
	stubby="TRANS";
}

SharesInfo::SharesInfo(){
	description="Shares of Stock";
	stubby="SHARES";
}
