/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "xmlplugin.h"
#include "qhacctable.h"
#include "qhaccutils.h"
#include "qhacc.h"

#include <sys/stat.h>
#include <zlib.h>

#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>

// plugin factory calls
extern "C" {
	QHaccPlugin * create(){	return new XMLDBPlugin; }
	void destroy( XMLDBPlugin * p ){ delete p; }
}

const XMLInfo XMLDBPlugin::pinfo;

// depths of xml elements
const int XMLDBPlugin::TABLED=2;
const int XMLDBPlugin::ROWD=3;
const int XMLDBPlugin::FIELDD=4;

XMLDBPlugin::XMLDBPlugin() : LocalFileDBPlugin(), QXmlDefaultHandler(){}
XMLDBPlugin::~XMLDBPlugin(){}

const PluginInfo& XMLDBPlugin::info() const { return pinfo; }

bool XMLDBPlugin::iload( QString& err ){ 
	depth=0;
	tableid=-1;

	// read the file as raw bytes because it might be compressed (binary) data
	QFile xmlf( home );
	uint fsz=xmlf.size();
	char * raw=new char[fsz];
	if( xmlf.open( IO_ReadOnly ) ){
		QDataStream( &xmlf ).readRawBytes( raw, fsz );
		xmlf.close();
	}
	else{
		err="could not file file: "+home;
		return false;
	}

	// at this point, we don't know if the file is compressed
	// try to open it as a compressed file initially, but fall back to 
	// an uncompressed file if that doesn't work
	ByteBuffer bb( ( unsigned char * )raw, fsz );
	delete [] raw;

	bool okay=false;
	char * emsg=0;
	ByteBuffer * data=bb.uncompress( okay, emsg );

	if( !okay ){
		// uncompression failed, so assume we have a plain text file
		delete data;
		data=new ByteBuffer( bb );
	}
	
	// hopefully, at this point we have some nice text data
	// we still need to get it into a QTextStream so the
	// XML reader can handle it.
	const uint ds=data->length();
	const char * darr=( const char * )data->array();
	QByteArray arr( ds );
	for( uint i=0; i<ds; i++ ) arr[i]=darr[i];
	QTextStream stream( arr, IO_ReadOnly );

	// I think we're ready to read the data...
	QXmlInputSource source( stream );
	QXmlSimpleReader reader;
	reader.setContentHandler( this );
	reader.parse( source );
	delete data;
	return true;
}

bool XMLDBPlugin::save( QString& err ){
	// for some unexplained reason, we're using SAX to parse the file from
	// disk, and DOM to write the file to disk.
	bool good=true;

	QDomDocument doc;
	QDomProcessingInstruction instr=doc.createProcessingInstruction( "xml", "version=\"1.0\" encoding=\"UTF-8\"" );
	doc.appendChild(instr);

	QDomElement root=doc.createElementNS( "http://qhacc.sourceforge.net",
																				"qhacc" );
	doc.appendChild( root );
	
	for( int i=0; i<QC::NUMTABLES; i++ )
		good=( good && writeTable( doc, root, ( Table )i ) );

	if( good ){
		// compress the output before writing
		ByteBuffer uncomp( doc.toCString() );
		char * emsg=0;
		int cl=engine->getIP( "XMLCOMPRESSIONLEVEL" );
		ByteBuffer * comp=uncomp.compress( cl, good, emsg );
		
		if( good ){
			const unsigned char * arr=comp->array();
			const uint sz=comp->length();

			QFile file( home );
			if( file.open( IO_WriteOnly ) ){
				QDataStream out( &file );
				out.writeRawBytes( ( const char * )arr, sz );
			}
			else err=emsg;
			file.close();
		}
		else good=false;
		delete comp;
	}

	if( good ){
		if( engine->getBP( "KEEPFILEPERMS" ) ) chmod( home, S_IRUSR | S_IWUSR );
	}
	else err="could not save to "+home;
  return good;
}

bool XMLDBPlugin::startElement( const QString&, const QString&,
																const QString& elename, 
																const QXmlAttributes& ){	
	depth++;
	//cout<<depth<<"<-depth elename->"<<elename<<endl;

	// The depths we're expecting are
	// 1) namespace information
	// 2) table name
	// 3) row delimiter
	// 4) field name
	// but use the defined integers TABLED, ROWD, FIELDD
	
	if( depth==TABLED ){
		tableid=-1;
		for( int i=0; i<QC::NUMTABLES; i++ ){
			if( QC::TABLENAMES[i]==elename ){
				tableid=i;
				dbs[tableid]->startLoad();
			}
		}
	}
	else if( depth==ROWD ) currow=TableRow( Utils::tcols( ( Table )tableid ) );
	else if( depth==FIELDD && tableid>=0 ){
		rowidx=Utils::tcnum( ( Table )tableid, elename );
	}
	return true;
}

bool XMLDBPlugin::characters( const QString& str ){
	if( tableid>=0 && depth==FIELDD ) currow.set( rowidx, str );
	return true;
}

bool XMLDBPlugin::endElement( const QString&, const QString&, const QString& ){
	// we're closing a row definition, so add the row to the current table
	if( depth==ROWD ) dbs[tableid]->add( currow );
	else if( depth==TABLED ) dbs[tableid]->stopLoad();
	depth--;
	return true;
}

bool XMLDBPlugin::writeTable( QDomDocument& doc, QDomElement& parent, 
															Table t ) const {
	QString tablen=QC::TABLENAMES[( int )t];

	// the top-level table element
	QDomElement tab=doc.createElement( tablen );
	parent.appendChild( tab );

	uint rr=dbs[( int )t]->rows();
	// each row is another level down
	for( uint i=0; i<rr; i++ ){
		// field delimiters are the tablename minus the plural "s"
		QDomElement row=doc.createElement( tablen.left( tablen.length()-1 ) );
		tab.appendChild( row );

		// write each individual field as its own element
		const int FIELDS=Utils::tcols( t );
		for( int j=0; j<FIELDS; j++ ){
			QDomElement field=doc.createElement( Utils::tcname( t, j ) );
			row.appendChild( field );
			QDomText data=doc.createTextNode( dbs[( int )t]->at( i )[j].gets() );
			field.appendChild( data );
		}
	}
	
	return true;
}

/*** byte buffer stuff ***/
const unsigned int ByteBuffer::BUFFERSIZE=1048576; // un/compress in 1MB chunks
const unsigned int ByteBuffer::WINDOWBITS=31;
const unsigned int ByteBuffer::MAXSIZE=0xffffffff;

ByteBuffer::ByteBuffer( unsigned int i, unsigned int g ){ init( 0, i, g ); }
ByteBuffer::ByteBuffer( const unsigned char * arr, unsigned int asz,
												unsigned int g ){
	init( arr, asz, g );
}
ByteBuffer::ByteBuffer( const ByteBuffer& model ){
	if( this!=&model ){
		init( model.data, model.size, model.growby );
	}
}
ByteBuffer::ByteBuffer( const QByteArray& arr, unsigned int growby ){ 
	const char * arra=arr;
	init( (const unsigned char *)arra, arr.size(), growby ); 
}
ByteBuffer::~ByteBuffer(){ delete [] data; }

void ByteBuffer::add( unsigned char b ){
	if( size+1>=capacity ) grow( 1 );
	data[size++]=b;
}

void ByteBuffer::add( const unsigned char * arr, unsigned int arrsz ){
	if( size+arrsz>=capacity ) grow( arrsz );
	for( unsigned int l=0; l<arrsz; l++ ) data[size++]=arr[l];
}

void ByteBuffer::add( const QByteArray& arr ){
	unsigned int arrsz=arr.size();
	if( size+arrsz>=capacity ) grow( arrsz );
	for( unsigned int i=0; i<arrsz; i++ ) data[size++]=arr[i];
}

void ByteBuffer::grow( unsigned int atleast ){
	// enlarge the internal array by at least atleast indices. We'll do this by
	// growing using the growby size, but if that's 0, we're just going to 
	// double the size of the array. If the additional indices are still less than
	// atleast, we'll just use atleast.
	unsigned int gb=( growby>0 ? growby : size );
	if( gb<atleast ) gb=atleast;

	capacity=size+gb;
	//cout<<"building new internal array, capacity: "<<capacity<<endl;

	unsigned char * tdata=new unsigned char[capacity];
	for( unsigned int l=0; l<size; l++ ) tdata[l]=data[l];
	delete [] data;
	data=tdata;
}

unsigned int ByteBuffer::length() { return size; }
const unsigned char * ByteBuffer::array() { return data; }
unsigned char ByteBuffer::operator[] ( unsigned int i ) const {
	if( i<size ) return data[i];
	return 0;
}
void ByteBuffer::init( const unsigned char * arr, unsigned int arrsz,
											 unsigned int grow ){
	growby=grow;
	bool copyarr=( arr!=0 );
	if( copyarr ){
		size=arrsz;
		capacity=arrsz;
	}
	else{
		size=0;
		capacity=arrsz;
	}
	data=new unsigned char [capacity];
	if( copyarr ) for( unsigned int i=0; i<size; i++ ) data[i]=arr[i];
}

ByteBuffer * ByteBuffer::compress( int level, bool& okay, char * emsg ){
	// if we're not compressing, just return the data
	if( level<1 || level>9 ) return new ByteBuffer( *this );

	ByteBuffer * comp=new ByteBuffer( size/15 ); // assume 15X compression ratio
	okay=true;
	
	z_stream c_stream; // compression stream

	c_stream.zalloc=Z_NULL;
	c_stream.zfree=Z_NULL;
	c_stream.opaque=Z_NULL;

	if( deflateInit2( &c_stream, level, Z_DEFLATED, 
										WINDOWBITS, 8, Z_DEFAULT_STRATEGY )==Z_OK ){
		unsigned char * compr=new unsigned char[BUFFERSIZE];
		c_stream.next_out = compr;
		c_stream.avail_out = BUFFERSIZE;

		c_stream.next_in=( Bytef * )data;
		c_stream.avail_in=size;

		// we only want to append new stuff to the bytebuffer, 
		// so keep track of the last append mark
		unsigned int lasttotout=0; 
		while( c_stream.total_in<size && c_stream.total_out<MAXSIZE ) {
			if( deflate( &c_stream, Z_NO_FLUSH )==Z_OK ){
				// flush the buffer to the ByteBuffer
				comp->add( compr, c_stream.total_out-lasttotout );
				c_stream.avail_out = BUFFERSIZE;
				c_stream.next_out=compr;
			}
			else{
				okay=false;
				emsg=c_stream.msg;
			}

			lasttotout=c_stream.total_out;
		}

		if( deflate( &c_stream, Z_FINISH )==Z_STREAM_END ){
			comp->add( compr, c_stream.total_out-lasttotout );
		}
		else{
			okay=false;
			emsg=c_stream.msg;
		}

		if( deflateEnd( &c_stream )!=Z_OK ){
			okay=false;
			emsg=c_stream.msg;
		}
		delete [] compr;
	}
	else{
		okay=false;
		emsg=c_stream.msg;
	}
	return comp;
}

ByteBuffer * ByteBuffer::uncompress( bool& okay, char * emsg ){
	ByteBuffer * uncomp=new ByteBuffer( size*22 ); // assume 22X inflation ratio
	okay=true;

	z_stream d_stream; // uncompression stream

	d_stream.zalloc=Z_NULL;
	d_stream.zfree=Z_NULL;
	d_stream.opaque=Z_NULL;

	d_stream.next_in=( Bytef * )data;
	d_stream.avail_in=size;
 
	if( inflateInit2( &d_stream, WINDOWBITS )==Z_OK ){
		unsigned char * uncompr=new unsigned char[BUFFERSIZE];

		d_stream.next_out=uncompr;
		d_stream.avail_out=BUFFERSIZE;
		unsigned int lasttotout=0;
		while( d_stream.total_out<ByteBuffer::MAXSIZE && d_stream.total_in<size ) {
			int err=inflate( &d_stream, Z_NO_FLUSH );
			if( err==Z_STREAM_END || err==Z_OK ){
				// flush the buffer to the ByteByffer
				uncomp->add( uncompr, d_stream.total_out-lasttotout );
				if( err==Z_STREAM_END ){
					//cout<<"stream ended"<<endl;
					break;
				}

				d_stream.avail_out=BUFFERSIZE;
				d_stream.next_out=uncompr;
			}
			else{
				okay=false;
				emsg=d_stream.msg;
				break;
			}
			lasttotout=d_stream.total_out;
		}
		if( inflateEnd( &d_stream )!=Z_OK ){
			okay=false;
			emsg=d_stream.msg;
		}
		delete [] uncompr;
	}
	else{
		okay=false;
		emsg=d_stream.msg;
	}

	return uncomp;
}



XMLInfo::XMLInfo(){
	description=stubby="XML";
	targ=SINGLEFILE;
	auto_ptr<QHaccResultSet> tp=LocalFileDBPlugin::pinfo.prefs();
	piprefs=tp;
	TableRow row( QC::IPICOLS );
	row.set( QC::IPIPREF, "XMLCOMPRESSIONLEVEL" );
	row.set( QC::IPITYPE, CTINT );
	row.set( QC::IPILABEL, "XML Compression Level" );
	piprefs->add( row );
}

