/************************** * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * ***************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#else
#define VERSION 3.1
#endif

#include "qhacccli.h"
#include "qhacc.h"
#include "qhaccext.h"
#include "qhaccutils.h"
#include "qhacctable.h"
#include "guiconstants.h"
#include "qhaccsegmenter.h"
#include "qhaccconstants.h"
#include "localfileplugin.h"

#include <qdir.h>

#include <stdlib.h>
#include <sys/stat.h> 

QHaccCLI::QHaccCLI( QHacc * eng ){ engine=eng; }
QHaccCLI::~QHaccCLI(){}

bool QHaccCLI::save(){
	QString err;
	std::ostream * str=0;
	bool good=engine->save( err );
	if( !good && Utils::error( Utils::ERRFATAL, str ) ) *str<<err<<endl;
	return good;
}

bool QHaccCLI::verify( bool andFix ){
	if( QHaccExt( engine ).verify( andFix ) )	return save();
	return false;
}

bool QHaccCLI::exp( const char * dirname ){
	if( QHaccExt( engine ).exp( dirname ) ) return save();
	return false;
}

bool QHaccCLI::imp( const char * dirname ){
	if( QHaccExt( engine ).imp( dirname ) ) return save();
	return false;
}

bool QHaccCLI::reco( const char * fname ){
	if( QHaccExt( engine ).reco( fname ) ) return save();
	return false;
}

void QHaccCLI::create( const char * homename ){
	std::ostream * str=0;
	if( Utils::debug( Utils::PRINT, str ) )
		*str<<QHaccExt( engine ).create( homename )<<endl;
}

void QHaccCLI::report( const char * homename ){
	std::ostream * str=0;
	
	if( Utils::debug( Utils::PRINT, str ) ){
		std::ostream * err=0;
		bool writerrs=Utils::error( Utils::ERRFATAL, err );
		bool cont=true;
		
		QHaccPlugin * tpi=0;
		QString homer=engine->getPluginFor( QHacc::PIREPORTER, homename, tpi );
		if( tpi==0 ){
			if( writerrs ) *err<<"No plugin for: "<<homename<<endl;
			cont=false;
		}
		
		if( cont ){
			QHaccReportPlugin * pi=( QHaccReportPlugin * )tpi;
			pi->setup( engine );

			QString tit, emsg;
			auto_ptr<QHaccResultSet> data=pi->generate( homer, tit, emsg, cont );

			if( cont ) *str<<pi->writereport( tit, data.get() )<<endl;
			else if( writerrs ) *err<<emsg<<endl;

			engine->destroyPlugin( QHacc::PIREPORTER, tpi );
		}
	}
}

void QHaccCLI::plugins(){
	std::ostream * str=0;
	if( Utils::debug( Utils::PRINT, str ) ){
		QString output( "QHacc Available Plugins" );
		const char * titles[]={ "Database", "Import", "Export",
														"Reporting", "Graphing" };
		int types[]={ QHacc::PIDATABASE, QHacc::PIIMPORTER, QHacc::PIEXPORTER,
									QHacc::PIREPORTER, QHacc::PIGRAPHER };
		
		for( int p=0; p<5; p++ ){
			int curr=0;
			vector<PluginInfo>pis( engine->getPluginInfo( types[p], &curr ) );
			
			output.append( "\n" ).append( titles[p] ).append( ":" );
			int i=0;
			for ( vector<PluginInfo>::iterator it=pis.begin(); it!=pis.end(); it++ ){
				output+="\n  ";
				if( i++==curr ) output+="*";
				else output+=" ";
				output+=QString( "%1 (%2): %3" ).arg( it->stub() ).arg( it->descr() ).arg( it->fname() );
			}
		}	
		
		*str<<output<<endl;
	}
}

void helpQuit( const char * progname ){
  cout<<"usage: "<<progname<<" [options]"<<endl
			<<"options:"<<endl
			<<" [-l             <Language Directory>   set QHacc's locale"<<endl
			<<" [--debug        <Level>]               set debugging output"<<endl
      <<" [--report       <Report ID>]           generate report"<<endl
			<<" [--import       <QHACC_HOME>]          import data from QHACC_HOME"<<endl
			<<" [--export       <Directory>]           export data to directory"<<endl
			<<" [--reconcile    <QHACC_HOME>           reconcile transactions from QHACC_HOME"<<endl
			<<" [--create       <QHACC_HOME>]          make a QHACC_HOME creation script"<<endl
			<<" [--verifydb]                           verify the integrity of the database"<<endl
			<<" [--fixdb]                              fix database, if possible"<<endl
      <<" [--plugins]                            list available plugins"<<endl;
  exit ( EXIT_SUCCESS );
}

char * opt( int argc, char ** argv, int& i ){
	i++;
	if( i<argc ) return argv[i];
	else helpQuit( argv[0] );
	return 0;
}

bool cmp( const char * model, const char * test ){
	return ( strcmp( model, test )==0 );
}

int main( int argc, char **argv ){
	//for( int i=0; i<argc; i++ ) cout<<argv[i]<<" ";
	//cout<<endl;

	char * qroot=getenv( "QHACC_ROOT" );
	char * qhome=getenv( "QHACC_HOME" );

	if( !qroot ){
		cerr<<"QHACC_ROOT environment variable not set"<<endl;
		return EXIT_FAILURE;
	}
	if( argc<2 ) helpQuit( argv[0] );

	QHacc qhacc( qroot );
	QHaccCLI cli( &qhacc );

	const char * langdir=qhacc.languagedir();
	if( !langdir ) langdir=getenv( "QHACC_LANGDIR" );

	if( cmp( argv[1], "--create" ) ){
		int i=1;
		cli.create( opt( argc, argv, i ) );
	}
	else if( cmp( argv[1], "--plugins" ) ) cli.plugins();
	else{
		//if( !qhacc.setHome( qhome ) ) return EXIT_FAILURE;

		// parse the command line. If debug is the first arg, set the home AFTER
		// setting debug. If not, set home before doing anything else
		for( int i=1; i<argc; i++ ){
			if( i==1 ){
				if( cmp( argv[i], "--debug" ) ){
					if( i+1>=argc ) helpQuit( argv[0] );
					Utils::setDebug( QString( argv[i+1] ).toInt() );
				}
				if( !qhacc.setHome( qhome ) ) return EXIT_FAILURE;
			}

			if( cmp( argv[i], "--debug" ) ){
				Utils::setDebug( QString( opt( argc, argv, i ) ).toInt() );
			}		
			else if( cmp( argv[i], "-l" ) ) langdir=opt( argc, argv, i );
			else if( cmp( argv[i], "--report" ) )	cli.report( opt( argc, argv, i ) );
			else if( cmp( argv[i], "--reconcile" ) )cli.reco( opt( argc, argv, i ) );
			else if( cmp( argv[i], "--export" ) ) cli.exp( opt( argc, argv, i ) );
			else if( cmp( argv[i], "--import" ) ) cli.imp( opt( argc, argv, i ) );
			else if( cmp( argv[i], "--verifydb" ) ) cli.verify( false );
			else if( cmp( argv[i], "--fixdb" ) ) cli.verify( true );
			else helpQuit( argv[0] );
		}
	} // else ( if is --create loop)

	return EXIT_SUCCESS;
}
