/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2005 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************* * * * * * * * * * * * * ***************************/

#include "qhaccsubsetwin.h"
#include "qhacc.h"
#include "plugins.h"
#include "qhaccview.h"
#include "qhaccutils.h"
#include "qhacctable.h"
#include "qhacclineedits.h"
#include "qhaccconstants.h"
#include "qhaccsubsetwin.moc"

#include <qlabel.h>
#include <qlayout.h>
#include <qcombobox.h>
#include <qpushbutton.h>

const int QHaccSubsetWin::NTRANS=5;
const int QHaccSubsetWin::ADELTA=6;
const int QHaccSubsetWin::LTRANS=7;
const int QHaccSubsetWin::STRANS=8;
const int QHaccSubsetWin::ATRANS=9;

const int QHaccSubsetWin::NUMDISPLAYS=5;

QHaccSubsetWin::QHaccSubsetWin( QHacc * e, QWidget * p, const char * n )
	: QWidget( p, n, WDestructiveClose ) {
  
	engine=e;
	journalid=0;
  view=new QHaccView( e, this );
	view->setPrefBase( "SUBVIEW" );

  QPushButton * upd=new QPushButton( tr( "Update" ), this );
  QPushButton * ok=new QPushButton( tr( "Dismiss" ), this );

  connect( ok, SIGNAL( clicked() ), SLOT( close() ) );
  connect( upd, SIGNAL( clicked() ), SLOT( changeAll() ) );
  
  QGridLayout * layout=new QGridLayout( this, 15, 20, 5 );
	int row=0;

	sels=new TableSelectDisplay * [NUMDISPLAYS];
	for( int i=0; i<NUMDISPLAYS; i++ ){
		sels[i]=new TableSelectDisplay( TableSelect(), this );
		connect( sels[i], SIGNAL( updated() ), SLOT( changeAll() ) );
		layout->addMultiCellWidget( sels[i], row, row, 0, 2 );		
		row++;
	}
	
	layout->addMultiCellWidget( upd, row, row, 0, 2 );
	row++;
	layout->addMultiCellWidget( ok, row, row, 0, 2 );
	row++;
	
	plugin=0;
	QHaccPlugin * pi=0;
	engine->getPluginFor( QHacc::PIREPORTER, "AVES:", pi );
	if( pi!=0 ){
		plugin=( QHaccReportPlugin * )pi;
		plugin->setup( engine );
		infosize=NUMDISPLAYS;
	}
	else infosize=0;
	
	const char * infos[]={ QT_TR_NOOP( "Transactions:" ),
												 QT_TR_NOOP( "Delta:" ),
												 QT_TR_NOOP( "Largest:" ),
												 QT_TR_NOOP( "Smallest:" ),
												 QT_TR_NOOP( "Average:" ) };
	
	info=new QLabel * [infosize*2];
	for( uint i=0; i<infosize; i++ ){
		info[i]=new QLabel( infos[i], this );
		info[i]->setAlignment( AlignLeft );
		info[i+infosize]=new QLabel( this );
		info[i+infosize]->setAlignment( AlignRight );
		layout->addMultiCellWidget( info[i], row, row, 0, 1, AlignLeft );
		layout->addMultiCellWidget( info[i+infosize], row, row, 1, 2, AlignRight );
		row++;
	}


	layout->addMultiCellWidget( view, 0, 14, 3, 19 );

	readPrefs( true );
}

QHaccSubsetWin::~QHaccSubsetWin(){
	delete [] info; 
	engine->destroyPlugin( QHacc::PIREPORTER, plugin );
}

void QHaccSubsetWin::setAL( const Account& a, const Journal& l ){
	view->setJournal( l );
	view->setAccount( a );
	if( !( a.isNull() || l.isNull() ) ){
		setCaption( tr( "Subset View of %1/%2" ).arg( l.gets( QC::LNAME ) ).arg( a.gets( QC::ANAME ) ) );
		account=a;
		journalid=l.getu( QC::LID );
	}
}

void QHaccSubsetWin::readPrefs( bool initial ){
	view->readPrefs( initial );
	if( initial ){
		connect( engine, SIGNAL( changedP( const QString&, QString ) ),
						 this, SLOT( changeP( const QString&, QString ) ) );
		connect( engine, SIGNAL( changedP( const QString&, int ) ),
						 this, SLOT( changeP( const QString&, int ) ) );
		connect( engine, SIGNAL( removedA( const Account& ) ),
						 this, SLOT( removeA( const Account& ) ) );
	}

	QString str=engine->getSP( "SUBSETWINSIZE" );
	if( !str.isEmpty() ){
		QString vals[2];
		Utils::parser( str, " ", 0, vals, 2 );
		resize( vals[0].toInt(), vals[1].toInt() );
	}

	// set the starting date equal to the graphtime limit
	//QDate de=enddate->getDate();
	//startdate->setDate( de.addMonths( 0-engine->getIP( "GRAPHTIME" ) ) );
}

void QHaccSubsetWin::save(){
	engine->setSP( "SUBSETWINSIZE",	QString::number( width() )+
								 " "+QString::number( height() ) );
	view->save();
}

void QHaccSubsetWin::changeP( const QString& str, QString ){
	if( str=="DATESEPARATOR" ){
		//enddate->refresh();
		//startdate->refresh();
	}
}

void QHaccSubsetWin::changeP( const QString& str, int ){
	if( str=="DATEFORMAT" ){
		//enddate->refresh();
		//startdate->refresh();
	}
}

void QHaccSubsetWin::changeAll() {
	vector<TableSelect> ts;
	for( int i=0; i<NUMDISPLAYS; i++ ){
		TableSelect s=sels[i]->get();
		if( s.column()>=0 )	ts.push_back( s );
	}

	setSelectors( ts );
}

void QHaccSubsetWin::setSelectors( vector<TableSelect> tss ){
	uint tssz=tss.size();
	uint vecidx=0;
	uint selidx=0;

	//for( uint i=0; i<tssz; i++ ) cout<<tss[i].toString()<<endl;

	while( selidx<( uint )NUMDISPLAYS && vecidx<tssz ){
		int col=tss[vecidx].column();
		// don't display column selections that can't be changed
		if( !( col==QC::XTLID || col==QC::XSACCTID || col==QC::XTTYPE ) )
			sels[selidx++]->set( tss[vecidx] );
		vecidx++;
	}
	
	view->setSelectors( tss );
	setInfo( tss );
}

void QHaccSubsetWin::setInfo( vector<TableSelect> tss ){
	if( !plugin ) return;

	if( !account.isNull() ) tss.push_back( TableSelect( QC::XSACCTID,
																											account[QC::AID] ) );
	tss.push_back( TableSelect( QC::XTLID, TableCol( journalid ) ) );

	QString title;
	QHaccResultSet accts( QC::ACOLS, QC::ACOLTYPES );
	accts+=account;
	auto_ptr<QHaccResultSet> infor=plugin->generate( &accts, tss, title );
		
	// we only want to display some of the rows from the averages report
	uint rows[]={ 0, 1, 2, 3, 4 };
	uint lkp[]={ NTRANS, ADELTA, LTRANS, STRANS, ATRANS };
	for( uint i=0; i<infosize; i++ ){
		TableRow data=infor->at( rows[i] );
		info[lkp[i]]->setText( data[1].gets() );
	}
}

void QHaccSubsetWin::close(){
	save();
	QWidget::close( true );
}

void QHaccSubsetWin::removeA( const Account& a ){
	if( a[QC::AID]==account[QC::AID] ) close();
}

QHaccView * QHaccSubsetWin::getView() const { return view; }








TableSelectDisplay::TableSelectDisplay( const TableSelect& t, QWidget * w,
																				const char * n ) : QWidget( w, n ) {

	QGridLayout * layout=new QGridLayout( this, 1, 6 );
	col=new QComboBox( false, this );
	col->insertItem( tr( "<none>" ) );
	int XCOLS[]={ QC::XTID, QC::XSID, QC::XSRECO, QC::XTNUM, QC::XTPAYEE,
								QC::XTMEMO, QC::XSSUM, QC::XTDATE, QC::XSRECODATE };
	for( int i=0; i<9; i++ )
		col->insertItem( Utils::tcname( XTRANS, XCOLS[i] ) );

	sel=new QComboBox( false, this );
	const char * sels[]={ "=", "!=", "<" ,"<=", ">", ">=" };
	for( int i=0; i<6; i++ ) sel->insertItem( sels[i] );
	
	val=new QHaccLineEdit( this );
	val->setEmptyText( tr( "<value>" ) );
	connect( val, SIGNAL( returnPressed() ), SLOT( upd() ) );

	layout->addMultiCellWidget( col, 0, 0, 0, 0 );
	layout->addMultiCellWidget( sel, 0, 0, 1, 1 );
	layout->addMultiCellWidget( val, 0, 0, 2, 2 );
	
	set( t );
}

TableSelectDisplay::~TableSelectDisplay(){}

void TableSelectDisplay::upd(){ emit updated(); }

TableSelect TableSelectDisplay::get() const {
	int mycol=-1;
	QString column=col->currentText();
	
	for( int i=0; i<QC::XCOLS; i++ )
		if( Utils::tcname( XTRANS, i )==column ) mycol=i;
	
	int mycheck=TableSelect::EQ;
	QString check=sel->currentText();
	const char * sels[]={ "=", "!=", "<" ,"<=", ">", ">=" };
	const int chks[]={ TableSelect::EQ, TableSelect::NE, TableSelect::LT,
										 TableSelect::LE, TableSelect::GT, TableSelect::GE };
	for( int i=0; i<6; i++ ) if( check==sels[i] ) mycheck=chks[i];

	TableCol mytc( val->text() );
	
	return TableSelect( mycol, mytc, mycheck );
}

void TableSelectDisplay::set( const TableSelect& t ){
	int column=t.column();
	if( column<0 ){
		col->setCurrentText( tr( "<none>" ) );
		return;
	}
	col->setCurrentText( Utils::tcname( XTRANS, column ) );


	int check=t.check();
	const char * sels[]={ "=", "!=", "<" ,"<=", ">", ">=" };
	const int chks[]={ TableSelect::EQ, TableSelect::NE, TableSelect::LT,
										 TableSelect::LE, TableSelect::GT, TableSelect::GE };
	for( int i=0; i<6; i++ ) if( check==chks[i] ) sel->setCurrentText( sels[i] );


	val->setText( t.model().gets() );
}
