/***************************************************************************
 *   Copyright (C) 2005 by Krzysztof Zawadyl                               *
 *   k.zawadyl@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "messagetable.h"
#include "metamonitor.h"
#include "message.h"

#include <qapplication.h>
#include <qcursor.h>
#include <qlistview.h>
#include <qclipboard.h>
#include <qpushbutton.h>

#include <kpopupmenu.h>
#include <klistview.h>
#include <klocale.h>
#include <kiconloader.h>
#include <qwidget.h>

/**
 * @brief Constructor
 * 
 * @param parent window's parent
 * @param name window's name
 */
MessageTable::MessageTable( MetaMonitor *parent, const char *name ) : MessageTableBase( parent, name )
{
	// Context menu
	menu = new KPopupMenu(this,"menu");
	
	menu->insertItem( i18n("&Copy to clipboard"), this, SLOT( copySelected() ) );
	menu->insertSeparator();
	menu->insertItem( i18n("&Select all"), this, SLOT( selectAll() ) );
	menu->insertItem( i18n("&Invert selection"), this, SLOT( invertSelection() ) );
	menu->insertSeparator();
	menu->insertItem( i18n( "&Delete messages"), this, SLOT( deleteMessages() ) );
	
	connect( configureButton, SIGNAL(clicked()), parent, SLOT(configure()) );
	connect( mMessageTable, SIGNAL( rightButtonPressed(QListViewItem *, const QPoint &, int) ), 
	         this, SLOT( showMenu( QListViewItem *, const QPoint &, int) ) );
}

/**
 * @brief Destructor
 */
MessageTable::~MessageTable()
{
}

/**
 * @brief Adds a new message
 * 
 * Message is added to the ListView
 * 
 * @param m Message's object
 */
void MessageTable::addMessage( Message *m )
{
	QListViewItem *item = new QListViewItem( (QListView *)mMessageTable, m->getTime(), m->getFrom(), m->getText() );
	//if( m->getIcon() != "metamonitor" )
	{
		QPixmap pixmap( MainBarIcon(m->getIcon()) );
		item->setPixmap( 0, pixmap );
	}
	
	mMessageTable->insertItem( item );
}

/**
 * @brief Shows the context menu of a ListView
 * 
 * @param item selected item (unused)
 * @param pnt point of click (unused)
 * @param col column's number of click (unused)
 */
void MessageTable::showMenu( QListViewItem *item, const QPoint &pnt, int col )
{
	menu->exec( QCursor::pos() );
}

/**
 * @brief Copies selected rows to clipboard
 */
void MessageTable::copySelected()
{
	QClipboard *cb = QApplication::clipboard();
	QString selection = "";
 
	QListViewItemIterator it( mMessageTable );
	while ( it.current() ) 
	{
		QListViewItem *item = it.current();
		
		if ( item->isSelected() )
			selection += item->text( 0 ) + " [" + item->text( 1 ) + "] " + item->text( 2 ) + "\n";
		
		++it;
	}
	
	cb->setText( selection, QClipboard::Clipboard );
}

/**
 * @brief Selects all rows
 */
void MessageTable::selectAll()
{
	mMessageTable->selectAll( true );
}
/**
 * @brief Inverts selection
 */
void MessageTable::invertSelection()
{
	mMessageTable->invertSelection();
}

/**
 * @brief Clears the message table
 */
void MessageTable::deleteMessages()
{
	mMessageTable->clear();
}
#include "messagetable.moc"
