/***************************************************************************
 *   Copyright (C) 2005 by Krzysztof Zawadyl                               *
 *   k.zawadyl@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "metamonitor.h"
#include "filemonitor.h"
#include "message.h"
#include "messagetable.h"
#include "popup.h"
#include "configure.h"
#include "dcoplistener.h"

#include <qpixmap.h>
#include <qcursor.h>
#include <qptrlist.h>

#include <ksystemtray.h>
#include <kpopupmenu.h>
#include <kaction.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kapp.h>
#include <kaboutapplication.h>
#include <kstandarddirs.h>
#include <kglobal.h>
#include <kconfig.h>


/**
 * @brief MetaMonitor's constructor
 * Creates application, applies settings and so on :)
 */
MetaMonitor::MetaMonitor() : KSystemTray(), configDlg(0)
{
	mIcon = loadIcon( "metamonitor" );
    setPixmap( mIcon );

	Config().mm = this;
	Config().read();
	soundPlayProcess = new KProcess( this );
	
    // Context menu
    KPopupMenu* menu = contextMenu();
    menu->setCheckable( TRUE );
    menu->insertItem( i18n( "&View logs" ), this, SLOT( messageTable() ) );
    pauseId = menu->insertItem( i18n( "&Pause popups" ),
                                this, SLOT( switchPausePopups() ) );
	menu->setItemChecked( pauseId, Config().pausePopups );
    menu->insertSeparator();
    KAction* configure = KStdAction::preferences( this, SLOT( configure() ),
                                                  actionCollection(), "configure" );
    configure->plug( menu );
    menu->insertItem( i18n("&About"), this, SLOT( about() ) );
    connect( this, SIGNAL( quitSelected() ), kapp, SLOT( quit(void) ) );
    messageViewer = new MessageTable( this );

	// prepare popup
	pop = new Popup(this);

    // initialize log file to watch
	fileMonitor = new FileMonitor( this, Config().logFile, Config().updateInterval );
    connect( fileMonitor, SIGNAL(messageReceived(Message *)),
             this, SLOT(addMessage(Message *)) );

	// launch DCOP listener
    listener = new DCOPListener( this );
}

/**
 * @brief MetaMonitor's descructor
 * Closes timer, file and popup
 */
MetaMonitor::~MetaMonitor()
{
    if( fileMonitor )
        delete fileMonitor;

    if( pop )
        delete pop;
	
	if( soundPlayProcess->isRunning() )
		soundPlayProcess->kill();
	delete soundPlayProcess;
	
	if( configDlg )
		delete configDlg;

	delete listener;
}

/**
 * @brief mouse click on tray-icon reaction
 *
 * If it's left button - show the messages list
 * If right - show the context menu
 */
void MetaMonitor::mousePressEvent(QMouseEvent* ev)
{
	if( ev->button() == Qt::RightButton )
		contextMenu()->exec( QCursor::pos() );
	else
	if( ev->button() == Qt::LeftButton )
			messageTable();
}

/**
 * @brief Pause/unpause popup messaging
 * 
 * It saves the configuration after changing the pausePopups' value
 */
void MetaMonitor::switchPausePopups()
{
	Config().pausePopups = !Config().pausePopups;
	contextMenu()->setItemChecked( pauseId, Config().pausePopups );
	Config().write();
}

/**
 * @brief show/hide the message list window
 */
void MetaMonitor::messageTable()
{
	messageViewer->setHidden( messageViewer->isVisible() );
}

/**
 * @brief Adds a single message
 * 
 * Adds a single message and shows popup. Message is deleted after adding to
 * widgets.
 */
void MetaMonitor::addMessage( Message *m )
{
	m->setLifetime( Config().popupsLifetime );
	m->setFont( this->font() );
	
    // check if we got some triggers
    // only the first matching trigger's action is performed!
	MTrigger *mt;
	QPtrListIterator<MTrigger> it( Config().triggerList );
	while( (mt=it.current()) != 0 )
	{
		if( mt->check( m ) )
		{
			mt->perform( m );
			break;
		}
		++it;
	}
	
	messageViewer->addMessage(m);
	
	if( !Config().pausePopups && m->showPopup() )
	{
		if( Config().groupMessages )
			pop->addMessage( m, Config().maxMessages, Config().resetPopupsTimers );
		else
			Popup::message( this, m );
	}
	
	delete m;
}

/**
 * @brief runs the configure window
 */
void MetaMonitor::configure()
{
	if( !configDlg )
	{
		configDlg = new Configure(this);
		connect(configDlg, SIGNAL(applyClicked()), this, SLOT(applyConfig()) );
	}
	
	int result = configDlg->exec();
	
	if( result == QDialog::Accepted )
	{
		applyConfig();
	}
}

void MetaMonitor::applyConfig()
{
	if( !configDlg )
		return;
	
	QString oldFile = Config().logFile;
	int oldUpdateInterval = Config().updateInterval;
	
	configDlg->updateConfig();
	
	if( oldUpdateInterval != Config().updateInterval )
		fileMonitor->changeInterval( Config().updateInterval );
	
	if( oldFile != Config().logFile )
	{
		delete fileMonitor;
		fileMonitor = new FileMonitor( this, Config().logFile, Config().updateInterval );
		connect( fileMonitor, SIGNAL(messageReceived(Message *)),
		         this, SLOT(addMessage(Message *)) );
	}
	
	Config().write();
}
/**
 * @brief runs the about dialog
 */
void MetaMonitor::about()
{
    KAboutApplication dlg( this );
    dlg.exec();
}

void MetaMonitor::playSoundFile( QString file )
{
	/*	if( soundPlayProcess->isRunning() )
	{
		if( !soundPlayProcess->kill() )
		{
			qDebug( "Couldn't kill soundProcess, trying SIGKILL" );
			if( !soundPlayProcess->kill(SIGKILL) )
				qDebug( "Couldn't kill soundProcess" );
		}
	}*/
	
	QString cmd = Config().soundPlayerCmd;
	QStringList sl = QStringList::split( " ", cmd );
	
	soundPlayProcess->clearArguments();
	*soundPlayProcess << sl << file;
	
	if( !soundPlayProcess->start() )
		qDebug( "Failed to run soundPlayProcess" );
}

#include "metamonitor.moc"
