/***************************************************************************
 *   Copyright (C) 2005 by Krzysztof Zawadyl                               *
 *   k.zawadyl@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "popup.h"
#include "message.h"
#include "mconfiguration.h"
#include "popuppixmap.h"

#include <qlabel.h>
#include <qvbox.h>
#include <qhbox.h>
#include <qlayout.h>
#include <qpixmap.h>
#include <qptrlist.h>
#include <qevent.h>
#include <qpainter.h>
#include <qrect.h>
#include <qtooltip.h>
#include <qpushbutton.h>
#include <qimage.h>
#include <qtimer.h>
#include <qwidget.h>

#include <kdialog.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kpixmapeffect.h>
#include <ksharedpixmap.h>
#include <kglobalsettings.h>

#include <netwm.h> 
/**
 * @brief Constructor
 *
 * Creates a popup
 *
 * @param parent pointer to MetaMonitor
 */
Popup::Popup( QWidget *parent ) : KPassivePopup( parent , 0, Qt::WStyle_Customize |
                                                 Qt::WDestructiveClose | WResizeNoErase |
                                                 WNoAutoErase | WPaintDesktop |
                                                 Qt::WX11BypassWM | Qt::WStyle_StaysOnTop |
                                                 Qt::WStyle_NoBorder ),
window( parent ? parent->winId() : 0L )
{
	clearWFlags(Qt::WStyle_Tool);
	
	setTimeout(0);
	messageQueue.clear();
	
	setFocusPolicy(NoFocus);
	
    //	setBackgroundColor(QColor(0,0,0));
	setBackgroundMode( QWidget::NoBackground );
	setPalette( QToolTip::palette() );
	
	resize( 0, 0 );
	hide();
	
	topLayout = new QVBoxLayout( this, 0, 0 );
	topLayout->setAutoAdd( true );
	//topLayout->setSpacing( 0 );
	//topLayout->setMargin( KDialog::marginHint() );
}

/**
 * @brief Destructor
 */
Popup::~Popup()
{
	messageQueue.clear();
	
	if( topLayout )
		delete topLayout;
}

void Popup::positionSelf()
{
	NETWinInfo ni( qt_xdisplay(), window, qt_xrootwin(), NET::WMKDESystemTrayWinFor );
	NETRect frame, win;
	ni.kdeGeometry( frame, win );

	int x = win.pos.x;
	int y = win.pos.y;
	int w = size().width();
	int h = size().height();
	
	QRect r = KGlobalSettings::desktopGeometry(QPoint(x+w/2,y+h/2));

	// determine the tray's position and move the popup so it
	// doesn't override the kicker
	if( x > r.width()/2 )
		x -= w;
	else
		x = win.size.width;
	
	if( y > r.height()/2 )
		y -= h;
	else
		y += win.size.height;
	
	move( x, y );
}

/**
 * @brief Paint event handler
 *
 * To make the popup nice (some code taken from kickerTip :) )
 *
 * @param e the paint event
 */
void Popup::paintEvent( QPaintEvent *e )
{
	QRect r = rect();
	
	if( !Config().themesEnabled )
	{
		int w = size().width();
		int h = size().height();
		
		m_mask.resize( w, h );
		m_pixmap.resize( w, h );
		
		QPainter maskPainter(&m_mask);
		m_mask.fill( Qt::black );
		maskPainter.setBrush( Qt::white );
		maskPainter.setPen( Qt::white );
		maskPainter.drawRoundRect( r, 1600/w, 1600/h );
		setMask( m_mask );
		
		QPainter bufferPainter(&m_pixmap);
		//bufferPainter.setPen(Qt::black);
		bufferPainter.setBrush( colorGroup().brush( QColorGroup::Background ) );
		bufferPainter.drawRoundRect( r, 1600/w, 1600/h );
	}
	
	bitBlt( this, r.topLeft(), &m_pixmap, r, Qt::CopyROP );
}

void Popup::render(PopupTheme *theme)
{
	int w = size().width();
	int h = size().height();
	
	unsetPalette();
	topLayout->setMargin( theme->marginWidth );

	m_pixmap = m_screenshot;
	
	QPainter bufferPainter(&m_pixmap);

	QRect tlRect( 0, 0,
	              theme->leftMarginWidth, theme->topMarginHeight );
	bufferPainter.drawPixmap( tlRect, theme->topLeftPix );
	
	QRect trRect( w-theme->rightMarginWidth, 0,
	              theme->rightMarginWidth, theme->topMarginHeight );
	bufferPainter.drawPixmap( trRect, theme->topRightPix );
	
	QRect tRect( theme->rightMarginWidth, 0,
	             w-theme->rightMarginWidth-theme->leftMarginWidth, theme->topMarginHeight );
	bufferPainter.drawTiledPixmap( tRect, theme->topCenterPix );
	
	QRect blRect( 0, h-theme->bottomMarginHeight,
	              theme->leftMarginWidth, theme->bottomMarginHeight );
	bufferPainter.drawPixmap( blRect, theme->bottomLeftPix );
	
	QRect brRect( w-theme->rightMarginWidth, h-theme->bottomMarginHeight,
	              theme->rightMarginWidth, theme->bottomMarginHeight );
	bufferPainter.drawPixmap( brRect, theme->bottomRightPix );
	
	QRect bRect( theme->rightMarginWidth, h-theme->bottomMarginHeight,
	             w-theme->rightMarginWidth-theme->leftMarginWidth, theme->bottomMarginHeight );
	bufferPainter.drawTiledPixmap( bRect, theme->bottomCenterPix );
	
	QRect lRect( 0, theme->topMarginHeight,
	             theme->leftMarginWidth, h-theme->topMarginHeight-theme->bottomMarginHeight );
	bufferPainter.drawPixmap( lRect, theme->centerLeftPix );
	
	QRect rRect( w-theme->rightMarginWidth, theme->topMarginHeight,
	             theme->rightMarginWidth, h-theme->topMarginHeight-theme->bottomMarginHeight );
	bufferPainter.drawPixmap( rRect, theme->centerRightPix );
	
	QRect cRect( theme->rightMarginWidth, theme->topMarginHeight,
	             w-theme->rightMarginWidth-theme->leftMarginWidth,
	             h-theme->topMarginHeight-theme->bottomMarginHeight );
	bufferPainter.drawPixmap( cRect, theme->centerCenterPix );
	
	QPtrListIterator<PopupPixmap> apit( theme->additionalPixmapsList );
	PopupPixmap *pp;
	while( (pp = apit.current()) != 0 )
	{
		int px=0;
		int py=0;
		int pw = pp->width();
		int ph = pp->height();

		switch( pp->xAlign )
		{
			case PopupPixmap::xRight:
				px = w-pw+pp->xOffset;
				break;
			case PopupPixmap::xCenter:
				px = w/2-pw/2+pp->xOffset;
				break;
			case PopupPixmap::xLeft:
				px = pp->xOffset;
				break;
		}

		switch( pp->yAlign )
		{
			case PopupPixmap::yBottom:
				py = h-ph+pp->yOffset;
				break;
			case PopupPixmap::yCenter:
				py = h/2-ph/2+pp->yOffset;
				break;
			case PopupPixmap::yTop:
				py = pp->yOffset;
				break;
		}

		QRect pRect( px, py, pw, ph );
		bufferPainter.drawPixmap( pRect, *pp );
		++apit;
	}

	bufferPainter.end();
	
	QPtrListIterator<PopupMessage> it( messageQueue );
	PopupMessage *pm;
	while( (pm = it.current()) != 0 )
	{
		pm->setPaletteBackgroundPixmap(m_pixmap);
		++it;
	}
	
	// now, let's draw the mask
	QBitmap mask( w, h );
	mask.fill( Qt::white );
	QPainter maskPainter( &mask );
	maskPainter.drawPixmap( QPoint( w-theme->topRightMask.width(), 0 ),
	                        theme->topRightMask );
	maskPainter.drawPixmap( QPoint( 0, 0 ),
	                        theme->topLeftMask );
	maskPainter.drawPixmap( QPoint( 0, h-theme->bottomLeftMask.height() ),
	                        theme->bottomLeftMask );
	maskPainter.drawPixmap( QPoint( w-theme->bottomRightMask.width(),
	                                h-theme->bottomRightMask.height() ),
	                        theme->bottomRightMask );
	
	maskPainter.drawTiledPixmap( theme->leftMarginWidth, 0,
	                        w-theme->leftMarginWidth-theme->rightMarginWidth, theme->topMarginHeight,
	                        theme->topMask );
	maskPainter.drawTiledPixmap( theme->leftMarginWidth,
	                        h-theme->bottomMarginHeight,
	                        w-theme->leftMarginWidth-theme->rightMarginWidth, theme->bottomMarginHeight,
	                        theme->bottomMask );
	maskPainter.drawTiledPixmap( 0, theme->topMarginHeight,
	                        theme->leftMarginWidth,
	                        h-theme->topMarginHeight-theme->bottomMarginHeight,
	                        theme->leftMask );
	maskPainter.drawTiledPixmap( w-theme->rightMarginWidth, theme->topMarginHeight,
	                        theme->rightMarginWidth,
	                        h-theme->topMarginHeight-theme->bottomMarginHeight,
	                        theme->rightMask );
	maskPainter.end();
	setMask( mask );
}

/**
 * @brief Adds a message to MessageQueue
 *
 * This constructs a PopupMessage object from a Message and appends it to the messageQueue list
 *
 * @param m Pointer to a message object
 */
void Popup::addMessage(Message *m, int max=0, bool resetTimers=false)
{
	if( max && messageQueue.count() >= max )
	{
		PopupMessage *pm = messageQueue.take( 0 );
		topLayout->remove( pm );
		delete pm;
	}
	
	PopupMessage *pm;
	if( resetTimers )
	{
		QPtrListIterator<PopupMessage> it( messageQueue );
		while( (pm = it.current()) != 0 )
		{
			pm->resetTimer();
			++it;
		}
	}
	pm = new PopupMessage( m, this );
	
	messageQueue.append( pm );
	
	connect( pm, SIGNAL( clicked(PopupMessage *) ),
	         this, SLOT( removeMessage(PopupMessage *) ) );
	connect( pm, SIGNAL( timeout(PopupMessage *) ),
	         this, SLOT( removeMessage(PopupMessage *) ) );
	
	pm->setTimeout(m->getLifetime());
	pm->startTimer();
	pm->show();
	
	updateBg();
}

/**
 * @brief Removes the message
 *
 * The message is removed from the list and deleted.
 * It shows undeleted messages once again after that.
 *
 * @param pm PopupMessage pointer
 */
void Popup::removeMessage(PopupMessage *pm)
{
	topLayout->remove( pm );
	topLayout->setMargin( KDialog::marginHint() );
	
	messageQueue.remove( pm );
	delete pm;

	if( messageQueue.isEmpty() )
	{
		hide();
	}
	else
	{
		updateBg();
	}
}

/**
 * @brief Static method to show single popup
 *
 * Creates the single popup with one message and shows it
 */
Popup * Popup::message( QWidget *parent, Message *m )
{
	Popup *p = new Popup( parent );
	p->setTimeout( m->getLifetime() );
	p->addMessage( m );
	return p;
}

void Popup::updateBg()
{
	adjustSize();
	PopupTheme *theme = Config().currentTheme;

	if( Config().themesEnabled && theme )
	{
		topLayout->setMargin( theme->marginWidth );
		hide();

		//		adjustSize();
		QPtrListIterator<PopupMessage> it( messageQueue );
		PopupMessage *pm;
		while( (pm = it.current()) != 0 )
		{
			pm->resetWidth(size().width()-2*getMargin());
			++it;
		}
		adjustSize();
		
		positionSelf();

		m_screenshot=QPixmap::grabWindow( qt_xrootwin(), x(), y(), width(), height() );
		render(theme);

		// we use timer to slow down the popup's show
		// this makes popup appear nicer than just flashing when the new messages appears
		QTimer *t = new QTimer( this );
		connect( t, SIGNAL(timeout()), this, SLOT(showPopup()) );
		t->start( 160, true );
	}
	else
	{
		topLayout->setMargin( KDialog::marginHint() );
		setPalette( QToolTip::palette() );
		
		//adjustSize();
		QPtrListIterator<PopupMessage> it( messageQueue );
		PopupMessage *pm;
		while( (pm = it.current()) != 0 )
		{
			pm->resetWidth(size().width()-2*getMargin());
			++it;
		}
		adjustSize();
		
		positionSelf();
		
		update();
		
		if( !isVisible() )
			showPopup();
	}
}

void Popup::showPopup()
{
	if( !messageQueue.isEmpty() )
	{
		show();
	}
}

QString Popup::getBody()
{
	PopupTheme *theme = Config().currentTheme;

	if( Config().themesEnabled && theme && !theme->body.isEmpty() )
		return theme->body;

	return "<nobr>%IMG <nobr><b>%SENDER</b></nobr></nobr><br />%TEXT";
}

QString Popup::getTextColor(Message *m)
{
	PopupTheme *theme = Config().currentTheme;
	QString color;
	
	if( m && m->getTextColor() )
		color = m->getTextColor()->name();
	else
	if( Config().themesEnabled && theme && !theme->textColor.isEmpty() )
		color = theme->textColor;
	else
		color = QToolTip::palette().active().foreground().name();
	
	return color;
}

QString Popup::getSenderColor(Message *m)
{
	PopupTheme *theme = Config().currentTheme;
	QString color;

	if( Config().themesEnabled && theme && !theme->senderColor.isEmpty() )
		color = theme->senderColor;
	else
		color = QToolTip::palette().active().foreground().name();

	return color;
}
#include "popup.moc"
