/***************************************************************************
 *   Copyright (C) 2004 by Dominik Seichter                                *
 *   domseichter@web.de                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qfile.h>
#include <qlabel.h>
#include <qtimer.h>

#include <kaction.h>
#include <kactioncollection.h>
#include <kapplication.h>
#include <kaboutapplication.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kstdaction.h>

#include "kcpuspeed.h"
#include "preferencesdlg.h"

#define MAX_LINE_LENGTH 256

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif 

KCpuSpeed::KCpuSpeed(const QString& configFile, Type type, int actions, QWidget *parent, const char *name)
    : KPanelApplet(configFile, type, actions, parent, name)
{
    loadConfig();
    
    m_mhz = new QLabel(this);
    m_mhz->setAlignment( Qt::AlignVCenter | Qt::AlignHCenter );
    
    m_timer = new QTimer( this, "m_timer" );
    connect( m_timer, SIGNAL( timeout() ), this, SLOT( parseCpuinfo() ) );
    m_timer->start( m_frequency, false ); // 0.5 minute

    KActionCollection* actionCollection = new KActionCollection( this, "actionCollection", kapp );
    mnu = new KPopupMenu( this );
    mnu->insertTitle( SmallIcon( "kcpuspeed" ), i18n("KCpuSpeed") );
    KAction* act = KStdAction::preferences( this, SLOT( preferences() ), actionCollection );
    act->setText( i18n("&Configure KCpuSpeed...") );
    act->plug( mnu );
    mnu->insertSeparator();
    act = KStdAction::aboutApp( this, SLOT( about() ), actionCollection );
    act->setText( i18n("&About KCpuSpeed...") );
    act->plug( mnu );
    mnu->insertSeparator();
    KStdAction::quit( kapp, SLOT( quit() ), actionCollection )->plug( mnu );
    // Why does kicker start out with a size of 800x409?
    // Kicker bug?
    resize(60,42);
    
    parseCpuinfo();
}


KCpuSpeed::~KCpuSpeed()
{
}

void KCpuSpeed::setText( const int m )
{
    m_mhz->setText( QString("<qt>%1MHz</qt>").arg(m) );
}

void KCpuSpeed::parseCpuinfo()
{
    QFile file( "/proc/cpuinfo" );
    QString line;
    
    if( !file.open( IO_ReadOnly ) )
    {
        KMessageBox::error( this, i18n("Can't read /proc/cpuinfo which is required for KCpuSpeed to work.") );
        m_timer->stop();
        return;
    }
    
    while( file.readLine( line, MAX_LINE_LENGTH ) != -1 )
    {
        if( line.startsWith( "cpu MHz") )
        {
            QString mhz = line.section( ':', 1, 1 ).stripWhiteSpace();
            setText( (int)mhz.toDouble() );
            break;
        }
    }
    file.close();
}

void KCpuSpeed::about()
{
    KAboutData about
        (
        "kcpuspeed",
        I18N_NOOP("KCpuSpeed"),
        VERSION,
        I18N_NOOP("<qt>This simple applet display the current cpu frequency. This is useful for notebooks with speedstepping technology.</qt>"),
        KAboutData::License_GPL_V2,
        "(C) 2004 Dominik Seichter", 0,  "domseichter@web.de"
    );
    
    about.addAuthor("Dominik Seichter", 0, "domseichter@web.de", "http://www.krename.net" );
        
    KAboutApplication a(&about);
    a.exec();
}

void KCpuSpeed::preferences()
{
    PreferencesDlg prefs( this, "prefs" );
    prefs.setUpdateFrequency( m_frequency );
    
    if( prefs.exec() == QDialog::Accepted )
    {
        m_frequency = prefs.updateFrequency();
        
        saveConfig();
    }
}

int KCpuSpeed::widthForHeight(int h) const
{
    m_mhz->setFixedSize( width(), h );
    return width();
}

int KCpuSpeed::heightForWidth(int w) const
{
    m_mhz->setFixedSize( w, height() );
    return height();
}

void KCpuSpeed::loadConfig()
{
    KConfig* cfg = config();
    cfg->setGroup( "KCpuSpeed" );
    m_frequency = cfg->readNumEntry( "UpdateFrequency", 30*1000 );
    
}

void KCpuSpeed::saveConfig()
{
    KConfig* cfg = config();
    cfg->setGroup( "KCpuSpeed" );
    cfg->writeEntry( "UpdateFrequency", m_frequency );
    
    cfg->sync();
}

void KCpuSpeed::mousePressEvent( QMouseEvent* e )
{
    if( e->button() == Qt::RightButton )
        mnu->popup( e->globalPos() );
}

extern "C"
{
    KPanelApplet* init( QWidget *parent, const QString& configFile)
    {
        KGlobal::locale()->insertCatalogue("kcpuspeed");
        return new KCpuSpeed(configFile, KPanelApplet::Normal,
                             KPanelApplet::About | KPanelApplet::Preferences,
                             parent, "kcpuspeed");
    }
}
