/**
 * @file
 * Main application implementation.
 *
 * Kisa provides spell checking as you type and displays the result in a small
 * window on your Desktop.
 *
 * Copyright (c) 2007 by Pete Black <theblackpeter@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * @author Pete Black <theblackpeter@gmail.com>
 */

#include <QtGui>
#include <QtDebug>
#include <QMenu>
#include <QProcess>
#include <QScrollBar>

#include <iostream>

#include "kisa.h"
#include "kisalib.h"
#include "kisawidget.h"
#include "kisasettings.h"
#include "kisathread.h"


/**
 * The current project version.
 */
#ifndef PROJECT_VERSION
#define PROJECT_VERSION "0.43"
#endif


/**
 * The project name.
 */
#ifndef PROJECT_NAME
#define PROJECT_NAME "Kisa"
#endif


/**
 * Instances of this class will have this name.
 * @note this is not a unique identifier, just a common name.
 */
#ifndef OBJECT_NAME
#define OBJECT_NAME "kisa"
#endif


Kisa::Kisa(int& argc, char** argv) : QApplication(argc, argv) {
  // check for system tray icon availability
  if (!QSystemTrayIcon::isSystemTrayAvailable())
    qFatal("No system tray available.");

  // input arguments
  QString display;
  QString language;
  QString iconsPath;
  QString flagsPath;
  QString dictDir;
  QString aspellPrefix;
  QString extraDicts;

  // parse the input argument's values
  // as they are on the form "-option value" we always want the next string in
  // in the array (++i)
  // there are more arguments available then these underneath, so we need not
  // worry about an arrayOutOfBounds, try a arg[15] for instance...
  for(int i = 1; i < argc; i++) {
    QString argument = argv[i];

    if(argument == tr("-help"))
      showCommandLineHelp();

    else if(argument == tr("-display"))
      display = argv[++i];

    else if(argument == tr("-language"))
      language = argv[++i];

    else if(argument == tr("-icons"))
      iconsPath = argv[++i];

    else if(argument == tr("-flags"))
      flagsPath = argv[++i];

    else if(argument == tr("-dict-dir"))
      dictDir = argv[++i];

    else if(argument == tr("-aspell-prefix"))
      aspellPrefix = argv[++i];

    else if(argument == tr("-extra-dicts"))
      extraDicts = QString(argv[++i]).simplified();

    else {
      QString message = tr("Unknown option") + " " + QString(argv[i]);
      message += "\n";
      message += tr("Use -help to get a list of available command line options");
      qFatal(message.toLocal8Bit());
    }
  }

  // load the translator according the current locale
  QString locale = QLocale::system().name();
  QTranslator translator;

  if(translator.load("kisa_" + locale, "/usr/share/kisa/translations/")) {
    installTranslator(&translator);

    qDebug() << "kisa: Using translation given by locale" << locale;
  }
  // ignore default (American) locale
  else if(locale != "en_US" || locale != "C")
    qDebug() << "kisa: No translation found for locale" << locale;

  // done with that crap, now lets start the application
  qDebug() << "Starting" << PROJECT_NAME << "version" << PROJECT_VERSION;
  setApplicationName(PROJECT_NAME);
  setOrganizationName(PROJECT_NAME);
  setObjectName(OBJECT_NAME);

  // instantiate the library
  kisaLib = KisaLib::instance(display);

  // initilizing the library
  kisaLib->initialize(iconsPath, flagsPath, language, dictDir, aspellPrefix, extraDicts);

  // main GUI window
  kisaWidget = new KisaWidget();
  kisaWidget->setWindowIcon(QIcon(":/icons/kisa.png"));

  // event filer used for checking if the keyboard input layout has changed
  kisaWidget->installEventFilter(this);

  // event filer used for catching mouse click events
  kisaWidget->lineEdit->installEventFilter(this);

  // settings window
  kisaSettings = new KisaSettings(kisaWidget);

  // main worker thread that handles all I/O, all GUI is done from here
  kisaThread = new KisaThread(this);

  // system tray icon
  kisaTrayIcon = new QSystemTrayIcon(QIcon(":/icons/kisa.png"), this);
  kisaTrayIcon->setObjectName("kisaTrayIcon");

  // and its menu
  trayIconMenu = new QMenu(tr("Kisa Tray Icon Menu"), kisaWidget);
  trayIconMenu->setIcon(QIcon(":/icons/kisa.png"));
  trayIconMenu->setObjectName("kisaTrayIconMenu");
  kisaTrayIcon->setContextMenu(trayIconMenu);
  updateSystemTrayIconMenu();

  // the tool button pop-up menu containing the dictionaries used so far
  toolButtonMenu = new QMenu(tr("Kisa Tool Button Menu"), kisaWidget);
  toolButtonMenu->setObjectName("kisaToolButtonMenu");
  kisaWidget->toolButton->setMenu(toolButtonMenu);

  // word has been passed to the spell checker flag
  wordIsChecked = false;

  // used as a pop-up to show possible suggestions
  completer = new QCompleter(kisaWidget->lineEdit);

  // set the timer that clears the input if idle for more then 20 s
  kisaTimer = new QTimer(this);
  kisaTimer->setSingleShot(true);

  // clear text field and reset cursor after 20 s
  connect(kisaTimer, SIGNAL(timeout()), this, SLOT(clear()));

  // input from any application window (worker thread) to GUI connection
  connect(kisaThread, SIGNAL(gotNewCharacter(const QChar&)), this,
          SLOT(addCharacter(const QChar&)));

  // application visibility state change
  connect(kisaThread, SIGNAL(visibilityStateChange(const int&)), kisaWidget,
          SLOT(updateVisibilityState(const int&)));

  // restart the worker thread if the settings are changed
  connect(kisaSettings, SIGNAL(accepted()), this, SLOT(updateSettings()));

  // clicking the toolButton switches to the next dictionary in the pop-up list
  connect(kisaWidget->toolButton, SIGNAL(clicked()), this,
          SLOT(switchToNextDictionary()));

  // clicking on one of the items in the tool button menu
  connect(toolButtonMenu, SIGNAL(triggered(QAction*)), this,
          SLOT(toolButtonMenuActivated(QAction*)));

  // show or hide the application whenever someone clicks on the icon
  connect(kisaTrayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
          this, SLOT(systemTrayIconActivated(QSystemTrayIcon::ActivationReason)));

  // clear the dictionary popup menu whenever changing the "show flag"
  // setting, so that we clear dictinaries showing or hiding flag icons
  // it's a cheap hack i know...
  connect(kisaSettings->checkBox_13, SIGNAL(toggled(bool)), this,
          SLOT(resetToolButtonMenu()));

  // adds the custom command to the bottom of the context menu of the lineEdit
  // whenever a context menu is requested, that is the custim context menu
  connect(kisaWidget->lineEdit, SIGNAL(customContextMenuRequested(const QPoint&)),
          this, SLOT(contextMenuEvent(const QPoint&)));

  // check if to restore a previous session's settings
  // make sure this is among the last things to be done
  if(isSessionRestored())
    restoreSession();

  else {
    kisaSettings->readSettings();
    updateSettings();
  }

  // start worker thread
  startThread();

  // make sure this is the last thing we do
  kisaWidget->show();
  kisaTrayIcon->show();
}


Kisa::~Kisa() {
  delete kisaTrayIcon; // not needed, owned by kisaWidget
  delete completer; // not needed, owned by kisaWidget (via lineEdit)
  delete kisaSettings; // not needed owned by kisaWidget
  delete kisaThread; // thread knows how to stop itself, owned by this
  delete kisaTimer; // not needed, owend by this
  delete kisaWidget;
  delete kisaLib;

  qDebug() << "kisa: Exiting program";
}


void Kisa::addCharacter(const QChar& key) {
  QString word = kisaWidget->lineEdit->text();

  // if the word is checked already, clear it and start over
  if(wordIsChecked) {
    wordIsChecked = false;

    // TODO must be a better way to change color, now the font needs to be re-set
    kisaWidget->lineEdit->setStyleSheet("QLineEdit { color: black }");
    kisaWidget->lineEdit->setFont(kisaSettings->font());

    // change the mouse pointer to default
    kisaWidget->lineEdit->setCursor(Qt::ArrowCursor);

    completer->popup()->hide();

    // just in case the user changes their mind and wants to retype the word,
    // add an extra space (will be cleared later), else clear the word
    if(key == '\b')
      word += " ";
    else
      word.clear();
  }

  // letter
  if(key.isLetter())
    word += key;

  // hyphen (ignore inital and duplicate hyphens)
  else if(!word.isEmpty() && key == '-' && !hyphenAsPunctuation &&
          !word.endsWith('-'))
    word += key;

  // backspace erases only if there are characters there already
  else if(!word.isEmpty() && key == '\b')
    word.chop(1);

  // white space and punctuation will trigger a spell check on the word
  else if(!word.isEmpty() && (key.isSpace() || key.isPunct() ||
                              key == '-' && hyphenAsPunctuation)) {
    // remove any trailing hyphens
    if(word.endsWith('-'))
      word.chop(1);

    checkWord();
    wordIsChecked = true;
  }

  // any other key clears the input
  else
    word.clear();

  kisaWidget->lineEdit->setText(word);

  // start the timer to clear any input if idle for more then 20 s
  kisaTimer->start(20000);
}


void Kisa::checkWord() {
  const QString word = kisaWidget->lineEdit->text();

  if(kisaLib->isMisspelled(word)) {
    // TODO must be a better way to change color, now the font needs to be re-set
    kisaWidget->lineEdit->setStyleSheet("QLineEdit { color: red }");
    kisaWidget->lineEdit->setFont(kisaSettings->font());

    // change the mouse pointer so it looks clickable
    kisaWidget->lineEdit->setCursor(Qt::PointingHandCursor);

    QStringList suggestions = kisaLib->getSuggestions(word);

    // the auto completer used as a pop-up to show the possible candidates
    if(kisaSettings->showSuggestionsPopup()) {
      completer->disconnect();
      delete completer;

      completer = new QCompleter(suggestions, kisaWidget->lineEdit);

      completer->setCompletionMode(QCompleter::PopupCompletion);
      completer->setCaseSensitivity(Qt::CaseInsensitive);

      QAbstractItemView* popup = completer->popup();

      popup->setFont(kisaWidget->lineEdit->font());

      kisaWidget->lineEdit->setCompleter(completer);

      connect(completer, SIGNAL(activated(const QString&)), this,
              SLOT(updateText(const QString&)));

      // only show pop-up if main window is actually visible
      if(!kisaWidget->isPartiallyObscured() && kisaWidget->isVisible() &&
         !kisaWidget->isMinimized()) {
        // force the pop-up twice, this is needed in order to steal back focus
        completer->complete();
        popup->hide();
        completer->complete();
      }

      // we need the duration in ms
      int popupDuration = kisaSettings->suggestionsPopupDuration() * 1000;

      // we'll use a timer to close the pop-up after a given time
      // only set it if the pop-up duration is > 0
      if(popupDuration)
        QTimer::singleShot(popupDuration, popup, SLOT(hide()));
    }

    // the system tray icon pop-up
    if(kisaSettings->showSystrayPopup()) {
      QString title = tr("Kisa says:");

      QString message = tr("%1 is unknown").arg(word);

      // we need the duration in ms
      int popupDuration = kisaSettings->systrayPopupDuration() * 1000;

      kisaTrayIcon->showMessage(title, message, QSystemTrayIcon::Information,
                                popupDuration);
    }
  }
}


void Kisa::systemTrayIconActivated(QSystemTrayIcon::ActivationReason  aReason) {
  if(aReason == QSystemTrayIcon::Trigger)
    // show/hide the main widget
    if(kisaWidget->isVisible()) {
      kisaWidget->hide();

      kisaTrayIcon->setIcon(QIcon(":/icons/kisa_shaded.png"));

      stopThread();
    }
    else {
      kisaWidget->showNormal();

      kisaTrayIcon->setIcon(QIcon(":/icons/kisa.png"));

      startThread();
    }
}


void Kisa::updateSystemTrayIconMenu() {
  // clear the menu completely and rebuild it, much easier then trying to
  // replace items at certain locations
  trayIconMenu->clear();

  // if the thread is running we want the stop action at the top
  if(kisaThread->isRunning()) {
    kisaWidget->lineEdit->setEnabled(true);
    kisaWidget->toolButton->setEnabled(true);

    trayIconMenu->addAction(QIcon("icons:stop.png"), tr("&Stop"), this,
                            SLOT(stopThread()));

    // update the tooltips
    QString message = tr("Kisa -- using dictionary %1").arg(kisaLib->currentDictionary());
    kisaTrayIcon->setToolTip(message);
    kisaWidget->setToolTip(message);
    kisaWidget->toolButton->setToolTip(kisaLib->currentDictionary());

    // update the icon
    kisaTrayIcon->setIcon(QIcon(":/icons/kisa.png"));
    kisaWidget->setWindowIcon(QIcon(":/icons/kisa.png"));
  }
  // if not then make the run action available instead
  else {
    kisaWidget->lineEdit->setEnabled(false);
    kisaWidget->toolButton->setEnabled(false);

    trayIconMenu->addAction(QIcon("icons:run.png"), tr("&Run"), this,
                            SLOT(startThread()));

    // update the tooltips
    QString message = tr("Kisa is not running");
    kisaTrayIcon->setToolTip(message);
    kisaWidget->setToolTip(message);
    kisaWidget->toolButton->setToolTip(message);

    // update the icon
    kisaTrayIcon->setIcon(QIcon(":/icons/kisa_shaded.png"));
    kisaWidget->setWindowIcon(QIcon(":/icons/kisa_shaded.png"));
  }

  trayIconMenu->addSeparator();
  trayIconMenu->addAction(QIcon("icons:kwin.png"), tr("R&estore"), kisaWidget,
                          SLOT(showNormal()));
  trayIconMenu->addAction(QIcon("icons:configure.png"), tr("Se&ttings"),
                          kisaSettings, SLOT(showNormal()));
  trayIconMenu->addSeparator();
  trayIconMenu->addAction(QIcon("icons:help.png"), tr("&Help"), this,
                          SLOT(getHelp()));
  trayIconMenu->addSeparator();
  trayIconMenu->addAction(QIcon(":/icons/kisa.png"), tr("&About Kisa"), this,
                          SLOT(showAbout()));
  trayIconMenu->addAction(QIcon(":/icons/qt-logo.png"), tr("A&bout Qt"), this,
                          SLOT(aboutQt()));
  trayIconMenu->addSeparator();
  trayIconMenu->addAction(QIcon("icons:exit.png"), tr("&Quit"), this,
                          SLOT(quit()));
}


void Kisa::startThread() {
  if(!kisaThread->isRunning())
    toggleThread();
}


void Kisa::stopThread() {
  if(kisaThread->isRunning())
    toggleThread();
}


void Kisa::toggleThread() {
  kisaLib->setIgnoreList(kisaSettings->ignoreListRegExp());

  kisaThread->startToggle();

  updateSystemTrayIconMenu();
}


void Kisa::restartThread() {
  if(kisaThread->isRunning()) {
    qDebug() << "kisa: Restarting thread...";
    // call the thread method isntead of toggleThread as we don't need to update
    // the icons
    kisaThread->startToggle();
    kisaThread->startToggle();
  }
}


void Kisa::showAbout() {
  // some fancy html for the author email
  QString msg = "<html><head><meta name=\"qrichtext\" content=\"1\" /><style type=\"text/css\">, li { white-space: pre-wrap; }</style></head><body style=\" font-family:'Sans Serif'; font-size:9pt; font-weight:400; font-style:normal;\">Kisa &#8212; live! spell checker ";
  msg += PROJECT_VERSION;
  msg += "<br><br>(c) 2007 <a href=\"mailto:theblackpeter@gmail.com?subject=Something%20about%20the%20great%20application%20Kisa\">Pete Black</a></body></html>";

  QMessageBox::about(kisaWidget, tr("About Kisa"), msg);
}


void Kisa::updateText(const QString& newWord) {
  qDebug() << "kisa: Selected" << newWord << "from pop-up";

  // save the current text before changing it
  QString word = kisaWidget->lineEdit->text();

  // add to system clipboard
  if(kisaSettings->addToClipboard()) {
    QClipboard *clipboard = QApplication::clipboard();
    clipboard->setText(newWord);
    qDebug() << "kisa: Selected word" << newWord << "added to clipboard";
  }

  // replace misspelled word in client application with the selected
  // as it is the worker thread that does all the interaction with the kisa
  // events, and remembers where they came from, it's the only one that can do
  // this (except for a call to the library directly)
  if(kisaSettings->replaceInClient())
    kisaThread->replaceInClient(word, newWord);

  // TODO must be a better way to change color, now the font needs to be re-set
  kisaWidget->lineEdit->setStyleSheet("QLineEdit { color: black }");
  kisaWidget->lineEdit->setFont(kisaSettings->font());

  // change the mouse pointer to default
  kisaWidget->lineEdit->setCursor(Qt::ArrowCursor);

  kisaWidget->lineEdit->setText(newWord);
}


void Kisa::updateSettings() {
  // update the dictionary, always do this first
  int currentDictionaryIndex = kisaLib->currentDictionaryIndex();
  int dictionaryIndex = kisaSettings->dictionaryIndex();
  if(currentDictionaryIndex != dictionaryIndex)
    updateDictionary(dictionaryIndex);

  // stay on top or not
  if(kisaSettings->stayOnTop()) {
    // only make changes if needed as this hides the window for some reason
    if(!(kisaWidget->windowFlags() & Qt::WindowStaysOnTopHint)) {
      kisaWidget->setWindowFlags(Qt::Tool | Qt::WindowStaysOnTopHint);

      // NOTE Qt <= 4.3.1, when changing the window flags the window gets
      // minimized, or the application can even shut down for some reason
      kisaWidget->show();
    }
  }
  else
    // only make changes if needed as this hides the window for some reason
    if(kisaWidget->windowFlags() & Qt::WindowStaysOnTopHint) {
      kisaWidget->setWindowFlags(Qt::Tool);

      // NOTE Qt <= 4.3.1, when changing the window flags the window gets
      // minimized, or the application can even shut down for some reason
      kisaWidget->show();
    }

  // update the font
  kisaWidget->lineEdit->setFont(kisaSettings->font());

  // show tool button or not?
  if(kisaSettings->showButton()) {
    updateToolButtonMenu();

    kisaWidget->toolButton->show();
  }
  else
    kisaWidget->toolButton->hide();

  // convenience variable to reduce function calls
  hyphenAsPunctuation = kisaSettings->hyphenAsPunctuation();

  qDebug() << "kisa: Settings updated";

  if(kisaWidget->isVisible())
    restartThread();
}


bool Kisa::eventFilter(QObject* object, QEvent* event) {

#ifdef FULL_DEBUG
  qDebug() << "kisa: Got event type:" << event->type() << "from object:" << object->objectName() << ws << "(" << object->metaObject()->className() << ")";
#endif

  // keyboard layout change
  if(object == kisaWidget && event->type() == QEvent::KeyboardLayoutChange) {
    kisaWidget->lineEdit->clear();

    // temporary disable the event filter
    kisaWidget->removeEventFilter(this);

    if(kisaSettings->updateOnNewLayout()) {
      // get the new layout code
      QString layout = kisaLib->getKeyboardInputLayout();

      if(!layout.isEmpty()) {
        // update the dictionary accordingly to the new language
        updateDictionary(layout);

        // update the tool button's pop-up menu
        updateToolButtonMenu();
      }
    }

    // remove the event from the event queue
    event->accept();

    // flush the event queue
    QCoreApplication::processEvents();

    // reinstall the event filter
    kisaWidget->installEventFilter(this);

    // event was handled
    return true;
  }

  // left-click on misspelled word executes custom command
  else if(object == kisaWidget && event->type() == QEvent::MouseButtonRelease) {
    if(((QMouseEvent *) event)->button() == Qt::LeftButton &&
       kisaWidget->lineEdit->cursor().shape() == Qt::PointingHandCursor)
      runCustomCommand();

    // remove the event from the event queue
    event->accept();

    // event was handled
    return true;
  }

  // destroy the event
  event->ignore();

  // event was ignored
  return false;
}


// make sure the dictionary has been updated first
void Kisa::updateToolButtonMenu() {
  // first time eh?, create a new menu then
  if(toolButtonMenu->isEmpty()) {
    resetToolButtonMenu();

    // resetToolButtonMenu() will take care of the rest by calling this method..
    return;
  }

  // get the current dictionary and index
  QString dictionary = kisaLib->currentDictionary();
  int dictionaryIndex = kisaLib->currentDictionaryIndex();

  // update the toolButton icon, we don't need the returned icon here
  updateToolButtonIcon();

  // add the dictionary to the popup
  addDictionaryToMenu(dictionaryIndex);

  // if there are more the two dictionaries show menu arrow
  if(toolButtonMenu->actions().size() > 3)
    kisaWidget->toolButton->setStyleSheet("::menu-indicator { image: url(:/icons/downarrow.png); }");
}


bool Kisa::dictionarySeenBefore(const int& index) {
  foreach(QAction* action, toolButtonMenu->actions())
    if(index == action->data())
      return true;

  return false;
}


void Kisa::toolButtonMenuActivated(QAction* action) {
  int index = action->data().toInt();
  updateDictionary(index);

  // no need to update the entire pop-up menu, just the icon
  updateToolButtonIcon();
}


void Kisa::switchToNextDictionary() {
  QList<QAction*> actions = toolButtonMenu->actions();

  // ignore the separator and the "clear" action that where added at the end
  int actionsSize = actions.size() - 2;

  // don't bother if there is only one dictionary
  if(actionsSize <= 1)
    return;

  // get the current dictionary index
  int dictionaryIndex = kisaLib->currentDictionaryIndex();

  // yes a regular for loop, need the index incremented here...
  for(int i = 0; i < actionsSize; i++) {
    if(dictionaryIndex == actions.at(i)->data()) {
      // we want the subsequent element, but if we are at the end then the next
      // is the first... I know, it's getting late here...
      if(i == actionsSize - 1)
        i = 0;

      else
        i++;

      // use the toolButtonMenuActivated slot to update both the flag and the
      // dictionary
      toolButtonMenuActivated(actions.at(i));

      return;
    }
  }

  qDebug() << "kisa: Sorry, there is no new dictionary to switch to, try adding some to the list";
}


void Kisa::resetToolButtonMenu() {
  toolButtonMenu->clear();

  toolButtonMenu->addSeparator();
  toolButtonMenu->addAction(QIcon("icons:history_clear.png"),
                            tr("Clear used dictionaries"), this,
                            SLOT(resetToolButtonMenu()));

  kisaWidget->toolButton->setStyleSheet("");

  qDebug() << "kisa: Tool button menu has been cleared and updated";

  updateToolButtonMenu();
}


void Kisa::updateDictionary(const QString& layout) {
  // try to update both the dictionary in the library and the one in the
  // settings menu based on a 5 char language_COUNTRY layout
  if(kisaLib->updateDictionary(layout)) {
    kisaSettings->setDictionaryIndex(kisaLib->currentDictionaryIndex());

    QString message =
        tr("Kisa -- using dictionary %1").arg(kisaLib->currentDictionary());
    kisaTrayIcon->setToolTip(message);
    kisaWidget->setToolTip(message);
    kisaWidget->toolButton->setToolTip(kisaLib->currentDictionary());

    // update the text input direction
    updateTextDirection();
  }
}


void Kisa::updateDictionary(const int& index) {
  // try to update both the dictionary in the library and the one in the
  // settings menu based on index
  if(kisaLib->updateDictionary(index)) {
    kisaSettings->setDictionaryIndex(index);

    QString message =
        tr("Kisa -- using dictionary %1").arg(kisaLib->currentDictionary());
    kisaTrayIcon->setToolTip(message);
    kisaWidget->setToolTip(message);
    kisaWidget->toolButton->setToolTip(kisaLib->currentDictionary());

    // update the text input direction
    updateTextDirection();
  }
}


void Kisa::runCustomCommand() {
  QString customCommand = kisaSettings->customCommand();

  // substitute language and word if user specified any
  customCommand.replace("$LANG", kisaLib->currentDictionaryCode().left(2));
  customCommand.replace("$WORD", kisaWidget->lineEdit->text());

  // check if the command is a web url and use the sytem default method of
  // opening it, else execute the command
  QUrl url(customCommand);
  if(!url.isRelative()) {
    if(!QDesktopServices::openUrl(customCommand))
      qDebug() << "kisa: Unable to open url" << customCommand << "using current system settings";
  }
  else {
    // block and wait for process to start
    if(!QProcess::startDetached(customCommand))
      qDebug() << "kisa: Unable to execute custom command" << customCommand;
  }
}


void Kisa::showCommandLineHelp() {
  QString message;
  message = QString(PROJECT_NAME) + QString(" ") + QString(PROJECT_VERSION) + QString(" Copyright (c) 2007 by Pete Black\n");
  message += QString(PROJECT_NAME) + " " + tr("-- live! spell checker") + "\n";
  message += "\n";
  message += tr("Usage: ") + OBJECT_NAME + " " + tr("[options]") + "\n";
  message += "\n";
  message += tr("Options:") + "\n";
  message += "  " + tr("-help\t\tShows this help") + "\n";
  message += "  " + tr("-display <display>\tThe display on the X server to use") + "\n";
  message += "  " + tr("-language <language>\tThe default language code") + "\n";
  message += "  " + tr("-icons <path>\tWhere to look for action icons") + "\n";
  message += "  " + tr("-flags <path>\tWhere to look for country flag icons") + "\n";
  message += "  " + tr("-dict-dir <path>\tMain word list location") + "\n";
  message += "  " + tr("-aspell-prefix <path>\tAspell prefix directory") + "\n";
  message += "  " + tr("-extra-dicts <list>\tList of extra dictionaries to use") + "\n";
  message += "\n";
  message += tr("Example:") + "\n";
  message += "\n";
  message += "  " + QString(OBJECT_NAME) + " " + tr("-language sv -icons ~/myTheme -flags /usr/share/locale/l10n") + "\n";
  message += "\n";
  message += tr("Will use Swedish as a default language, the icons from the theme located under\n\"~/myTheme\" and the flag icons from \"/usr/share/locale/l10n\"") + "\n";
  message += "\n";

  std::cout << message.toLocal8Bit().data() << std::endl;

  std::exit(1);
}


QIcon Kisa::updateToolButtonIcon() {
  // get the icon of the current dictionary
  QIcon icon;

  if(kisaSettings->showFlag())
    icon = kisaSettings->currentCountryFlag();

  // if we don't want an icon or none is found show text
  if(icon.isNull()) {
    // show the button with the country as text
    kisaWidget->toolButton->setAutoRaise(false);
    kisaWidget->toolButton->setToolButtonStyle(Qt::ToolButtonTextOnly);
    kisaWidget->toolButton->setText(kisaLib->currentDictionaryCode().left(2));

    qDebug() << "kisa: No icon for the selected dictionary found, using text instead";
  }
  else {
    kisaWidget->toolButton->setToolButtonStyle(Qt::ToolButtonIconOnly);
    kisaWidget->toolButton->setIcon(icon);

    // make sure the icon is no larger then 21x14 (which most flag icons in
    // /usr/share/locale/l10n/ are)
    kisaWidget->toolButton->setIconSize(icon.actualSize(QSize(21, 14)));

    qDebug() << "kisa: Updated tool button icon";
  }

  return icon;
}


void Kisa::aboutQt() {
  QMessageBox::aboutQt(kisaWidget);
}


void Kisa::commitData(QSessionManager& manager) {
  const QString sessionID = manager.sessionId();

  // get the default settings file name
  QString fileName = QSettings().fileName();

  // add the session id
  fileName.replace(QRegExp("\\.conf$"), "_" + sessionID + ".conf");

  qDebug() << "kisa: Saving session...";

  // write the settings information from the settings window
  kisaSettings->writeSettings(fileName);

  // store the used dictionaries in revers order as we need fifo order later
  QStringList dictionaryIndexes;
  foreach(QAction* action, toolButtonMenu->actions())
    if(!action->data().isNull())
      dictionaryIndexes.prepend(action->data().toString());

  // and some extra stuff from the main window widget
  QSettings settings(fileName, QSettings::IniFormat, this);
  settings.beginGroup("Session");
  settings.setValue("dictionaryIndexes", dictionaryIndexes);
  settings.endGroup();
}


void Kisa::restoreSession() {
  const QString sessionID = sessionId();

  // get the default settings file name
  QString fileName = QSettings().fileName();

  // add the session id
  fileName.replace(QRegExp("\\.conf$"), "_" + sessionID + ".conf");

  // stop if the file doesn't exist
  if(!QFile::exists(fileName)) {
    qDebug() << "kisa: Could not find session file, using previous settings";

    kisaSettings->readSettings();
    updateSettings();

    return;
  }

  qDebug() << "kisa: Restoring session...";

  // get the dictionaries
  QSettings settings(fileName, QSettings::IniFormat, this);
  settings.beginGroup("Session");
  QStringList dictionaryIndexes = settings.value("dictionaryIndexes").toStringList();
  settings.endGroup();

  // if there is only 1 dictionary or less no need to add it to the list
  if(dictionaryIndexes.size() > 1) {
    // clear everything so we get the previous order intact
    toolButtonMenu->clear();
    toolButtonMenu->addSeparator();
    toolButtonMenu->addAction(QIcon("icons:history_clear.png"),
                              tr("Clear used dictionaries"), this,
                              SLOT(resetToolButtonMenu()));

    foreach(QString dictionaryIndex, dictionaryIndexes)
      addDictionaryToMenu(dictionaryIndex.toInt());
  }

  // the rest of the settings
  kisaSettings->readSettings(fileName);

  // this also calls updateToolButtonMenu
  updateSettings();

  // don't need the file any more
  QFile::remove(fileName);
}


void Kisa::addDictionaryToMenu(const int& dictionaryIndex) {
  // name of the dictionary
  QString dictionary = kisaLib->dictionaries().at(dictionaryIndex);

  // check if the dictionary is already in the menu
  if(!dictionarySeenBefore(dictionaryIndex)) {
    // get the icon, if any is used
    QIcon icon = kisaSettings->currentCountryFlag();

    // have to do it this way as we need to store the dictionary index value
    QAction* dictionaryAction = new QAction(icon, dictionary, toolButtonMenu);
    dictionaryAction->setData(dictionaryIndex);

    // insert action at the top
    toolButtonMenu->insertAction(toolButtonMenu->actions().first(),
                                 dictionaryAction);

    qDebug() << "kisa: Added new dictionary" << dictionary << "to pop-up menu";
  }
  else
    qDebug() << "kisa: Dictionary" << dictionary << "is already in menu pop-up, not adding";
}


void Kisa::contextMenuEvent(const QPoint& point) {
  // get the original menu
  QMenu* menu = kisaWidget->lineEdit->createStandardContextMenu();

  QString word = "\"";
  word += kisaWidget->lineEdit->text();
  word += "\"";

  QString message = tr("Add %1 to personal world list").arg(word);

  // add to personal dictionary option
  menu->addSeparator();
  QAction* actionAdd = menu->addAction(QIcon("icons:edit_add.png"), message,
                                       this, SLOT(addToPersonalWordList()));

  // ignore option (add to session list)
  message = tr("Ignore %1").arg(word);

  QAction* actionIgnore = menu->addAction(QIcon("icons:edittrash.png"), message,
                                          this, SLOT(ignoreWord()));

  // only make the add/ignore word actio available if there is a word
  if(kisaWidget->lineEdit->text().isEmpty()) {
    actionAdd->setEnabled(false);
    actionIgnore->setEnabled(false);
  }

  // direction change option
  menu->addSeparator();

  if(kisaWidget->lineEdit->layoutDirection() == Qt::LeftToRight)
    menu->addAction(QIcon("icons:keyboard_layout.png"), tr("Set right-to-left layout"),
                    kisaWidget, SLOT(toggleLayoutDirection()));
  else
    menu->addAction(QIcon("icons:keyboard_layout.png"), tr("Set left-to-right layout"),
                    kisaWidget, SLOT(toggleLayoutDirection()));

  // the custom command
  QString customCommandName = kisaSettings->customCommandName();
  QIcon customCommandIcon = kisaSettings->customCommandIcon();

  if(customCommandName.isEmpty()) {
    customCommandName = tr("Run Custom Command");
    customCommandIcon = QIcon("icons:launch.png");
  }

  menu->addSeparator();
  menu->addAction(customCommandIcon, customCommandName, this,
                  SLOT(runCustomCommand()));

  // Use popup() to execute asynchronously or exec() synchronously
  menu->exec(kisaWidget->mapToGlobal(point));

  delete menu;
}


void Kisa::getHelp() {
  QDesktopServices::openUrl(QUrl("http://kisaspell checker.googlepages.com/manual"));
}


void Kisa::clear() {
  kisaWidget->lineEdit->clear();

    // change the mouse pointer to default
  kisaWidget->lineEdit->setCursor(Qt::ArrowCursor);
}


void Kisa::addToPersonalWordList() {
  QString word = kisaWidget->lineEdit->text();

  if(!word.isEmpty())
    kisaLib->addToPersonalWordList(word);
}


void Kisa::ignoreWord() {
  QString word = kisaWidget->lineEdit->text();

  if(!word.isEmpty())
    kisaLib->addToSessionWordList(word);
}


void Kisa::updateTextDirection() {
  /// NOTE make these calls to avoid problems associated with kxkb and setxkbmap
  /// when selecting dictionary at startup
  kisaWidget->removeEventFilter(this);

  kisaLib->getKeyboardInputLayout();

  // flush the event queue
  QCoreApplication::processEvents();

  // reinstall the event filter
  kisaWidget->installEventFilter(this);

  // now update the direction
  kisaWidget->lineEdit->setLayoutDirection(keyboardInputDirection());

  // update the locale just in case
  kisaWidget->lineEdit->setLocale(keyboardInputLocale());
}
