/**
 * @file
 * Main application header file.
 *
 * Kisa provides spell checking as you type and displays the result in a small
 * window on your Desktop.
 *
 * Copyright (c) 2007 by Pete Black <theblackpeter@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * @author Pete Black <theblackpeter@gmail.com>
 */

#ifndef KISA_H
#define KISA_H

#include <QApplication>
#include <QSystemTrayIcon>

class QAction;
class QCompleter;
class QProcess;

class KisaLib;
class KisaWidget;
class KisaSettings;
class KisaThread;


/**
 * @class Kisa
 * The Kisa class provides the main application, using a reimplementation of
 * QApplication. It manages the GUI application's control flow and main
 * settings. All the algorithms, logic and data structures that constitute the
 * core of the application are kept here.
 *
 * It also contains the main event loop, which is the final destination for all
 * other events from the window system and other sources.
 *
 * This is not the main window or worker thread. See KisaWidget and KisaThread
 * respectively. External API function calls, such as X11 and Aspell are made
 * through the library, KisaLib, which is instantiated here.
 *
 * If for some reason you didn't know, Kisa is a small widget window for X11
 * based desktops, that spell checks everything that is typed and shows a pop-up
 * if the word is not found in the dictionary. Kisa uses Aspell as a default
 * spell checker. Kisa has has a variety of settings and interactivity levels
 * and it's also possible to replace misspelled words in the client application.
 *
 * @image html kisa_suggestion_popup1.png "Kisa main widget window and pop-up"
 *
 * @author Pete Black <theblackpeter@gmail.com>
 */
class Kisa: public QApplication {
  Q_OBJECT

  public:
    /**
     * Creates a new application instance using @p argc command line arguments
     * given by @p argv.
     *
     * Does all object instantiation and initialization, for example the library,
     * KisaLib, the worker thread, KisaThread, the main window KisaWidget,
     * the settings window, KisaSettings and many more.
     *
     * @note Qt has a set of default arguments, these are not processes here.
     *
     * @param argc the number of command line arguments
     * @param argv the command line arguments
     *
     * @see showCommandLineHelp
     */
    Kisa(int& argc, char** argv);


    /**
     * Destroys the application.
     */
    ~Kisa();


    /**
     * Updates the System Tray menu. For instance, if the worker thread is
     * stopped we want to change the top action from "Stop" to "Run".
     */
    void updateSystemTrayIconMenu();


    /**
     * Updates the tool button's menu pop-up with new dictionary information.
     * If the dictionary has not been used before it's added to the menu.
     */
    void updateToolButtonMenu();


    /**
     * Checks if the selected dictionary has been used before. If so @c true is
     * returned, otherwise @c false.
     *
     * @param index the position of the dictionary in the dictionary list or
     * combo box in the settings window
     * @return @c true if the dictionary has been used, otherwise @c false
     */
    bool dictionarySeenBefore(const int& index);


    /**
     * Convenience function to update the current dictionary based on a
     * locale layout.
     *
     * @param layout the locale on ISO 639/ISO 3166 standard form, that is:
     * @c language[_COUNTRY]
     */
    void updateDictionary(const QString& layout);


    /**
     * Updates the current dictionary to that given by the index. Updates both
     * the current dictionary in the library and that in the system settings
     * window.
     *
     * @param index the position of the dictionary in the dictionary list or
     * combo box in the settings window
     */
    void updateDictionary(const int& index);


    /**
     * Shows which command line options are available for the application.
     */
    void showCommandLineHelp();


  protected:
    /**
     * Reimplementation of the the main event filter.
     *
     * Filters QEvent::KeyboardLayoutChange events for the main widget window,
     * so it knows when to update the current dictionary. If the event has been
     * handled the function will return @c return true and stop it from
     * being handled further, otherwise @c false is returned.
     *
     * @param object pointer to object that generated the event
     * @param event pointer to the event object
     * @return true if the event was handled, @c false otherwise
     */
    bool eventFilter(QObject *object, QEvent *event);



    /**
     * Updates the icon or text showing the current dictionary in use. If an
     * icon is set to be used and found, it will be displayed and returned. If
     * not, the language code of the current dictionary will be displayed.
     *
     * @return the current country flag icon, or none if no icon is found
     */
    QIcon updateToolButtonIcon();


    /**
     * Saves the current application settings and state if requested by the
     * sessions manager. The settings are written to the directory and file
     * specified by the organization, application name and session ID, which by
     * default is @c "~/.config/Kisa/Kisa_SESIONID.conf"
     *
     * @note This method always uses the QSettings::IniFormat.
     *
     * @note This differs from KisaSettings::writeSettings() as it also stores
     * the used dictionaries (in revers order) under the sessions group. These
     * settings can only be written and read from here.
     *
     * @note Using this method over QApplication::saveState() even though the
     * latter is used when answering to local calls from the sessions manager.
     * This method does not get called on application start when the session
     * manager tries to determine if the application is session aware. Hopefully
     * this will reduce the amount of residual session files left, and there
     * will be no need for any maintenance cleanup.
     *
     * @param manager the desktops session manager
     *
     * @see restoreSession()
     */
    void commitData(QSessionManager& manager);


    /**
     * Reads the current application settings and state when restoring from a
     * previous session. The settings are read from the directory and file
     * specified by the organization, application name and session ID, which by
     * default is @c "~/.config/Kisa/Kisa_SESSIONID.conf"
     *
     * The state consists of the used dictionaries.
     *
     * @note This method always uses the QSettings::IniFormat.
     *
     * @see commitData()
     */
    void restoreSession();


    /**
     * Adds a dictionary given by the index @p dictionaryIndex to the tool
     * button's pop-up menu. An country flag icon is added if available in the
     * settings dictionary combo box.
     *
     * @param dictionaryIndex the index used to locate the dictionary
     */
    void addDictionaryToMenu(const int& dictionaryIndex);


    /**
     * Updates the text direction for the text field to the current keyboard
     * input layout direction and locale.
     */
    void updateTextDirection();


  public slots:
    /**
     * Adds a word character to the text field. If the character is a backspace,
     * '\\b', then the last character is removed from the word. If
     * the character is a space character then spell checking is preformed using
     * checkWord().
     *
     * @note Space characters and punctuations, such as space, enter, tab, ".",
     * ",", "!" and so on will trigger a spell check on the word. Any character
     * other then a word character, including hyphen, '-', will be ignored and
     * erases the word.
     *
     * @param character the character to add to the word
     */
    void addCharacter(const QChar& character);


    /**
     * Starts the worker thread if it's not running. When the thread is running
     * the main window widget is enabled.
     *
     * @image html kisa_enabled.png "Enabled main window widget"
     */
    void startThread();


    /**
     * Stops the worker thread if it's not running. When the thread is stopped
     * the main window widget is disabled.
     *
     * @image html kisa_disabled.png "Disabled main window widget"
     */
    void stopThread();

    /**
     * Toggles the worker thread. That is, if the thread is running it will be
     * stopped and if not it will be started.
     */
    void toggleThread();


    /**
     * Restarts the thread, but only if it is running.
     */
    void restartThread();


    /**
     * Spell checks the word currently in the text field. Convenience function
     * to make library calls that spell check the contents of the text field.
     * Used whenever there is a new word available, that is, when input is
     * completed by a space character or punctuation.
     *
     * @see addCharacter()
     */
    void checkWord();


    /**
     * Called whenever the system tray icon has been activated, for example
     * when the user clicks on the system tray icon.
     *
     * @image html systray_menu.png "Kisa system tray menu"
     *
     * @param aReason how the system tray was activated
     */
    void systemTrayIconActivated(QSystemTrayIcon::ActivationReason aReason);


    /**
     * Convenience slot that is called whenever the text in the text field
     * is changed. For instance, when replaced by a suggestion from the pop-up
     * menu.
     *
     * @note If the option "add to clipboard" has been selected the, the text in
     * @p newText is added to the clipboard.
     *
     * @note If the option "replace in client application" has been selected the
     * text in @p newText is sent back to the client application.
     *
     * @param newText the new word to use when replacing the old text with
     */
    void updateText(const QString& newText);


    /**
     * Called whenever the settings in the settings window have been changed.
     *
     * Updates, the main window hint ("stay on top"), the font, the current
     * dictionary used and the "show country flag" and "custom command" options.
     */
    void updateSettings();


    /**
     * Shows information about the application whenever the user selects "About"
     * from the system tray icon menu.
     */
    void showAbout();


    /**
     * Used when an action from the tool button menu is selected. For instance,
     * if the users changes dictionary or clears the used dictionaries history.
     *
     * @param action the action selected by the user
     */
    void toolButtonMenuActivated(QAction* action);


    /**
     * Called whenever the tool button is clicked and switches to the next seen
     * dictionary. If only one dictionary has been used so far, no event occurs.
     *
     * @image html kisa_dictionary_popup.png "Kisa main widget with a pop-up showing the recently used dictionaries"
     */
    void switchToNextDictionary();


    /**
     * Convenience slot to removes all the seen dictionaries from the dictionary
     * history and add the default "clear history" command. After calling this
     * slot the tool button pop-up menu will be reset and contain no
     * dictionaries except the one currently used.
     */
    void resetToolButtonMenu();


    /**
     * Slot that preforms variable substitution and runs the custom command
     * given in the settings as a detached process.
     *
     * The command can be any known system command or a protocol like @c http.
     * The command has to be either given by its full path or the path has to be
     * in the @c $PATH system variable.
     *
     * The Desktop's settings determine which application is used when executing
     * the protocol.
     *
     * @image html kisa_context_menu.png "The custom command in the context menu pop-up"
     *
     * Available varaible substitutions are:
     *   @li @c $LANG is substituted with the current dictionary language
     *
     *   @li @c $WORD is substituted with the current word in the text field
     *
     * @see KisaSettings::cusomCommand()
     */
    void runCustomCommand();


    /**
     * Convenience slot to call the static versions of aboutQt. The
     * QApplication::aboutQt slot closes the program for some reason.
     *
     * @todo remove this when the slot version starts working
     */
    void aboutQt();


    /**
     * Shows customized context menu for the text field that has action icons
     * (added by createLineEditContextMenu()) and also the custom command name
     * from the settings menu.
     *
     * The slot is called when the QWidget::customContextMenuRequested() signal
     * is emitted, that is, when right-clicking on the text field.
     *
     * @note Reimplementing the standard context menu for QLineEdit stopped
     * working for some reason.
     *
     * @image html kisa_context_menu.png "Kisa's custom context menu"
     *
     * @param point used to obtain the position where the mouse cursor
     * was when the event was generated
     */
    void contextMenuEvent(const QPoint& point);


    /**
     * Convenience slot to open the external Kisa help page.
     */
    void getHelp();


    /**
     * Convenience slot to clear and reset the mouse pointer. Used by the
     * kisaTimer when application has been idle too long.
     */
    void clear();


    /**
     * Adds the current word in the text field to the personal word list.
     *
     */
    void addToPersonalWordList();


    /**
     * Ignores the current word in the text field by adding it to the sessions
     * list. Subsequent occurrences of this word will be ignored by the
     * spell checker for the reminder of the session.
     */
    void ignoreWord();


  private:
    /**
     * Pointer to the main window.
     */
    KisaWidget* kisaWidget;


    /**
     * Pointer to the settings window.
     */
    KisaSettings* kisaSettings;


    /**
     * Pointer to the main worker thread.
     */
    KisaThread* kisaThread;


    /**
     * Pointer to the library object.
     */
    KisaLib* kisaLib;


    /**
     * Pointer to the applications system tray entry.
     */
    QSystemTrayIcon* kisaTrayIcon;


    /**
     * Pointer to the system tray icon menu containing all the possible actions.
     */
    QMenu* trayIconMenu;


    /**
     * Pointer to the tool button menu containing the used dictionaries.
     */
    QMenu* toolButtonMenu;


    /**
     * Completer pointer, that is used to display the suggestions pop-up.
     */
    QCompleter* completer;


    /**
     * Flag that if set to @c true the current word has been spell checked,
     * otherwise not.
     */
    bool wordIsChecked;

    /**
     * Timer that clears the text field if idle for 20 s.
     */
    QTimer* kisaTimer;


    /**
     * Store this setting to reduce the number of function calls.
     */
    bool hyphenAsPunctuation;
};

#endif
