/**
 * @file
 * Library implementation.
 *
 * Kisa provides spell checking as you type and displays the result in a small
 * window on your Desktop.
 *
 * Copyright (c) 2007 by Pete Black <theblackpeter@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * @author Pete Black <theblackpeter@gmail.com>
 */

#include <QDir>
#include <QLocale>
#include <QFile>
#include <QProcess>
#include <QSettings>
#include <QtDebug>

#include "kisalib.h"


/**
 * The current library version.
 */
#ifndef KISALIB_VERSION
#define KISALIB_VERSION "0.43"
#endif


// define and initialize static variables
KisaLib* KisaLib::m_instance = NULL;
XErrorHandler KisaLib::m_oldXErrorHandler = (XErrorHandler) 0;


KisaLib::KisaLib(QString displayString) {
  qDebug() << "kisalib: Using KisaLib version" << KISALIB_VERSION;

  if(displayString.isEmpty()) {
    // try the system display from the environment variable $DISPLAY
    displayString = getenv("DISPLAY");

    if(displayString.isEmpty())
      // use system default display on :0
      displayString = ":0";
  }

  m_display = XOpenDisplay(displayString.toAscii());

  if(!m_display) {
    QString message = "kisalib: Could not connect to X server on " + displayString;
    qFatal(message.toUtf8());
  }
  else
    qDebug() << "kisalib: Connected to X server on display" << displayString;

  m_rootWindow = XDefaultRootWindow(m_display);

  if(!m_rootWindow) {
    QString message = "kisalib: Could not get root window on display " + displayString;
    qFatal(message.toUtf8());
  }
  else
    qDebug() << "kisalib: Root window is" << showbase << hex << m_rootWindow;

  // instance variables
  m_total = 0;
  m_currentDictionaryIndex = 0;

  // create the list of X event types that we use
  m_eventTypes[0] = "None";
  m_eventTypes[2] = "KeyPress";
  m_eventTypes[15] = "VisibilityNotify";
  m_eventTypes[16] = "CreateNotify";

  // and the list of all the possible X visibility states
  m_visibilityStates[0] = "VisibilityUnobscured";
  m_visibilityStates[1] = "VisibilityPartiallyObscured";
  m_visibilityStates[2] = "VisibilityFullyObscured";

  // override default X11 error handler
  m_oldXErrorHandler = XSetErrorHandler(setErrorHandler);
}


KisaLib::~KisaLib() {
  // get the old X11 error handler back
  (void) XSetErrorHandler(m_oldXErrorHandler);

  qDebug() << "kisalib: Closing connection to X server";
  XCloseDisplay(m_display);

  delete_aspell_speller(m_aspellSpeller);
}


KisaLib* KisaLib::instance(const QString& displayString) {
  if(!m_instance)
    m_instance = new KisaLib(displayString);

  return m_instance;
}


extern void KisaLib::initialize(const QString& iconspath,
                                const QString& flagspath,
                                const QString& language,
                                const QString& dictDir,
                                const QString& aspellPrefix,
                                const QString& extraDicts) {
  if(!m_instance)
    m_instance = new KisaLib();

  // set the search paths for where to look for icons
  setIconsSearchPaths(iconspath);

  // set the search path for where to look for flag icons
  setFlagsSearchPaths(flagspath);

  // create the Aspell spell checker and load dictionaries
  initializeAspell(language, dictDir, aspellPrefix, extraDicts);
}


void KisaLib::changeDeepNotification(const Window& window,
                                      const EventMask& eventMask) {
  // make sure we have don't any crappy windows
  if(!isWindowDecent(window))
    return;

  Window dummyWindow; // root window, dummy return, not used
  Window parentWindow; // not used either
  Window* childWindows;
  unsigned int totalChildWindows = 0;
  XQueryTree(m_display, window, &dummyWindow, &parentWindow, &childWindows,
              &totalChildWindows);

  // update the event subscription
  changeNotification(window, eventMask);

  // now get all the children of the window
  for(unsigned int i = 0; i < totalChildWindows; i++) {
    // recursive step
    changeDeepNotification(childWindows[i], eventMask);
  }

  if(!childWindows)
    XFree((char *)childWindows);
}

void KisaLib::changeNotification(const Window& window, EventMask eventMask) {
  // make sure we have don't have a crappy window
  if(!isWindowDecent(window))
    return;

  // if there is no event mask, remove any notification right away
  if(eventMask == NoEventMask) {
    removeNotification(window);
    return;
  }

  // skip our own window for key and create events but not notification of
  // window visibility change
  if(isOurOwnWindow(window))
    eventMask = VisibilityChangeMask;

  // OK then, every thing is as it should be, change the notification...
  XSelectInput(m_display, window, eventMask);
  XFlush(m_display);

  // increment the total windows we have subscription to
  m_total++;

#ifdef FULL_DEBUG
  printNotificationOutput(window, eventMask);
#endif
}


void KisaLib::addNotification(const Window& window, KisaEvent::Type type) {
  EventMask eventMask;

  switch(type) {
    case KisaEvent::None:
      eventMask = NoEventMask;
      break;

    // just a keypress event
    case KisaEvent::KeyPress:
      eventMask = KeyPressMask;
      break;

    // a VisibilityNotify
    case KisaEvent::WindowStateChange:
      eventMask = VisibilityChangeMask;
      break;

    // CreateNotify, MapNotify events, a window was either
    // created, made visible or re-configured
    case KisaEvent::WindowActivate:
      eventMask = KeyPressMask | SubstructureNotifyMask;
      break;

    default:
      ;
  }

  // careful not to call this function too often, ignore the root window
  if(window != m_rootWindow)
    changeDeepNotification(window, eventMask);
}


void KisaLib::reAddNotification() {
  // use the default event masks, KeyPressMask and SubstructureNotifyMask
  changeDeepNotification(m_rootWindow);
}


void KisaLib::removeNotification(const Window& window) {
  XSelectInput(m_display, window, NoEventMask);
  XFlush(m_display);

  m_total--;

#ifdef FULL_DEBUG
  printNotificationOutput(window, NoEventMask);
#endif
}

void KisaLib::removeAllNotifications() {
  changeDeepNotification(m_rootWindow, NoEventMask);

  m_total = 0;
}


KisaEvent* KisaLib::nextEvent() {
  // get the event from the X server event queue
  XEvent xEvent;
  XNextEvent(m_display, &xEvent);

  switch(xEvent.type) {
    case XKeyPress: {
      Window window = xEvent.xkey.window;

      // get the key string from the XEvent which containing one unicode
      // (utf-8) character
      QChar key = lookupKey(xEvent.xkey);

#ifdef FULL_DEBUG
      if(key.isNull())
        qDebug() << "kisalib: New KeyPress event, keycode:" << showbase << hex << xEvent.xkey.keycode << "key sym:" << XKeycodeToKeysym(m_display, xEvent.xkey.keycode, 0) << "from id:" << window;

      else
        qDebug() << "kisalib: New KeyPress event, key:" << key << "keycode:" << showbase << hex << xEvent.xkey.keycode << "key sym:" << XKeycodeToKeysym(m_display, xEvent.xkey.keycode, 0) << "from id:" << window;
#endif

      return new KisaEvent(KisaEvent::KeyPress, window, key);
    }

    // visibility changed for out application
    case VisibilityNotify: {
      Window window = xEvent.xvisibility.window;

      int state = xEvent.xvisibility.state;
#ifdef FULL_DEBUG
      qDebug() << "kisalib: New VisibilityNotify event" <<  m_visibilityStates[state] << "from id:" << showbase << hex << window;
#endif
      return new KisaEvent(KisaEvent::WindowStateChange, window, state);
    }

    // a window was made "visible"
    case MapNotify: {
      Window window = xEvent.xmap.window;
#ifdef FULL_DEBUG
      qDebug() << "New MapNotify event from from id:" << showbase << hex << window;
#endif
      return new KisaEvent(KisaEvent::WindowActivate, window);
    }

    // a new window was created, note that all windows are not really useful
    // as they are not always mapped
    // this event gets thrown around quite a bit, but we'll include it anyway
    case CreateNotify: {
      Window window = xEvent.xcreatewindow.window;
#ifdef FULL_DEBUG
      qDebug() << "kisalib: New CreateNotify event from id:" << showbase << hex << window;
#endif
      return new KisaEvent(KisaEvent::WindowActivate, window);
    }

    default:
      return new KisaEvent();
  }

  return new KisaEvent();
}


QChar KisaLib::lookupKey(XKeyEvent& xKey) {
  // utf-8 is variable-length character encoding that uses 4 bytes at the most
  // remember we are only interested in the character, so null-termination
  // ('\0') is unimportant (Qt takes care of that)
  const int KEY_SIZE = 4;
  char keyBuffer[KEY_SIZE];

  // XLookupString() returns the number of characters that are stored in the
  // buffer. It should be utf-8 safe, this may change though.
  int nChars = XLookupString(&xKey, keyBuffer, KEY_SIZE, NULL, NULL);

  // modifiers (shift, alt, ctrl...)  and arrowkeys will result in nChars = 0
  if(nChars > 0 && keyBuffer != NULL)
    return QString::fromUtf8(keyBuffer, nChars).at(0);

  return QChar();
}


void KisaLib::sendBogusXEvent() {
  // create the event struct
  XMapEvent xMapEvent;
  xMapEvent.type = MapNotify;
  xMapEvent.display = m_display;
  xMapEvent.window = m_rootWindow;
  xMapEvent.override_redirect = true;

  qDebug() << "kisalib: Sending bogus XEvent...";

  XSendEvent(xMapEvent.display, xMapEvent.window, xMapEvent.override_redirect,
              SubstructureNotifyMask, (XEvent *)&xMapEvent);
  XFlush(m_display);
}


bool KisaLib::isMisspelled(const QString& word) {
  const char* aspell_word = word.toUtf8();

  // aspell_speller_check returns 0 if it is not in the dictionary, 1 if it is,
  // or -1 on error
  int spellerResult = aspell_speller_check(m_aspellSpeller, aspell_word, -1);

  if(!spellerResult) {
    qDebug() << "kisalib: Can't find" << word;
    return true;
  }
  else if(spellerResult == -1)
    qWarning() << "kisalib: Aspell error accrued when looking up" << word;

  return false;
}


QStringList KisaLib::getSuggestions(const QString& word) {
  const char* aspell_word = word.toUtf8();

  const AspellWordList* aspell_suggestions =
      aspell_speller_suggest(m_aspellSpeller, aspell_word, -1);

  AspellStringEnumeration* aspell_elements =
      aspell_word_list_elements(aspell_suggestions);

  qDebug() << "kisalib: Suggestions:";

  const char* aspell_suggestion =
      aspell_string_enumeration_next(aspell_elements);

  QStringList suggestions;

  while(aspell_suggestion != NULL) {
    QString suggestion = QString::fromUtf8(aspell_suggestion);

    qDebug() << suggestion;

    suggestions << suggestion;

    aspell_suggestion = aspell_string_enumeration_next(aspell_elements);
  }

  delete_aspell_string_enumeration(aspell_elements);

  return suggestions;
}


void KisaLib::initializeAspell(QString language,
                               const QString& dictDir,
                               const QString& aspellPrefix,
                               const QString& extraDicts) {
  if(language.isEmpty()) {
    qDebug() << "kisalib: No language given, getting language from keyboard layout";

    // get the current input language from keyboard layout
    // this is usually a 5 character string in the form country_LANGUAGE
    // but sometimes we only get the two-letter ISO 639 country code
    language = getKeyboardInputLayout();
  }

  // init aspell with system default
  AspellConfig* aspell_config = new_aspell_config();
  aspell_config_replace(aspell_config, "lang", language.toAscii());
  aspell_config_replace(aspell_config, "encoding", "utf-8");

  // add user overrides
  if(!dictDir.isEmpty())
    aspell_config_replace(aspell_config, "dict-dir", dictDir.toAscii());

  if(!aspellPrefix.isEmpty())
    aspell_config_replace(aspell_config, "prefix", aspellPrefix.toAscii());

  if(!extraDicts.isEmpty())
    aspell_config_replace(aspell_config, "extra-dicts", extraDicts.toAscii());

  // check for errors, if none create the speller
  AspellCanHaveError* aspell_possible_err = new_aspell_speller(aspell_config);
  if (aspell_error_number(aspell_possible_err) != 0)
    printAspellErrorAndExit(aspell_error_message(aspell_possible_err));

  else
    m_aspellSpeller = to_aspell_speller(aspell_possible_err);

  // we need to iterate over all the dictionaries aspell knows about
  AspellDictInfoList* dictInfoList = get_aspell_dict_info_list(aspell_config);
  AspellDictInfoEnumeration* dictInfoEnum =
      aspell_dict_info_list_elements(dictInfoList);

  // make sure we actual found dictionaries
  if(aspell_dict_info_list_empty(dictInfoList))
    qFatal("kisalib: No dictionaries found");

  const AspellDictInfo* entry;
  while((entry = aspell_dict_info_enumeration_next(dictInfoEnum)) != 0) {

    // available entry struct variables are
    //    entry->name
    //    entry->code
    //    entry->jargon
    //    entry->size_str
    //    entry->module->name;

    // creat a full language_COUNTRY code based on either the language only or
    // on the full 5 characters in the code variable of the dictionary
    // ISO 639/ISO 3166 standard is on the form
    // language[_COUNTRY][.codeset][@modifier]
    QString code = entry->code;
    QLocale locale(code);

    // build up a format similar to that of KSpell, that is something more
    // intuitive for the user
    QString dictionary;
    // in case we only have the language of the dictionary
    if(code.size() == 2) {
      dictionary = QLocale::languageToString(locale.language());
      dictionary += " (common";
    }

    // both language and country coded in 5 characters (language_COUNTRY)
    else {
      // just in case there is other crap there...
      code = code.left(5);

      dictionary = QLocale::languageToString(locale.language());
      dictionary += " (";
      dictionary += QLocale::countryToString(locale.country());
    }

    // every dictionary dosn't have a jargon
    QString jargon = entry->jargon;
    if(!jargon.isEmpty())
      dictionary += " - ";

    dictionary += jargon;
    dictionary += ")";

    // add the user readable dictionary string to the list
    m_dictionaries << dictionary;

    // add the ISO standard to the list, will need it later too...
    m_dictionaryCodes << code;

    // QLocale tries to guess the country based on the language
    m_dictionaryCodesGuessed << locale.name();
  }

  delete_aspell_dict_info_enumeration(dictInfoEnum);

  qDebug() << "kisalib: Available dictionaries" << m_dictionaries;

  updateDictionary(language);

  // no need for the config any more
  delete_aspell_config(aspell_config);
}


bool KisaLib::updateDictionary(const QString& newLanguage) {
  // try the full language_COUNTRY format first
  int index = m_dictionaryCodes.indexOf(QRegExp(newLanguage,
                                        Qt::CaseInsensitive));

  if(index == -1)
    // try just the language
    index = m_dictionaryCodes.indexOf(QRegExp(newLanguage.left(2),
                                      Qt::CaseInsensitive));

  // don't bother checking the index here, it's done in the other method
  return updateDictionary(index);
}


bool KisaLib::updateDictionary(const int& index) {
  // check the index
  if(index < 0 || index >= m_dictionaries.size()) {
    qDebug() << "kisalib:" << index << "is not a valid dictionary index";
    return false;
  }

  // use the actual language code and not our dictionary name
  QString dictionaryCode = m_dictionaryCodes.at(index);

  qDebug() << "kisalib: Using language code" << dictionaryCode;
  qDebug() << "kisalib: Using dictionary" << m_dictionaries.at(index);

  // update the spell checker
  AspellConfig* aspell_config = new_aspell_config();
  aspell_config_replace(aspell_config, "lang", dictionaryCode.toUtf8());
  aspell_config_replace(aspell_config, "encoding", "utf-8");

  AspellCanHaveError* aspell_possible_err = new_aspell_speller(aspell_config);

  // check for errors, if none create the spell checker
  if (aspell_error_number(aspell_possible_err) != 0)
    printAspellErrorAndExit(aspell_error_message(aspell_possible_err));

  else
    m_aspellSpeller = to_aspell_speller(aspell_possible_err);

  // no need for the config any more
  delete_aspell_config(aspell_config);

  // keep track of the index
  m_currentDictionaryIndex = index;

  return true;
}


QString KisaLib::getKeyboardInputLayout() {
  /// cheap HACK to overcome the bugs associated with kxkb and setxkbmap
  QProcess process;
  process.start("dcop kxkb kxkb getCurrentLayout");
  QString output;

  // block for a maximum of 5 s
  if(process.waitForFinished(5000))
    output = process.readAllStandardOutput().simplified();

  // if kxkb is not running we'll get a No such application...
  if(!output.isEmpty() || !output.startsWith("No such application")) {
    // run setxkbmap -layout "<output from kxkb>" && setxkbmap
    QProcess::execute("setxkbmap -layout \"" + output + "\"");
  }
  /// end cheap HACK

  // flush the event queue, this is needed to register any new xkb settings
    QCoreApplication::processEvents();

  // get the current keyboard layout from the system instead
  QString keyboardLayout = qApp->keyboardInputLocale().name();

  // the "C" locale is identical to English/UnitedStates (en_US) and is used as
  // a default or if an error occurred when creating the QLocale()
  if(keyboardLayout == "C" || keyboardLayout == "c") {
    qDebug() << "kisalib: System returned default locale, using \"en_US\" instead";
    keyboardLayout = "en_US";
  }
  else
    qDebug() << "kisalib: Got keyboard layout locale" << keyboardLayout;

  return keyboardLayout;
}

QString KisaLib::getCountryFlagURL(QString countryCode) {
  // make sure we have the right format
  if(countryCode.size() != 2) {
    qDebug() << "kisalib:" << countryCode << "is a wrong country code";
    return "";
  }
  // just in case...
  countryCode = countryCode.toLower();

  QString url;

  // get flags paths from resources
  QStringList searchPaths =  QDir::searchPaths("flags");
  if(!searchPaths.isEmpty())
    url = searchPaths.at(0);
  else
    return "";

  // try with country code as a sub directory first
  if(QFile::exists(url + countryCode + "/flag.png"))
    // create the full url using the resource "flags" directory location
    url += countryCode + "/flag.png";
  else
    url += "/flag.png";

  if(QFile::exists(url))
    return url;

  return "";
}


void KisaLib::replaceInClient(const Window& window, const QString& misspelled,
                              const QString& suggestion) {
  // create the event struct
  XKeyEvent xKey;
  xKey.type = XKeyPress; // of event
  xKey.display = m_display; // Display the event was read from
  xKey.window = window; // "event" window it is reported relative to
  xKey.root = m_rootWindow; // root window that the event occurred on
  xKey.time = CurrentTime; // in ms
  xKey.x = xKey.y = 0; // pointer x, y coordinates in event window
  xKey.x_root = xKey.y_root = 0; // coordinates relative to root
  xKey.same_screen = true;

  bool propagate = false; // send to every client, is supposed to be false

  // set the keycode to the backspace key
  xKey.keycode = XKeysymToKeycode(xKey.display, XK_BackSpace);

  // erase the misspelled word in the client application window by sending back
  // equal amount of backspace keys as characters in the word plus one for the
  // space key entered by the user (to get the word spell check)
  for(int i = 0; i <= misspelled.size(); i++)
    XSendEvent(m_display, window, propagate, KeyPressMask, (XEvent *) &xKey);

    // now send the characters of the new word
    for(int i = 0; i < suggestion.size(); i++) {
      // get the single character and its string representaion
      const QChar keyChar = suggestion.at(i);

    // set the modifier mask, that helps us get the character's right case
    if(keyChar.isUpper())
      xKey.state = ShiftMask;
    else
      xKey.state = XNone; // just in case ;)

    // last, get the keycode that we need for sending the key event, using the
    // Keysyms from X11/keysymdef.h
    // the mechanisms by which Xlib obtains them is implementation-dependent, as
    // there is there is no 1-to-1 mapping between keysyms and keycodes, however
    // Qt helps out here as we can just use the raw utf-8 value of the character
    KeySym keySym = keyChar.unicode();
    xKey.keycode = XKeysymToKeycode(xKey.display, keySym);

    XSendEvent(m_display, window, propagate, KeyPressMask, (XEvent *) &xKey);

#ifdef FULL_DEBUG
    qDebug() << "kisalib: Sent key:" << keyChar << "key code:" << showbase << hex << xKey.keycode << "key sym:" << keySym << "back to client on id:" << xKey.window;
#endif
  }

  // add a space to the end as it gets removed
  xKey.keycode = XKeysymToKeycode(xKey.display, XK_space);

  XSendEvent(m_display, window, propagate, KeyPressMask, (XEvent *) &xKey);

  qDebug() << "kisalib: Sent back" << suggestion << "to client window id:" << showbase << hex << xKey.window;
}


int KisaLib::setErrorHandler(Display* d, XErrorEvent* xe) {
  // create a return buffer for the errror messsage, 256 bytes should do
  const int BUFFER_SIZE = 256;
  char errorMessage[BUFFER_SIZE];

  int errorCode = xe->error_code;
  int requestCode = xe->request_code;

  // get the messsage
  XGetErrorText(d, errorCode, errorMessage, BUFFER_SIZE);

  // don't us a qWarning here as we can generate a lot of BadWindow errors
  qDebug() << "kisalib: X error on display:" << XDisplayString(d) << errorMessage << requestCode;

  return 0;
}


void KisaLib::printAspellErrorAndExit(QString aspellError) {
  // aspell uses punctuation, remove it!
  aspellError.chop(1);

  QString message = "kisalib: " + aspellError;
  message += "\nkisalib: Check your Aspell installation!";

  qFatal(message.toUtf8());
}


void KisaLib::setIconsSearchPaths(QString path) {
  // check if the given icons directory actually exists
  if(!QDir(path).exists()) {
    path.clear();
    qDebug() << "kisalib: Unable to find icons directory" << path;
  }

  // if no command line argument was given, skip it
  if(!path.isEmpty())
    QDir::addSearchPath("icons", path);

  // use as settings object to get the current KDE icons theme located under
  // the group [Icons] with value "Theme in" the kdeglobals config file
  // make sure QSettings dosn't creates a missing file
  QString kdeGlobals = QDir::homePath() + "/.kde/share/config/kdeglobals";
  if(QFile().exists(kdeGlobals)) {
    QSettings kdeSettings(kdeGlobals, QSettings::IniFormat);
    kdeSettings.beginGroup("Icons");
    QString theme = kdeSettings.value("Theme").toString();
    kdeSettings.endGroup();

    // now get the icons path
    QProcess process;
    process.start("kde-config --path icon");
    QString output;

    // block for a maximum of 5 s
    if(process.waitForFinished(5000))
      output = process.readAllStandardOutput().simplified();

    // creat a string list of the paths
    QStringList kdePaths = output.split(":");

    // finally try to find the full path of the icons theme
    foreach(QString kdePath, kdePaths)
      if(QDir(kdePath + theme).exists())
        path = kdePath + theme;
  }

  /// NOTE the order in which the pats are added matters
  if(QDir(path + "/16x16/actions").exists())
    QDir::addSearchPath("icons", path + "/16x16/actions");

  if(QDir(path + "/16x16/apps").exists())
    QDir::addSearchPath("icons", path + "/16x16/apps");

  // add system default paths just in case
  path = "/usr/share/icons/default.kde";
  if(QDir(path + "/16x16/actions").exists())
    QDir::addSearchPath("icons", path + "/16x16/actions");

  if(QDir(path + "/16x16/apps").exists())
    QDir::addSearchPath("icons", path + "/16x16/apps");

  qDebug() << "kisalib: Using search paths for icons:" << QDir::searchPaths("icons");

  /// TODO add GNOME support, gnome-config --datadir
}


void KisaLib::setFlagsSearchPaths(QString path) {
  // check if the given icons directory actually exists
  if(path.isEmpty() || !QDir(path).exists()) {

    // try the default locale path first, used by most distros
    path = "/usr/share/locale/l10n/";

    if(!QDir(path).exists()) {
      // use the last path given by "kde-config --path locale"
      QProcess process;
      process.start("kde-config --path locale");
      QString output;

      // block for a maximum of 5 s
      if(process.waitForFinished(5000))
        output = process.readAllStandardOutput().simplified();

      // creat a string list of the paths
      QStringList kdeLocalePaths = output.split(":");

      path = kdeLocalePaths.last() + "l10n/";

      QDir::addSearchPath("flags", path);
    }

    if(!QDir(path).exists())
      qWarning() << "kisalib: Search path for flags:" << path << "doesn't exist";
  }

  else
    QDir::addSearchPath("flags", path);

  qDebug() << "kisalib: Using search paths for flags:" << QDir::searchPaths("flags");

  /// TODO add GNOME support
}


QString KisaLib::getWindowName(const Window& window) {
  XTextProperty textProperty;
  if(XGetWMName(m_display, window, &textProperty))
    return QString::fromUtf8((const char *)textProperty.value);

  // if we can't get the name return an empty string
  return "";
}


bool KisaLib::isWindowVisible(const Window& window) {
  // get the window attributes, if the window has none this will generate a BadWindow error
  XWindowAttributes windowAttributes;
  if(!XGetWindowAttributes(m_display, window, &windowAttributes))
    return false;

  // special case for our own application window that for some reason stays
  // unmapped, so ignore that
  if(isOurOwnWindow(window))
    return true;

  // a window is not visible if it is unmapped and unviewable
  if(windowAttributes.map_state == IsUnmapped ||
     windowAttributes.map_state == IsUnviewable)
    return false;

  return true;
}


bool KisaLib::isWindowDecent(const Window& window) {
  // just in case some idiot sends the wrong window id, like a default 0 instead
  if(window < m_rootWindow) {
#ifdef FULL_DEBUG
    qDebug() << "kisalib: Skipping window" << "id:" << showbase << hex << window << "(not a window)";
#endif
    return false;
  }

  // skip unmapped and unviewable windows
  if(!isWindowVisible(window)) {
#ifdef FULL_DEBUG
    qDebug() << "kisalib: Skipping window" << "id:"<< showbase << hex << window << "(window not visible)";
#endif
    return false;
  }

  QString windowName = getWindowName(window);

  // skip windows from the ignore listen
  if(windowName.contains(QRegExp(m_ignoreList, Qt::CaseInsensitive))) {
#ifdef FULL_DEBUG
    qDebug() << "kisalib: Skipping window" << windowName << "id:" << showbase << hex <<  window << "(regexp match)";
#endif
    return false;
  }

  return true;
}

void KisaLib::printNotificationOutput(const Window& window,
                                      const EventMask& eventMask) {
  QString windowName = getWindowName(window);

  if(eventMask == NoEventMask)
    qDebug() << "kisalib: Removed notification for" << windowName << "id:" << showbase << hex << window;

  else if(eventMask == (KeyPressMask | SubstructureNotifyMask))
    qDebug() << "kisalib: Added KeyPress and CreateNotify notification for" << windowName << "id:" << showbase << hex << window;

  else if(eventMask == KeyPressMask)
    qDebug() << "kisalib: Added KeyPress notification for" << windowName << "id:" << showbase << hex << window;

  else if(eventMask == SubstructureNotifyMask)
    qDebug() << "kisalib: Added CreateNotify notification for" << windowName << "id:" << showbase << hex << window;

  else if(eventMask == VisibilityChangeMask)
    qDebug() << "kisalib: Added VisibilityNotify notification for" << windowName << "id:" << showbase << hex << window;

  else
    qDebug() << "kisalib: Added notification for" << windowName << "id:" << showbase << hex << window;
}


bool KisaLib::isOurOwnWindow(const Window& window) {
  // compare returns 0 on match
  QString windowName = getWindowName(window);
  return (!windowName.isEmpty() && !windowName.compare(qApp->applicationName(),
          Qt::CaseInsensitive));
}


void KisaLib::addToPersonalWordList(const QString& word) {
  if(aspell_speller_add_to_personal(m_aspellSpeller, word.toUtf8(), word.size()))
    qDebug() << "kisalib: Added" << word << "to personal word list";
  else
    qDebug() << "kisalib: Could not add" << word << "to personal word list";

  if(aspell_speller_save_all_word_lists(m_aspellSpeller))
    qDebug() << "kisalib: Updated to personal word list";
  else
    qDebug() << "kisalib: Could not write to personal word list";
}


void KisaLib::addToSessionWordList(const QString& word) {
  if(aspell_speller_add_to_session(m_aspellSpeller, word.toUtf8(), word.size()))
    qDebug() << "kisalib: Added" << word << "to session word list (word will be ignored)";
  else
    qDebug() << "kisalib: Could not add" << word << "to session word list";
}
