/* This file is part of the KDE project

   Copyright (C) 2006 Tom Albers <tomalbers@kde.nl>

   This file is based on digikams albumdb which is
   Copyright 2004 by Renchi Raju

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

// C Ansi includes.

extern "C"
{
#include <sqlite3.h>
#include <sys/time.h>
#include <time.h>
}

// C++ includes.

#include <cstdio>
#include <cstdlib>

// Qt includes.

#include <qdir.h>

// KDE includes.

#include <kdebug.h>
#include <klocale.h>

#include "db.h"

typedef struct sqlite3_stmt sqlite3_stmt;

KTUDB::KTUDB()
{
    m_valid = false;
    m_db    = 0;
}

KTUDB::~KTUDB()
{
    if (m_db)
        sqlite3_close(m_db);
}

void KTUDB::setDBPath(const QString& path)
{
    if (m_db)
    {
        sqlite3_close(m_db);
        m_db = 0;
    }

    m_valid = false;

    sqlite3_open(QFile::encodeName(path), &m_db);
    if (m_db == 0)
    {
        kdWarning() << "Cannot open database: "
                    << sqlite3_errmsg(m_db)
                    << endl;
    }
    else
    {
        initDB();
    }
}

void KTUDB::initDB()
{
    m_valid = false;

    // Check if we have the required tables

    QStringList values;

    if (!execSql( QString("SELECT name FROM sqlite_master"
                          " WHERE type='table'"
                          " ORDER BY name;"),
                  &values ))
    {
        return;
    }

    if (!values.contains("InstalledFiles"))
    {
        if (!execSql( QString("CREATE TABLE InstalledFiles\n"
                              " (fileName TEXT PRIMARY KEY,\n"
                              "  isInstalled BOOL);") ))
        {
            return;
        }
    }
    m_valid = true;
}

bool KTUDB::execSql(const QString& sql, QStringList* const values,
                    const bool debug)
{
    kdDebug() << "SQL-query: " << sql << endl;

    if ( !m_db )
    {
        kdWarning() << k_funcinfo << "SQLite pointer == NULL"
                << endl;
        return false;
    }

    const char*   tail;
    sqlite3_stmt* stmt;
    int           error;

    //compile SQL program to virtual machine
    error = sqlite3_prepare(m_db, sql.utf8(), -1, &stmt, &tail);
    if ( error != SQLITE_OK )
    {
        kdWarning() << k_funcinfo
                << "sqlite_compile error: "
                << sqlite3_errmsg(m_db)
                << " on query: "
                << sql << endl;
        return false;
    }

    int cols = sqlite3_column_count(stmt);

    while ( true )
    {
        error = sqlite3_step( stmt );

        if ( error == SQLITE_DONE || error == SQLITE_ERROR )
            break;

        //iterate over columns
        for ( int i = 0; values && i < cols; i++ )
        {
            *values << QString::fromUtf8( (const char*)sqlite3_column_text( stmt, i ) );
        }
    }

    sqlite3_finalize( stmt );

    if ( error != SQLITE_DONE )
    {
        kdWarning() << "sqlite_step error: "
                << sqlite3_errmsg( m_db )
                << " on query: "
                << sql << endl;
        return false;
    }

    return true;
}

QString KTUDB::escapeString(QString str) const
{
    str.replace( "'", "''" );
    return str;
}

Q_LLONG KTUDB::lastInsertedRow()
{
    return sqlite3_last_insert_rowid(m_db);
}

void KTUDB::setInstalled(const QString& file, bool installed)
{
    if (!m_db)
        return;

    execSql( QString("REPLACE INTO InstalledFiles (fileName, isInstalled)"
                          " VALUES( '%1', %2)")
            .arg(escapeString(file)).arg(installed));
}


bool KTUDB::isInstalled(const QString& file)
{
    QStringList values;
    execSql( QString("SELECT isInstalled"
                     " FROM InstalledFiles where fileName='%1'")
            .arg(escapeString(file)), &values );

    if (values.isEmpty())
        return false;

    return values[0]=="1" ? true:false;
}

void KTUDB::removeInstalled(const QString& file)
{
    execSql( QString("DELETE FROM InstalledFiles where fileName='%1'")
            .arg(escapeString(file)));
}



