/* This file is part of the KDE project
   Copyright (C) 2006 Tom Albers <tomalbers@kde.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qtimer.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
#include <kprogress.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>

#include "ktufirstrunwidget.h"
#include "ktufetch.h"
#include "ktuincludes.h"

KTUFirstRunWidget::KTUFirstRunWidget( QWidget* parent  )
    : KDialogBase(Plain, i18n("First Run"), Ok|Cancel, Ok,
                               parent, 0, true, true)
    , m_ready(0)
{
    KTUIncludes::checkRequirements();

    QGridLayout* grid = new QGridLayout(plainPage(), 2, 2, 0, spacingHint());
    QLabel* label1 = new QLabel(
            i18n("Welcome to the KDE Translation Update Tool\n\n") +
            i18n("With this tool you can install translations straight "
                 "from the KDE translators. If you ever decide to not use KTU "
                 "anymore, be sure you delete all installed translations. "
                 "Otherwise updates coming from your distribution will be "
                 "ignored.\n\n") +
            i18n("Before you can use KTU, we need to fetch the "
                 "available languages and applications. Press the button to "
                 "start fetching it, the progressbar is not very accurate..."),
            plainPage());
    label1->setAlignment(Qt::WordBreak);

    m_fetchButton = new QPushButton(i18n("Fetch"),plainPage());
    m_fetchButton->setFocus();
    connect(m_fetchButton, SIGNAL(clicked()), SLOT(slotFetchedClicked()));

    QFont(font);
    font.setItalic(true);

    QLabel* label2 = new QLabel(i18n("Languages:"), plainPage());
    label2->setAlignment(Qt::AlignRight);

    m_labelLang = new QLabel(i18n("Not yet fetched"), plainPage());
    m_labelLang->setFont( font );

    QLabel* label5 = new QLabel(i18n("Default language:"), plainPage());
    label5->setAlignment(Qt::AlignRight);

    m_selectLang = new QComboBox(plainPage());
    m_selectLang->setEnabled(false);

    QLabel* label4 = new QLabel(i18n("Applications:"), plainPage());
    label4->setAlignment(Qt::AlignRight);

    m_labelApp = new QLabel(i18n("Not yet fetched"), plainPage());
    m_labelApp->setFont( font );

    m_progress = new KProgress(plainPage());
    m_progress->setTextEnabled(false);
    m_progress->setEnabled(false);
    m_progress->setTotalSteps(1000);
    m_progress->setValue(0);

    grid->addMultiCellWidget(label1, 0, 0, 0, 3);
    grid->addMultiCellWidget(m_fetchButton, 1, 1, 3, 3);
    grid->addMultiCellWidget(label2, 2, 2, 1, 1);
    grid->addMultiCellWidget(m_labelLang, 2, 2, 2, 3);
    grid->addMultiCellWidget(label5, 3, 3, 1, 1);
    grid->addMultiCellWidget(m_selectLang, 3, 3, 2, 3);
    grid->addMultiCellWidget(label4, 4, 4, 1, 1);
    grid->addMultiCellWidget(m_labelApp, 4, 4, 2, 3);
    grid->addMultiCellWidget(m_progress, 5, 5, 2, 3);

    m_timer = new QTime();
    m_ticking= new QTimer(plainPage());
    connect(m_ticking, SIGNAL(timeout()), SLOT(slotUpdateProgress()));

    enableButtonOK(false);
    setMinimumSize(600,300);
}

KTUFirstRunWidget::~KTUFirstRunWidget()
{
    delete m_timer;
}

void KTUFirstRunWidget::slotOk()
{
    KConfig* config = kapp->config();
    config->setGroup("General");
    config->writeEntry("Lang",
                       KTUIncludes::findShortCode( m_languages,
                               m_selectLang->currentText()));

    KDialogBase::accept();

    QString ErrorMsg, URL;
    if (kapp->startServiceByDesktopName("ktu", URL , &ErrorMsg) > 0)
    {
        kdError() << ErrorMsg << endl;
        KMessageBox::sorry(this, i18n("Cannot restart KTU automatically.\n"
                                      "Please restart KTU manually."));
    }
    delete m_timer;
}

void KTUFirstRunWidget::slotFetchedClicked()
{
    kdDebug() << "Start fetching" << endl;

    m_timer->start();

    m_fetchButton->setEnabled(false);
    m_labelLang->setText(i18n("Fetching"));

    QString output;
    KTUFetch* fetch1 = new KTUFetch( this );
    if (fetch1->fetch("svn://anonsvn.kde.org/home/kde/trunk/l10n/teamnames",
        locateLocal("appdata","")+"teamnames", output))
    {
        languagesFinished();
    }

    m_labelApp->setText(i18n("Fetching"));
    KTUFetch* fetch2 = new KTUFetch( this );
    if (fetch2->fetchDirContents(
            "svn://anonsvn.kde.org/home/kde/trunk/l10n/templates/messages",
    locateLocal("appdata","")+"unstabletemplates", output))
    {
        appsFinished();
    }
}

void KTUFirstRunWidget::languagesFinished()
{
    int elapsed = m_timer->elapsed();
    // fetching the apps, taks roughly 10 times longer
    kdDebug() << "Time elapsed to fetch languages: " << elapsed << endl;
    m_progress->setTotalSteps(elapsed*10);
    m_progress->setEnabled(true);
    m_ticking->start(1000);

    m_labelLang->setText(i18n("Fetched!"));
    m_labelLang->setAlignment(Qt::AlignRight);
    kapp->config()->setGroup("General");
    kapp->config()->writeEntry("LastBuildLanguages",QDateTime::currentDateTime());

    m_selectLang->setEnabled(true);

    QStringList localLangs;
    int i = 0;
    KTUIncludes::getLanguageList(m_languages, localLangs, i);
    m_selectLang->insertStringList(localLangs);
    m_selectLang->setCurrentItem(i);

    m_ready++;
    if (m_ready == 2)
        enableButtonOK(true);
}

void KTUFirstRunWidget::appsFinished()
{
    m_ticking->stop();
    int elapsed = m_timer->elapsed();
    m_progress->setTotalSteps(elapsed);
    m_progress->setProgress(elapsed);

    kdDebug() << "Time elapsed to fetch apps: " << elapsed << endl;

    m_labelApp->setText(i18n("Fetched!"));
    m_labelApp->setAlignment(Qt::AlignRight);
    kapp->config()->setGroup("General");
    kapp->config()->writeEntry("LastBuildTemplates",QDateTime::currentDateTime());

    m_ready++;
    if (m_ready == 2)
        enableButtonOK(true);
}

void KTUFirstRunWidget::slotUpdateProgress()
{
    m_progress->setProgress( m_timer->elapsed() );
}

#include "ktufirstrunwidget.moc"
