/* This file is part of the KDE project
   Copyright (C) 2006 Tom Albers <tomalbers@kde.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <qfile.h>
#include <qregexp.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>

#include "ktuincludes.h"

void KTUIncludes::getLanguageList(QMap<QString,QString>& list,
                                  QStringList& langSorted,
                                  int& defaultLangIndex)
{
    QString curLang;

    kapp->config()->setGroup("General");
    curLang = kapp->config()->readEntry("Lang");
    kdDebug() << "According to the config file " << curLang << endl;
    if (curLang.isEmpty())
    {
        KConfig *config = KGlobal::config();
        config->setGroup("Locale");
        curLang = config->readEntry("Language").section(':',0,0);
        kdDebug() << "According to KGlobal you prefer: " << curLang << endl;
    }

    kdDebug() << "Set to language: " << curLang << endl;

    // Fetch all languages in the local language. There is no KDE access to it ;-(
    QString file = locate("locale", "all_languages");
    QFile g(file);
    g.open(IO_ReadOnly);
    QString mainlang,s;
    QMap<QString,QString> translatedLang;
    QString firstpart = "Name["+curLang+"]=";
    while (!g.atEnd())
    {
        if (g.readLine(s,1024) < 1) break;
        if (s.startsWith("["))
        {
            mainlang = QString::fromUtf8(s.remove("]").remove("[").stripWhiteSpace());
            QString u = KGlobal::locale()->twoAlphaToLanguageName(mainlang);
            translatedLang[mainlang] = !u ? mainlang : u;
        }
        if (s.startsWith(firstpart))
        {
            QString t = QString::fromUtf8(s.right(
                    s.length()-firstpart.length()).stripWhiteSpace());
            translatedLang[mainlang] = t;
        }
    }

    // We now parse the fetched languages
    QRegExp re("([A-Z_@]+)=([A-Z ]+)",false);
    int pos;

    QFile f(locateLocal("appdata","") + "teamnames");
    if (!f.open(IO_ReadOnly))
        return;

    while (!f.atEnd())
    {
        if (f.readLine(s,1024) < 1) break;
        pos = re.search(s);
        if (pos >= 0)
        {
            QString abbr = re.cap(1);
            QString t = translatedLang[abbr];
            list[t] = abbr;
        }
    }

    // localise the dafaultLang
    QString defaultLang=translatedLang[curLang];

    // Now sort all the keys to get a localized alfabetic list.
    langSorted = list.keys();
    qHeapSort(langSorted);

    // Iterate one more time to find the index of this lang.
    int i=0;
    QStringList::Iterator it;
    for (it = langSorted.begin(); it != langSorted.end(); ++it)
    {
        if ((*it) == defaultLang)
            defaultLangIndex=i;
        i++;
    }
}

QString KTUIncludes::findShortCode(QMap<QString,QString>& languages,
                                         const QString& lang)
{
    QMap<QString,QString>::Iterator it;
    for (it = languages.begin(); it != languages.end(); ++it)
    {
        if (it.key() == lang)
            return it.data();
    }
    return QString::QString();
}

int KTUIncludes::getRealBranch(int branch, const QString& module)
{
    if (branch == 0)
    {
        if (module.startsWith("playground-") ||
            module.startsWith("extragear-") ||
            module.startsWith("others") ||
            module.startsWith("kdenonbeta"))
            return(1);
        else
            return(2);
    }
    else
        return(branch);
}

void KTUIncludes::checkRequirements()
{
    if ( !KStandardDirs::findExe("msgfmt"))
    {
        KMessageBox::error(0,i18n("The application 'msgfmt' is not found. "
                "Please install it, it is part of the GNU Gettext package."));
        exit(0);
    }

    if ( !KStandardDirs::findExe("svn"))
    {
        KMessageBox::error(0,i18n("The application 'svn' is not found. "
                "Please install it, it is part of the Subversion package."));
        exit(0);
    }
}
