/* This file is part of the KDE project
   Copyright (C) 2006 Tom Albers <tomalbers@kde.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <qfile.h>

#include <kstandarddirs.h>
#include <kdebug.h>
#include <kprocio.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <klistview.h>
#include <ktempfile.h>

#include "ktustatitem.h"
#include "ktuincludes.h"
#include "ktuwidget.h"
#include "ktustats.h"
#include "ktufetch.h"
#include "db.h"

KTUStatItem::KTUStatItem(const QString& lang, const QString& app,
                         const QString& module, int branch)
    : QObject(0,app)
    , m_trans(0)
    , m_untrans(0)
    , m_validlv(false)
{
    m_lang=lang;
    m_app=app;
    m_module=module;
    m_branch=KTUIncludes::getRealBranch(branch, module);
}

KTUStatItem::~KTUStatItem()
{
}

void KTUStatItem::insertInTable(QListView* table)
{
    if (m_validlv)
        return;

    QString branch;
    m_branch == 1 ? branch=i18n("Trunk") :  branch=i18n("Stable");
    m_lv = new KListViewItem(table, m_app, m_module, branch );

    setInstalled(isInstalled(), false /* no need to update db */);
    m_lv->setText(7, QString::number(m_branch));
    m_validlv=true;
    setStats();
}

void KTUStatItem::slotUpdateStats()
{
    if (QFile::exists(poFile()))
    {
        KTUStats* t = new KTUStats(this);
        t->getStats(poFile(),m_trans,m_untrans);
        setStats();
    }
}

void KTUStatItem::slotShowStats(int trans, int untrans)
{
    m_trans = trans;
    m_untrans = untrans;
    setStats();
}

void KTUStatItem::remove( bool keepPoFile )
{
    if (!keepPoFile)
    {
        QFile::remove(poFile());
        KTUWidget::i()->dbI()->removeInstalled(poFile());
    }
    else
        setInstalled(false);

    QFile::remove(moFile());
}

void KTUStatItem::setStats()
{
    if (!m_validlv)
    {
        kdDebug() << "Stats received, but the listviewitem is gone" << endl;
        return;
    }

    m_lv->setText(4,QString::number(m_trans));
    m_lv->setText(5,QString::number(m_untrans));
    if (m_untrans > 0)
        m_lv->setText(6,QString::number(m_trans*100/(m_trans+m_untrans))+'%');
    else
        m_trans ? m_lv->setText(6,"100%"): m_lv->setText(6,"0%");
}

bool KTUStatItem::isInstalled()
{
    return KTUWidget::i()->dbI()->isInstalled(poFile());
}

void KTUStatItem::setInstalled(bool isInstalled, bool updateDB)
{
    m_app.startsWith("desktop_") ?
            m_lv->setText(3, i18n("Not possible to install", "-")) :
            isInstalled ? m_lv->setText(3, i18n("Installed","Yes")) :
                m_lv->setText(3, i18n("Installed","No"));

    if (updateDB)
        KTUWidget::i()->dbI()->setInstalled(poFile(), isInstalled);
}

void KTUStatItem::mailDiff()
{
    QString output;
    KTUFetch* fetch = new KTUFetch( this );
    fetch->diff(svnFile(), poFile(), output);

    bool completeFile=false;
    if (output.isEmpty())
    {
        int i = KMessageBox::warningContinueCancel(0,
               i18n("There is no difference between your "
                    "local version and the one located at KDE. There is no need to "
                    "mail anything. If you feel this is in error, please send "
                    "the complete file to the coordinator. Press continue to "
                       "do so."),
               i18n("No difference found"), KStdGuiItem::cont(),
               "send_complete_file");

        if (i == KMessageBox::Continue)
            completeFile=true;
        else
            return;
    }

    QString subject;
    QString filename;
    QString body;

    if (!completeFile)
    {
        KTempFile file = KTempFile( locateLocal("tmp","ktu_"), ".diff");
        QFile commentsFile( file.name() );
        QTextStream stream( &commentsFile );
        stream.setEncoding( QTextStream::UnicodeUTF8 );
        commentsFile.open( IO_WriteOnly );
        stream << output << "\n";
        commentsFile.close();

        filename = file.name();
        subject  = i18n("Subject of the mail with application(1), module(2) "
                        "and branch(3)", "[PATCH] Modifications to %1 (%2/%3)")
                       .arg( m_app, m_module, m_lv->text(2));
        body      = i18n("Body of the email","Please review attached patch");
    }
    else
    {
        filename = poFile();
        subject  = i18n("Subject of the mail with application(1), module(2) "
                        "and branch(3)", "[TRANSLATION] %1 (%2/%3)")
                       .arg( m_app, m_module, m_lv->text(2));
        body      = i18n("Body of the email",
                         "Please review attached translation.");
    }

    KProcess* mailAgentProc = new KProcess;
    *mailAgentProc << "kmail";
    *mailAgentProc << "-s" << subject;
    *mailAgentProc << "--body" << body;
    *mailAgentProc << "--attach" << QFile::encodeName(filename);
    connect(mailAgentProc , SIGNAL(processExited(KProcess *)),
            SLOT(slotFinished(KProcess *)));
    if ( mailAgentProc->start() == false )
        KMessageBox::error(0, i18n("Cannot start KMail.\nplease "
                        "check your installation. Currently KMail is the only "
                        "supported mail application"));

    // I know this doesn't work with most other mailers, but when it does, this
    // is better:

//     KApplication::kApplication()->invokeMailer(
//             QString::null,                     // Destination address.
//             QString::null,                     // Carbon Copy address.
//             QString::null,                     // Blind Carbon Copy address
//             QString::null,                     // Message Subject.
//             QString::null,                     // Message Body.
//             QString::null,                     // Message Body File.
//             file.name());   // attachments
}

void KTUStatItem::slotFinished(KProcess *p)
{
    delete p;
}

QString KTUStatItem::poFile()
{
    return locateLocal("appdata","checkout/" + m_lang + '/' +
            QString::number(m_branch) + '/' + m_module +'/' ) +
            m_app + ".po";
}

QString KTUStatItem::moFile()
{
    return locateLocal("locale", m_lang + "/LC_MESSAGES/")
            + m_app + ".mo";
}

QString KTUStatItem::svnFile()
{
    QString result;
    int type = m_branch;
    switch (type)
    {
        case 1:
        {
            result = "svn://anonsvn.kde.org/home/kde/trunk/l10n/" +
                     m_lang + "/messages/" + m_module + '/' + m_app  + ".po";
            break;
        }
        case 2:
        {
            result = "svn://anonsvn.kde.org/home/kde/branches/stable/l10n/" +
                     m_lang + "/messages/" + m_module + "/" + m_app  + ".po";
            break;
        }
    }
    return result;
}

void KTUStatItem::slotInstall()
{
    if (m_app.startsWith("desktop_"))
        return;

    KProcIO msgfmt;
    msgfmt << "msgfmt" << "-o" << moFile() << poFile();
    msgfmt.start(KProcess::Block);
    if (msgfmt.normalExit())
    {
        if (msgfmt.exitStatus())
        {
            KMessageBox::detailedError(0,i18n("Installation failed"),
                                       "Msgfmt returned "+msgfmt.exitStatus() );
            kdWarning() << "Msgfmt returned " << msgfmt.exitStatus() << endl;
            setInstalled(false);
        }
        else
        {
            kdDebug() << m_app << " is installed at: " << moFile() << endl;
            KMessageBox::information(0, i18n("The translation for %1 is now "
                    "installed, to see it in the application you need to "
                    "restart it.").arg(getApp()),
                    i18n("Installed"),"install_success");
            setInstalled(true);
        }
    }
    else
    {
        KMessageBox::detailedError(0,i18n("Installation failed"),
                                   "Msgfmt died" );
        kdWarning() << "Msgfmt died." << endl;
        setInstalled(false);
    }
}

void KTUStatItem::slotFailed(const QString &error)
{
    KMessageBox::detailedError(0,i18n("Download failed, this can "
            "happen when the file in that language is not available "
                    "or when there are connection problems. See details"),
            error);
    KTUWidget::i()->dbI()->removeInstalled(poFile());
    emit removeMe(this);
}

#include "ktustatitem.moc"
