/* This file is part of the KDE project

   Copyright (C) 2006 Tom Albers <tomalbers@kde.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

//#include <qlistview.h>

#ifndef KTUSTATITEM_H
#define KTUSTATITEM_H

/**
 * @class KTUStatItem
 * This class is responsible for holding info on an item. It can recalculate
 * statistics, is responsible for all the text in the columns of the listview
 * and you can order it to make a diff and ivoke the mailer.
 *
 * @author Tom Albers <tomalbers@kde.nl>
 */
class KTUStatItem : public QObject
{
    Q_OBJECT

    public:
        /**
         * Contructor
         * @param lang the language
         * @param app the name of the application
         * @param module the module where the application belongs
         * @param branch pass 1 for trunk and 2 for branch, 0 for auto.
         */
        KTUStatItem(const QString& lang, const QString& app,
                    const QString& module, int branch);


        /**
        * Destructor
        */
        ~KTUStatItem();

        /**
         * Pointer to the qlistviewitem
         */
        KListViewItem* lvitem(){ return m_lv; }

        /**
         * reinsert it in the @p table
         */
        void insertInTable(QListView *table);

        /**
         * remove the translation, when @p removePo is true, the pofile
         * will be removed as well.
         */
        void remove( bool );

        /**
         * Will find out what the diff is and invoke KMail.
         */
        void mailDiff();

        /**
         * returns the full path to the po-file
         */
        QString poFile();

        /**
         * returns the full path to the go-file
         */
        QString moFile();

        /**
         * returns the full path to the svn-file
         */
        QString svnFile();

        /**
         * returns the languageAbbr
         */
        QString getLang() {return m_lang;}

        /**
         * returns the name of the application
         */
        QString getApp()  {return m_app;}

        /**
         * returns the module of the application
         */
        QString getMod()  {return m_module;}

        /**
         * returns true if it is installed
         */
        bool getInstalled()  {return m_lv->text(3);}

        /**
         * returns the branch of this file, this is either trunk(1) or stable(2)
         */
        int getBranch()  {return m_branch;}

        /**
         * @p valid can be set to indicate that this item is removed from the
         * listview. If you don't use that, thinks like poFile() will crash.
         * It will be set to true automatically when insertInTable is used.
         */
        void setValidlv(bool valid) { m_validlv = valid; }

        /**
         * returns if this is a valid listview item.
         */
        bool isValidlv() { return m_validlv; }

        /**
         * returns if this item is installed.
         */
        bool isInstalled();

         /**
          * sets the installed column / db
          */
        void setInstalled(bool isInstalled, bool updateDB=true);


    public slots:
        void slotInstall();
        void slotFailed(const QString &error);
        void slotFinished(KProcess *p);
        /**
         * Updates the statistics
         */
        void slotUpdateStats();


    private:
        void setStats();


        KListViewItem*  m_lv;
        QString         m_lang;
        QString         m_app;
        QString         m_module;
        int             m_branch;
        int             m_trans;
        int             m_untrans;
        bool            m_validlv;

    private slots:
        void slotShowStats(int trans, int untrans);

    signals:
        void removeMe(KTUStatItem*);
};

#endif // KTUSTATITEM_H

