/* This file is part of the KDE project
   Copyright (C) 2006 Tom Albers <tomalbers@kde.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <qdir.h>
#include <qlayout.h>
#include <qpushbutton.h>

#include <kactionclasses.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kdirwatch.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <klistview.h>
#include <klistviewsearchline.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kprocio.h>
#include <kprogress.h>
#include <kstandarddirs.h>
#include <kstatusbar.h>

#include "db.h"
#include "ktufetch.h"
#include "ktuincludes.h"
#include "ktustatitem.h"
#include "ktustats.h"
#include "ktuwidget.h"

class KTUWidgetPriv
{
    public:
        KTUDB   *db;
};

KTUWidget* KTUWidget::m_instance=0;

KTUWidget* KTUWidget::i()
{
    return m_instance;
}

KTUWidget::KTUWidget(   )
  : KMainWindow(0)
{
    m_instance = this;
    d=new KTUWidgetPriv;

    checkRequirements();

    kapp->config()->setGroup("General");
    m_currentLang = kapp->config()->readEntry("Lang");

    d->db = new KTUDB;
    d->db->setDBPath(m_dataDir+"/ktu.db");

    m_box = new QWidget(this);
    QGridLayout* gbox = new QGridLayout(m_box,0,0);

    QLabel *l2 = new QLabel(i18n("Module:")+' ',m_box);
    m_mod = new QComboBox(m_box);
    m_mod->insertItem(i18n("Being loaded"));
    m_mod->insertItem("   playground-sysadmin   "); // size
    l2->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    l2->setBuddy(m_mod);
    connect(m_mod, SIGNAL(activated(int)),
            SLOT(slotPullDownChanged()));

    QLabel *l3 = new QLabel(i18n("Application:")+' ',m_box);
    m_app = new QComboBox(m_box);
    m_app->insertItem(i18n("Being loaded"));
    m_app->insertItem("   digikamimageplugin_lensdistortion   "); // size
    l3->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    l3->setBuddy(m_app);

    m_fetch = new QPushButton(i18n("Get this"), m_box);
    connect(m_fetch, SIGNAL(clicked()), SLOT(slotFetchClicked()));

    m_table = new KListView(m_box);
    m_table->addColumn( i18n("Application") );
    m_table->addColumn( i18n("Module") );
    m_table->addColumn( i18n("Branch") );
    m_table->addColumn( i18n("Installed") );
    m_table->addColumn( i18n("Translated") );
    m_table->setColumnAlignment(4, Qt::AlignRight);
    m_table->addColumn( i18n("Untranslated") );
    m_table->setColumnAlignment(5, Qt::AlignRight);
    m_table->addColumn( i18n("Percentage") );
    m_table->setColumnAlignment(6, Qt::AlignRight);
    m_table->setAllColumnsShowFocus(true);
    m_table->setSelectionMode(QListView::Extended);
    connect( m_table,
             SIGNAL( contextMenuRequested(QListViewItem*,const QPoint&, int)),
             SLOT( slotTableContext(QListViewItem*,const QPoint&)));

    KListViewSearchLineWidget* searchLine =
            new KListViewSearchLineWidget(m_table, m_box);

    gbox->addWidget(l2,0,0);
    gbox->addWidget(m_mod,0,1);
    gbox->addWidget(l3,0,2);
    gbox->addWidget(m_app,0,3);
    gbox->addWidget(m_fetch,0,4);
    gbox->addMultiCellWidget(searchLine,1,1,0,4);
    gbox->addMultiCellWidget(m_table,2,2,0,4);

    // KTU Menu
    KStdAction::quit(this, SLOT(slot_exit()), actionCollection(),  "app_exit");

    // Update menu
    new KAction(i18n("Update available &applications"), "updateapps", Key_F4,
                this, SLOT(slotUpdateApplications()), actionCollection(),
                "update_apps");
    new KAction(i18n("Update available &languages"), "updatelangs", Key_F3,
                this, SLOT(slotUpdateTeamnames()), actionCollection(),
                "update_langs");
    m_delete = new KAction(i18n("&Delete all locally installed translations"),
                           "updatelangs", Key_F5, this,
                           SLOT(slotDeleteAllTranslations()),
                           actionCollection(), "delete_all_installed_apps");
    slotEnableDeleteAllTranslations();

    // Settings menu
    m_typeMenuList = new KSelectAction(i18n("&Default branch"), 0, 0,
                                       actionCollection(), "select_version");
    QStringList typesList;
    typesList.append(i18n("Menu option to select the translation tree",
                          "Automatic"));
    typesList.append(i18n("Menu option to select the translation tree",
                          "Trunk"));
    typesList.append(i18n("Menu option to select the translation tree",
                          "Stable"));

    m_typeMenuList->setItems(typesList);
    connect(m_typeMenuList, SIGNAL(activated(int)), SLOT(slotTypeChanged(int)));

    m_langMenuList = new KSelectAction(i18n("&Language"), 0, 0,
                                       actionCollection(), "select_lang");
    connect(m_langMenuList, SIGNAL(activated(int)),
            SLOT(slotChangeLang()));

    m_box->show();
    setCentralWidget(m_box);

    setupGUI(Keys | StatusBar | Create | Save);

    // Read config for m_type
    kapp->config()->setGroup("General");
    m_currentBranch = kapp->config()->readNumEntry("Type_of_translation", 0);
    m_typeMenuList->setCurrentItem(m_currentBranch);

    statusBar()->insertItem(i18n("Ready"),1,10);
    statusBar()->insertItem(i18n("Statusbar", "Unknown"),2,0);
    statusBar()->insertItem(m_typeMenuList->currentText(),3,0);
    statusBar()->setItemAlignment(1,Qt::AlignLeft);

    QTimer::singleShot(0, this, SLOT(slotGetTeamnames()));
    QTimer::singleShot(0, this, SLOT(slotGetApplications()));

    // DirWatch to enable/disable the menuitem.
    m_dir = new KDirWatch(m_box);
    m_dir->addDir(locateLocal("locale", currentLanguage() + "/LC_MESSAGES/"));
    connect(m_dir, SIGNAL(dirty(const QString&)),
            SLOT(slotEnableDeleteAllTranslations()));

    // This will check for the m_needStats array to see if there are any
    // stats needed. It will call the timerEvent
    startTimer(20);
}

QString KTUWidget::currentLanguage()
{
    if (m_currentLang.isEmpty())
        m_currentLang=KTUIncludes::findShortCode( m_languages,
                                             m_langMenuList->currentText());
    return m_currentLang;
}

KTUWidget::~KTUWidget()
{
    QValueListIterator< KTUStatItem* > ita;
    for (ita = m_statItems.begin(); ita != m_statItems.end(); ++ita)
        delete (*ita);

    kapp->config()->setGroup("General");
    kapp->config()->writeEntry("Lang", currentLanguage());
    kapp->config()->writeEntry("Type_of_translation", m_currentBranch);
    kapp->config()->sync();

    delete d->db;
    delete d;
    m_instance=0;
}

void KTUWidget::checkRequirements()
{
    KTUIncludes::checkRequirements();

    kapp->config()->setGroup("General");
    m_edit = m_currentLang = kapp->config()->
            readEntry("KBabelCustomPath", KStandardDirs::findExe("kbabel"));
    kdDebug() << "Location of KBabel: " << m_edit << endl;
    if ( m_edit.isEmpty() )
        KMessageBox::information(0,i18n("The application KBabel is not found. "
                "If you want to edit files, you should install it."),
                i18n("KBabel not found"),"no_kbabel");

    m_dataDir = locateLocal("appdata","");
}

void KTUWidget::getTeamnames(bool force)
{
    QFile f(locateLocal("appdata","") + "teamnames");
    kapp->config()->setGroup("General");
    QDateTime* t = new QDateTime( QDateTime::currentDateTime().addSecs((-86400*7)-1));
    QDateTime lastUpdate = kapp->config()->readDateTimeEntry("LastBuildLanguages",t);
    if ((lastUpdate.secsTo(QDateTime::currentDateTime()) > 86400*7) ||
         !f.exists() || force)
    {
        slotStatusbarBusy(i18n("Updating languages"));

        KTUFetch* fetch = new KTUFetch( this );
        QString output;
        if (fetch->fetch("svn://anonsvn.kde.org/home/kde/trunk/l10n/teamnames",
            m_dataDir+"teamnames", output))
        {
            readTeamnames();
            kapp->config()->writeEntry("LastBuildLanguages",QDateTime::currentDateTime());
        }
        else
        {
            KMessageBox::detailedError(0,i18n("It was not possible to fetch "
                    "the languagelist..."),
                    output );
        }
        slotStatusbarReady();
    }
    else
        readTeamnames();

    delete t;
}

void KTUWidget::getApplications(bool force)
{
    // Check if there is a checkout of the templates, else fetch it and place it.
    QFile f(m_dataDir + "unstabletemplates");
    kapp->config()->setGroup("General");
    QDateTime* t = new QDateTime( QDateTime::currentDateTime().addSecs((-86400*7)-1));
    QDateTime lastUpdate = kapp->config()->readDateTimeEntry("LastBuildTemplates",t);
    if ((lastUpdate.secsTo(QDateTime::currentDateTime()) > 86400*7) ||
         !f.exists() || force)
    {
        slotStatusbarBusy(i18n("Updating modules and applications"));

        QString output;
        KTUFetch* fetch = new KTUFetch( this );
        if (fetch->fetchDirContents(
            "svn://anonsvn.kde.org/home/kde/trunk/l10n/templates/messages",
            m_dataDir+"unstabletemplates", output))
        {
            readApplications();
            kapp->config()->writeEntry("LastBuildTemplates",QDateTime::currentDateTime());
        } else {
            KMessageBox::detailedError(0,i18n("It was not possible to fetch "
                    "the applicationlist..."),
                    output );
        }
        slotStatusbarReady();
    }
    else
        readApplications();

    delete t;
}

void KTUWidget::getInstalledApplications()
{
    // If this is the first run, teamnames does not have to be executed yet.
    if (currentLanguage().isEmpty())
        return;

    int modIndex = m_mod->currentItem();
    QString mod = m_modcomboitems[m_mod->currentItem()];

    for (int branch=1; branch <= 2 ; branch++)
    {
        QDir d;
        if (!d.cd(m_dataDir + "checkout/" + currentLanguage()
             + '/' + QString::number(branch) + '/'))
        {
            kdDebug() << "Nothing installed in "<< branch << " for "
                    << currentLanguage() << endl;
        }
        else
        {
            d.setFilter( QDir::Dirs | QDir::NoSymLinks);
            const QFileInfoList *list = d.entryInfoList();
            QFileInfoListIterator it( *list );
            QFileInfo *fi;
            while ((fi = it.current()) != 0)
            {
                QString module = fi->fileName();
                if ((module != ".") &&
                    ( module != "..") &&
                    ( module != ".svn"))
                {
                    QDir e;
                    e.cd(fi->absFilePath());
                    e.setFilter( QDir::Files | QDir::NoSymLinks) ;
                    const QFileInfoList *list2 = e.entryInfoList();
                    QFileInfoListIterator it2( *list2 );
                    QFileInfo *fi2;

                    while ((fi2 = it2.current()) != 0)
                    {
                        QString app = fi2->fileName().left( (fi2->fileName().length()-3));

                        // check for existing ones;
                        bool found=false;
                        QValueListIterator<KTUStatItem* > ita;
                        for (ita = m_statItems.begin(); ita != m_statItems.end(); ++ita)
                        {
                            if ( (*ita)->getApp() == app &&
                                    (*ita)->getBranch() == branch &&
                                    (*ita)->getLang() == currentLanguage())
                                found=true;
                        }

                        if (!found)
                        {
                            KTUStatItem* statItem = new KTUStatItem(
                                    currentLanguage(), app, module, branch);
                            m_statItems.append(statItem);
                            if ( module == mod || modIndex == 0)
                            {
                                statItem->insertInTable(m_table);
                                m_statItemsVisible.append(statItem);
                                m_needStat.append(statItem);
                            }
                        }
                        ++it2;
                    }
                }
                ++it;
            }
        }
    }
}

void KTUWidget::readTeamnames()
{
    kdDebug() << "Current language is " << currentLanguage() << endl;

    m_languages.clear();
    m_langMenuList->clear();

    QStringList localLangs;
    int i = 0;
    KTUIncludes::getLanguageList(m_languages, localLangs, i);
    m_langMenuList->setItems(localLangs);
    m_langMenuList->setCurrentItem(i);

    slotPullDownChanged();
}

void KTUWidget::readApplications()
{
    QFile f(m_dataDir + "unstabletemplates");
    if (!f.open(IO_ReadOnly))
    {
        kdWarning() << "Cannot open unstabletemplates for reading."  << endl;
    }
    else
    {
        m_appcomboitems.clear();
        m_app->clear();
        m_appcomboitemsshown[0] = qMakePair( QString("All"), QString("All"));
        m_app->insertItem(i18n("All"),0);

        m_modcomboitems.clear();
        m_mod->clear();
        m_mod->insertItem(i18n("All modules"), 0);

        QMap<QString, QString> appModuleMap;
        QString s;
        int i=1;
        while (!f.atEnd())
        {
            if (f.readLine(s,1024) < 1) break;
            QStringList appMod = QStringList::split('/',s);
            QString module=appMod[0];
            QString filename=appMod[1].stripWhiteSpace();
            if (!filename.isEmpty())
            {
                appModuleMap[filename] = module;
                //TODO: searching should be easier. We only want unique
                // modules in the array. Is that so hard?
                QValueList<QString> list=m_modcomboitems.values();
                QValueList<QString>::iterator it =
                qFind( list.begin(), list.end(), module );
                if ( it == list.end() )
                {
                    m_modcomboitems[i] = module;
                    m_mod->insertItem(module,i);
                    ++i;
                }
            }
        }
        m_mod->setEnabled( true );
        m_mod->setCurrentItem(0);

        i=1;
        QMap<QString,QString>::Iterator ita;
        for (ita = appModuleMap.begin(); ita != appModuleMap.end(); ++ita)
        {
            QString application = ita.key().left( (ita.key().length()-4) );
            m_appcomboitems[i] = qMakePair(ita.data(),application);
            m_app->insertItem(application,i);
            i++;
        }
        m_appcomboitemsshown = m_appcomboitems;
    }
}

void KTUWidget::slotPullDownChanged()
{
    // disable all current pointers.
    QValueListIterator< KTUStatItem* > ita;
    for (ita = m_statItems.begin(); ita != m_statItems.end(); ++ita)
        (*ita)->setValidlv(false);

    int modIndex = m_mod->currentItem();
    QString curMod = m_modcomboitems[modIndex];
    m_appcomboitemsshown.clear();
    m_app->clear();

    m_appcomboitemsshown[0] = qMakePair(curMod, QString("All"));
    m_app->insertItem(i18n("All"),0);
    int i2=1;

    // Update applications combo
    for (uint i=0; i < m_appcomboitems.count(); ++i)
    {
        QPair<QString, QString> temp = m_appcomboitems[i];
        if (temp.first == curMod || modIndex==0)
        {
            m_appcomboitemsshown[i2] = qMakePair(temp.first,temp.second);
            m_app->insertItem(temp.second,i2);
            i2++;
        }
    }

    m_table->clear();
    m_statItemsVisible.clear();

    // See if there are already some pointers.
    kapp->config()->setGroup("InstalledApplications");
    for (ita = m_statItems.begin(); ita != m_statItems.end(); ++ita)
    {
        if (((*ita)->getMod() == curMod || modIndex==0) &&
               ((*ita)->getLang() == currentLanguage()))
        {
            (*ita)->insertInTable(m_table);
            m_statItemsVisible.append(*ita);
        }
    }

    // no pointers, check folder...
    if (!m_statItemsVisible.count())
        getInstalledApplications();

    // Update statusbar.
    statusBar()->changeItem(m_langMenuList->currentText(),2);
}

void KTUWidget::installAllApps()
{

    int i = KMessageBox::warningContinueCancel(this,
            i18n("Retrieving all translations can take a while and causes a "
                    "load on the translation servers, you should only do this "
                    "when you have no other option."),
            i18n("Are you sure?"));
    if (i == KMessageBox::Cancel)
        return;

    KProgressDialog* progress = new KProgressDialog(this,"progress",
                                                   i18n("Retrieving all translations"),
                                                   i18n("Retrieving all translations"),
                                                   true);
    progress->setAutoClose(true);
    progress->progressBar()->setTotalSteps(m_modcomboitems.count());
    progress->progressBar()->setValue(0);
    progress->show();
    connect(progress,SIGNAL(slotCancel()), SLOT(slotStatusbarReady()));

    kdDebug() << "Retrieving all translations" << endl;
    QMap<int,QString>::Iterator ita;
    for (ita = m_modcomboitems.begin(); ita != m_modcomboitems.end(); ++ita)
    {
        if (!ita.data().isEmpty() && !progress->wasCancelled())
        {
            progress->setLabel(i18n("Retrieving translations for %1")
                    .arg(ita.data()));
            progress->progressBar()->setValue(ita.key());
            kapp->processEvents();
            installAllAppsInModule(ita.data());
        }
    }
    delete progress;
}

void KTUWidget::installAllAppsInModule(const QString& module)
{
    slotStatusbarBusy(i18n("Retrieving all translations for module %1")
            .arg(module));

    // first fetch the module
    int branch=KTUIncludes::getRealBranch(m_currentBranch, module);
    QString svnDir;
    if (branch == 1)
        svnDir = "svn://anonsvn.kde.org/home/kde/trunk/l10n/" +
                currentLanguage() + "/messages/" + module;
    else if (branch == 2)
        svnDir = "svn://anonsvn.kde.org/home/kde/branches/stable/l10n/" +
                currentLanguage() + "/messages/" + module;
    else
        return;

    QString m = "svn/" + QString::number(branch) + '/' +
            currentLanguage() + "/messages/" + module;
    QString localFolder = locateLocal("appdata", m);

    kdDebug() << svnDir << "->" << localFolder << endl;

    QString output;
    KTUFetch* fetch = new KTUFetch( this );
    if (!fetch->fetchDir(svnDir, localFolder, output))
    {
        KMessageBox::detailedError(0,i18n("It was not possible to retrieve "
                                         "all the requested translations..."),
                                   output );
        slotStatusbarReady();
        return;
    }

    // install them piece by piece.
    QDir e;
    e.cd(localFolder);
    kdDebug() << "QDIR " << e.absPath () << endl;

    e.setFilter( QDir::Files | QDir::NoSymLinks) ;
    const QFileInfoList *list2 = e.entryInfoList();
    QFileInfoListIterator it2( *list2 );
    QFileInfo *fi2;

    while ((fi2 = it2.current()) != 0)
    {
        QString app = fi2->fileName().left( (fi2->fileName().length()-3));

        KTUStatItem* statItem=0;

        // Find existing statItem
        QValueListIterator<KTUStatItem* > ita;
        for (ita = m_statItemsVisible.begin(); ita != m_statItemsVisible.end(); ++ita)
            if ( (*ita)->getApp() == app && (*ita)->getBranch() == branch &&
                   (*ita)->getLang() == currentLanguage() )
                statItem = (*ita);

        if (!statItem)
        {
            statItem = new KTUStatItem(currentLanguage(), app, module, branch);
            statItem->insertInTable(m_table);
            m_statItems.append(statItem);
            m_statItemsVisible.append(statItem);
        }

        KIO::NetAccess::file_copy(fi2->absFilePath(),statItem->poFile());
        statItem->slotUpdateStats();
        statItem->slotInstall();
        ++it2;
    }
    slotStatusbarReady();
}

void KTUWidget::slotFetchClicked()
{
    QPair<QString,QString> installApp = m_appcomboitemsshown[m_app->currentItem()];

    kdDebug() << "Install all for: " << installApp.first
            << " - " << installApp.second << endl;

    if (m_app->currentItem() == 0)
    {
        if (m_mod->currentItem() == 0)
            installAllApps();
        else
            installAllAppsInModule(installApp.first);
        return;
    }

    slotStatusbarBusy(i18n("Retrieving %1 translation of %2")
            .arg(currentLanguage(), installApp.second));

    // search an existing listviewitem...
    int branch = KTUIncludes::getRealBranch(m_currentBranch,installApp.first);
    KTUStatItem* statItem=0;
    QValueListIterator<KTUStatItem* > ita;
    for (ita = m_statItems.begin(); ita != m_statItems.end(); ++ita)
        if ( (*ita)->getApp() == installApp.second &&
               (*ita)->getLang() == currentLanguage())
        {
            (*ita)->remove(true);
            if ((*ita)->getBranch() == branch)
               statItem = (*ita);
        }

    if (!statItem)
    {
        statItem = new KTUStatItem(currentLanguage(),
                                   installApp.second, installApp.first,
                                   m_currentBranch);
        connect(statItem, SIGNAL(removeMe(KTUStatItem*)),
                SLOT(slotRemoveItem(KTUStatItem*)));
        statItem->insertInTable(m_table);
        m_statItems.append(statItem);
        m_statItemsVisible.append(statItem);
    }

    QString output;
    KTUFetch* fetch = new KTUFetch( this );
    if (fetch->fetch(statItem->svnFile(), statItem->poFile(), output))
    {
        statItem->slotInstall();
        statItem->slotUpdateStats();
    } else {
        statItem->slotFailed(output);
    }
    slotStatusbarReady();
}

void KTUWidget::slotRemoveItem(KTUStatItem* statItem)
{
    statItem->remove( false );
    m_table->removeItem(statItem->lvitem());
    m_statItems.remove(statItem);
    m_statItemsVisible.remove(statItem);
    delete statItem;
}

void KTUWidget::removeInstallation(KTUStatItem* statItem)
{
    statItem->remove( true /* keep po file */);
}

void KTUWidget::slotTableContext( QListViewItem* item,  const QPoint & pos)
{
    if (!item)
        return;

    // retrieve the statItems selected. If the same app is selected in
    // different branches, we need to disable install...
    bool disableInstall = false;
    bool oneInstallable = false;
    bool oneUnInstallable = false;
    QValueList<KTUStatItem*> visible;
    QValueList<QString> visibleApps;

    QValueListIterator< KTUStatItem* > ita;
    for (ita = m_statItemsVisible.begin(); ita != m_statItemsVisible.end(); ++ita)
    {
        if ( (*ita)->lvitem()->isSelected() && (*ita)->lvitem()->isVisible())
        {
            visible.append(*ita);
            if (visibleApps.findIndex((*ita)->getApp()) == -1)
                visibleApps.append((*ita)->getApp());
            else
                disableInstall = true;

            if (!(*ita)->isInstalled())
                oneInstallable = true;
            else
                oneUnInstallable = true;

            if ((*ita)->getApp().startsWith("desktop_"))
                disableInstall = true;
        }
    }

    if (!visible.count())
      return;

    QPopupMenu *menu = new QPopupMenu(m_box);

    menu->insertItem(i18n("&Edit"), 0);
    menu->insertItem(i18n("E&dit && Install"), 1);
    menu->insertItem(i18n("&Install"), 2);
    menu->insertSeparator();
    menu->insertItem(i18n("&Remove translation"), 3);
    menu->insertItem(i18n("&Complete Remove"), 4);
    menu->insertSeparator();
    menu->insertItem(i18n("&Update stats"), 5);
    menu->insertSeparator();
    menu->insertItem(i18n("&Mail difference"), 6);

    if (m_edit.isEmpty())
    {
        menu->setItemEnabled(0, false);
        menu->setItemEnabled(1, false);
    }

    if (disableInstall)
        menu->setItemEnabled(1, false);

    if (disableInstall || !oneInstallable)
        menu->setItemEnabled(2, false);

    if (!oneUnInstallable)
        menu->setItemEnabled(3, false);

    if (visible.count() > 1)
        menu->setItemEnabled(6, false);

    int i = menu->exec(pos);

    delete menu;

    QValueListIterator< KTUStatItem* > it;
    for (it = visible.begin(); it != visible.end(); ++it)
    {
        KTUStatItem* statItem=(*it);

        if (i == 0 || i ==1)
        {
            // Remove existing installs in all branches.
            QValueListIterator< KTUStatItem* > ita;
            for (ita = m_statItemsVisible.begin(); ita != m_statItemsVisible.end(); ++ita)
                if ( (*ita)->getApp() == statItem->getApp() &&
                       (*ita)->getLang() == currentLanguage())
                    removeInstallation((*ita));

            KProcess *kbabel = new KProcess();
            *kbabel << m_edit << "--nofork" << statItem->poFile();
            if (i == 1)
                connect(kbabel, SIGNAL(processExited(KProcess *)),
                        statItem, SLOT(slotInstall()));

            connect(kbabel, SIGNAL(processExited(KProcess *)),
                    statItem, SLOT(slotUpdateStats( )));

            connect(kbabel, SIGNAL(processExited(KProcess *)),
                    SLOT(slotFinished(KProcess *)));
            kbabel->start();
        }
        else if (i == 2)
        {
            // Remove existing installs in all branches.
            QValueListIterator< KTUStatItem* > ita;
            for (ita = m_statItemsVisible.begin(); ita != m_statItemsVisible.end(); ++ita)
                if ( (*ita)->getApp() == statItem->getApp() &&
                       (*ita)->getLang() == currentLanguage())
                    removeInstallation((*ita));
            QTimer::singleShot(0, statItem, SLOT(slotInstall()));
            m_needStat.prepend(statItem);
        }
        else if (i == 3)
            removeInstallation(statItem);
        else if (i == 4)
            slotRemoveItem(statItem);
        else if (i == 5)
            m_needStat.append(statItem);
        else if (i == 6)
            statItem->mailDiff();
    }
}

void KTUWidget::slotEnableDeleteAllTranslations()
{
    kdDebug() << "Evaluating: " << currentLanguage() << endl;
    QDir e;
    e.cd(locateLocal("locale", currentLanguage() + "/LC_MESSAGES/"));
    e.setFilter( QDir::Files | QDir::NoSymLinks);
    m_delete->setEnabled(e.count() > 0);
}

void KTUWidget::slotDeleteAllTranslations()
{
    QDir e;
    e.cd(locateLocal("locale", currentLanguage() + "/LC_MESSAGES/"));
    e.setFilter( QDir::Files | QDir::NoSymLinks) ;
    const QFileInfoList *list2 = e.entryInfoList();
    QFileInfoListIterator it2( *list2 );
    QFileInfo *fi2;

    QStringList toBeDeleted;
    while ((fi2 = it2.current()) != 0)
    {
        toBeDeleted.append(fi2->absFilePath());
        ++it2;
    }

    if (toBeDeleted.count() == 0)
      return;

    int i = KMessageBox::warningContinueCancelList   (  0,
            i18n("This option will delete all locally installed translations "
                    "which are not installed by your distribution. It is also "
                    "possible this contains translations installed outside KTU. "
                    "Please review the list below."), toBeDeleted,
            i18n("Confirm deletion"), KStdGuiItem::cont(),
            "delete_all_locally_installed_files");
    if (i == KMessageBox::Continue)
    {
        QStringList::Iterator it;
        for (it = toBeDeleted.begin(); it != toBeDeleted.end(); ++it)
        {
            kdDebug() << "removing " << (*it) << endl;
            QFile::remove(*it);
        }

        // As a consquence, nothing is installed anymore
        QValueListIterator< KTUStatItem* > ita;
        for (ita = m_statItemsVisible.begin(); ita != m_statItemsVisible.end(); ++ita)
            (*ita)->setInstalled(false);
    }
}

void KTUWidget::slotChangeLang()
{
    m_dir->removeDir(locateLocal("locale", currentLanguage() + "/LC_MESSAGES/"));
    m_currentLang=QString::null;
    m_currentLang=currentLanguage();

    kapp->config()->setGroup("General");
    kapp->config()->writeEntry("Lang", currentLanguage());
    kapp->config()->sync();

    m_dir->addDir(locateLocal("locale", currentLanguage() + "/LC_MESSAGES/"));
    slotEnableDeleteAllTranslations();
    readTeamnames();
}

void KTUWidget::slotTypeChanged(int id)
{
    m_currentBranch = id;
    statusBar()->changeItem(m_typeMenuList->currentText(),3);
}

void KTUWidget::slotGetTeamnames()
{
    getTeamnames(false);
}

void KTUWidget::slotGetApplications()
{
    getApplications(false);
}

void KTUWidget::slotUpdateTeamnames()
{
    getTeamnames(true);
}

void KTUWidget::slotUpdateApplications()
{
    getApplications(true);
}

void KTUWidget::slotStatusbarBusy(const QString& msg)
{
    m_fetch->setEnabled( false );
    m_table->setEnabled( false );
    m_app->setEnabled(false);
    m_mod->setEnabled(false);
    statusBar()->changeItem(msg,1);
    kapp->processEvents();
}

void KTUWidget::slotStatusbarReady()
{
    m_fetch->setEnabled( true );
    m_table->setEnabled( true );
    m_app->setEnabled( true );
    m_mod->setEnabled( true );
    statusBar()->changeItem(i18n("Ready"),1);
}

void KTUWidget::timerEvent( QTimerEvent * )
{
    if (!m_needStat.count())
        return;

    KTUStatItem* statItem = m_needStat.first();
    if (statItem->isValidlv())
        statItem->slotUpdateStats();
    else
        kdDebug() << "Stats request was queued, but is gone: "
            << statItem->name() << endl;

    m_needStat.pop_front();
}

void KTUWidget::slotFinished(KProcess *p)
{
    delete p;
}

KTUDB* KTUWidget::dbI()
{
    return d->db;
}

void KTUWidget::slot_exit()
{
    close();
}
#include "ktuwidget.moc"
