//
// C++ Interface: kpgdatatable
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#ifndef KPGDATATABLE_H
#define KPGDATATABLE_H

// Note that for new code which you wish to be strictly Unicode-clean, you can define the macro QT_NO_ASCII_CAST when compiling your code.
#define QT_NO_ASCII_CAST

// include files for Qt
#include <qtable.h>
#include <qvaluelist.h>

// include files for KDE
#include <kurl.h>
#include <kxmlguiclient.h>

// application specific includes
#include "../kpgconnectioninthread.h"
#include "../DbObjects/kpgtableindex.h"

// include files for libpgxx 
#include <pqxx/largeobject.hxx>

class QTextCodec;
class QTable;
class KFind;
class KXMLGUIFactory;

class KPoGreView;
class KPGConnection;
class KPGDatabase;
class KPGDataTableChildView;
class KPGKateXmlEditorDialog;

/**
 * Stores information about column in data table
 *
 */
class KPGDataTableColumn
{
public:
	KPGDataTableColumn() {};
	KPGDataTableColumn(const QString &strName, const QString &strTypName, bool bNotNull, bool bHasDef, const QString &strAdSrc, const QString &strDescription, const QString &strCreatedAs)
	{
		m_strName = strName;
		m_strTypName = strTypName;
		m_bNotNull = bNotNull;
		m_bHasDef = bHasDef;
		m_strAdSrc = strAdSrc;
		m_strDescription = strDescription;
		m_strCreatedAs = strCreatedAs;
	}
	
	const QString &name() const { return m_strName; }
	const QString &typName() const { return m_strTypName; }
	bool hasDefault() const { return m_bHasDef; }
	bool isNotNull() const { return m_bNotNull; }
	const QString &defaultValue() const { return m_strAdSrc; }
	const QString &description() const { return m_strDescription; }
	const QString createdAs() const { return m_strCreatedAs; }
	
protected:
    QString m_strName;
    QString m_strTypName;
	bool m_bNotNull;
	bool m_bHasDef;
	QString m_strAdSrc;
	QString m_strDescription;
	QString m_strCreatedAs;
};

/**
 * Stores information about field in data table
 *
 */
class KPGDataTableField
{
public:
	KPGDataTableField() {};
	KPGDataTableField(const QString &strOldValue)
	{
		m_strOldValue = strOldValue;
		m_bIsDirty = false;
	}
	
	void setDirty() { m_bIsDirty = true; }
	void clearDirty() { m_bIsDirty = false; }
	
	const QString & oldValue() const { return m_strOldValue; }
	const bool isDirty() const { return m_bIsDirty; }
	
protected:

	QString m_strOldValue; 	// Old value of field, used for cancelChanges()
	bool m_bIsDirty;		// Is field changed by user ?
};

// List of fields for edited row 	
typedef QValueList<KPGDataTableField> ListRowFields;
     
   // List of table columns 	
typedef QValueList<KPGDataTableColumn> ListTableColumns;

/**
  * Editable datatable to edit database table content
  *
  * @author Lumir Vanek <lvanek@users.sourceforge.net>
  */
class KPGDataTable : public QTable, virtual public KXMLGUIClient
{
	Q_OBJECT
public:
    KPGDataTable(KPGDataTableChildView *, 
		KPoGreView *, 
		KXMLGUIFactory *, 
		const PGSTD::string &, 
		const QString &, 
		const QString &, 
		const ListTableColumns &,
		const MapIndexKey &, 
		const QString &, 
		const QString &, 
		KPGKateXmlEditorDialog *);
    
    ~KPGDataTable();
   
	// Add yourself to GUI factory
    void addToGuiFactory();
    
    // Remove yourself from GUI factory
    void removeFromGuiFactory();

     // Datatable states
     enum EState 
	 { 	
     	view = 0, 	// default state
     	update, 	// current row is edited (at least one field is dirty), we pending for UPDATE
     	insert, 	// last row is edited, we pending for INSERT 
     	refresh		// refreshing data, waiting for resultset
     };
     			
     // DML types
     enum EDmlType 
	 { 	
     	dmlInsert = 0, 
     	dmlUpdate, 	
     	dmlDelete 
     };
	
	// Filters events
	bool eventFilter(QObject *, QEvent *);
	
	// Run SQL query for reload data from database table
	void reloadData();
	
	// Terminate thread that run SQL query
	void stopReload();
	
	// Returns TRUE if the connection thread is running; otherwise returns FALSE.
	bool running() const { return  m_connectionInThread.running(); }
	
	// Move current row in data table
	void firstRow();
	void previousRow();
	void nextRow();
	void lastRow();
	
	// Edit selected cell in data table
	void editCell();
	
	// Clear selected cell in data table
    void clearCell();
    
    // Insert new row into data table
    void insertRow();
    
    // Delete selected row from data table
    void deleteRow();
    
    // UPDATE or INSERT changes from data table current row to database
    bool commitChanges();
    
    // Cancel changes in data table
    void cancelChanges();
	
	// LOB manipulation functions
	void loadLob(const KURL &);
	void saveLob(const KURL &);
	void editLobInEditor();
	
	// Find first occurence of text
    void findFirst(QStringList &);
    
    // Find next occurence of text
    void findNext();
	
protected:

	// This event handler is called whenever the QScrollView receives a mousePressEvent(): the press position in e is translated to be a point on the contents.
	virtual void contentsMousePressEvent( QMouseEvent* );
	
	// Reimplemented for update actions state purpose
	virtual QWidget *beginEdit(int, int, bool);
    virtual void endEdit(int, int, bool, bool);

    // Disable all actions
    void disableAllActions();
    
    // Enable/disable actions with regards to datatable state
    void setActions();
    
    // Receive event from KPGConnectionInThread
    virtual void customEvent( QCustomEvent * );
    
    // Display popup menu
	void popupContextMenu(const QString &, const QPoint &);

    // Display SQL result
    void displayResult();  
    
    // Display data from one reultset row
    void displayOneRow(int, unsigned int, unsigned int, pqxx::result &);
    
    // Refresh current row from database
    void refreshRow();
    
    // Generate new primary key value from sequence
	int generateSerialPrimaryKeyValue(pqxx::work &);
	long generateBigSerialPrimaryKeyValue(pqxx::work &);
    
    // Prepare DML statements
    void prepareInsertStatement(const std::string &);
    void prepareUpdateStatement(const std::string &);
    void prepareDeleteStatement(const std::string &);
    void prepareSelectStatement(const std::string &);
        
    // Build WHERE predicate for primary key columns. 
    const QString wherePredicateForPrimaryKey(int) const;
    
    // Fill key values into vParams, from iParamsIndex position
    void fillParametersForPrimaryKey(int, pqxx::prepare::invocation &) const;
    
    // Run DML string on DB server
	bool runDml(EDmlType);
	
	// Return SQL query for selecting edited row
    const QString sqlForSelectOneRow() const;
	
	// Return SQL query for selecting edited row
    const QString sqlForSelectAllRows() const;
    
    // Retrieve largeobject of LOB for given row and column
    pqxx::largeobject getLargeObject(int, int, std::string &, work &);
    
    // Store largeobject of LOB to database, for given row and column
    bool setLargeObject(pqxx::largeobject &, int, int, std::string &, work &);
      
    virtual void virtual_hook( int id, void* data );
    
private:
	
	void fixRow2( int &row, int y );
    
protected:

	// Edit actions
	KAction* m_pActEditFind;
	KAction* m_pActEditFindNext;
	
	// Data table actions
    KAction* m_pActCopyCell;
	KAction* m_pActCopyRow;
	KAction* m_pActCopyTableCsv;
	KAction* m_pActCopyTableXml;
    KAction* m_pActFirstRow;
    KAction* m_pActPreviousRow;
    KAction* m_pActNextRow;
    KAction* m_pActLastRow;
    KAction* m_pActEditCell;
    KAction* m_pActClearCell;
    KAction* m_pActInsertRow;
    KAction* m_pActDeleteRow;
    KAction* m_pActCommitChanges;
    KAction* m_pActCancelChanges;
    KAction* m_pActReloadData;
    KAction* m_pActStopReload;
    KAction* m_pActLoadLob;
    KAction* m_pActSaveLob;
    KAction* m_pActEditLobInEditor;
    
	// XML GUI factory
    KXMLGUIFactory * m_pXmlGuiFactory;
    
    // True, if this is in GUI factory
    bool m_bIsAddedToGuiFactory;

	// Datatable state
	EState m_state;
	
	// True, if current cell is edited
	bool m_bInplaceEditing;

	// Number of row with edited data
	int m_editedRow;
	
	// List of fields for m_editedRow - that user edited it's value
	ListRowFields m_listRowFields;
		
	// List of columns in table
	ListTableColumns m_listTableColumns;	
    
	// Connection options
	PGSTD::string m_strConnectionOptions;
	
    // Database table to edit its data
    QString m_strTableName;
        
    // Table namespace
    QString m_strNamespaceName;
          	
  	// Map of primary key index columns - pair of <int iColumnNum, QString strColumnName> 
  	MapIndexKey m_mapIndexKey;
  	
  	// Is PK autogenerated using sequence ?
  	bool m_bAutoGeneratedPrimaryKey;
  	
  	// If m_bAutoGeneratedPrimaryKey = true, contain SQL for generate new PK value
  	QString m_strNextValueForPrimaryKey;
    
    // If m_bAutoGeneratedPrimaryKey = true, contain PK column name
    QString m_strPKColumnName;
    
    // Thread that run queries on background
	KPGConnectionInThread m_connectionInThread;
	
	// The WHERE predicate, used to reduce anmount of edited data
	QString m_strWherePredicate;
	
	// The ORDER BY predicate, used to sort edited data
	QString m_strOrderByPredicate;
	
	// True, is show DML statements for user inspection, before executing it
	bool m_bShowDml;
	
	// True, if it's first load data 
	bool m_bOpeningReload;
	
	// Resultset to display
	pqxx::result m_pqxxResult;
	
	// For searching in data tables and result tables
	QStringList m_listOfResultQuerySearchHistory;
	
private:
	// A generic implementation of the "find" function
	KFind *m_pFind;
	
	int m_iRowToSearch;
	int m_iColToSearch;
	
	// Dialog for editing XML datatypes
	KPGKateXmlEditorDialog *m_pKateXmlEditorDialog;
	
protected slots:

	//--- Edit main menu actions
	void slotEditFind();
    void slotEditFindNext();

	//--- Datatable actions
    void slotFirstRow();
    void slotPreviousRow();
    void slotNextRow();
    void slotLastRow();
    void slotEditCell();
    void slotClearCell();
    void slotInsertRow();
    void slotDeleteRow();
    void slotCommitChanges();
    void slotCancelChanges();
    void slotReloadData();
    void slotStopReload();
    void slotLoadLob();
    void slotSaveLob();
    void slotEditLobInEditor();

    // Called, when the current cell in m_pTableData has changed to row, col
    void slotCurrentChanged(int, int);
    
    // Called, when user changed the value in the m_pTableData
	void slotValueChanged (int, int);
	
	// Called when mouse button button is double-clicked over m_pTableData
	void slotDoubleClicked(int, int, int, const QPoint &);
	
	// Called, when user make right-clickj over DataTable
	void slotContextMenuRequested(int, int, const QPoint &);
	
	// Called when user click on horizontal header 
	void slotHorzHeaderClicked(int);
    
    // Highligth found text
    void slotHighlight( const QString &, int, int);
    
    // Find next occurence of text
    void slotFindNext();
};

#endif
