//
// C++ Implementation: kpgdebugger
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2007
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kpgdebugger.h"

// include files for Qt
#include <qwidgetstack.h> 
#include <qtextcodec.h>
#include <qregexp.h>
#include <qtimer.h>

// include files for KDE
#include <klocale.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kcursor.h>
#include <kdebug.h>
#include <kcombobox.h>
#include <ktextedit.h>

// include files for KatePart
#include <kate/document.h>  // KatePart document

// application specific includes
#include "../kpgutil.h"
#include "../kpogreview.h"
//#include "../DbObjects/kpgconnection.h"
#include "../DbObjects/kpgdatabase.h"
#include "../DbObjects/kpgfunction.h"
#include "../kpgsqldialog.h"
#include "../DbObjects/kpgtablecolumn.h"
#include "kpgdebuggerchildview.h"

#define NUM_OF_RECORDS_PER_PAGE 500 // How many records are loaded to result table 
#define UPDATE_TIMER_INTERVAL 1000 // 1 sec for updatiing code completion

#define COL_NAME 0
#define COL_TYPE_NAME 1
#define COL_MODE 2
#define COL_VALUE 3

KPGDebugger::KPGDebugger(KPGDebuggerChildView *pParent, 
	KPoGreView *pPoGreView,
	KParts::Factory *pKatePartFactory,
	KXMLGUIFactory *pXmlGuiFactory,
 	const PGSTD::string& strConnectionOptions,
	const QString & strNamespaceName, 
	KPGFunction *pFunction
	)
	: 
	KXMLGUIClient(pPoGreView),
	KPGDebuggerBase(pParent, "KPGDebugger")
{
	// Setup GUI
	setXMLFile("kpgdebugger.rc", true);
	m_pXmlGuiFactory = pXmlGuiFactory;
	m_bIsAddedToGuiFactory = false;
	
	// Initialize members
	m_strConnectionOptions = strConnectionOptions;
	m_strNamespaceName = strNamespaceName;
	m_strFunctionName = pFunction->text(0);
	m_oidFunction = pFunction->oid();
	
	m_nTotalRows = 0;
	m_nTotalCols = 0;
	m_nFetchedRows = 0;
		
	//-----------------------------------------------------------------
	// Debugger actions
	m_pActRun = new KAction(i18n("Start in debugger"), "dbgrun", Key_F9, this, SLOT(slotExecute()), actionCollection(), "debugger_start");
	
	m_pActStop = new KAction(i18n("Stop"), "stop", 0, this, SLOT(slotStop()), actionCollection(), "debugger_stop");
  	
  	m_pActStepOver = new KAction(i18n("Step &Over"), "dbgnext", Key_F10, this, SLOT(slotStepOver()), actionCollection(), "debugger_step_over");

	m_pActStepInto = new KAction(i18n("Step &Into"), "dbgstep", Key_F11, this, SLOT(slotStepInto()), actionCollection(), "debugger_step_into");
	
	m_pActToggleBreakPoint = new KAction(i18n("Toggle &Breakpoint"), "kpg_togglebp", 0, this, SLOT(slotToggleBreakPoint()), actionCollection(), "debugger_toggle_bpoint");

	m_pActRun->setEnabled(true);
    m_pActStop->setEnabled(false);
    m_pActStepOver->setEnabled(false);
	m_pActStepInto->setEnabled(false);
	m_pActToggleBreakPoint->setEnabled(false);
    
    // Create Kate part editor
	m_pKateView = createKatePart(pKatePartFactory);
    m_pFrameSourceCodeLayout->addWidget(m_pKateView, 0, 0 );
        
			
	// install a working kate part popup dialog thingy
  	if (static_cast<Kate::View*>(m_pKateView->qt_cast("Kate::View")))
  	{
    	static_cast <Kate::View*> (m_pKateView->qt_cast("Kate::View"))->installPopup ((QPopupMenu*)(pXmlGuiFactory->container("ktexteditor_popup", pPoGreView)) );
  	}
    
    m_pKateView->setLineNumbersOn(true);
    m_pKateView->setIconBorder(true);
        
    KTextEditor::HighlightingInterface *pHighlightIface = KTextEditor::highlightingInterface(m_pKateView->getDoc());
        
    for(uint i = 0; i < pHighlightIface->hlModeCount(); i++) 
    {
    	//kdDebug() << "hlmode("<<i<<"): " << pHighlightIface->hlModeName(i) << endl;
        if (pHighlightIface->hlModeName(i).contains("SQL (PostgreSQL)", false))
        {
        	pHighlightIface->setHlMode(i);
        	break;
        }
     }
    
    setEditorText(pFunction->source());
	
	// Setup breakpoints icons	
	KIconLoader *loader = KGlobal::iconLoader();
	
	m_pKateView->getDoc()->setPixmap(KTextEditor::MarkInterface::BreakpointActive, loader->loadIcon("breakpoint_enabled", KIcon::Small));
	
	m_pKateView->getDoc()->setPixmap(KTextEditor::MarkInterface::BreakpointReached, loader->loadIcon("breakpoint_enabled", KIcon::Small));
	
	m_pKateView->getDoc()->setPixmap(KTextEditor::MarkInterface::BreakpointDisabled, loader->loadIcon("breakpoint_disabled", KIcon::Small));
				
	/*pMarkIface->setMark(1, KTextEditor::MarkInterface::Bookmark);
	
	pMarkIface->setMark(5, KTextEditor::MarkInterface::Execution);
	pMarkIface->setMark(6, KTextEditor::MarkInterface::Warning);
	pMarkIface->setMark(7, KTextEditor::MarkInterface::Error);*/
	
	// Setup tab icons
	m_pTabWidget->setTabIconSet(pTabParameters, QIconSet(QPixmap(UserIcon("argument"))));
	m_pTabWidget->setTabIconSet(pTabVariables, QIconSet(QPixmap(SmallIcon("math_brace"))));
	m_pTabWidget->setTabIconSet(pTabResult, QIconSet(QPixmap(SmallIcon("view_text"))));
	m_pTabWidget->setTabIconSet(pTabStack, QIconSet(QPixmap(SmallIcon("view_choose"))));
	m_pTabWidget->setTabIconSet(pTabTextResult, QIconSet(QPixmap(SmallIcon("openterm"))));
	
	//-----------------------------------------------------------------
	// Make own copy of arguments info
	m_listArguments = pFunction->listArguments();
	
	// Setup parameters table
	if(pFunction->listArguments().count() > 0)
	{
		m_pTableParameters->setNumRows(pFunction->listArguments().count());
		int nRow = 0;
		
		for(KPGFunction::ListArguments::const_iterator cit = pFunction->listArguments().begin(); cit != pFunction->listArguments().end(); ++cit)
		{ 
			m_pTableParameters->setText(nRow, COL_NAME, (*cit).name());
			
			if((*cit).typName().length() > 0)
			{
				m_pTableParameters->setText(nRow, COL_TYPE_NAME, (*cit).typName());
				
				QPixmap *pTypePixmap = KPGTableColumn::getPixmapForType((*cit).typName());
				m_pTableParameters->setPixmap(nRow, COL_NAME, *pTypePixmap);
			}
			
			
			if((*cit).mode() == "i") m_pTableParameters->setText(nRow, COL_MODE, "IN");
			else if((*cit).mode() == "o") m_pTableParameters->setText(nRow, COL_MODE, "OUT");
			else if((*cit).mode() == "b") m_pTableParameters->setText(nRow, COL_MODE, "INOUT");
			nRow++;
		}
	}
	
	m_pTableParameters->setColumnReadOnly(0, true);
	m_pTableParameters->setColumnReadOnly(1, true);
	m_pTableParameters->setColumnReadOnly(2, true);
	
	// Open connection
	m_connectionInThread.setEventReceiver(this, &m_pqxxResult);
	m_connectionInThread.setOperationType(KPGConnectionInThread::eRunPreparedStatement);
	
	try
	{
		m_connectionInThread.connectToServer(m_strConnectionOptions);
		
		QString strQuery("SET log_min_messages TO fatal");
		m_connectionInThread.connection()->runQuery(strQuery);
		
		KPGNoticer * pPgNoticer = m_connectionInThread.connection()->kpgNoticer();
		connect(pPgNoticer, SIGNAL(sigNotice(QString)), this, SLOT(slotProcessNotice(QString)));
	}
	catch (const std::exception &e)
	{
		kdError() << "Failed to open connection " << e.what() << endl;
		KMessageBox::sorry(this, e.what());
		return;
	}
	
	m_state = prepared;
	
	// Setup timer
	m_pTimer = new QTimer(this);
	connect(m_pTimer, SIGNAL(timeout()), this, SLOT(slotTimer()));
	m_pTimer->start( UPDATE_TIMER_INTERVAL ); 
}

KPGDebugger::~KPGDebugger()
{
	m_pTimer->stop(); // stop timer, but don't delete it
	
	KPGNoticer * pPgNoticer = m_connectionInThread.connection()->kpgNoticer();
	disconnect(pPgNoticer, SIGNAL(sigNotice(QString)), this, SLOT(slotProcessNotice(QString)));
	
	// Close connection
	m_connectionInThread.disconnectFromServer(); 
	
	if ( m_pKateView )
    {
        removeFromGuiFactory();

        // remove the view, then the doc
        Kate::Document *pDoc = m_pKateView->getDoc();
        delete m_pKateView;
        delete pDoc;
    }
}

// Add yourself and Kate view to GUI factory
void KPGDebugger::addToGuiFactory()
{
	if(m_pKateView && !m_bIsAddedToGuiFactory)
    {
        m_pXmlGuiFactory->addClient(this);
        m_pXmlGuiFactory->addClient(m_pKateView);
        m_bIsAddedToGuiFactory = true;
		m_pKateView->setFocus();
    }
}
    
// Remove yourself and Kate view from GUI factory
void KPGDebugger::removeFromGuiFactory()
{
    if(m_pKateView && m_bIsAddedToGuiFactory)
    {
        m_pXmlGuiFactory->removeClient(this);
        m_pXmlGuiFactory->removeClient(m_pKateView);
        m_bIsAddedToGuiFactory = false;
    }
}

// Create Kate part view
Kate::View* KPGDebugger::createKatePart(KParts::Factory* pFactory)
{
    // The library was found, so create the Kate::Document
    KTextEditor::Document *doc = (KTextEditor::Document *) pFactory->createPart(m_pFrameSourceCode, "", this, "", "KTextEditor::Document");

    // The document only represents the document, to view
    // the document's content
    // we have to create a view for the document.
	Kate::View *pView = (Kate::View *) doc->createView( m_pFrameSourceCode, 0L );

    // all went well, so return the view
	return pView;
}

// Set function source code
void KPGDebugger::setEditorText(const QString &strTest)
{
	KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
	pEditIface->setText(strTest);
		
	KTextEditor::UndoInterface *pUndoIface = KTextEditor::undoInterface(m_pKateView->document());
	pUndoIface->clearUndo();
	pUndoIface->clearRedo();
}

// Get function source code
const QString KPGDebugger::editorText() const 
{
	KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
    return pEditIface->text();
}

// Returns true, if breakpoint is set to given line
bool KPGDebugger::isBreakpointSet(unsigned int iLineNumber) const
{
	KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());
	
	QPtrList<KTextEditor::Mark> listOfMarks = pMarkIface->marks();
	KTextEditor::Mark *pMark = 0;
	for(pMark = listOfMarks.first(); pMark; pMark = listOfMarks.next())
	{
		if(pMark->line != iLineNumber) continue;
		
		if(pMark->type == KTextEditor::MarkInterface::BreakpointActive)
			return true;
	}
	return false;
}


void KPGDebugger::slotExecute()
{
	m_iProxyApiVersion = getProxyInfo();
	if(m_iProxyApiVersion < 3)
	{
		KMessageBox::sorry(this, "KPoGre requires debugging API version 3 or newer");
		return;
	}
	
	if(oidDebug() == false)
	{
		KMessageBox::sorry(this, "Activating debugger failed");
		return;
	}
	
	executeFunction();
	
	
}

void KPGDebugger::slotStop()
{
	stopDebugging();
}

void KPGDebugger::slotStepOver()
{

}

void KPGDebugger::slotStepInto()
{

}

void KPGDebugger::slotToggleBreakPoint()
{
	/*KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());
	
	
	pMarkIface->setMark(3, KTextEditor::MarkInterface::BreakpointReached);
	pMarkIface->setMark(4, KTextEditor::MarkInterface::BreakpointDisabled);*/
	
	uint uiLine, uiCol;
	viewCursorInterface(m_pKateView)->cursorPositionReal(&uiLine, &uiCol);
	
	if(isBreakpointSet(uiLine))
	{
		setBreakpoint(uiLine);
	}
	else
	{
		clearBreakpoint(uiLine);
	}
}

// Process notice from connection in thread
void KPGDebugger::slotProcessNotice(QString strNotice)
{
	qApp->lock();
	m_pTextEditResult->append(strNotice);
	qApp->unlock();
	
	if((m_state == waitingForDebuggerPort) && strNotice.startsWith("NOTICE:") && (strNotice.contains("PLDBGBREAK:")))
	{
		QRegExp regexpNumber("PLDBGBREAK:\\d+", true);
		int iPos = regexpNumber.search(strNotice);
		if(iPos > 0)
		{
			int iLen = regexpNumber.matchedLength();
			
			
			m_strDebuggerPort = strNotice.mid(iPos + 11, iLen - 11);
			
			qApp->lock();
			m_pTextEditResult->append(QString("Assigned debugger port: %1 \n").arg(m_strDebuggerPort));
			qApp->unlock();
			m_state = obtainedDebuggerPort;
			
		}
	}
}

void KPGDebugger::slotTimer()
{
	bool locked = mutex.tryLock();
	if(!locked) return;
	
	if(m_state == obtainedDebuggerPort)
	{
		attachToDebugger();
	}
	
	mutex.unlock();
}

void KPGDebugger::executeFunction()
{
	// clear previous result
	m_pTableResult->setNumRows(0);
	m_pTableResult->setNumCols(0);
	//m_pTextEditResult->clear();
	//m_pTextLabelExecutionTime->setText("");
	m_pqxxResult.clear(); // free memory
		
	//int iTransType = m_pComboBoxTransaction->currentItem();
	KPGConnection::ETransType transType = KPGConnection::eTransNormal;
	/*switch(iTransType)
	{
		case 0: transType = KPGConnection::eTransNormal;
		break;
				
		case 1: transType = KPGConnection::eTransRobust;
		break;
				
		case 2: transType = KPGConnection::eTransNone;
		break;
	}*/
	
	m_pTableParameters->setEnabled(false);
	//m_pComboBoxTransaction->setEnabled(false);
	//emit sigSetTerminalIcon(this, 1); // red terminal icon
	((QWidget *) parent())->setCursor(KCursor::waitCursor());
		
	m_pActRun->setEnabled(false);
	m_pActStop->setEnabled(true);
	m_pActStepOver->setEnabled(false); 
	m_pActStepInto->setEnabled(false);
	m_pActToggleBreakPoint->setEnabled(false); 
	
	bool bSuccess = runSql(transType);
	if(!bSuccess)
	{
		/*m_pActExecute->setEnabled(true);
		m_pActStop->setEnabled(false);
		m_pTableParameters->setEnabled(true);
		m_pComboBoxTransaction->setEnabled(true);*/
	}
}

// Create prepared statement for SQL SELECT operation
void KPGDebugger::prepareSelectStatement(const std::string &stdstrPrepStmtName)
{
	QString strSql("SELECT * FROM ");
	strSql.append(KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strFunctionName));
            
	strSql.append("(");        
    
	int nRows = m_pTableParameters->numRows();
	for(int i = 0; i < nRows; i++)
	{
		if(i > 0) strSql.append(", ");
    	
		if((m_pTableParameters->text(i, COL_VALUE) == QString::null) || m_pTableParameters->text(i, COL_VALUE).isEmpty())
		{
			strSql.append("NULL");
		}
		else
		{
			strSql.append(QString("$%1").arg(i + 1));
		}
	}
    
	strSql.append(")"); 
    
	kdDebug() << strSql << endl;
    
	pqxx::prepare::declaration pqxxPrepDecl = m_connectionInThread.connection()->prepare(stdstrPrepStmtName, strSql);
    
    // Add declaring parameters to prepared statements
	for(int i = 0; i < nRows; i++)
	{
		if((m_pTableParameters->text(i, COL_VALUE) != QString::null) && m_pTableParameters->text(i, COL_VALUE).isEmpty() == false)
		{
			m_connectionInThread.connection()->setPrepareDeclaration(pqxxPrepDecl, m_listArguments[i].typName());
		}
	}
}

// Run SQL to execute function on DB server
bool KPGDebugger::runSql(KPGConnection::ETransType transType)
{
	m_connectionInThread.setPrepStatementName("S_" + KPGUtil::unspacedName(m_strNamespaceName) + "_" + KPGUtil::unspacedName(m_strFunctionName), transType); 

	try
	{
		prepareSelectStatement(m_connectionInThread.prepStatementName()); 
	}
	catch(const std::exception &e)
	{
		m_pTextEditResult->append(e.what());
		m_pTabWidget->setCurrentPage(4);

		return false;
	} 
	
	//***********************
	ListArgumentsWithValues listArgumentsWithValues;
			
	int currentArg = 0;
	for(KPGFunction::ListArguments::iterator it = m_listArguments.begin(); it != m_listArguments.end(); ++it)
	{ 
		if((m_pTableParameters->text(currentArg, COL_VALUE) == QString::null) 
				  || m_pTableParameters->text(currentArg, COL_VALUE).isEmpty()) continue;
					
		listArgumentsWithValues.append(KPGFunctionArgumentWithValue(currentArg + 1, m_listArguments[currentArg].name(), m_listArguments[currentArg].mode(), m_listArguments[currentArg].typName(), m_pTableParameters->text(currentArg, COL_VALUE)));
				
		currentArg++;
	}
	
	m_connectionInThread.setPrepStatementArgumentValues(listArgumentsWithValues); 
	m_connectionInThread.start();
	
	return true;
}

void KPGDebugger::customEvent(QCustomEvent *pEvent)
{
	KPGQueryResultEvent *pQueryResultEvent = static_cast <KPGQueryResultEvent *> (pEvent);
	
	//kdDebug() << QString("Result size is: %1").arg(m_pqxxResult.size()) << endl;
	
	// Display result
	if(m_pqxxResult.size() > 0)
	{
		//displayExecutionTime(pQueryResultEvent->executionMiliseconds());   
		displayResult(); 
		m_pTabWidget->setCurrentPage(2);
	}
	else
	{
		m_pTabWidget->setCurrentPage(3);
	}
	
	if(pQueryResultEvent->error().isEmpty() == false)
	{
		m_pTextEditResult->append(pQueryResultEvent->error());
	}
	
	//displayExecutionTime(pQueryResultEvent->executionMiliseconds());      		
	m_pTextEditResult->append(i18n("\nOperation finished"));
			
	
	m_pTableParameters->setEnabled(true);
	//m_pComboBoxTransaction->setEnabled(true);
	//emit sigSetTerminalIcon(this, 2); // blue terminal icon
	((QWidget *) parent())->setCursor(KCursor::arrowCursor());
	
	m_connectionInThread.wait();
		
	m_pActRun->setEnabled(true);
	m_pActStop->setEnabled(false);
    /*
	m_pActFetchNext->setEnabled((fetchedAllRows()) ? false : true); 
	m_pActFetchAll->setEnabled((fetchedAllRows()) ? false : true);
	m_pActFindInResult->setEnabled((resultEmpty()) ?  false : true); */
}

// Display SQL result
void KPGDebugger::displayResult()
{
	// Fill output area with query result
	m_nTotalRows = m_pqxxResult.size();
	m_nTotalCols = m_pqxxResult.columns();
	m_nFetchedRows = 0;
	bool bAllRowsFetched = true;
		
	m_pTableResult->setNumRows(min(m_nTotalRows, NUM_OF_RECORDS_PER_PAGE));
	m_pTableResult->setNumCols(m_nTotalCols);
	
	// Field names
	QHeader* hHeader = m_pTableResult->horizontalHeader();
	
	for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
	{
		hHeader->setLabel(nCol, m_pqxxResult.column_name(nCol));
	}
	
	
	// Data
	for(unsigned int nRow = 0; nRow < m_nTotalRows; nRow++)
	{
		for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
		{
			QString strValue(m_connectionInThread.connection()->textCodec()->toUnicode(m_pqxxResult[nRow][nCol].c_str()));
			m_pTableResult->setText(nRow, nCol, strValue);
		}
		
		m_nFetchedRows++;
		if(nRow + 1 >= NUM_OF_RECORDS_PER_PAGE)
		{
			bAllRowsFetched = false;
			break;
		}
	}
	
	for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
	{
		m_pTableResult->adjustColumn(nCol);
	}
	
	if(bAllRowsFetched) m_pqxxResult.clear(); // free memory
}

int KPGDebugger::getProxyInfo()
{
	QString strQuery("SELECT proxyapiver FROM pldbg_get_proxy_info()");
	int iProxyApiVersion = 0;
	
	m_pTextEditResult->append(i18n("Obtaining debugger API version ... "));
	
	try
	{
		pqxx::result pqxxResult = m_connectionInThread.connection()->runQuery(strQuery);
		
		pqxxResult[0][0].to(iProxyApiVersion);
		m_pTextEditResult->append(QString("OK, it's: %1\n").arg(iProxyApiVersion));
		return iProxyApiVersion;
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		KMessageBox::sorry(this, e.what());
		m_state = attachFailed;
		m_pTextEditResult->append(i18n("... failed\n"));
	}
	
	return -1;
}

bool KPGDebugger::oidDebug()
{
	QString strQuery = QString("SELECT plpgsql_oid_debug(%1)").arg(m_oidFunction);
	
	m_pTextEditResult->append(i18n("Setting debugger for function ... "));
	
	try
	{
		m_connectionInThread.connection()->runQuery(strQuery);
				
		m_pTextEditResult->append(QString("OK\n"));
		m_state = waitingForDebuggerPort;
		return true;
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		KMessageBox::sorry(this, e.what());
		m_state = attachFailed;
		m_pTextEditResult->append(i18n("... failed\n"));
	}
	
	return false;
}


void KPGDebugger::attachToDebugger()
{
	QString strQuery = QString("SELECT * FROM pldbg_attach_to_port(%1)").arg(m_strDebuggerPort);
	
	try
	{
		pqxx::result pqxxResult = m_connectionInThread.connection()->runQuery(strQuery);
		
		m_strSessionHandle = pqxxResult[0][0].c_str();
		m_state = attached;
		m_pTextEditResult->append(i18n("Atached to debugger on port: ") + m_strDebuggerPort + "\n");
		m_pTextEditResult->append(i18n("Session handle is: ") + m_strSessionHandle + "\n");
		
		m_pActRun->setEnabled(false);
		m_pActStop->setEnabled(true);
		m_pActStepOver->setEnabled(true);
		m_pActStepInto->setEnabled(true);
		m_pActToggleBreakPoint->setEnabled(true);
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		KMessageBox::sorry(this, e.what());
		m_state = attachFailed;
		return;
	}
}

void KPGDebugger::stopDebugging()
{
	QString strQuery("SELECT * FROM pldbg_abort_target(" + m_strSessionHandle + ")");
	
	try
	{
		pqxx::result pqxxResult = m_connectionInThread.connection()->runQuery(strQuery);
		bool bResult;
		pqxxResult[0][0].to(bResult);
		
		if(bResult)
		{
			m_strSessionHandle = QString::null;
			m_state = prepared;
			m_pActRun->setEnabled(true);
			m_pActStop->setEnabled(false);
			m_pActStepOver->setEnabled(false);
			m_pActStepInto->setEnabled(false);
			m_pActToggleBreakPoint->setEnabled(false);
		}
		else
		{
			KMessageBox::sorry(this, "Stop debugging failed");
		}
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		KMessageBox::sorry(this, e.what());
		return;
	}
}

void KPGDebugger::setBreakpoint(uint uiLineNumber)
{
	QString strQuery = QString("SELECT * FROM pldbg_set_breakpoint(%1, %2, %3)").arg(m_strSessionHandle).arg(m_oidFunction).arg(uiLineNumber + 1);
	
	try
	{
		pqxx::result pqxxResult = m_connectionInThread.connection()->runQuery(strQuery);
		KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());
		pMarkIface->setMark(uiLineNumber, KTextEditor::MarkInterface::BreakpointActive);
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		KMessageBox::sorry(this, e.what());
		return;
	}
}

void KPGDebugger::clearBreakpoint(uint uiLineNumber)
{
	QString strQuery = QString("SELECT * FROM pldbg_drop_breakpoint(%1, %2, %3)").arg(m_strSessionHandle).arg(m_oidFunction).arg(uiLineNumber + 1);
	
	try
	{
		pqxx::result pqxxResult = m_connectionInThread.connection()->runQuery(strQuery);
		KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());
		pMarkIface->removeMark(uiLineNumber, KTextEditor::MarkInterface::BreakpointActive);
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
		KMessageBox::sorry(this, e.what());
		return;
	}
}
