//
// C++ Implementation: kpgexecutor
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kpgexecutor.h"

// include files for Qt
#include <qtable.h>
#include <qwidgetstack.h> 
#include <qtextcodec.h>
#include <qcheckbox.h>
#include <qtimer.h>

// include files for KDE
#include <klocale.h>
#include <kmessagebox.h>
#include <kcursor.h>
#include <kdebug.h>
#include <ktextedit.h>
#include <kcombobox.h>
#include <kfinddialog.h>
#include <kfind.h>
#include <kiconloader.h>

// application specific includes
#include "../kpgutil.h"
#include "../kpogreview.h"
#include "../kpgconfiguration.h"
#include "../DbObjects/kpgdatabase.h"
#include "../DbObjects/kpgfunction.h"
#include "../kpgsqldialog.h"
#include "../DbObjects/kpgtablecolumn.h"
#include "kpgexecutorchildview.h"

#define NUM_OF_RECORDS_PER_PAGE 500 // How many records are loaded to result table 
#define UPDATE_TIMER_INTERVAL 1000 // 1 sec for updatiing code completion

#define COL_NAME 0
#define COL_TYPE_NAME 1
#define COL_MODE 2
#define COL_VALUE 3

KPGExecutor::KPGExecutor(KPGExecutorChildView *parent,
	KPoGreView *pPoGreView,
	KXMLGUIFactory *pXmlGuiFactory,
 	const PGSTD::string& strConnectionOptions,
	const QString & strNamespaceName, 
	KPGFunction *pFunction)
	: 
	KXMLGUIClient(pPoGreView),
	KPGExecutorBase(parent, "Executor"),
 	m_pFind(0)
{
	// Setup GUI
	setXMLFile("kpgexecutor.rc", true);
	m_pXmlGuiFactory = pXmlGuiFactory;
	m_bIsAddedToGuiFactory = false;
	
	// Initialize members
	m_strConnectionOptions = strConnectionOptions;
	m_strNamespaceName = strNamespaceName;
	m_strFunctionName = pFunction->text(0);
	m_bReturnSet = pFunction->returnSet();
	m_nTotalRows = 0;
	m_nTotalCols = 0;
	m_nFetchedRows = 0;
	
	//-----------------------------------------------------------------
	// Executor actions       
	m_pActExecute = new KAction(i18n("Execute"), "exec", Key_F8, this, SLOT(slotExecute()), actionCollection(), "executor_execute");
	
	m_pActStop = new KAction(i18n("Stop"), "stop", 0, this, SLOT(slotStop()), actionCollection(), "executor_stop");
  	
  	m_pActFetchNext = new KAction(i18n("Fetch &next"), "1downarrow", 0, this, SLOT(slotFetchNext()), actionCollection(), "executor_fetch_next");
	
	m_pActFetchAll = new KAction(i18n("Fetch &all"), "2downarrow", 0, this, SLOT(slotFetchAll()), actionCollection(), "executor_fetch_all");
	
	m_pActFindInResult = new KAction(i18n("&Find in result..."), "find", 0, this, SLOT(slotFindInResult()), actionCollection(), "edit_find_in_result");
	m_pActFindInResult->setWhatsThis(i18n("Find in result table."));
	
	m_pActFindNextInResult = new KAction(i18n("Find next in result"), "next", 0, this, SLOT(slotFindNextInResult()), actionCollection(), "edit_find_next_in_result");
	m_pActFindNextInResult->setWhatsThis(i18n("Find next in result table."));
    
    //-----------------------------------------------------------------
	// Common clipboard actions
	m_pActCopyCell = new KAction(i18n("&Copy cell"), "editcopy", 0, this, SLOT(slotCopyCell()), actionCollection(), "copy_cell");
	
	m_pActCopyRow = new KAction(i18n("Copy &row"), 0, 0, this, SLOT(slotCopyRow()), actionCollection(), "copy_row");
	
	m_pActCopyTableCsv = new KAction(i18n("Copy table to C&SV"), 0, 0, this, SLOT(slotCopyTableCsv()), actionCollection(), "copy_table_csv");
	
	m_pActCopyTableXml = new KAction(i18n("Copy table to &XML"), 0, 0, this, SLOT(slotCopyTableXml()), actionCollection(), "copy_table_xml");
    
    //-----------------------------------------------------------------
    connect(m_pTableResult, SIGNAL(contextMenuRequested(int, int, const QPoint &)), this, SLOT(slotContextMenuRequested(int, int, const QPoint &)));
    
	// Setup tab icons
	m_pTabWidget->setTabIconSet(pTabResult, QIconSet(QPixmap(SmallIcon("view_text"))));
	m_pTabWidget->setTabIconSet(pTabTextResult, QIconSet(QPixmap(SmallIcon("openterm"))));
	
    //-----------------------------------------------------------------
	// Make own copy of arguments info
	m_listArguments = pFunction->listArguments();
	
	// Set-up parameters table
	if(pFunction->listArguments().count() > 0)
	{
		m_pTableParameters->setNumRows(pFunction->listArguments().count());
		int nRow = 0;
		
		for(KPGFunction::ListArguments::const_iterator cit = pFunction->listArguments().begin(); cit != pFunction->listArguments().end(); ++cit)
		{ 
			m_pTableParameters->setText(nRow, COL_NAME, (*cit).name());
			
			if((*cit).typName().length() > 0)
			{
				m_pTableParameters->setText(nRow, COL_TYPE_NAME, (*cit).typName());
				
				QPixmap *pTypePixmap = KPGTableColumn::getPixmapForType((*cit).typName());
				m_pTableParameters->setPixmap(nRow, COL_NAME, *pTypePixmap);
			}
			
			
			if((*cit).mode() == "i") m_pTableParameters->setText(nRow, COL_MODE, "IN");
			else if((*cit).mode() == "o") m_pTableParameters->setText(nRow, COL_MODE, "OUT");
			else if((*cit).mode() == "b") m_pTableParameters->setText(nRow, COL_MODE, "INOUT");
			nRow++;
		}
		
		// We use prepared statement, if function has arguments
		m_connectionInThread.setOperationType(KPGConnectionInThread::eRunPreparedStatement);
	}
	else
	{
		m_connectionInThread.setOperationType(KPGConnectionInThread::eRunSql);
	}
	
	m_pTableParameters->setColumnReadOnly(0, true);
	m_pTableParameters->setColumnReadOnly(1, true);
	m_pTableParameters->setColumnReadOnly(2, true);
	
	// Open connection
	m_connectionInThread.setEventReceiver(this, &m_pqxxResult);
		
	try
	{
		m_connectionInThread.connectToServer(m_strConnectionOptions);
		
		KPGNoticer * pPgNoticer = m_connectionInThread.connection()->kpgNoticer();
		connect(pPgNoticer, SIGNAL(sigNotice(QString)), this, SLOT(slotProcessNotice(QString)));
	}
	catch (const std::exception &e)
	{
		kdError() << "Failed to open connection " << e.what() << endl;
		KMessageBox::sorry(this, e.what());
		return;
	}
	
	m_pActExecute->setEnabled(true);
    m_pActStop->setEnabled(false);
    m_pActFetchNext->setEnabled(false); 
    m_pActFetchAll->setEnabled(false);
}

KPGExecutor::~KPGExecutor()
{
	KPGNoticer * pPgNoticer = m_connectionInThread.connection()->kpgNoticer();
	disconnect(pPgNoticer, SIGNAL(sigNotice(QString)), this, SLOT(slotProcessNotice(QString)));
	
	// Close connection
	m_connectionInThread.disconnectFromServer(); // close connection if any
	
	if(m_pFind != 0) 
    {
    	disconnect( m_pFind, 0, this, 0);
    	delete m_pFind;
    }

	removeFromGuiFactory();
}

// Add yourself to GUI factory
void KPGExecutor::addToGuiFactory()
{
	if(!m_bIsAddedToGuiFactory)
    {
        m_pXmlGuiFactory->addClient(this);
        m_bIsAddedToGuiFactory = true;
    }
}
    
// Remove yourself from GUI factory
void KPGExecutor::removeFromGuiFactory()
{
    if(m_bIsAddedToGuiFactory)
    {
        m_pXmlGuiFactory->removeClient(this);
        m_bIsAddedToGuiFactory = false;
    }
}

// Display popup menu
void KPGExecutor::slotContextMenuRequested(int, int, const QPoint &pos)
{
	QWidget * pContainer = m_pXmlGuiFactory->container("popupExecutorTable", this);

	if ( ! pContainer )
	{
		kdError() << k_funcinfo << " Couldn't get a container widget for the given menu name popupQueryResultTable" << endl;
		return;
	}

	if ( ! pContainer->inherits("KPopupMenu") )
	{
		kdError() << k_funcinfo << " Wrong container widget" << endl;
		return;
	}

	KPopupMenu * pMenu = static_cast <KPopupMenu*> (pContainer);
	pMenu->popup( pos );
}

void KPGExecutor::slotExecute()
{
    executeFunction(); 
}

void KPGExecutor::slotStop()
{
    stopExecute(); 
}

void KPGExecutor::slotFetchNext()
{
    fetchNext();
}

void KPGExecutor::slotFetchAll()
{
    fetchAll();     
}

// Find first occurence of text in result table  
void KPGExecutor::slotFindInResult()
{
    findInResultFirst(m_listOfResultQuerySearchHistory);     
}

// Find next occurence of text in result table  
void KPGExecutor::slotFindNextInResult()
{
    findInResultNext();     
}

void KPGExecutor::slotCopyCell()
{
	KPGUtil::copyCell(m_pTableResult);
}

void KPGExecutor::slotCopyRow()
{
	KPGUtil::copyRow(m_pTableResult);
}

void KPGExecutor::slotCopyTableCsv()
{
	KPGUtil::copyTableCsv(m_pTableResult);
}

void KPGExecutor::slotCopyTableXml()
{
	KPGUtil::copyTableXml(m_pTableResult);
}

void KPGExecutor::updateActions()
{
	//emit sigExecutorStateChanged(this);
}

bool KPGExecutor::resultEmpty() const
{ 
	return m_pTableResult->numRows() == 0; 
}

void KPGExecutor::executeFunction()
{
	// clear previous result
	m_pTableResult->setNumRows(0);
	m_pTableResult->setNumCols(0);
	m_pTextEditResult->clear();
	
	
	m_pTextLabelExecutionTime->setText("");
	m_pqxxResult.clear(); // free memory
		
	int iTransType = m_pComboBoxTransaction->currentItem();
	KPGConnection::ETransType transType = KPGConnection::eTransNormal;
	switch(iTransType)
	{
		case 0: transType = KPGConnection::eTransNormal;
				break;
				
		case 1: transType = KPGConnection::eTransRobust;
				break;
				
		case 2: transType = KPGConnection::eTransNone;
				break;
	}
	
	m_pTableParameters->setEnabled(false);
	m_pComboBoxTransaction->setEnabled(false);
	//emit sigSetTerminalIcon(this, 1); // red terminal icon
	
	m_pTableParameters->setCursor(KCursor::waitCursor());
		
	m_pActExecute->setEnabled(false);
    m_pActStop->setEnabled(true);
    m_pActFetchNext->setEnabled(false); 
    m_pActFetchAll->setEnabled(false);
    m_pActFindInResult->setEnabled(false); 
    m_pActFindNextInResult->setEnabled(false);
	m_pTableParameters->setEnabled(false);
	
	bool bSuccess = runSql(transType);
	if(!bSuccess)
	{
		m_pActExecute->setEnabled(true);
		m_pActStop->setEnabled(false);
		m_pTableParameters->setEnabled(true);
		m_pComboBoxTransaction->setEnabled(true);
		m_pTableParameters->setCursor(KCursor::arrowCursor());
	}
}

void KPGExecutor::customEvent(QCustomEvent *pEvent)
{
	KPGQueryResultEvent *pQueryResultEvent = static_cast <KPGQueryResultEvent *> (pEvent);
	
	//kdDebug() << QString("Result size is: %1").arg(m_pqxxResult.size()) << endl;
	
	// Display result
	if(m_pqxxResult.size() > 0)
	{
		displayExecutionTime(pQueryResultEvent->executionMiliseconds());   
		displayResult(); 
		m_pTabWidget->setCurrentPage(0);
	}
	else
	{
		m_pTabWidget->setCurrentPage(1);
	}
	
	
	if(pQueryResultEvent->error().isEmpty() == false)
	{
		m_pTextEditResult->append(pQueryResultEvent->error());
	}
	
	displayExecutionTime(pQueryResultEvent->executionMiliseconds());      
	m_pTextEditResult->append(i18n("\nOperation finished"));
	
	
	m_pTableParameters->setEnabled(true);
	m_pComboBoxTransaction->setEnabled(true);
	//emit sigSetTerminalIcon(this, 2); // blue terminal icon
	m_pTableParameters->setCursor(KCursor::arrowCursor());
	
	m_connectionInThread.wait();
		
	m_pActExecute->setEnabled(true);
    m_pActStop->setEnabled(false);
    
    m_pActFetchNext->setEnabled((fetchedAllRows()) ? false : true); 
    m_pActFetchAll->setEnabled((fetchedAllRows()) ? false : true);
    m_pActFindInResult->setEnabled((resultEmpty()) ?  false : true); 
}

void KPGExecutor::stopExecute()
{
	if(m_connectionInThread.running ())
	{
		m_connectionInThread.terminate();
		m_connectionInThread.wait();
		m_connectionInThread.leaveConnection();
		
		m_pTextEditResult->setText(i18n("Operation terminated"));
		m_pTabWidget->setCurrentPage(1);
		
		m_pqxxResult.clear(); // free memory
		m_nTotalRows = 0;
		m_nTotalCols = 0;
		
		try
		{
			m_connectionInThread.connectToServer(m_strConnectionOptions);
			
			m_pTableParameters->setEnabled(true);
			m_pComboBoxTransaction->setEnabled(true);
			//emit sigSetTerminalIcon(this, 0); // black terminal icon
			//m_bQueryTextChanged = false;
		
			m_pActExecute->setEnabled(true);
			m_pActStop->setEnabled(false);
			m_pActFetchNext->setEnabled(false); 
			m_pActFetchAll->setEnabled(false);
		}
		catch (const std::exception &e)
		{
			kdError() << "Failed to open connection " << e.what() << endl;
			KMessageBox::sorry(this, e.what());
			return;
		}

		m_pTableParameters->setCursor(KCursor::arrowCursor());
	} 
}


void KPGExecutor::displayExecutionTime(int iMiliseconds)
{
    int hour = 0; int min = 0; int sec = 0;
    int milis = iMiliseconds;
    while(milis >= 1000)
    {
      milis -= 1000;
      sec++;
    }
    while(sec >= 60)
    {
      sec -=60;
      min++;
    }
    while(min >= 60)
    {
      min -= 60;
      hour++;
    }
    
    QString execTime;
    if(hour > 0) execTime.append(QString("%1 h :").arg(hour));
    execTime.append(QString("%1 m : ").arg(min));
    execTime.append(QString("%1 s : ").arg(sec));
    execTime.append(QString("%1 ms").arg(milis));
      
    m_pTextLabelExecutionTime->setText(execTime);
}

// Display SQL result
void KPGExecutor::displayResult()
{
	// Fill output area with query result
	m_nTotalRows = m_pqxxResult.size();
	m_nTotalCols = m_pqxxResult.columns();
	m_nFetchedRows = 0;
	bool bAllRowsFetched = true;
		
	m_pTableResult->setNumRows(min(m_nTotalRows, NUM_OF_RECORDS_PER_PAGE));
	m_pTableResult->setNumCols(m_nTotalCols);
	
	// Field names
	QHeader* hHeader = m_pTableResult->horizontalHeader();
	
	for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
	{
		hHeader->setLabel(nCol, m_pqxxResult.column_name(nCol));
	}
	
	
	// Data
	for(unsigned int nRow = 0; nRow < m_nTotalRows; nRow++)
	{
		for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
		{
			QString strValue(m_connectionInThread.connection()->textCodec()->toUnicode(m_pqxxResult[nRow][nCol].c_str()));
			m_pTableResult->setText(nRow, nCol, strValue);
		}
		
		m_nFetchedRows++;
		if(nRow + 1 >= NUM_OF_RECORDS_PER_PAGE)
		{
			bAllRowsFetched = false;
			break;
		}
	}
	
	for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
	{
		m_pTableResult->adjustColumn(nCol);
	}
	
	if(bAllRowsFetched) m_pqxxResult.clear(); // free memory
}

// Fetch next X rows from large result
void KPGExecutor::fetchNext()
{
	unsigned int nRows = m_nTotalRows - m_nFetchedRows;
	nRows = min(nRows, NUM_OF_RECORDS_PER_PAGE);
	m_pTableResult->setNumRows(nRows + m_pTableResult->numRows());
	
	// Data
	unsigned int nRow = m_nFetchedRows;
	unsigned int nFetchedRows = 0;
	
	for(; nRow < m_nTotalRows; nRow++)
	{
		for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
		{
			QString strValue(m_connectionInThread.connection()->textCodec()->toUnicode(m_pqxxResult[nRow][nCol].c_str()));
			m_pTableResult->setText(nRow, nCol, strValue);
		}
		
		m_nFetchedRows++;
		nFetchedRows++;
		if(nFetchedRows >= NUM_OF_RECORDS_PER_PAGE)
		{
			m_pActFetchNext->setEnabled(true); 
            m_pActFetchAll->setEnabled(true);
            break;
		}
	}
	
	if(nRow == m_nTotalRows)
	{
		m_pActFetchNext->setEnabled(false); 
        m_pActFetchAll->setEnabled(false);
		m_pqxxResult.clear(); // free memory
	}
}
  
// Fetch all rows from large result  
void KPGExecutor::fetchAll()
{
	unsigned int nRows = m_nTotalRows - m_nFetchedRows;
	
	if(nRows > 10000)
	{ 
		if( KMessageBox::questionYesNo(this, QString("There are %1 rows in result. Fething them may take long time and much anmount of memory. Continue ?").arg(nRows)) != KMessageBox::Yes )
		{
			return;
		}
	}
	
	((QWidget *) parent())->setCursor(KCursor::waitCursor());
	m_pTableResult->setNumRows(nRows + m_pTableResult->numRows());
	
	// Data
	unsigned int nRow = m_nFetchedRows;
	for(; nRow < m_nTotalRows; nRow++)
	{
		for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
		{
			QString strValue(m_connectionInThread.connection()->textCodec()->toUnicode(m_pqxxResult[nRow][nCol].c_str()));
			m_pTableResult->setText(nRow, nCol, strValue);
		}
		
		m_nFetchedRows++;
	}
	
	((QWidget *) parent())->setCursor(KCursor::arrowCursor());
	
	m_pActFetchNext->setEnabled(false); 
    m_pActFetchAll->setEnabled(false);
    
	m_pqxxResult.clear(); // free memory
}  


// Run SQL to execute function on DB server
bool KPGExecutor::runSql(KPGConnection::ETransType transType)
{
	QString strSql;
    
	if(m_pCheckBoxExplain->isChecked())
	{
		strSql.append("EXPLAIN ");
    
		if(m_pCheckBoxAnalyze->isChecked())
			strSql.append("ANALYZE ");
    		
		if(m_pCheckBoxVerbose->isChecked())
			strSql.append("VERBOSE ");
	}
    
	strSql.append("SELECT ");
	
	if(m_bReturnSet)
	{
		strSql.append("* FROM ");
	}
	
	strSql.append(KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strFunctionName));
            
	strSql.append("(");
    
	int nArgs = m_pTableParameters->numRows();
	for(int i = 0; i < nArgs; i++)
	{
		if(i > 0) strSql.append(", ");
    	
		if((m_pTableParameters->text(i, COL_VALUE) == QString::null) || m_pTableParameters->text(i, COL_VALUE).isEmpty())
		{
			strSql.append("NULL");
		}
		else
		{
			strSql.append(QString("$%1").arg(i + 1));
		}
	}
    
	strSql.append(")"); 
    
	kdDebug() << strSql << endl;
	
	if(nArgs == 0)
	{
		m_connectionInThread.setSQL(strSql, transType);
	}
	else
	{
		QString strStatementName("exec_" + KPGUtil::unspacedName(m_strNamespaceName) + "_" + KPGUtil::unspacedName(m_strFunctionName));
		if(m_pCheckBoxExplain->isChecked())
		{
			strStatementName.prepend("expl_"); // Choose different name for EXPLAIN, to avoid error: Inconsistent redefinition of prep. stmt.
		}
		
		m_connectionInThread.setPrepStatementName(strStatementName, transType);
	
		try
		{
			pqxx::prepare::declaration pqxxPrepDecl = m_connectionInThread.connection()->prepare(strStatementName, strSql);
		
			// Add declaring parameters to prepared statements
			for(int i = 0; i < nArgs; i++)
			{
				if((m_pTableParameters->text(i, COL_VALUE) != QString::null) && m_pTableParameters->text(i, COL_VALUE).isEmpty() == false)
				{
					m_connectionInThread.connection()->setPrepareDeclaration(pqxxPrepDecl, m_listArguments[i].typName());
				}
			}
		}
		catch(const std::exception &e)
		{
			m_pTextEditResult->append(e.what());
			m_pTabWidget->setCurrentPage(1);
	
			return false;
		} 
		
		//***********************
		ListArgumentsWithValues listArgumentsWithValues;
				
		int currentArg = 0;
		for(KPGFunction::ListArguments::iterator it = m_listArguments.begin(); it != m_listArguments.end(); ++it)
		{ 
			if((m_pTableParameters->text(currentArg, COL_VALUE) == QString::null) 
					|| m_pTableParameters->text(currentArg, COL_VALUE).isEmpty()) continue;
						
			listArgumentsWithValues.append(KPGFunctionArgumentWithValue(currentArg + 1, m_listArguments[currentArg].name(), m_listArguments[currentArg].mode(), m_listArguments[currentArg].typName(), m_pTableParameters->text(currentArg, COL_VALUE)));
					
			currentArg++;
		}
		
		m_connectionInThread.setPrepStatementArgumentValues(listArgumentsWithValues); 
	}
	
	m_connectionInThread.start();
	return true;
}

void KPGExecutor::slotExplainToggled(bool bToggled)
{
	m_pCheckBoxAnalyze->setEnabled(bToggled/* && m_listArguments.count() == 0*/); 
	m_pCheckBoxVerbose->setEnabled(bToggled);
}

// Find first occurence of text in result table
void KPGExecutor::findInResultFirst(QStringList &listOfSearchHistory)
{
    KFindDialog dlg(this, "", 0, listOfSearchHistory, false);
    dlg.setHasCursor(false);
        
    int c = dlg.exec();
    
    if(c != QDialog::Accepted)
        return;

    listOfSearchHistory = dlg.findHistory();
        
    
    if(m_pFind != 0) 
    {
    	disconnect( m_pFind, 0, this, 0);
    	delete m_pFind;
    }
    m_pFind = new KFind(dlg.pattern(), dlg.options(), this);

    // Connect highlight signal to code which handles highlighting
    // of found text.
    connect( m_pFind, SIGNAL( highlight( const QString &, int, int ) ),
             this, SLOT( slotHighlight( const QString &, int, int ) ) );
             
    // Connect findNext signal - called when pressing the button in the dialog
    connect( m_pFind, SIGNAL( findNext() ), this, SLOT( slotFindInResultNext() ) );

    m_iRowToSearch = (m_pFind->options() & KFindDialog::FindBackwards) ? m_pTableResult->numRows() - 1 : 0;
    m_iColToSearch = (m_pFind->options() & KFindDialog::FindBackwards) ? m_pTableResult->numCols() - 1 : 0;
    m_pFind->setData(m_pTableResult->text(m_iRowToSearch, m_iColToSearch));

    slotFindInResultNext();
}

// Find next occurence of text in result table
void KPGExecutor::findInResultNext()
{
    m_iRowToSearch = m_pTableResult->currentRow();
	m_iColToSearch = m_pTableResult->currentColumn();
	if(m_iRowToSearch < 0) return;
	if(m_iColToSearch < 0) return;
	
	slotFindInResultNext();
}

// Find next occurence of text
void KPGExecutor::slotFindInResultNext()
{
	if (!m_pFind) // shouldn't be called before find is activated
        return;

    KFind::Result res = KFind::NoMatch;
    while ( res == KFind::NoMatch &&
    		((m_pFind->options() & KFindDialog::FindBackwards) ? 
    				(m_iRowToSearch >= 0) : (m_iRowToSearch < m_pTableResult->numRows())
    		) 
          ) 
    	{
    		//kdDebug() << "searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
        	
        	Q_ASSERT(m_iRowToSearch >= 0);
            Q_ASSERT(m_iRowToSearch < m_pTableResult->numRows());
            Q_ASSERT(m_iColToSearch >= 0);
            Q_ASSERT(m_iColToSearch < m_pTableResult->numCols());
            	
        	if(m_pFind->needData()) 
        	{
            	m_pFind->setData(m_pTableResult->text(m_iRowToSearch, m_iColToSearch));
        	}
        
        	// Let KFind inspect the text fragment, and display a dialog if a match is found
        	res = m_pFind->find();
        
        	if( res == KFind::NoMatch ) 
        	{
            	if((m_pFind->options() & KFindDialog::FindBackwards))
            	{
            		if(m_iColToSearch == 0)
            		{
            			m_iColToSearch = m_pTableResult->numCols() - 1;
            			m_iRowToSearch--;
            		}
            		else
            		{
            			m_iColToSearch--;
            		}
            	}
            	else
            	{
            		if(m_iColToSearch == m_pTableResult->numCols() - 1)
            		{
            			m_iColToSearch = 0;
            			m_iRowToSearch++;
            		}
            		else
            		{
            			m_iColToSearch++;
            		}
            	}
        	}
        	//kdDebug() << "2searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
		}

    if( res == KFind::NoMatch ) 
    { // i.e. at end
        m_pFind->displayFinalDialog();
        m_pFind->resetCounts();
        m_pTableResult->removeSelection(0);
    }
}

// Highligth found text
void KPGExecutor::slotHighlight( const QString &, int, int)
{
    //kdDebug() << "highlight: " << index << " " << length << endl;
    m_pTableResult->setCurrentCell(m_iRowToSearch, m_iColToSearch);
}

// Process notice from connection in thread
void KPGExecutor::slotProcessNotice(QString strNotice)
{
	qApp->lock();
	m_pTextEditResult->append(strNotice);
	qApp->unlock();
}
