//
// C++ Implementation: kpgsqledit
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kpgsqledit.h"

// include files for KDE
#include <klocale.h>
#include <kdebug.h>
#include <kfinddialog.h>
#include <kreplacedialog.h>
#include <kfind.h>
#include <kreplace.h>

// include files for Qt
#include <private/qrichtext_p.h> // We need QTextCursor
#include <qtimer.h>
#include <qregexp.h>

// application specific includes
#include "kpgsyntaxhighlighter.h"
#include "kpgcompletionbox.h"
#include "../kpogreview.h"
#include "../kpgconfiguration.h"
#include "kpgsqleditorsettings.h"
#include "kpgsqlparser.h"

#define UPDATE_TIMER_INTERVAL 1000 // 1 sec

KPGSqlEdit::KPGSqlEdit(QWidget* pParent, const char* szName)
 : KTextEdit(pParent, szName),
 m_pUpdateCodeCompletionListTimer(0),
 m_bCodeCompletionListUpdated(true),
 m_pFind(0),
 m_pReplace(0),
 m_bIgnoreSelectionClearing(false)
{
	m_pSyntaxHighlighter = new KPGSyntaxHighlighter(this);
	m_pCompletionBox = new KPGCompletionBox(this);
	m_eCompletionMode = modeNone;
	
	// Add SQL keywords to list
    QStringList strKeyWords = KPGSyntaxHighlighter::listOfKeyWords();
    
    for(QStringList::const_iterator cit = strKeyWords.begin(); cit != strKeyWords.end(); ++cit)
    {
        // SQL keywords are identified as nodeUnselected
		m_listOfCodeCompletionObjects.append(KPGOidName(0, *cit, QString::null, KPGTreeItem::nodeUnselected, KPGTreeItem::m_pIconNull, QString::null));
    }
	completionObject()->insertItems(KPGSyntaxHighlighter::listOfKeyWords());
		
	// SQL is case insensitive language
	completionObject()->setIgnoreCase(true); 
		
	// Apply current configuration
	slotSqlEditorSettingsChanged();
	
 	// and make sure to be informed about its changes.
	connect(KPoGreView::configuration()->sqleditor(), SIGNAL(sigChanged()), this, SLOT(slotSqlEditorSettingsChanged()));
	connect(m_pCompletionBox, SIGNAL(activated(const QString &)), this, SLOT(setCompletedText(const QString &)));
	connect(this, SIGNAL(textChanged()), this, SLOT(slotTextChanged()));
	
	// Setup timer
	m_pUpdateCodeCompletionListTimer = new QTimer(this);
    connect(m_pUpdateCodeCompletionListTimer, SIGNAL(timeout()), this, SLOT(slotUpdateCodeCompletionList()));
	m_pUpdateCodeCompletionListTimer->start( UPDATE_TIMER_INTERVAL ); 
}

KPGSqlEdit::~KPGSqlEdit()
{
	delete m_pSyntaxHighlighter;
	if(m_pFind != 0) 
    {
    	disconnect( m_pFind, 0, this, 0);
    	delete m_pFind;
    }
	if(m_pReplace != 0) delete m_pReplace;
	
	m_pUpdateCodeCompletionListTimer->stop(); // stop timer, but don't delete it
}

void KPGSqlEdit::slotSqlEditorSettingsChanged()
{
	m_pSyntaxHighlighter->setColorDefaultText( KPoGreView::configuration()->sqleditor()->colorDefaultText() );
	m_pSyntaxHighlighter->setColorKeyWords( KPoGreView::configuration()->sqleditor()->colorKeyWords() );
	m_pSyntaxHighlighter->setColorDataTypes( KPoGreView::configuration()->sqleditor()->colorDataTypes() );
	m_pSyntaxHighlighter->setColorOperators( KPoGreView::configuration()->sqleditor()->colorOperators() );
	m_pSyntaxHighlighter->setColorQuotedStrings( KPoGreView::configuration()->sqleditor()->colorQuotedStrings() );
	m_pSyntaxHighlighter->setColorNumbers( KPoGreView::configuration()->sqleditor()->colorNumbers() );
    m_pSyntaxHighlighter->setColorComments( KPoGreView::configuration()->sqleditor()->colorComments() );
	
    setFont( KPoGreView::configuration()->sqleditor()->font() );
    
	m_pSyntaxHighlighter->rehighlight();
}

// Filters events - for watching Ctrl+SPACE
bool KPGSqlEdit::eventFilter(QObject *pObject, QEvent *pEvent)
{
	if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->stop(); 
	
	switch(pEvent->type()) 
	{
		case QEvent::KeyPress: 
		{
			QKeyEvent *pKeyEvent = (QKeyEvent*) pEvent;
				
			if((pKeyEvent->key() == Key_Space) && (pKeyEvent->state() & ControlButton) && (m_pCompletionBox->isVisible() == false))
			{
			    updateCtrlSpaceCompletion(true); 
                return true;
			}
			else if((pKeyEvent->key() == Key_Period) && (m_pCompletionBox->isVisible() == false))
			{
			    updatePeriodCompletion(true); 
			}
			
			break;
		}
		default: break;
	}
	
	bool bRetVal = KTextEdit::eventFilter(pObject, pEvent);
	
	if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->start( UPDATE_TIMER_INTERVAL ); 
	
	return bRetVal;
}

// Update completion when text changed
void KPGSqlEdit::slotTextChanged()
{
    if(m_pCompletionBox->isVisible())
	{
		if(m_eCompletionMode == modeMainObjects)
		{
		  updateCtrlSpaceCompletion(false);
		}
		else if(m_eCompletionMode == modeSchemaChilds) 
		{
		  updatePeriodCompletion(false);
		}
		else if(m_eCompletionMode == modeTableColumns) 
		{
		  updatePeriodCompletion(false);
		}
		else
		{
		  m_bCodeCompletionListUpdated = false;
		  Q_ASSERT(m_pUpdateCodeCompletionListTimer->isActive());
		}
	}
	else
	{
	   m_bCodeCompletionListUpdated = false;
	   Q_ASSERT(m_pUpdateCodeCompletionListTimer->isActive());
	}
}

// TODO multiline comments !
const QString KPGSqlEdit::uncompletedWord()
{
	// Get cursor position 
	int para, index;
	getCursorPosition(&para, &index );								
												
	// Get text at cursor position, look at last word before cursor
	QString strText(text(para));
	QString strOneWord("");
	bool bInsideComment = false;
	
	for(unsigned int i = 0; i < strText.length(); i++)
	{
		if(i >= (unsigned int) index) break;
					
	    QChar ch = strText[i];
		
		if((ch == '-') && (i > 0) && (strText[i-1] == '-') && (!bInsideComment))
		{
		    bInsideComment = true;
            continue;
		}
		
		if(bInsideComment) 
		{
          if(ch == '\n')
		  {   bInsideComment = false;
		      continue;
		  }
		  else continue;
		}
		
		if((ch.isLetter()) || (ch.isDigit()) || (ch == '$') || (ch == '_'))
		{
		    strOneWord.append(ch);
		}
		else
		{
		    strOneWord.truncate(0);
		}
	}
	
	QRegExp patternIdentifier("^[\\w][\\w$]*$"); // SQL identifiers and key words must begin with a letter 
    // (a-z, but also letters with diacritical marks and non-Latin letters) or an underscore (_). 
    // Subsequent characters in an identifier or key word can be letters, underscores, digits (0-9), or dollar signs ($). 
	
	if((strOneWord.length() > 0) && (!strOneWord[0].isDigit()) && (patternIdentifier.search(strOneWord, 0) == 0))
		return strOneWord; // Return uncompleted word, only if it's valid identifier
	else
	   return QString("");
}

// Get uncompleted word, set it to completion and set completed items to box, eventually show it
void KPGSqlEdit::updateCtrlSpaceCompletion(bool bShow)
{
    QString strWord(uncompletedWord());
													
	// Set-up and eventually show completion box
	QStringList strListMatches;
	if(strWord.length() > 0)
	{											
		kdDebug() << k_funcinfo << " for: " << strWord << endl;
		strListMatches = completionObject()->allMatches(strWord.upper());
	}	
	else
	{
	    for(KPGOidNameAliasesList::const_iterator cit = m_listOfCodeCompletionObjects.begin(); cit != m_listOfCodeCompletionObjects.end(); ++cit)
	    {
            strListMatches.append((*cit).name());
        }
	}	
		
	m_eCompletionMode = modeMainObjects;
	updateCompletion(bShow, strListMatches);
}

// Restore completion words from list of main objects
void KPGSqlEdit::restoreCtrlSpaceCompletion()
{
    completionObject()->clear();
    
    // Create list of all completions
    QStringList strAllCompletions;
    for(KPGOidNameAliasesList::const_iterator cit = m_listOfCodeCompletionObjects.begin(); cit != m_listOfCodeCompletionObjects.end(); ++cit)
	{
        strAllCompletions.append((*cit).name());
    }
        
    completionObject()->insertItems(strAllCompletions);
}
	
// Set columns into completion if table/function name is left before period: aaa.
void KPGSqlEdit::updatePeriodCompletion(bool bShow)
{
    QString strWord(uncompletedWord());	
		
	if(strWord.length() > 0)
	{											
		// If code completion is not up-to date, update it first
        if(!m_bCodeCompletionListUpdated)
        {
        	slotUpdateCodeCompletionList();
        }
        
        kdDebug() << k_funcinfo << " for: " << strWord << endl;
		if(bShow)
		{
		    // Uncompleted word is table or alias
		      
            // Find table or function, 1st by name, 2nd by alias
            const KPGOidNameAliases * pOidNameAliases = m_listOfCodeCompletionObjects.getConstItemByName(strWord); 
            if(pOidNameAliases == 0)
            {
                pOidNameAliases = m_listOfCodeCompletionObjects.getConstItemByAlias(strWord); 
            }
            
            if(pOidNameAliases && (pOidNameAliases->type() == KPGTreeItem::nodeSchema))
            {
                emit sigRequestSchemaChildsListsForCodeCompletion(pOidNameAliases->oid());
                QStringList strListMatches;
                
                for(KPGOidNameList::const_iterator cit = m_listOfCodeCompletionSchemaChilds.begin(); cit != m_listOfCodeCompletionSchemaChilds.end(); ++cit)
                {
                    strListMatches.append((*cit).name());
                }
                
                m_eCompletionMode = modeSchemaChilds;
                updateCompletion(bShow, strListMatches);
            }
            else if(pOidNameAliases && (pOidNameAliases->type() == KPGTreeItem::nodeTable))
            {
                emit sigRequestTableColumnsListsForCodeCompletion(pOidNameAliases->oid());
                
                QStringList strListMatches;
                for(KPGListTableColumns::const_iterator cit = m_listOfCodeCompletionColumns.begin(); cit != m_listOfCodeCompletionColumns.end(); ++cit)
                {
                    strListMatches.append((*cit).name());
                }
                
                m_eCompletionMode = modeTableColumns;
                updateCompletion(bShow, strListMatches);
            }
            else if(pOidNameAliases && (pOidNameAliases->type() == KPGTreeItem::nodeFunction))
            {
                emit sigRequestFunctionReturnTypeAttributesListsForCodeCompletion(pOidNameAliases);
                
                QStringList strListMatches;
                for(KPGListTableColumns::const_iterator cit = m_listOfCodeCompletionColumns.begin(); cit != m_listOfCodeCompletionColumns.end(); ++cit)
                {
                    strListMatches.append((*cit).name());
                }
                
                m_eCompletionMode = modeTableColumns;
                updateCompletion(bShow, strListMatches);
            }
		}
		else
		{
		    // Uncompleted word is table column
		    
		    // Set-up and eventually show completion box
            QStringList strListMatches;
            if(strWord.length() > 0)
            {											
                strListMatches = completionObject()->allMatches(strWord.upper());
            }	
            else
            {
                /*for(KPGOidNameList::const_iterator cit = m_listOfCodeCompletionObjects.begin(); cit != m_listOfCodeCompletionObjects.end(); ++cit)
                {
                    strListMatches.append((*cit).name());
                }*/
                Q_ASSERT(false);
            }
		    
		    Q_ASSERT(m_eCompletionMode == modeTableColumns);
            updateCompletion(bShow, strListMatches);
		}
	}
}

// Set completed items to box, eventually show it
void KPGSqlEdit::updateCompletion(bool bShow, const QStringList &strListMatches)
{
	if(strListMatches.count() > 0)
	{
		QTextCursor *pTextCursor = textCursor();
												
		// Get cursor place, make correction to display completion box on proper place
		QPoint point(pTextCursor->globalX(), pTextCursor->globalY() + 18); // TODO: 18 -> fontHeigth
		point = viewport()->mapToGlobal(contentsToViewport(point));
										
		setCompletedItems(strListMatches);
		m_pCompletionBox->move(point);
			
		if(bShow)
		    m_pCompletionBox->show();
	}
	else
	{
	    if(m_pCompletionBox->isVisible())
	    {
	        m_pCompletionBox->hide();
	        
	        if(m_eCompletionMode != modeMainObjects)
                restoreCtrlSpaceCompletion();
                
	        m_eCompletionMode = modeNone;
	    }
	}
}

void KPGSqlEdit::setCompletionObject(KCompletion* comp, bool hsig)
{
    KCompletion *oldComp = compObj();
    if( oldComp && handleSignals() )
       disconnect( oldComp, SIGNAL( matches( const QStringList& )), this, SLOT( setCompletedItems( const QStringList& )));

    if( comp && hsig )
      connect( comp, SIGNAL( matches( const QStringList& )), this, SLOT( setCompletedItems( const QStringList& )));

    KCompletionBase::setCompletionObject( comp, hsig );
}

// Insert code completion
void KPGSqlEdit::setCompletedText(const QString& strCompletion)
{
	// Get cursot position 
	int para, index;
	getCursorPosition(&para, &index);								
												
	// Get text at cursor position, look at last word before cursor
	QString strText = text(para);
	for(int i = index; i > 0; i--)
	{
	    QChar ch = strText[i - 1];
		if((ch == ' ') || (ch == '.') || (ch == '\t') || (ch == '\n') || (ch == '(') || (ch == ')')) break;
		//kdDebug() << "Removing: " << strText[i-1] << endl;
		doKeyboardAction(ActionBackspace); // remove uncomplete word
	}
	
	getCursorPosition(&para, &index );	
	insertAt(strCompletion, para, index ); // insert completed word
	KTextEdit::setCursorPosition(para, index +  strCompletion.length());
		
	if(m_eCompletionMode != modeMainObjects)
        restoreCtrlSpaceCompletion();
        
    m_eCompletionMode = modeNone;
}

void KPGSqlEdit::setCompletedItems(const QStringList& items)
{
    QString txt;
    if (m_pCompletionBox && m_pCompletionBox->isVisible() )
    {
        // The popup is visible already - do the matching on the initial string,
        // not on the currently selected one.
        txt = m_pCompletionBox->cancelledText();
    } 
    else 
    {
        txt = uncompletedWord();
    }
    
    if ( !items.isEmpty() && !(items.count() == 1 && txt == items.first()) )
    {
    	if ( m_pCompletionBox->isVisible() )
        {
        	bool wasSelected = m_pCompletionBox->isSelected( m_pCompletionBox->currentItem() );
            const QString currentSelection = m_pCompletionBox->currentText();
            
            switch(m_eCompletionMode)
            {
            	case modeMainObjects: 
            			m_pCompletionBox->setItems(items, m_listOfCodeCompletionObjects);
            			break;
            			
            	case modeSchemaChilds: 
            			m_pCompletionBox->setItems(items, m_listOfCodeCompletionSchemaChilds);
            			break;
            			
            	case modeTableColumns:
            			m_pCompletionBox->setItems(items, m_listOfCodeCompletionColumns);
            			break;
            			
            	case modeNone: 
            	        Q_ASSERT(false);
            	        break;	
            }
            
            QListBoxItem* item = m_pCompletionBox->findItem( currentSelection, Qt::ExactMatch );
            // If no item is selected, that means the listbox hasn't been manipulated by the user yet,
            // because it's not possible otherwise to have no selected item. In such case make
            // always the first item current and unselected, so that the current item doesn't jump.
            if( !item || !wasSelected )
            {
                wasSelected = false;
                item = m_pCompletionBox->item( 0 );
            }
            if ( item )
            {
                m_pCompletionBox->blockSignals( true );
                m_pCompletionBox->setCurrentItem( item );
                m_pCompletionBox->setSelected( item, wasSelected );
                m_pCompletionBox->blockSignals( false );
            }
        }
        else // completion box not visible yet -> show it
        {
            if ( !txt.isEmpty() )
	            m_pCompletionBox->setCancelledText( txt );
            
            switch(m_eCompletionMode)
            {
            	case modeMainObjects: 
            			m_pCompletionBox->setItems(items, m_listOfCodeCompletionObjects);
            			break;
            			
            	case modeSchemaChilds: 
            			m_pCompletionBox->setItems(items, m_listOfCodeCompletionSchemaChilds);
            			break;
            			
            	case modeTableColumns:
            			m_pCompletionBox->setItems(items, m_listOfCodeCompletionColumns);
            			break;
            			
            	case modeNone: 
            	        Q_ASSERT(false);
            	        break;
            	
            }
            m_pCompletionBox->popup();
        }
    }
    else
    {
        if(m_pCompletionBox && m_pCompletionBox->isVisible())
        {
            m_pCompletionBox->hide();
            
            if(m_eCompletionMode != modeMainObjects)
                restoreCtrlSpaceCompletion();
            
            m_eCompletionMode = modeNone;
        }    
    }
}

// Set list of main DB objects for code completion
void KPGSqlEdit::setListOfObjectsForCodeCompletion(const KPGOidNameList &listOfCodeCompletionObjects)
{
	if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->stop(); 
    clearListOfObjectsForCodeCompletion();
    
    // Add given DB objects
    for(KPGOidNameList::const_iterator cit = listOfCodeCompletionObjects.begin(); cit != listOfCodeCompletionObjects.end(); ++cit)
    {
        m_listOfCodeCompletionObjects.append(KPGOidNameAliases(*cit));
    }
    
    // Sort list of code completions
    m_listOfCodeCompletionObjects.sort();
    
    // Create list of all completions
    QStringList strAllCompletions;
    for(KPGOidNameAliasesList::const_iterator cit = m_listOfCodeCompletionObjects.begin(); cit != m_listOfCodeCompletionObjects.end(); ++cit)
	{
        strAllCompletions.append((*cit).name());
    }
        
    completionObject()->insertItems(strAllCompletions);
    
    m_bCodeCompletionListUpdated = false;
    if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->start( UPDATE_TIMER_INTERVAL ); 
}

// Set list of schema childs for code completion
void KPGSqlEdit::setListOfSchemaChildsForCodeCompletion(const KPGOidNameList &listOfCodeCompletionObjects)
{
	if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->stop(); 
    clearListOfSchemaChildsForCodeCompletion();
    
    // Add given DB objects
    for(KPGOidNameList::const_iterator cit = listOfCodeCompletionObjects.begin(); cit != listOfCodeCompletionObjects.end(); ++cit)
    {
        m_listOfCodeCompletionSchemaChilds.append(*cit);
    }
    
    // Sort list of code completions
    m_listOfCodeCompletionSchemaChilds.sort();
    
    // Create list of all completions
    QStringList strAllCompletions;
    for(KPGOidNameList::const_iterator cit = m_listOfCodeCompletionSchemaChilds.begin(); cit != m_listOfCodeCompletionSchemaChilds.end(); ++cit)
	{
        strAllCompletions.append((*cit).name());
    }
        
    completionObject()->insertItems(strAllCompletions);
    
    m_bCodeCompletionListUpdated = false;
    if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->start( UPDATE_TIMER_INTERVAL ); 
}

// Clear list of DB objects for code completion
void KPGSqlEdit::clearListOfObjectsForCodeCompletion()
{
	m_listOfCodeCompletionObjects.clear();
    completionObject()->clear();
    
    // Add SQL keywords to list
    QStringList strKeyWords = KPGSyntaxHighlighter::listOfKeyWords();
    
    for(QStringList::const_iterator cit = strKeyWords.begin(); cit != strKeyWords.end(); ++cit)
    {
        // SQL keywords are identified as nodeUnselected
		m_listOfCodeCompletionObjects.append(KPGOidName(0, *cit, QString::null, KPGTreeItem::nodeUnselected, KPGTreeItem::m_pIconNull, QString::null));
    }
}

// Clear list of schema childs for code completion
void KPGSqlEdit::clearListOfSchemaChildsForCodeCompletion()
{
    m_listOfCodeCompletionSchemaChilds.clear();
    completionObject()->clear();
}

// Set list of table columns/type attributes for code completion
void KPGSqlEdit::setListOfColumnsForCodeCompletion(const KPGListTableColumns &listOfTableColumns)
{
	clearListOfColumnsForCodeCompletion();
    
    // Add given table columns, create list of all completions
    QStringList strAllCompletions;
    for(KPGListTableColumns::const_iterator cit = listOfTableColumns.begin(); cit != listOfTableColumns.end(); ++cit)
    {
        m_listOfCodeCompletionColumns.append(*cit);
        strAllCompletions.append((*cit).name());
    }
            
    completionObject()->insertItems(strAllCompletions);
}

// Clear list of table columns/type attributes for code completion
void KPGSqlEdit::clearListOfColumnsForCodeCompletion()
{
	m_listOfCodeCompletionColumns.clear();
    completionObject()->clear();
}


// Fired from m_pUpdateCodeCompletionListTimer. Updates aliases 
// in m_listOfCodeCompletionObjects according to current SQL
void KPGSqlEdit::slotUpdateCodeCompletionList()
{
	if(m_bCodeCompletionListUpdated) return;
	kdDebug() << k_funcinfo << endl;
	m_pUpdateCodeCompletionListTimer->stop(); 

	KPGSqlParser::updateAliases(text(), m_listOfCodeCompletionObjects); 

	m_bCodeCompletionListUpdated = true;
	m_pUpdateCodeCompletionListTimer->start( UPDATE_TIMER_INTERVAL );
}

// Set cursor position to given character offset, select character
void KPGSqlEdit::highlightCharacter(int iPosition)
{
	for(int iParagraph = 0; iParagraph < paragraphs(); iParagraph++)
	{
		if(iPosition > paragraphLength(iParagraph))
		{
			iPosition -= paragraphLength(iParagraph) + 1;
		}
		else
		{
			KTextEdit::setCursorPosition(iParagraph, iPosition);
			KTextEdit::setSelection(iParagraph, iPosition-1, iParagraph, iPosition);
			KTextEdit::ensureCursorVisible();
			return;
		}
	}
}

// Find first occurence of text
void KPGSqlEdit::findFirst(QStringList &listOfSearchHistory)
{
    // don't listen to selection changes
    m_bIgnoreSelectionClearing = true;
    
    KFindDialog dlg(this, "", 0, listOfSearchHistory, hasSelectedText());
    dlg.setHasCursor(false);
        
    int c = dlg.exec();
    // listen to selection changes
    m_bIgnoreSelectionClearing = false;
    
    if(c != QDialog::Accepted)
        return;

    listOfSearchHistory = dlg.findHistory();
    
    if(m_pFind != 0) 
    {
    	disconnect( m_pFind, 0, this, 0);
    	delete m_pFind;
    }
    m_pFind = new KFind(dlg.pattern(), dlg.options(), this);

    // Connect highlight signal to code which handles highlighting
    // of found text.
    connect( m_pFind, SIGNAL( highlight( const QString &, int, int ) ),
             this, SLOT( slotHighlight( const QString &, int, int ) ) );
             
    // Connect findNext signal - called when pressing the button in the dialog
    connect( m_pFind, SIGNAL( findNext() ), this, SLOT( slotFindNext() ) );

	setUpFindBoundaries(dlg.options(), &m_iStartPara, &m_iStartIdx, &m_iEndPara, &m_iEndIdx);
    
    m_iParaToSearch = m_iStartPara;
    m_pFind->setData(text(m_iStartPara), m_iStartIdx);

    slotFindNext();
}

// Find next occurence of text
void KPGSqlEdit::findNext()
{
	if (!m_pFind) // shouldn't be called before find is activated
        return;
        
	getCursorPosition(&m_iParaToSearch, &m_iStartIdx);
	m_pFind->setData(text(m_iParaToSearch), m_iStartIdx);
	
	slotFindNext();
}

// Find next occurence of text
void KPGSqlEdit::slotFindNext() 
{
    if (!m_pFind) // shouldn't be called before find is activated
        return;
	
    KFind::Result res = KFind::NoMatch;
    while ( res == KFind::NoMatch &&
            m_iParaToSearch <= QMAX(m_iStartPara, m_iEndPara) &&
            m_iParaToSearch >= QMIN(m_iStartPara, m_iEndPara) 
          ) 
    	{
    		//kdDebug() << "searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
        
        	if(m_pFind->needData()) 
        	{
            	int idx = (m_iParaToSearch == m_iEndPara) ? m_iEndIdx : -1;
            	m_pFind->setData(text(m_iParaToSearch), idx);
        	}
        
        	// Let KFind inspect the text fragment, and display a dialog if a match is found
        	res = m_pFind->find();
        
        	if( res == KFind::NoMatch ) 
        	{
            	m_iParaToSearch += (m_pFind->options() & KFindDialog::FindBackwards) ? -1 : 1;
        	}
        	//kdDebug() << "2searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
		}

    if( res == KFind::NoMatch ) 
    { // i.e. at end
        m_pFind->displayFinalDialog();
        m_pFind->resetCounts();
        //delete m_pFind;
        //m_pFind = 0L;
        removeSelection(0);
    }
}

// Highligth found text
void KPGSqlEdit::slotHighlight( const QString &, int index, int length)
{
    //kdDebug() << "highlight: " << index << " " << length << endl;
    setSelection(m_iParaToSearch, index, m_iParaToSearch, index + length );
}

void KPGSqlEdit::selectAll( bool b )
{
    if (!b && m_bIgnoreSelectionClearing)
        return;
    
    KTextEdit::selectAll(b);
}

// Replace first occurence of text
void KPGSqlEdit::replace(QStringList &listOfSearchHistory, QStringList &listOfReplacementHistory)
{
	// don't listen to selection changes
    m_bIgnoreSelectionClearing = true;
    
    KReplaceDialog dlg(this, "", 0, listOfSearchHistory, listOfReplacementHistory, hasSelectedText());
    dlg.setHasCursor(false);
        
    int c = dlg.exec();
    // listen to selection changes
    m_bIgnoreSelectionClearing = false;
    
    if(c != QDialog::Accepted)
        return;

    listOfSearchHistory = dlg.findHistory();
    listOfReplacementHistory = dlg.replacementHistory();
        
    
    if(m_pReplace != 0) delete m_pReplace;
    m_pReplace = new KReplace(dlg.pattern(), dlg.replacement(), dlg.options(), this);

    // Connect highlight signal to code which handles highlighting
    // of found text.
    connect( m_pReplace, SIGNAL( highlight( const QString &, int, int ) ),
             this, SLOT( slotHighlight( const QString &, int, int ) ) );
             
    // Connect findNext signal - called when pressing the button in the dialog
    connect( m_pReplace, SIGNAL( findNext() ), this, SLOT( slotReplaceNext() ) );

	// Connect replace signal - called when doing a replacement
  	connect( m_pReplace, SIGNAL( replace(const QString &, int, int, int) ),
          this, SLOT( slotReplace(const QString &, int, int, int) ) );

	setUpFindBoundaries(dlg.options(), &m_iStartPara, &m_iStartIdx, &m_iEndPara, &m_iEndIdx);
	
    m_iParaToSearch = m_iStartPara;
    m_pReplace->setData(text(m_iStartPara ), m_iStartIdx);

    slotReplaceNext();
}

// Replace next occurence of text
void KPGSqlEdit::slotReplaceNext()
{
    if (!m_pReplace) // shouldn't be called before find is activated
        return;

	KFind::Result res = KFind::NoMatch;
    while ( res == KFind::NoMatch &&
            m_iParaToSearch <= QMAX(m_iStartPara, m_iEndPara ) &&
            m_iParaToSearch >= QMIN(m_iStartPara, m_iEndPara ) ) 
    {
    	//kdDebug() << "searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
        
       	if(m_pReplace->needData()) 
       	{
           	int idx = (m_iParaToSearch == m_iEndPara) ? m_iEndIdx : -1;
           	m_pReplace->setData( text(m_iParaToSearch), idx);
       	}
        
       	// Let KFind inspect the text fragment, and display a dialog if a match is found
       	res = m_pReplace->replace();
        
       	if( res == KFind::NoMatch ) 
       	{
           	m_iParaToSearch += (m_pReplace->options() & KFindDialog::FindBackwards) ? -1 : 1;
       	}
       	//kdDebug() << "2searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
	}

    if( res == KFind::NoMatch ) 
    { // i.e. at end
        m_pReplace->displayFinalDialog();
        delete m_pReplace;
        m_pReplace = 0L;
        removeSelection(0);
    }
    
    m_pSyntaxHighlighter->rehighlight();
}

// Implement updating of replaced text during the replace operation
void KPGSqlEdit::slotReplace(const QString &strText, int replacementIndex, int replacedLength, int /*matchedLength*/)
{
	kdDebug() << k_funcinfo << " " << strText << " replacementIndex: " << replacementIndex << " replacedLength: " << replacedLength<< endl;
	
	// don't listen to selection changes
    m_bIgnoreSelectionClearing = true;
    m_pSyntaxHighlighter->disable();
    
	removeParagraph(m_iParaToSearch);
	insertParagraph(strText, m_iParaToSearch);
	
	// listen to selection changes
    m_bIgnoreSelectionClearing = false;
    m_pSyntaxHighlighter->enable();
}
  
// Set up boudaries for find/replace
void KPGSqlEdit::setUpFindBoundaries(long lOptions, int *piStartPara, int *piStartIdx, int *piEndPara, int *piEndIdx)
{ 
    if(lOptions & KFindDialog::SelectedText ) 
    {
        Q_ASSERT(hasSelectedText());
        if(lOptions & KFindDialog::FindBackwards) 
        	getSelection(piEndPara, piEndIdx, piStartPara, piStartIdx);
        else
        	getSelection(piStartPara, piStartIdx, piEndPara, piEndIdx);
    } 
    else 
    {
    	*piStartIdx  = -1; // Interpreted as "process all data" by KFind
    	*piEndIdx    = -1; // Interpreted as "process all data" by KFind
    	
    	if(lOptions & KFindDialog::FindBackwards) 
		{
			*piStartPara = paragraphs();
			*piEndPara   = 0;
		}
    	else
    	{
        	*piStartPara = 0;
        	*piEndPara   = paragraphs();
        }
    }
}


void KPGSqlEdit::virtual_hook( int id, void* data )
{ KCompletionBase::virtual_hook( id, data ); }
