/**************************************************************************
*   Copyright (C) 2007 by Remi Villatel <maxilys@tele2.fr>                *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/


#include <qcheckbox.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qradiobutton.h>
#include <qslider.h>
#include <qspinbox.h>

#include <dcopclient.h>
#include <kapp.h>
#include <kcolorbutton.h>
#include <kconfig.h>
#include <kfontrequester.h>
#include <kglobal.h>
#include <klocale.h>
#include <knuminput.h>
#include <kparts/genericfactory.h>

#include "desktoptext.h"
#include "textconfig.h"
#include "textpreviewwidget.h"

typedef KGenericFactory<DesktopText, QWidget> DesktopTextFactory;
K_EXPORT_COMPONENT_FACTORY(kcm_desktoptext, DesktopTextFactory("kcmdesktoptext"))

DesktopText::DesktopText(QWidget *parent, const char *name, const QStringList&)
	: KCModule(parent, name), p_myAboutData(0)
{
	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setAutoAdd(true);

	p_configWidget = new TextConfig(this);

	p_client = kapp->dcopClient();

	p_client->attach();

	load();
};

DesktopText::~DesktopText() {}

void DesktopText::load()
{
	const TextPreviewWidget* preview = p_configWidget->standardPreview;

	p_configWidget->standardPreview->init("kdesktoprc");

	p_configWidget->backColorButton->setColor(preview->backColor());
	p_configWidget->customColor->setChecked(preview->useCustomShadow());
	p_configWidget->decayCombo->setCurrentItem(TextPreviewWidget::shadowAlgorithm(preview->algorithm()));
	p_configWidget->enableShadowsCheck->setChecked(preview->shadowEnabled());
	p_configWidget->factorDoubleSpin->setValue(preview->factor());
	p_configWidget->fontRequester->setFont(preview->deskFont());
	p_configWidget->foreColorButton->setColor(preview->foreColor());
	p_configWidget->offsetXSpin->setValue(preview->offsetX());
	p_configWidget->offsetYSpin->setValue(preview->offsetY());
	p_configWidget->presetCombo->setCurrentItem(0);
	p_configWidget->textHeightSpin->setValue(preview->textHeight());
	p_configWidget->textWidthSpin->setValue(preview->textWidth());
	p_configWidget->thickSpin->setValue(preview->thickness());
	p_configWidget->opacitySlider->setValue(preview->opacity());

	p_configWidget->backColorButton->setEnabled( p_configWidget->customColor->isChecked() );
	p_configWidget->shadowFrame->setEnabled( p_configWidget->enableShadowsCheck->isChecked() );
	if (preview->selectionType() == KShadowSettings::InverseVideoOnSelection)
	{
		p_configWidget->inverseRadio->setChecked(true);
	}
	else
	{
		p_configWidget->backgroundRadio->setChecked(true);
	}

	connect(p_configWidget->backColorButton, SIGNAL(changed(const QColor &)), this, SLOT(colorChanged()));
	connect(p_configWidget->customColor, SIGNAL(toggled(bool)), this, SLOT(colorChanged()));
	connect(p_configWidget->enableShadowsCheck, SIGNAL(toggled(bool)), this, SLOT(colorChanged()));
	connect(p_configWidget->decayCombo, SIGNAL(activated(int)), this, SLOT(presetCancelled()));
	connect(p_configWidget->factorDoubleSpin, SIGNAL(valueChanged(int)), this, SLOT(presetModified()));
	connect(p_configWidget->fontRequester, SIGNAL(fontSelected(const QFont &)), this, SLOT(colorChanged()));
	connect(p_configWidget->foreColorButton, SIGNAL(changed(const QColor &)), this, SLOT(colorChanged()));
	connect(p_configWidget->inverseRadio, SIGNAL(toggled(bool)), this, SLOT(configChanged()));
	connect(p_configWidget->offsetXSpin, SIGNAL(valueChanged(int)), this, SLOT(presetModified()));
	connect(p_configWidget->offsetYSpin, SIGNAL(valueChanged(int)), this, SLOT(presetModified()));
	connect(p_configWidget->opacitySlider, SIGNAL(valueChanged(int)), this, SLOT(presetModified()));
	connect(p_configWidget->presetCombo, SIGNAL(activated(int)), this, SLOT(presetSelected()));
	connect(p_configWidget->textHeightSpin, SIGNAL(valueChanged(int)), this, SLOT(configChanged()));
	connect(p_configWidget->textWidthSpin, SIGNAL(valueChanged(int)), this, SLOT(configChanged()));
	connect(p_configWidget->thickSpin, SIGNAL(valueChanged(int)), this, SLOT(presetModified()));
}

void DesktopText::defaults()
{
	emit changed(true);
}

void DesktopText::save()
{
	KConfig config("kdesktoprc");

	config.setGroup("FMSettings");
	if (p_configWidget->customColor->isChecked())
	{
		config.writeEntry("ItemTextBackground", p_configWidget->backColorButton->color());
	}
	else
	{
		config.writeEntry("ItemTextBackground", "invalid");
	}
	config.writeEntry("NormalTextColor", p_configWidget->foreColorButton->color());
	config.writeEntry("ShadowEnabled", p_configWidget->enableShadowsCheck->isChecked());
	config.writeEntry("ShadowParameters", p_configWidget->standardPreview->settingsString());
	config.writeEntry("StandardFont", p_configWidget->fontRequester->font());
	config.writeEntry("TextHeight", p_configWidget->textHeightSpin->value());
	config.writeEntry("TextWidth", p_configWidget->textWidthSpin->value());
	config.sync();

	p_client->send("kdesktop", "KDesktopIface", "configure()", "");

	emit changed(true);
}

int DesktopText::buttons()
{
	return KCModule::Apply;
}

void DesktopText::configChanged()
{
	emit changed(true);
}

void DesktopText::presetModified()
{
	int x = p_configWidget->offsetXSpin->value();
	int y = p_configWidget->offsetYSpin->value();
	int th = p_configWidget->thickSpin->value();
	int ds = p_configWidget->decayCombo->currentItem();
	int df = p_configWidget->factorDoubleSpin->value();
	int op = p_configWidget->opacitySlider->value();
	(p_configWidget->standardPreview)->TextPreviewWidget::preset(x, y, th, ds, df, op);

	p_configWidget->presetCombo->setCurrentItem(0);
	emit changed(true);
}

void DesktopText::presetSelected()
{
	int preset = p_configWidget->presetCombo->currentItem();
	if (preset)
	{
		int x, y, th, ds, df, op;
		switch (preset)
		{
			case 2:		// Fuzzy shadow
			{
				x = 2;
				y = 2;
				th = 10;
				ds = 1;	// Linear decay
				df = 30;
				op = 192; // 75%
				break;
			}
			case 3:		// Solid halo
			{
				x = 0;
				y = 0;
				th = 2;
				ds = 0;	// No decay
				df = 10;
				op = 192; // 75%
				break;
			}
			case 4:		// Fuzzy halo
			{
				x = 0;
				y = 0;
				th = 5;
				ds = 0;	// No decay
				df = 150;
				op = 128; // 50%
				break;
			}
			default:	// Solid shadow
			{
				x = 1;
				y = 1;
				th = 1;
				ds = 1;	// Linear decay
				df = 5;
				op = 192; // 75%
				break;
			}
		}
		p_configWidget->offsetXSpin->setValue(x);
		p_configWidget->offsetYSpin->setValue(y);
		p_configWidget->thickSpin->setValue(th);
		p_configWidget->decayCombo->setCurrentItem(ds);
		p_configWidget->factorDoubleSpin->setValue(df);
		p_configWidget->opacitySlider->setValue(op);

		(p_configWidget->standardPreview)->TextPreviewWidget::preset(x, y, th, ds, df, op);

		p_configWidget->presetCombo->setCurrentItem(preset); /// MXLS: Doesn't hold. Bug?
		emit changed(true);
	}
	else	// Preset = None
	{
		emit changed(false);
	}
}

void DesktopText::colorChanged()
{
	TextPreviewWidget* tpw = p_configWidget->standardPreview; 
	tpw->TextPreviewWidget::setDeskFont(p_configWidget->fontRequester->font());
	tpw->TextPreviewWidget::setForeColor(p_configWidget->foreColorButton->color());
	if (p_configWidget->customColor->isChecked()) 
	{
		p_configWidget->backColorButton->setEnabled(true);
		tpw->TextPreviewWidget::setCustomShadow(true);
		tpw->TextPreviewWidget::setBackColor(p_configWidget->backColorButton->color());
	}
	else
	{
		p_configWidget->backColorButton->setEnabled(false);
		tpw->TextPreviewWidget::setCustomShadow(false);
	}
	if (p_configWidget->enableShadowsCheck->isChecked()) 
	{
		p_configWidget->shadowFrame->setEnabled(true);
		tpw->TextPreviewWidget::setShadowEnabled(true);
	}
	else
	{
		p_configWidget->shadowFrame->setEnabled(false);
		tpw->TextPreviewWidget::setShadowEnabled(false);
	}
	tpw->TextPreviewWidget::updatePreview();

	emit changed(true);
}

QString DesktopText::quickHelp() const
{
	return i18n("This is the desktop text configuration module.");
}

#include "desktoptext.moc"
