/***************************************************************************
*   Copyright (C) 2007 by Remi Villatel <maxilys@tele2.fr>                *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/

#include <qbitmap.h>
#include <qpainter.h>

#include <kconfig.h>
#include <klocale.h>
#include "kshadowengine.h"
#include "kshadowsettings.h"

#include "textpreviewwidget.h"

TextPreviewWidget::TextPreviewWidget(QWidget *parent, const char *name)
	: QLabel(parent, name), p_engine(0),
	p_settings(0)
{
	setFrameShadow(Sunken);
	setFrameShape(NoFrame);
	setMargin(0);

 	p_contentRect.setX(0);
 	p_contentRect.setY(0);
 	p_contentRect.setWidth(300);
 	p_contentRect.setHeight(100);
}

TextPreviewWidget::~TextPreviewWidget()
{
	cleanup();
}

void TextPreviewWidget::cleanup(void)
{
	/**
		KShadowEngine deletes KShadowSettings
		in it's destructor.
	**/
	if (p_engine)
	{
		delete p_engine;
		p_engine = 0L;
	}
}

void TextPreviewWidget::init(QString configName)
{
	cleanup();

	KConfig config (configName, true);

	p_settings = new KShadowSettings();

	config.setGroup("FMSettings");

	QColor defaultColor = Qt::black;
	QString s = config.readEntry("ItemTextBackground", "invalid");
	p_useCustomShadow = strcmp(s, "invalid"); // Different from "invalid" 
	if (p_useCustomShadow)
	{
		p_backColor = config.readColorEntry("ItemTextBackground", &defaultColor);
	}
	else
	{
		p_backColor = defaultColor;
	}
	p_foreColor = config.readColorEntry("NormalTextColor", &defaultColor);
	p_shadowEnabled = config.readBoolEntry("ShadowEnabled", true);
	p_settings->fromString(config.readEntry("ShadowParameters", DEFAULT_SHADOW_CONFIGURATION));
	QFont f("Sans Serif", 10);
	p_deskFont = config.readFontEntry("StandardFont", &f);
	p_textHeight = config.readNumEntry("TextHeight", 1);
	p_textWidth = config.readNumEntry("TextWidth", 0);

	p_engine = new KShadowEngine(p_settings);
	updatePreview();
}

KShadowSettings::SelectionType TextPreviewWidget::selectionType(void) const
{
	QString temp;
	int tempNum;
	bool convert = true;

	if (p_settings)
	{
		return p_settings->selectionType();
	}
	temp = DEFAULT_SHADOW_CONFIGURATION.section('.',
		KShadowSettings::SELECTION_TYPE, KShadowSettings::SELECTION_TYPE);

	tempNum = temp.toInt(&convert);
	if (convert)
	{
		return static_cast<KShadowSettings::SelectionType>(tempNum);
	}
	return KShadowSettings::SelectionColorsOnSelection;
}

KShadowSettings::Algorithm TextPreviewWidget::algorithm(void) const
{
	QString temp;
	int tempNum;
	bool convert = true;

	if (p_settings)
	{
		return p_settings->algorithm();
	}

	temp = DEFAULT_SHADOW_CONFIGURATION.section('.',
		KShadowSettings::ALGORITHM, KShadowSettings::ALGORITHM);

	tempNum = temp.toInt(&convert);
	if (convert)
	{
		return static_cast<KShadowSettings::Algorithm>(tempNum);
	}
	return KShadowSettings::DoubleLinearDecay;
}

int TextPreviewWidget::factor(void) const
{
	QString temp;
	double tempNum;
	bool convert = true;

	if (p_settings)
	{
		return static_cast<int> (p_settings->multiplicationFactor());
	}
	temp = DEFAULT_SHADOW_CONFIGURATION.section('.',
		KShadowSettings::MULTIPLICATION_FACTOR,
		KShadowSettings::MULTIPLICATION_FACTOR);

	tempNum = temp.toDouble(&convert);
	if (convert)
	{
		return static_cast<int> (tempNum);
	}
	return 4;
}

int TextPreviewWidget::offsetX(void) const
{
	QString temp;
	int tempNum;
	bool convert = true;

	if (p_settings)
	{
		return p_settings->offsetX();
	}
	temp = DEFAULT_SHADOW_CONFIGURATION.section('.',
		KShadowSettings::OFFSET_X, KShadowSettings::OFFSET_X);

	tempNum = temp.toInt(&convert);
	if (convert)
	{
		return tempNum;
	}
	return 2;
}

int TextPreviewWidget::offsetY(void) const
{
	QString temp;
	int tempNum;
	bool convert = true;

	if (p_settings)
	{
		return p_settings->offsetY();
	}
	temp = DEFAULT_SHADOW_CONFIGURATION.section('.',
		KShadowSettings::OFFSET_Y, KShadowSettings::OFFSET_Y);

	tempNum = temp.toInt(&convert);
	if (convert)
	{
		return tempNum;
	}
	return 2;
}

int TextPreviewWidget::thickness(void) const
{
	QString temp;
	int tempNum;
	bool convert = true;

	if (p_settings)
	{
		return p_settings->thickness();
	}
	temp = DEFAULT_SHADOW_CONFIGURATION.section('.',
		KShadowSettings::THICKNESS, KShadowSettings::THICKNESS);

	tempNum = temp.toInt(&convert);
	if (convert)
	{
		return tempNum;
	}
	return 2;
}

int TextPreviewWidget::opacity(void) const
{
	QString temp;
	double tempNum;
	bool convert = true;

	if (p_settings)
	{
		return static_cast<int> (p_settings->maxOpacity());
	}
	temp = DEFAULT_SHADOW_CONFIGURATION.section('.',
		KShadowSettings::OFFSET_X, KShadowSettings::OFFSET_X);

	tempNum = temp.toDouble(&convert);
	if (convert)
	{
		return static_cast<int> (tempNum);
	}
	return 192;	// 75%
}

void TextPreviewWidget::preset(const int x, const int y, const int th, 
				const int ds, const int df, const int op)
{
	if (!p_settings) return;

	p_settings->setOffsetX(x);
	p_settings->setOffsetY(y);
	p_settings->setThickness(th);
	if (ds == 1)		// Linear
	{
		p_settings->setAlgorithm(KShadowSettings::DefaultDecay);
	}
	else if (ds == 2)	// Double Linear
	{
		p_settings->setAlgorithm(KShadowSettings::DoubleLinearDecay);
	}
	else if (ds == 3)	// Radial
	{
		p_settings->setAlgorithm(KShadowSettings::RadialDecay);
	}
	else			// None
	{
		p_settings->setAlgorithm(KShadowSettings::NoDecay);
	}
	p_settings->setMultiplicationFactor(df);
	p_settings->setMaxOpacity( static_cast<double> (op) );
	updatePreview();
}

void TextPreviewWidget::updatePreview(void)
{
	QString sampleText = i18n("Desktop Icon Text");
	QSize ts = QFontMetrics(p_deskFont).size(SingleLine, sampleText);
	
	QImage shadowImage;
	QPainter painter;
	if (p_shadowEnabled)
	{
		p_previewPixmap.resize(ts.width()+64, ts.height()+64);
		p_previewPixmap.fill(Qt::black);
		p_previewPixmap.setMask(p_previewPixmap.createHeuristicMask(true));
	
		painter.begin(&p_previewPixmap);
		painter.setPen(Qt::white);
		painter.setFont(p_deskFont);
		painter.drawText(32, 32, sampleText);
		painter.end();
	
		p_previewPixmap.setMask(QBitmap());
	
		shadowImage = p_engine->makeShadow(p_previewPixmap, Qt::black);
	}
	if (!p_useCustomShadow)
	{
		if (qGray(p_foreColor.rgb()) > 128)
			p_backColor = Qt::black;
		else
			p_backColor = Qt::white;
	}
	int rr, gg, bb;
	p_backColor.getRgb(&rr, &gg, &bb);
	//
	if ((p_shadowEnabled) && (p_backColor != Qt::black))	// Shadow color != black
	{
		register int x, y;
		uint* line;
		uint pxl;
		for (y = 0; y < shadowImage.height(); y++)
		{
			line = (uint*)(shadowImage.scanLine(y));
			for (x = 0; x < shadowImage.width(); x++)
			{
				pxl = line[x];
				if (qAlpha(pxl) != 0) // Colorize non-transparent pixels
				{
					line[x] = qRgba(rr, gg, bb, qAlpha(pxl));
				}
			}
		}
	}
	p_previewPixmap.resize(p_contentRect.size());
	if (p_useCustomShadow && !p_shadowEnabled)
	{
		p_previewPixmap.fill(p_backColor);
	}
	else
	{
		rr = qGray(p_foreColor.rgb());
		bb = qGray(p_backColor.rgb());
		//
		if ((rr > 160) && (bb > 160))
			gg = 32;
		else if ((rr < 96) && (bb < 96))
			gg = 224;
		else if ( ((rr > 160) && (bb < 96)) || ((rr < 96) && (bb > 160)) )
			gg = 128;
		else if ((rr+bb) > 256)
			gg = 64;
		else // if ((rr+bb) < 256)
			gg = 192;
		//
		p_previewPixmap.fill( QColor(gg, gg, gg) );
	}
	painter.begin(&p_previewPixmap);
	QPoint centerPos( (p_contentRect.width() / 2) - (ts.width() / 2),
			(p_contentRect.height() / 2) + (ts.height() / 4) );
	
	if (p_shadowEnabled)
	{
		painter.drawImage((centerPos.x() + p_settings->offsetX()) - 32,
				(centerPos.y() + p_settings->offsetY()) - 32,
				shadowImage);
	}
	
	if ( (!p_shadowEnabled) && (p_foreColor == p_backColor) )
	{
		sampleText = i18n("(foreground and background identical)");
		ts = fontMetrics().size(SingleLine, sampleText);
		gg = qGray(p_backColor.rgb()) ^ 255;
		painter.setPen( QColor(gg, gg, gg) );
		painter.drawText((p_contentRect.width() / 2) - (ts.width() / 2),
				(p_contentRect.height() / 2) + (ts.height() / 4), 
				sampleText);
	}
	else
	{
		painter.setPen(p_foreColor);
		painter.setFont(p_deskFont);
		painter.drawText(centerPos, sampleText);
	}
	painter.end();
	setPixmap(p_previewPixmap);
}

int TextPreviewWidget::shadowAlgorithm(KShadowSettings::Algorithm algo)
{
	switch(algo)
	{
		case KShadowSettings::DefaultDecay: return 1;
		case KShadowSettings::DoubleLinearDecay: return 2;
		case KShadowSettings::RadialDecay: return 3;
		default:
			return 0;
	}
}

QString TextPreviewWidget::settingsString(void) const
{
	if (!p_settings) return QString::null;

	return p_settings->toString();
}

#include "textpreviewwidget.moc"
