/*
** 1999-03-13 -	A (new) module for dealing with user dialogs. The old one was written during
**		my first week of GTK+ programming (back in May 1998, if you really care), and
**		that had started to show a little too much. Hopefully, this is cleaner.
** 1999-06-19 -	Since the initial cleanliness quickly wore down due to (I guess) bad design,
**		I did a complete rewrite and redesign. This new shiny version features clear
**		separation of dialogs into synchronous and asynchronous versions. The former
**		are way (way) more useful.
** 2002-02-08 -	Added window grouping. Very nice, keeps the number of icons in Window Maker
**		down. :)
*/

#include "gentoo.h"

#include <stdio.h>

#include <gdk/gdkkeysyms.h>

#include "dialog.h"

struct Dialog {
	GtkWidget	*dlg;		/* The GtkDialog widget used to display the dialog. */
	gint		last_button;
	gint		button;
	DlgAsyncFunc	func;		/* Used in asynchronous dialogs only. */
	gpointer	user;		/* This, too. */
	gboolean	stay_open;	/* This, on the other hand, only for synchronous ones. */
};

static struct {
	GdkWindow	*group_window;
	GtkWindowPosition window_pos;
} the_state;

#define	DLG_BUTTON_SIZE		(32)	/* Max length of a single button label. */

/* ----------------------------------------------------------------------------------------- */

/* 2002-02-08 -	Set the module's internal group pointer. All dialogs have their windows
**		set as being part of the group defined by that window. Very handy.
*/
void dlg_group_set(GdkWindow *group_window)
{
	the_state.group_window = group_window;
}

void dlg_position_set(GtkWindowPosition pos)
{
	the_state.window_pos = pos;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-19 -	Build a set of action buttons (like "OK", "Cancel" and so on), and pack them into <bo>. Buttons are
**		defined by writing the desired labels separated by vertical bars (e.g. "OK|Cancel" is very common).
**		Will connect <func,user> to the "clicked" signal of all buttons. Each button will have its user data
**		pointer set to the index (counting from 0) of the button.
**		Returns the index of the last button built (e.g. 1 for the "OK|Cancel" case).
** 2002-02-21 -	Added accelerator support, with pretty underlines even.
*/
static guint build_buttons(const gchar *buttons, GtkWidget *box, GtkSignalFunc func, gpointer user)
{
	gchar		*ptr, label[DLG_BUTTON_SIZE];
	gint		index = 0;
	GtkWidget	*btn;
	GtkAccelGroup	*ag = NULL;

	for(index = 0; *buttons; index++)
	{
		for(ptr = label; (ptr - label) < (sizeof label - 1) && *buttons && *buttons != '|';)
			*ptr++ = *buttons++;
		*ptr = '\0';
		if(*buttons == '|')
			buttons++;
		if(strchr(label, '_') != NULL)		/* Accelerator in use? */
		{
			GtkWidget	*lab;
			guint		key;

			lab = gtk_label_new("");
			key = gtk_label_parse_uline(GTK_LABEL(lab), label);
			btn = gtk_button_new();
			gtk_container_add(GTK_CONTAINER(btn), lab);
			if(ag == NULL)
				ag = gtk_accel_group_new();
			gtk_widget_add_accelerator(btn, "clicked", ag, key, 0, GTK_ACCEL_VISIBLE);
		}
		else
			btn = gtk_button_new_with_label(label);
		GTK_WIDGET_SET_FLAGS(btn, GTK_CAN_DEFAULT);
		gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(func), user);
		gtk_object_set_user_data(GTK_OBJECT(btn), GINT_TO_POINTER(index));
		gtk_box_pack_start(GTK_BOX(box), btn, TRUE, TRUE, 0);
		gtk_widget_show_all(btn);
		if(index == 0)
		{
			gtk_widget_grab_default(btn);
			gtk_widget_grab_focus(btn);
		}
	}
	if(ag != NULL)
		gtk_window_add_accel_group(GTK_WINDOW(((Dialog *) user)->dlg), ag);
	return index - 1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-19 -	Close dialog down in response to some event, and report it as if button <button> was clicked. */
static void sync_close(Dialog *dlg, gint button)
{
	dlg->button = button;
	if(!dlg->stay_open)
		gtk_widget_hide(dlg->dlg);
	gtk_grab_remove(dlg->dlg);
	gtk_main_quit();
}

/* 1999-06-19 -	User clicked the dialog's window close button. Hide it, and return non-button code (-1) to caller. */
static gboolean evt_sync_delete(GtkWidget *wid, GdkEvent *evt, gpointer user)
{
	sync_close(user, -1);

	return TRUE;
}

static gboolean evt_sync_key_pressed(GtkWidget *wid, GdkEventKey *evt, gpointer user)
{
	Dialog	*dlg = user;

	switch(evt->keyval)
	{
		case GDK_Return:
			sync_close(dlg, 0);
			return TRUE;
		case GDK_Escape:
			sync_close(dlg, dlg->last_button);
			return TRUE;
	}
	return FALSE;
}

/* 1999-06-19 -	This gets called when the user clicks a button in a synchronous dialog. */
static void evt_sync_button_clicked(GtkWidget *wid, gpointer user)
{
	sync_close(user, GPOINTER_TO_INT(gtk_object_get_user_data(GTK_OBJECT(wid))));
}

/* 1999-06-19 -	Create a new, synchronous dialog box. Creates and initializes the auxilary data as well as the
**		actual dialog window, with action buttons and stuff. Note that this window, when displayed by
**		dlg_dialog_sync_wait(), will be modal, and hang around (at least in memory) until you call
**		dlg_dialog_sync_destroy().
*/
Dialog * dlg_dialog_sync_new(GtkWidget *body, const gchar *title, const gchar *buttons)
{
	Dialog		*dlg;

	if(buttons == NULL)
		buttons = _("_OK|_Cancel");

	dlg = g_malloc(sizeof *dlg);
	dlg->dlg = gtk_dialog_new();
	gtk_window_set_position(GTK_WINDOW(dlg->dlg), the_state.window_pos);
	gtk_signal_connect(GTK_OBJECT(dlg->dlg), "delete_event", GTK_SIGNAL_FUNC(evt_sync_delete), dlg);
	gtk_signal_connect(GTK_OBJECT(dlg->dlg), "key_press_event", GTK_SIGNAL_FUNC(evt_sync_key_pressed), dlg);
	dlg->stay_open = FALSE;
	if(title != NULL)
		win_window_set_title(GTK_WIDGET(dlg->dlg), title);

	if(body != NULL)
	{
		GtkWidget	*vbox;

		vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
		gtk_box_pack_start(GTK_BOX(vbox), body, TRUE, TRUE, 0);
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg->dlg)->vbox), vbox, TRUE, TRUE, 0);
		gtk_widget_show_all(vbox);
	}
	dlg->last_button = build_buttons(buttons, GTK_DIALOG(dlg->dlg)->action_area, evt_sync_button_clicked, dlg);

	return dlg;
}

/* 1999-06-19 -	Create a synchronous dialog whose body is a plain text label. Extremely sugary. */
Dialog * dlg_dialog_sync_new_simple(const gchar *body, const gchar *title, const gchar *buttons)
{
	return dlg_dialog_sync_new(gtk_label_new(body), title, buttons);
}

/* 1999-06-19 -	This (bulkily-named) function is very handy for simple confirm dialogs. */
gint dlg_dialog_sync_new_simple_wait(const gchar *body, const gchar *title, const gchar *buttons)
{
	Dialog	*dlg;
	gint	ret;

	dlg = dlg_dialog_sync_new_simple(body, title, buttons);
	ret = dlg_dialog_sync_wait(dlg);
	dlg_dialog_sync_destroy(dlg);

	return ret;
}

/* 2002-08-20 -	Make the dialog stay open until destroyed. Used only by the generic command module,
**		but on the other hand that module is kind of important.
*/
void dlg_dialog_sync_stay_open(Dialog *dlg)
{
	dlg->stay_open = TRUE;
}

/* 1999-06-19 -	Display a previously created dialog, and wait for the user to click one of its buttons. Then
**		return the index (counting from zero which is the leftmost button) of the button that was
**		clicked.
*/
gint dlg_dialog_sync_wait(Dialog *dlg)
{
	if(dlg != NULL)
	{
		int	old_errno = errno;

		gtk_widget_realize(dlg->dlg);
		gdk_window_set_group(dlg->dlg->window, the_state.group_window);
		gtk_widget_show(dlg->dlg);
		gdk_window_raise(dlg->dlg->window);
		gtk_grab_add(dlg->dlg);
		gtk_main();
		errno = old_errno;
		return dlg->button;
	}
	return -1;
}

/* 1999-06-19 -	Close a synchronous dialog as if <button> was clicked. */
void dlg_dialog_sync_close(Dialog *dlg, gint button)
{
	if(dlg != NULL)
		sync_close(dlg, button);
}

/* 1999-06-19 -	Destroy a synchronous dialog. Don't expect to be able to access body widgets after calling this. */
void dlg_dialog_sync_destroy(Dialog *dlg)
{
	gtk_widget_destroy(dlg->dlg);

	g_free(dlg);
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-19 -	Execute the close-down policy of the asynchronous dialog. */
static void async_close(Dialog *dlg, gint button)
{
	dlg->button = button;
	if((dlg->button > -2 ) && (dlg->func != NULL))
		dlg->func(dlg->button, dlg->user);
	gtk_widget_destroy(dlg->dlg);
	g_free(dlg);
}

/* 1999-06-19 -	User closed an asynch. dialog window. */
static gboolean evt_async_delete(GtkWidget *wid, GdkEvent *evt, gpointer user)
{
	async_close(user, -1);

	return TRUE;
}

/* 1999-06-19 -	User pressed a key in an asynchronous dialog. */
static gboolean evt_async_key_pressed(GtkWidget *wid, GdkEventKey *evt, gpointer user)
{
	Dialog	*dlg = user;

	switch(evt->keyval)
	{
		case GDK_Return:
			async_close(dlg, 0);
			return TRUE;
		case GDK_Escape:
			async_close(dlg, dlg->last_button);
			return TRUE;
	}
	return FALSE;
}

/* 1999-06-19 -	User clicked one of the action buttons in an asynchronous dialog. */
static void evt_async_button_clicked(GtkWidget *wid, gpointer user)
{
	async_close(user, GPOINTER_TO_INT(gtk_object_get_user_data(GTK_OBJECT(wid))));
}

/* 1999-06-19 -	Create, and immediately display, an asynchronous dialog. The supplied callback will be called
**		as the user clicks a button (or closes the window). At that point, the dialog (and any user-
**		supplied body widgets) are still around.
*/
Dialog * dlg_dialog_async_new(GtkWidget *body, const gchar *title, const gchar *buttons, DlgAsyncFunc func, gpointer user)
{
	Dialog		*dlg;

	if(buttons == NULL)
		buttons = _("_OK|_Cancel");

	dlg = g_malloc(sizeof *dlg);
	dlg->func = func;
	dlg->user = user;
	dlg->dlg = gtk_dialog_new();
	gtk_window_set_position(&GTK_DIALOG(dlg->dlg)->window, the_state.window_pos);
	gtk_signal_connect(GTK_OBJECT(dlg->dlg), "delete_event", GTK_SIGNAL_FUNC(evt_async_delete), dlg);
	gtk_signal_connect(GTK_OBJECT(dlg->dlg), "key_press_event", GTK_SIGNAL_FUNC(evt_async_key_pressed), dlg);
	if(title != NULL)
		win_window_set_title(GTK_WIDGET(&GTK_DIALOG(dlg->dlg)->window), title);
	gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dlg->dlg)->vbox), 5);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg->dlg)->vbox), body, TRUE, TRUE, 0);
	dlg->last_button = build_buttons(buttons, GTK_DIALOG(dlg->dlg)->action_area, evt_async_button_clicked, dlg);
	gtk_widget_realize(dlg->dlg);
	gdk_window_set_group(dlg->dlg->window, the_state.group_window);
	gtk_widget_show_all(dlg->dlg);

	return dlg;
}

/* 1999-06-19 -	A simple sugary wrapper for text-only dialogs. */
Dialog * dlg_dialog_async_new_simple(const gchar *body, const gchar *title, const gchar *buttons, DlgAsyncFunc func, gpointer user)
{
	return dlg_dialog_async_new(gtk_label_new(body), title, buttons, func, user);
}

/* 1999-06-19 -	Provide a simple error reporting dialog. */
Dialog * dlg_dialog_async_new_error(const gchar *body)
{
	return dlg_dialog_async_new_simple(body, _("Error"), _("OK"), NULL, NULL);
}

/* 1999-06-19 -	Close an asynchronous dialog, as if button <button> was clicked. */
void dlg_dialog_async_close(Dialog *dlg, gint button)
{
	async_close(dlg, button);
}

/* 1999-06-19 -	Close an asynchronous dialog, but do not call the user's callback. */
void dlg_dialog_async_close_silent(Dialog *dlg)
{
	async_close(dlg, -2);
}
