/////////////////////////////////////////////////////////////////////////////
// groverclient.h
// -------------------
// Grover window decoration for KDE
// -------------------
// Copyright (c) 2003, 2004 David Johnson
// Please see the header file for copyright and license information.
//////////////////////////////////////////////////////////////////////////////

#include <kconfig.h>
#include <kdeversion.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <kimageeffect.h>
#include <klocale.h>

#include <qbitmap.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qtooltip.h>

#include "groverclient.h"
#include "images.h"

using namespace Grover;

static const int BUTTONSIZE      = 14;
static const int BUTTONSIZESMALL = 12;
static const int DECOSIZE        = 8;
static const int TITLESIZE       = 22;
static const int TITLESIZESMALL  = 18;
static const int HANDLESIZE      = 8;
static const int MARGIN          = 4;

static const unsigned char close_bits[] = {
    0x00, 0x66, 0x7e, 0x3c, 0x3c, 0x7e, 0x66, 0x00};

static const unsigned char help_bits[] = {
    0x7e, 0x7e, 0x60, 0x78, 0x78, 0x00, 0x18, 0x18};

static const unsigned char max_bits[] = {
    0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x00, 0x00};

static const unsigned char min_bits[] = {
    0x00, 0x00, 0xff, 0xff, 0x7e, 0x3c, 0x18, 0x00};

static const unsigned char minmax_bits[] = {
    0x00, 0x02, 0x06, 0x0e, 0x1e, 0x3e, 0x7e, 0x00};

static const unsigned char stickydown_bits[] = {
    0x00, 0x18, 0x18, 0x7e, 0x7e, 0x18, 0x18, 0x00};

static const unsigned char sticky_bits[] = {
    0x00, 0x00, 0x00, 0x7e, 0x7e, 0x00, 0x00, 0x00};

static const unsigned char above_on_bits[] = {
    0x7e, 0x18, 0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00 };

static const unsigned char above_off_bits[] = {
    0x18, 0x3c, 0x7e, 0x18, 0x7e, 0x00, 0x00, 0x00 };

static const unsigned char below_on_bits[] = {
    0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x7e };

static const unsigned char below_off_bits[] = {
    0x00, 0x00, 0x00, 0x7e, 0x18, 0x7e, 0x3c, 0x18 };

static const unsigned char shade_on_bits[] = {
    0xff, 0xff, 0x81, 0xa5, 0x81, 0xa5, 0x81, 0xff };

static const unsigned char shade_off_bits[] = {
    0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

//////////////////////////////////////////////////////////////////////////////
// GroverFactory Class                                                      //
//////////////////////////////////////////////////////////////////////////////

bool GroverFactory::initialized_ = false;
QString GroverFactory::effect_;
Qt::AlignmentFlags GroverFactory::titlealign_;
QColor GroverFactory::colors_[ColorTypeCount][2];
KPixmap GroverFactory::pix_[PixmapTypeCount][2][2];
int GroverFactory::bordersize_ = BorderTiny;

extern "C" KDecorationFactory* create_factory()
{
    return new Grover::GroverFactory();
}

//////////////////////////////////////////////////////////////////////////////
// GroverFactory()
// ---------------
// Constructor

GroverFactory::GroverFactory()
{
    readConfig();
    createPixmaps();
    initialized_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// ~GroverFactory()
// ----------------
// Destructor

GroverFactory::~GroverFactory()
{
    initialized_ = false;
}

//////////////////////////////////////////////////////////////////////////////
// createDecoration()
// ------------------
// Create the decoration

KDecoration* GroverFactory::createDecoration(KDecorationBridge* b)
{
    return new GroverClient(b, this);
}

//////////////////////////////////////////////////////////////////////////////
// reset()
// -------
// Reset the handler. Returns true if decorations need to be remade

bool GroverFactory::reset(unsigned long changed)
{
    initialized_ = false;
    changed |= readConfig();
    if (changed & (SettingColors | SettingDecoration | SettingBorder)) {
        createPixmaps();
    }
    initialized_ = true;

    if (changed & (SettingColors | SettingDecoration | SettingFont |
		   SettingButtons | SettingBorder)) {
        return true;
    } else {
        resetDecorations(changed);
        return false;
    }
}

//////////////////////////////////////////////////////////////////////////////
// readConfig()
// ------------
// Read in the configuration file

unsigned long GroverFactory::readConfig()
{
    KConfig config("kwingroverrc");
    config.setGroup("General");

    unsigned long changed = 0;

    // grab settings
    Qt::AlignmentFlags oldalign = titlealign_;
    QString value = config.readEntry("TitleAlignment", "AlignHCenter");
    if (value == "AlignLeft") titlealign_ = Qt::AlignLeft;
    else if (value == "AlignHCenter") titlealign_ = Qt::AlignHCenter;
    else if (value == "AlignRight") titlealign_ = Qt::AlignRight;
    if (oldalign != titlealign_) changed |= SettingFont;

    QString oldeffect = effect_;
    effect_ = config.readEntry("TitleEffect", effects[0]);
    if (oldeffect != effect_) changed |= SettingDecoration;

    int oldbordersize = bordersize_;
    switch(options()->preferredBorderSize(this)) {
      case BorderTiny:
          bordersize_ = 0; break;
      case BorderNormal:
          bordersize_ = 1; break;
      case BorderLarge:
          bordersize_ = 2; break;
      case BorderVeryLarge:
          bordersize_ = 4; break;
      case BorderHuge:
          bordersize_ = 6; break;
      case BorderVeryHuge:
      case BorderOversized:
          bordersize_ = 8; break;
      default:
          bordersize_ = 0;
    }
    if (oldbordersize != bordersize_) changed |= SettingBorder;

    // some colors that KWin doesn't handle
    // Note that KDE doesn't support these, so they will all default
    config.setGroup("WM");

    QColor oldcolor = colors_[TitleBtnBlend][true];
    colors_[TitleBtnBlend][true]
        = KDecoration::options()->color(ColorButtonBg, true);
    colors_[TitleBtnBlend][true]
        = config.readColorEntry("activeTitleBtnBlend",
                                &colors_[TitleBtnBlend][true]);
    if (oldcolor != colors_[TitleBtnBlend][true]) changed |= SettingColors;

    oldcolor = colors_[TitleBtnBlend][false];
    colors_[TitleBtnBlend][false]
        = KDecoration::options()->color(ColorButtonBg, false);
    colors_[TitleBtnBlend][false]
        = config.readColorEntry("inactiveTitleBtnBlend",
                                &colors_[TitleBtnBlend][false]);
    if (oldcolor != colors_[TitleBtnBlend][false]) changed |= SettingColors;

    oldcolor = colors_[TitleBtnText][true];
    bool light = (qGray(KDecoration::options()->color(ColorButtonBg, true).rgb()) > 128);
    colors_[TitleBtnText][true] = light ?
        KDecoration::options()->color(ColorButtonBg, true).dark() :
        KDecoration::options()->color(ColorButtonBg, true).light();
    colors_[TitleBtnFg][true] = light ? Qt::black : Qt::white;
    colors_[TitleBtnFg][true]
        = config.readColorEntry("activeTitleBtnFg",
                                &colors_[TitleBtnFg][true]);
    if (oldcolor != colors_[TitleBtnText][true]) changed |= SettingColors;

    oldcolor = colors_[TitleBtnText][false];
    light = (qGray(KDecoration::options()->color(ColorButtonBg, false).rgb()) > 128);
    colors_[TitleBtnText][false] = light ?
        KDecoration::options()->color(ColorButtonBg, false).dark() :
        KDecoration::options()->color(ColorButtonBg, false).light();
    colors_[TitleBtnFg][false] = light ? Qt::black : Qt::white;
    colors_[TitleBtnFg][false]
        = config.readColorEntry("inactiveTitleBtnFg",
                                &colors_[TitleBtnFg][false]);
    if (oldcolor != colors_[TitleBtnText][false]) changed |= SettingColors;

    return changed;
}

//////////////////////////////////////////////////////////////////////////////
// createPixmaps()
// ---------------
// Create all our pixmaps

void GroverFactory::createPixmaps()
{
    QPainter painter;
    QColorGroup  group;

    for (int active=0; active<=1; ++active) {
        for (int small=0; small<=1; ++small) {
            KPixmap &btn = pix_[Button][active][small];
            KPixmap &tbar = pix_[TitleBar][active][small];
            KPixmap &frame = pix_[Frame][active][small];
            KPixmap &hand = pix_[Handle][active][small];
            KPixmap &ahand = pix_[AltHandle][active][small];

            // resize pixmap
            if (small) {
                btn.resize(BUTTONSIZESMALL, BUTTONSIZESMALL);
                tbar.resize(100, TITLESIZESMALL);
                frame.resize(0, 0);
                hand.resize(0, 0);
                ahand.resize(0, 0);
            } else {
                btn.resize(BUTTONSIZE, BUTTONSIZE);
                tbar.resize(100, TITLESIZE);
                frame.resize(bordersize_, bordersize_);
                hand.resize(HANDLESIZE, HANDLESIZE);
                ahand.resize(HANDLESIZE, HANDLESIZE);
            }

            // create gradients and fills
            createGradient(btn, KDecoration::options()->
			   color(ColorButtonBg, active), effect_);
            createGradient(tbar, KDecoration::options()->
			   color(ColorTitleBar, active), effect_);
            createGradient(frame, KDecoration::options()->
			   color(ColorFrame, active), effect_);
            createGradient(hand, KDecoration::options()->
			   color(ColorHandle, active), effect_);
            createGradient(ahand, KDecoration::options()->
			   color(ColorFrame, active), effect_);

            // rotate frame pixmaps
            QImage temp = frame.convertToImage();
            frame.convertFromImage(KImageEffect::rotate
                                   (temp, KImageEffect::Rotate270));

            int x, y, w, h, x2, y2;

            // draw button frames
            btn.rect().rect(&x, &y, &w, &h);
            x2 = w-1; y2 = h-1;

            group = KDecoration::options()->colorGroup(ColorTitleBar, active);

            painter.begin(&btn);
            painter.setPen(group.light());
            painter.drawLine(x+1, y2, x2, y2);
            painter.drawLine(x2, y+1, x2, y2);

            painter.setPen(group.mid());
            painter.drawLine(x, y, x2, y);
            painter.drawLine(x, y, x, y2);
   
            painter.setPen(active ? group.shadow() : group.dark());
            painter.drawRect(x+1, y+1, w-2, h-2);
            painter.end();
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// borderSizes()
// -------------
// Return list of valid border sizes

QValueList<GroverFactory::BorderSize> GroverFactory::borderSizes() const
{
    return QValueList<BorderSize>() << BorderTiny << BorderNormal
                                    << BorderLarge <<  BorderVeryLarge
                                    << BorderHuge << BorderVeryHuge;
}

//////////////////////////////////////////////////////////////////////////////
// GroverButton Class                                                        //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// GroverButton()
// --------------
// Constructor

GroverButton::GroverButton(GroverClient *parent, const char *name,
                           const QString& tip, bool small, ButtonType type,
                           const unsigned char *bitmap)
    : QButton(parent->widget(), name), client_(parent), type_(type),
      deco_(0), icon_(), small_(small), lastmouse_(NoButton), mouseover_(false)
{
    setBackgroundMode(NoBackground);
    setCursor(arrowCursor);
    QToolTip::add(this, tip);

    if (small_) {
        setFixedSize(BUTTONSIZESMALL, BUTTONSIZESMALL);
    } else {
        setFixedSize(BUTTONSIZE, BUTTONSIZE);
    }

    if (bitmap) setBitmap(bitmap);
}

GroverButton::~GroverButton()
{
    if (deco_) delete deco_;
}

//////////////////////////////////////////////////////////////////////////////
// setBitmap()
// -----------
// Set the button decoration

void GroverButton::setBitmap(const unsigned char *bitmap)
{
    if (deco_) delete deco_;
    deco_ = new QBitmap(DECOSIZE, DECOSIZE, bitmap, true);
    deco_->setMask(*deco_);
    repaint(false);
}

//////////////////////////////////////////////////////////////////////////////
// setIcon()
// ---------
// Set the icon

void GroverButton::setIcon()
{
    if (deco_) { delete deco_; deco_ = 0; }

    // The standard 16x16 icon is too large, so we scale it
    icon_ = client_->icon().pixmap(QIconSet::Small, QIconSet::Normal);
    icon_.convertFromImage(icon_.convertToImage().smoothScale(width(),
                                                              height()));

    repaint(false);   
}

//////////////////////////////////////////////////////////////////////////////
// sizeHint()
// ----------
// Return size hint

QSize GroverButton::sizeHint() const
{
    if (small_)
        return QSize(BUTTONSIZESMALL, BUTTONSIZESMALL);
    else
        return QSize(BUTTONSIZE, BUTTONSIZE);
}

//////////////////////////////////////////////////////////////////////////////
// enterEvent()
// ------------
// Mouse has entered the button

void GroverButton::enterEvent(QEvent *e)
{
    mouseover_ = true;
    repaint(false);
    QButton::enterEvent(e);
}

//////////////////////////////////////////////////////////////////////////////
// leaveEvent()
// ------------
// Mouse has left the button

void GroverButton::leaveEvent(QEvent *e)
{
    mouseover_ = false;
    repaint(false);
    QButton::leaveEvent(e);
}

//////////////////////////////////////////////////////////////////////////////
// mousePressEvent()
// -----------------
// Button has been pressed

void GroverButton::mousePressEvent(QMouseEvent* e)
{
    lastmouse_ = e->button();
    
    // translate and pass on mouse event
    int button = LeftButton;
    if ((type_ != ButtonMax) && (e->button() != LeftButton)) {
        button = NoButton; // middle & right buttons inappropriate
    }
    QMouseEvent me(e->type(), e->pos(), e->globalPos(),
                   button, e->state());
    QButton::mousePressEvent(&me);
}

//////////////////////////////////////////////////////////////////////////////
// mouseReleaseEvent()
// -----------------
// Button has been released

void GroverButton::mouseReleaseEvent(QMouseEvent* e)
{
    lastmouse_ = e->button();
    
    // translate and pass on mouse event
    int button = LeftButton;
    if ((type_ != ButtonMax) && (e->button() != LeftButton)) {
        button = NoButton; // middle & right buttons inappropriate
    }
    QMouseEvent me(e->type(), e->pos(), e->globalPos(),
                   button, e->state());
    QButton::mouseReleaseEvent(&me);
}

//////////////////////////////////////////////////////////////////////////////
// drawButton()
// ------------
// Draw the button

void GroverButton::drawButton(QPainter *painter)
{
    if (!GroverFactory::initialized()) return;

    int dx, dy;
    bool active = client_->isActive();

    if (type_ == ButtonMenu) {
        if (small_) dx = dy = (TITLESIZESMALL - BUTTONSIZESMALL) / 2;
        else dx = dy = (TITLESIZE - BUTTONSIZE) / 2;

        painter->drawPixmap(0, 0, GroverFactory::pix(TitleBar, active, small_),
                            dx, dy);
        if (!icon_.isNull()) painter->drawPixmap(0, 0, icon_);
    } else {
        painter->drawPixmap(0, 0, GroverFactory::pix(Button, active, small_));

        // draw deco on active buttons
        if (active) {
            dx = (width()-DECOSIZE) / 2;
            dy = (height()-DECOSIZE) / 2;
            if (isDown()) { dx++; dy++; }
            if (mouseover_) {
                painter->setPen(GroverFactory::color(TitleBtnFg, true));
            } else {
                painter->setPen(GroverFactory::color(TitleBtnText, true));
            }
            if (deco_) painter->drawPixmap(dx, dy, *deco_);
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// GroverClient Class                                                        //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// GroverClient()
// -------------
// Constructor

GroverClient::GroverClient(KDecorationBridge *b, KDecorationFactory *f)
    : KDecoration(b, f) { ; }

GroverClient::~GroverClient()
{
    for (int n=0; n<ButtonTypeCount; n++) {
        if (button[n]) delete button[n];
    }
}

//////////////////////////////////////////////////////////////////////////////
// init()
// ------
// Real initialization

void GroverClient::init()
{
    createMainWidget(WResizeNoErase | WRepaintNoErase);
    widget()->installEventFilter(this);
    widget()->setBackgroundMode(NoBackground);

    if (isTool()) { // tool windows have tiny titlebars
  	titleheight_ = TITLESIZESMALL;
    } else {
  	titleheight_ = TITLESIZE;
    }

    // setup layout
    QVBoxLayout *mainlayout = new QVBoxLayout(widget());
    QHBoxLayout *titlelayout = new QHBoxLayout();
    QHBoxLayout *windowlayout = new QHBoxLayout();

    titlebar_ = new QSpacerItem(1, titleheight_, QSizePolicy::Expanding,
                                QSizePolicy::Fixed);
    handle_ = new QSpacerItem(1, HANDLESIZE, QSizePolicy::Expanding,
                              QSizePolicy::Fixed);

    mainlayout->setResizeMode(QLayout::FreeResize);
    mainlayout->addLayout(titlelayout);
    mainlayout->addLayout(windowlayout);
    mainlayout->addItem(handle_);

    windowlayout->addSpacing(GroverFactory::borderSize());
    if (isPreview()) {
        windowlayout->addWidget(
            new QLabel(i18n("<b><center>Grover preview</center></b>"),
            widget()));
    } else {
        windowlayout->addItem(new QSpacerItem(0, 0));
    }
    windowlayout->addSpacing(GroverFactory::borderSize());

    // setup titlebar
    for (int n=0; n<ButtonTypeCount; n++) button[n] = 0;
    addButtons(titlelayout, options()->titleButtonsLeft());
    titlelayout->addItem(titlebar_);
    addButtons(titlelayout, options()->titleButtonsRight());

    connect( this, SIGNAL(keepAboveChanged(bool)), SLOT(keepAboveChange(bool)));
    connect( this, SIGNAL(keepBelowChanged(bool)), SLOT(keepBelowChange(bool)));
}

//////////////////////////////////////////////////////////////////////////////
// addButtons()
// ------------
// Add buttons to title layout

void GroverClient::addButtons(QBoxLayout *layout, const QString& s)
{
    const unsigned char *bitmap;
    QString tip;

    if (s.length() > 0) {
        layout->addSpacing(MARGIN);
        for(unsigned n=0; n < s.length(); n++) {
            switch(s[n].latin1()) {
              case 'M': // Menu button
                  if (!button[ButtonMenu]) {
                      button[ButtonMenu] =
                          new GroverButton(this, "menu", i18n("Menu"),
                                           isTool(), ButtonMenu);
                      button[ButtonMenu]->setIcon();
                      connect(button[ButtonMenu], SIGNAL(pressed()),
                              this, SLOT(menuButtonPressed()));
                      layout->addWidget(button[ButtonMenu]);
                      if (n < s.length()-1) layout->addSpacing(1);
                  }
                  break;

              case 'S': // Sticky button
                  if (!button[ButtonSticky]) {
		      if (isOnAllDesktops()) {
			  bitmap = stickydown_bits;
			  tip = i18n("Un-Sticky");
		      } else {
			  bitmap = sticky_bits;
			  tip = i18n("Sticky");
		      }
                      button[ButtonSticky] =
                          new GroverButton(this, "sticky", tip,
                                           isTool(), ButtonSticky, bitmap);
                      connect(button[ButtonSticky], SIGNAL(clicked()),
                              this, SLOT(toggleOnAllDesktops()));
                      layout->addWidget(button[ButtonSticky]);
                      if (n < s.length()-1) layout->addSpacing(1);
                  }
                  break;

              case 'H': // Help button
                  if ((!button[ButtonHelp]) && providesContextHelp()) {
                      button[ButtonHelp] =
                          new GroverButton(this, "help", i18n("Help"),
                                           isTool(), ButtonHelp, help_bits);
                      connect(button[ButtonHelp], SIGNAL(clicked()),
                              this, SLOT(showContextHelp()));
                      layout->addWidget(button[ButtonHelp]);
                      if (n < s.length()-1) layout->addSpacing(1);
                  }
                  break;

              case 'I': // Minimize button
                  if ((!button[ButtonMin]) && isMinimizable()) 	{
                      button[ButtonMin] =
                          new GroverButton(this, "iconify", i18n("Minimize"),
                                           isTool(), ButtonMin, min_bits);
                      connect(button[ButtonMin], SIGNAL(clicked()),
                              this, SLOT(minimize()));
                      layout->addWidget(button[ButtonMin]);
                      if (n < s.length()-1) layout->addSpacing(1);
                  }
                  break;

              case 'A': // Maximize button
                  if ((!button[ButtonMax]) && isMaximizable()) {
		      if (maximizeMode() == MaximizeFull) {
			  bitmap = minmax_bits;
			  tip = i18n("Restore");
		      } else {
			  bitmap = max_bits;
			  tip = i18n("Maximize");
		      }
                      button[ButtonMax]  =
                          new GroverButton(this, "maximize", tip,
                                           isTool(), ButtonMax, bitmap);
                      connect(button[ButtonMax], SIGNAL(clicked()),
                              this, SLOT(maxButtonPressed()));
                      layout->addWidget(button[ButtonMax]);
                      if (n < s.length()-1) layout->addSpacing(1);
                  }
                  break;

              case 'X': // Close button
                  if ((!button[ButtonClose]) && isCloseable()) {
                      button[ButtonClose] =
                          new GroverButton(this, "close", i18n("Close"),
                                           isTool(), ButtonClose, close_bits);
                      connect(button[ButtonClose], SIGNAL(clicked()),
                              this, SLOT(closeWindow()));
                      layout->addWidget(button[ButtonClose]);
                      if (n < s.length()-1) layout->addSpacing(1);
                  }
                  break;

              case 'F': // Above button
                  if ((!button[ButtonAbove])) {
                      if (keepAbove()) {
                          bitmap = above_on_bits;
                          tip = i18n("Keep With Others");
                      } else {
                          bitmap = above_off_bits;
                          tip = i18n("Keep Above Others");
                      }
                      button[ButtonAbove] =
                          new GroverButton(this, "above", tip, isTool(),
                                           ButtonAbove, bitmap);
                      connect(button[ButtonAbove], SIGNAL(clicked()),
                              this, SLOT(aboveButtonPressed()));
                      layout->addWidget(button[ButtonAbove]);
                  }
                  break;

              case 'B': // Below button
                  if ((!button[ButtonBelow])) {
                      if (keepBelow()) {
                          bitmap = below_on_bits;
                          tip = i18n("Keep With Others");
                      } else {
                          bitmap = below_off_bits;
                          tip = i18n("Keep Below Others");
                      }
                      button[ButtonBelow] =
                          new GroverButton(this, "below", tip, isTool(),
                                           ButtonBelow, bitmap);
                      connect(button[ButtonBelow], SIGNAL(clicked()),
                              this, SLOT(belowButtonPressed()));
                      layout->addWidget(button[ButtonBelow]);
                  }
                  break;

              case 'L': // Shade button
                  if ((!button[ButtonShade && isShadeable()])) {
                      if ( isSetShade()) {
                          bitmap = shade_on_bits;
                          tip = i18n("Unshade");
                      } else {
                          bitmap = shade_off_bits;
                          tip = i18n("Shade");
                      }
                      button[ButtonShade] =
                          new GroverButton(this, "shade", tip, isTool(),
                                           ButtonShade, bitmap);
                      connect(button[ButtonShade], SIGNAL(clicked()),
                              this, SLOT(shadeButtonPressed()));
                      layout->addWidget(button[ButtonShade]);
                  }
                  break;

              case '_': // Spacer item
                  layout->addSpacing(MARGIN);
            }
	}
        layout->addSpacing(MARGIN);
    }
}

//////////////////////////////////////////////////////////////////////////////
// eventFilter()
// -------------
// Event filter

bool GroverClient::eventFilter(QObject *obj, QEvent *e)
{
    if (obj != widget()) return false;

    switch (e->type()) {
      case QEvent::MouseButtonDblClick: {
          mouseDoubleClickEvent(static_cast<QMouseEvent *>(e));
          return true;
      }
      case QEvent::MouseButtonPress: {
          processMousePressEvent(static_cast<QMouseEvent *>(e));
          return true;
      }
      case QEvent::Paint: {
          paintEvent(static_cast<QPaintEvent *>(e));
          return true;
      }
      case QEvent::Resize: {
          resizeEvent(static_cast<QResizeEvent *>(e));
          return true;
      }
      case QEvent::Show: {
          showEvent(static_cast<QShowEvent *>(e));
          return true;
      }
      default: {
          return false;
      }
    }

    return false;
}

//////////////////////////////////////////////////////////////////////////////
// mouseCoubleClickEvent()
// -----------------------
// Doubleclick on title

void GroverClient::mouseDoubleClickEvent(QMouseEvent *e)
{
    if (titlebar_->geometry().contains(e->pos())) titlebarDblClickOperation();
}

//////////////////////////////////////////////////////////////////////////////
// paintEvent()
// ------------
// Repaint the window

void GroverClient::paintEvent(QPaintEvent*)
{
    if (!GroverFactory::initialized()) return;

    QPainter painter(widget());
    QColorGroup group;

    QRect title(0, 0, width(), titlebar_->geometry().height());
    QRect handle(handle_->geometry());
    QRect left(0, title.height(),
               GroverFactory::borderSize(),
               height() - title.height() - handle.height());
    QRect right(width()-GroverFactory::borderSize(),
                title.height(), GroverFactory::borderSize(),
                height() - title.height() - handle.height());

    int x, y, x2, y2, h, w;

    // draw title bar
    painter.drawTiledPixmap(title,
                            GroverFactory::pix(TitleBar, isActive(), isTool()));

    title.rect(&x, &y, &w, &h);
    x2 = title.right(); y2 = title.bottom();

    group = options()->colorGroup(ColorTitleBar, isActive());
    painter.setPen(group.midlight());
    painter.drawLine(x+1, y+2, x+1, y2-2);
    painter.setPen(group.mid());
    painter.drawLine(x2-1, y+2, x2-1, y2-2);
    painter.setPen(group.shadow());
    painter.drawRect(title);

    // draw title text
    painter.setFont(options()->font(isActive(), isTool()));
    painter.setPen(options()->color(ColorFont, isActive()));

    titlebar_->geometry().rect(&x, &y, &w, &h);
    painter.drawText(x+MARGIN, y, w-MARGIN*2, h,
                     GroverFactory::titleAlign() | AlignVCenter,
                     caption());

    // draw outer frame
    if (!isShade()) {
        group = options()->colorGroup(ColorFrame, isActive());

        // left side
        left.rect(&x, &y, &w, &h);
        x2 = left.right(); y2 = left.bottom();
        painter.drawTiledPixmap(left,
                                GroverFactory::pix(Frame, isActive(), false));
        painter.setPen(group.shadow());
        painter.drawLine(x, y, x, y2);
        painter.drawLine(x2, y, x2, y2);

        // right side
        right.rect(&x, &y, &w, &h);
        x2 = right.right(); y2 = right.bottom();
        painter.drawTiledPixmap(right,
                                GroverFactory::pix(Frame, isActive(), false));
        painter.setPen(group.shadow());
        painter.drawLine(x, y, x, y2);
        painter.drawLine(x2, y, x2, y2);
    }

    // draw handle
    if (!isShade()) {
        // handle middle
        handle.rect(&x, &y, &w, &h);
        x += 23; w -= 46;
        x2 = handle.right()-23; y2 = handle.bottom();

        group = options()->colorGroup(ColorHandle, isActive());
        painter.drawTiledPixmap(x, y, w, h,
                                GroverFactory::pix(Handle, isActive(), false));

        painter.setPen(group.light());
        painter.drawLine(x+1, y+2, x+1, y2-2);
 
        painter.setPen(group.mid());
        painter.drawLine(x2-1, y+2, x2-1, y2-2);

        painter.setPen(group.shadow());
        painter.drawRect(handle);

        // handle left
        x = 0; w = 24; x2 = 23;

        group = options()->colorGroup(ColorFrame, isActive());
        painter.drawTiledPixmap(x, y, w, h,
                                GroverFactory::pix(AltHandle, isActive(), false));

        painter.setPen(group.light());
        painter.drawLine(x+1, y+2, x+1, y2-2);
 
        painter.setPen(group.mid());
        painter.drawLine(x2-1, y+2, x2-1, y2-2);

        painter.setPen(group.shadow());
        painter.drawRect(x, y, w, h);

        // handle right
        x = handle.right() - 23; x2 = handle.right();

        group = options()->colorGroup(ColorFrame, isActive());
        painter.drawTiledPixmap(x, y, w, h,
                                GroverFactory::pix(AltHandle, isActive(), false));

        painter.setPen(group.light());
        painter.drawLine(x+1, y+2, x+1, y2-2);
 
        painter.setPen(group.mid());
        painter.drawLine(x2-1, y+2, x2-1, y2-2);

        painter.setPen(group.shadow());
        painter.drawRect(x, y, w, h);

        if (GroverFactory::borderSize() > 2) {
            handle.rect(&x, &y, &w, &h);
            x2 = handle.right(); y2 = handle.bottom();

            // now blend in with frame
            painter.setClipping(true);
            int fs = GroverFactory::borderSize();
            QPointArray pa;

            pa.putPoints(0, 3,
                         x+1,y,
                         x+fs-2,y,
                         x+1,y+fs-3);
            QRegion mask(pa);
            painter.setClipRegion(mask);
            painter.drawPixmap(x, y,
                               GroverFactory::pix(Frame, isActive(), false));

            pa.putPoints(0, 3,
                         x2-fs+2,y,
                         x2-1,y,
                         x2-1,y+fs-3);
            mask = QRegion(pa);
            painter.setClipRegion(mask);
            painter.drawPixmap(x2-fs+1, y,
                               GroverFactory::pix(Frame, isActive(), false));
        } 
    }
}

//////////////////////////////////////////////////////////////////////////////
// resizeEvent()
// -------------
// Window is being resized

void GroverClient::resizeEvent(QResizeEvent*)
{
    if (widget()->isShown()) {
        QRegion region = widget()->rect();
        region = region.subtract(titlebar_->geometry());
	widget()->erase(region);
    }
}

//////////////////////////////////////////////////////////////////////////////
// showEvent()
// -----------
// Window is being shown

void GroverClient::showEvent(QShowEvent*)
{
    widget()->repaint();
}

//////////////////////////////////////////////////////////////////////////////
// activeChange()
// --------------
// window active state has changed

void GroverClient::activeChange()
{
    if (button[ButtonMenu]) button[ButtonMenu]->setIcon();

    for (int n=0; n<ButtonTypeCount; n++)
        if (button[n]) button[n]->reset();
    widget()->repaint(false);
}

//////////////////////////////////////////////////////////////////////////////
// captionChange()
// ---------------
// The title has changed

void GroverClient::captionChange()
{
    widget()->repaint(titlebar_->geometry(), false);
}

//////////////////////////////////////////////////////////////////////////////
// desktopChange()
// ---------------
// Sticky state has changed

void GroverClient::desktopChange()
{
    bool d = isOnAllDesktops();
    if (button[ButtonSticky]) {
        button[ButtonSticky]->setBitmap(d ? stickydown_bits : sticky_bits);
	QToolTip::remove(button[ButtonSticky]);
	QToolTip::add(button[ButtonSticky], d ? i18n("Un-Sticky") : i18n("Sticky"));
    }
}

//////////////////////////////////////////////////////////////////////////////
// iconChange()
// ------------
// The title has changed

void GroverClient::iconChange()
{
    if (button[ButtonMenu]) {
        button[ButtonMenu]->setIcon();
        button[ButtonMenu]->repaint(false);
    }
}

//////////////////////////////////////////////////////////////////////////////
// maximizeChange()
// ----------------
// Maximized state has changed

void GroverClient::maximizeChange()
{
    bool m = (maximizeMode() == MaximizeFull);
    if (button[ButtonMax]) {
        button[ButtonMax]->setBitmap(m ? minmax_bits : max_bits);
	QToolTip::remove(button[ButtonMax]);
	QToolTip::add(button[ButtonMax], m ? i18n("Restore") : i18n("Maximize"));
    }
}

//////////////////////////////////////////////////////////////////////////////
// shadeChange()
// -------------
// Called when window shading changes

void GroverClient::shadeChange()
{
    bool s = isSetShade();
    if (button[ButtonShade]) {
        button[ButtonShade]->setBitmap(s ? shade_on_bits : shade_off_bits);
        QToolTip::remove(button[ButtonShade]);
        QToolTip::add(button[ButtonShade], s ? i18n("Unshade") : i18n("Shade"));
    }
}

//////////////////////////////////////////////////////////////////////////////
// keepAboveChange()
// ------------
// The above state has changed

void GroverClient::keepAboveChange(bool a)
{
    if (button[ButtonAbove]) {
        button[ButtonAbove]->setBitmap(a ? above_on_bits : above_off_bits);
        QToolTip::remove(button[ButtonAbove]);
        QToolTip::add(button[ButtonAbove],
                      a ? i18n("Keep With Others") : i18n("Keep Above Others"));

    }
}

//////////////////////////////////////////////////////////////////////////////
// keepBelowChange()
// ------------
// The below state has changed

void GroverClient::keepBelowChange(bool b)
{
    if (button[ButtonBelow]) {
        button[ButtonBelow]->setBitmap(b ? below_on_bits : below_off_bits);
        QToolTip::remove(button[ButtonBelow]);
        QToolTip::add(button[ButtonBelow],
                      b ? i18n("Keep With Others") : i18n("Keep Below Others"));
    }
}

//////////////////////////////////////////////////////////////////////////////
// borders()
// ----------
// Get the size of the borders

void GroverClient::borders(int &l, int &r, int &t, int &b) const
{
    l = r = GroverFactory::borderSize();
    t = titleheight_;
    if (isShade()) {
        b = 0;
    } else {
        b = HANDLESIZE;
    }
}

//////////////////////////////////////////////////////////////////////////////
// resize()
// --------
// Called to resize the window

void GroverClient::resize(const QSize &size)
{
    widget()->resize(size);
}

//////////////////////////////////////////////////////////////////////////////
// minimumSize()
// -------------
// Return the minimum allowable size for this decoration

QSize GroverClient::minimumSize() const
{
    return widget()->minimumSize().expandedTo(QSize(16,16));
}

//////////////////////////////////////////////////////////////////////////////
// mousePosition()
// ---------------
// Return mouse position (for resizing)

KDecoration::Position GroverClient::mousePosition(const QPoint &point) const
{
    Position pos = PositionCenter;

    if (point.y() < (height() - HANDLESIZE)) { // inside client
        pos = KDecoration::mousePosition(point);
    } else {
        if (point.x() >= (width() - 24)) pos = PositionBottomRight;
        else if (point.x() <= 24)        pos = PositionBottomLeft;
        else                             pos = PositionBottom;
    }
    return pos;
}

//////////////////////////////////////////////////////////////////////////////
// maxButtonPressed()
// -----------------
// Max button was pressed

void GroverClient::maxButtonPressed()
{
    if (button[ButtonMax]) {
#if KDE_IS_VERSION(3, 3, 0)
        maximize(button[ButtonMax]->lastMousePress());
#else
        switch (button[ButtonMax]->lastMousePress()) {
          case MidButton:
              maximize(maximizeMode() ^ MaximizeVertical);
              break;
          case RightButton:
              maximize(maximizeMode() ^ MaximizeHorizontal);
              break;
          default:
              (maximizeMode() == MaximizeFull) ? maximize(MaximizeRestore)
                  : maximize(MaximizeFull);
        }
#endif
    }
}

//////////////////////////////////////////////////////////////////////////////
// shadeButtonPressed()
// -----------------
// Shade button was pressed

void GroverClient::shadeButtonPressed()
{
    if (button[ButtonShade]) {
        setShade(!isSetShade());
    }
}

//////////////////////////////////////////////////////////////////////////////
// aboveButtonPressed()
// -----------------
// Above button was pressed

void GroverClient::aboveButtonPressed()
{
    if (button[ButtonAbove]) {
        setKeepAbove(!keepAbove());
    }
}

//////////////////////////////////////////////////////////////////////////////
// belowButtonPressed()
// -----------------
// Below button was pressed

void GroverClient::belowButtonPressed()
{
    if (button[ButtonBelow]) {
        setKeepBelow(!keepBelow());
    }
}

//////////////////////////////////////////////////////////////////////////////
// menuButtonPressed()
// -------------------
// Menu button was pressed

void GroverClient::menuButtonPressed()
{
    if (button[ButtonMenu]) {
        QPoint pt(button[ButtonMenu]->rect().bottomLeft().x(),
                  button[ButtonMenu]->rect().bottomLeft().y()+4);
	KDecorationFactory* f = factory();
        showWindowMenu(button[ButtonMenu]->mapToGlobal(pt));
	if (!f->exists(this)) return; // we have been destroyed
	button[ButtonMenu]->setDown(false);
    }
}

//////////////////////////////////////////////////////////////////////////////
// isTool()
// --------
// Is this a tool window?

bool GroverClient::isTool() const
{
    static const unsigned long winmask = NET::DesktopMask | NET::DialogMask |
        NET::DockMask | NET::MenuMask |
        NET::NormalMask | NET::OverrideMask |
        NET::SplashMask | NET::ToolbarMask |
        NET::TopMenuMask | NET::UtilityMask;
      
    NET::WindowType type = windowType(winmask);
    return (type == NET::Menu || type == NET::Toolbar || type == NET::Utility);
}

#include "groverclient.moc"
