//////////////////////////////////////////////////////////////////////////////
// groverclient.h
// -------------------
// Grover window decoration for KDE
// -------------------
// Copyright (c) 2003, 2004 David Johnson <david@usermode.org>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
//////////////////////////////////////////////////////////////////////////////

#ifndef GROVERCLIENT_H
#define GROVERCLIENT_H

#include <qcolor.h>
#include <qbutton.h>
#include <kdecoration.h>
#include <kdecorationfactory.h>
#include <kpixmap.h>

class QSpacerItem;
class QPoint;

namespace Grover {

enum ButtonType {
    ButtonHelp=0,
    ButtonMax,
    ButtonMin,
    ButtonClose, 
    ButtonMenu,
    ButtonSticky,
    ButtonAbove,
    ButtonBelow,
    ButtonShade,
    ButtonTypeCount
};

enum PixmapType {
    Button=0,
    TitleBar,
    Frame,
    Handle,
    AltHandle,
    PixmapTypeCount
};

enum ColorType {
    TitleBtnBlend=0,
    TitleBtnFg,
    TitleBtnText,
    ColorTypeCount
};

class GroverClient;

// GroverFactory //////////////////////////////////////////////////////////////

class GroverFactory : public KDecorationFactory
{
public:
    GroverFactory();
    virtual ~GroverFactory();
    virtual KDecoration *createDecoration(KDecorationBridge *b);
    virtual bool reset(unsigned long changed);
    virtual QValueList<BorderSize> borderSizes() const;

    static bool initialized();
    static QColor &color(Grover::ColorType c, bool active=true);
    static KPixmap &pix(PixmapType p, bool active=true, bool small=false);
    static QString &effect();
    static Qt::AlignmentFlags titleAlign();
    static int borderSize();

private:
    unsigned long readConfig();
    void createPixmaps();

private:
    static bool initialized_;
    static QString effect_;
    static Qt::AlignmentFlags titlealign_;
    static QColor colors_[ColorTypeCount][2];
    static KPixmap pix_[PixmapTypeCount][2][2];
    static int bordersize_;
};

inline bool GroverFactory::initialized()
    { return initialized_; }

inline QColor &GroverFactory::color(Grover::ColorType c, bool active)
    { return colors_[c][active]; }

inline KPixmap &GroverFactory::pix(PixmapType p, bool active, bool small)
    { return pix_[p][active][small]; }

inline QString &GroverFactory::effect()
    { return effect_; }

inline Qt::AlignmentFlags GroverFactory::titleAlign()
    { return titlealign_; }

inline int GroverFactory::borderSize()
    { return bordersize_; }

// GroverButton ///////////////////////////////////////////////////////////////

class GroverButton : public QButton
{
public:
    GroverButton(GroverClient *parent=0, const char *name=0,
                 const QString &tip=NULL,
                 bool small=false, ButtonType type=ButtonHelp,
                 const unsigned char *bitmap=0);
    ~GroverButton();

    void setBitmap(const unsigned char *bitmap);
    void setIcon();
    QSize sizeHint() const;
    ButtonState lastMousePress() const;
    void reset();

private:
    void enterEvent(QEvent *e);
    void leaveEvent(QEvent *e);
    void mousePressEvent(QMouseEvent *e);
    void mouseReleaseEvent(QMouseEvent *e);
    void drawButton(QPainter *painter);

private:
    GroverClient *client_;
    ButtonType type_;
    QBitmap *deco_;
    KPixmap icon_;
    bool small_;
    ButtonState lastmouse_;
    bool mouseover_;
};

inline QButton::ButtonState GroverButton::lastMousePress() const
    { return lastmouse_; }

inline void GroverButton::reset()
    { repaint(false); }

// GroverClient ///////////////////////////////////////////////////////////////

class GroverClient : public KDecoration
{
    Q_OBJECT
public:
    GroverClient(KDecorationBridge *b, KDecorationFactory *f);
    virtual ~GroverClient();

    virtual void init();

    virtual void activeChange();
    virtual void desktopChange();
    virtual void captionChange();
    virtual void iconChange();
    virtual void maximizeChange();
    virtual void shadeChange();

    virtual void borders(int &l, int &r, int &t, int &b) const;
    virtual void resize(const QSize &size);
    virtual QSize minimumSize() const;
    virtual Position mousePosition(const QPoint &point) const;

private:
    void addButtons(QBoxLayout* layout, const QString& buttons);

    bool eventFilter(QObject *obj, QEvent *e);
    void mouseDoubleClickEvent(QMouseEvent *e);
    void paintEvent(QPaintEvent *);
    void resizeEvent(QResizeEvent *);
    void showEvent(QShowEvent *);

    bool isTool() const;

private slots:
    void maxButtonPressed();
    void menuButtonPressed();
    void aboveButtonPressed();
    void belowButtonPressed();
    void shadeButtonPressed();
    void keepAboveChange(bool);
    void keepBelowChange(bool);

private:
    GroverButton *button[ButtonTypeCount];
    QSpacerItem *titlebar_, *handle_;
    int  titleheight_;
};

} // namespace Grover

#endif // GROVERCLIENT_H
