/**************************************************************************

    kpager.cpp  - KPager2's main window
    Copyright (C) 2000  Antonio Larrosa Jimenez
			Matthias Ettrich
			Matthias Elter
    Copyright (c) 2003 Sebastian Wolff

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Send comments and bug fixes to larrosa@kde.org

***************************************************************************/

#include "config.h"
#include "kpager.h"
#include "version.h"

#include <qintdict.h>
#include <qptrlist.h>
#include <qlayout.h>
#include <qobjectlist.h>
#include <qmessagebox.h>
#include <qpainter.h>
#include <qsizepolicy.h>
#include <qtooltip.h>
#include <qtimer.h>

#include <kaboutapplication.h>
//#include <kaboutdata.h>
//#include <kaboutdialog.h>
//#include <kaction.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kglobalaccel.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
//#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kprocess.h>
//#include <kshortcut.h>
//#include <kstdaction.h>
#include <kwin.h>
#include <kwinmodule.h>
//#include <netwm.h>

#include "kpageraboutdlg.h"

#include "kpagerapplet.h"

#include "kpageranimation.h"
#include "kpagerdesktop.h"
#include "kpagerrmbmenu.h"
#include "kpagersingleswitchbutton.h"
#include "kpagertaskmanager.h"
//#include "taskbar.h"



extern KAboutData * m_aboutdata; // see main.cpp
extern KWinModule * kwin_module;
KPager2 * kpager2 = 0;
extern KConfig * kpagerglobalconfig;

KPager2::KPager2(KPagerMainWindow *parent, const char *name)
	: QFrame (parent, name)//, WStyle_Customize | WStyle_Tool)
	, m_menu (0)
    , m_layout(0)
{
	// initials
	m_singleDesktop = 0;
	m_emptyWidgetList.setAutoDelete(true);
	kpager2 = this;
	KPagerConfigDialog::initConfiguration();
	if ( kwin_module == NULL ) kwin_module = new KWinModule();

	// general widgets and layout
	setFrameStyle(NoFrame);
	setBackgroundMode(X11ParentRelative);
	m_parent=parent;
	m_layout=0;
	m_mainwidget2=new QFrame(this);
	m_mainwidget = new KPager2MainWidget(m_mainwidget2);
	m_mainwidget ->setBackgroundMode(X11ParentRelative);
	m_mainwidget2->setBackgroundMode(X11ParentRelative);

	// general layout
	QBoxLayout *lay1 = new QHBoxLayout(this,0,0);
	lay1->addWidget(m_mainwidget2,10);
	QBoxLayout *lay2 = new QVBoxLayout(m_mainwidget2,frameWidth(),frameWidth());
	lay2->addWidget(m_mainwidget,10);
	
	// initialize mini desktop widgets
	// 1st: for multiple desktop view mode 
	m_currentDesktop=kwin_module->currentDesktop();
	int numberOfDesktops=kwin_module->numberOfDesktops();
	for (int i=0;i<numberOfDesktops;i++)
	{
		KPagerDesktop *dsk=new KPagerDesktop(i+1,this,m_mainwidget);
		m_desktops.append(dsk);
		if (KPagerConfigDialog::m_showBackground) slotBackgroundChanged(i+1);
	}
	// 2nd: for single view mode:
	//m_singleDesktop = new KPagerDesktop(m_currentDesktop, this, m_mainwidget);
	m_singleDesktop=0;
	QPixmap pix;
	QImage img;
	pix = KGlobal::instance()->iconLoader()->loadIcon( "1leftarrow", KIcon::NoGroup, KIcon::SizeSmall );
	img = QImage(pix.convertToImage().smoothScale(8,8));
	pix.convertFromImage(img);
	m_prevDeskButton = new KPagerSingleSwitchButton(pix,"",this);
	pix = KGlobal::instance()->iconLoader()->loadIcon( "1rightarrow", KIcon::NoGroup, KIcon::SizeSmall );
	img = QImage(pix.convertToImage().smoothScale(8,8));
	pix.convertFromImage(img);
	m_nextDeskButton = new KPagerSingleSwitchButton(pix,"",this);
	if (m_singleDesktop) m_singleDesktop ->setHidden(true);
	m_nextDeskButton->setHidden(true);
	m_prevDeskButton->setHidden(true);
	lay1->insertWidget(0,m_prevDeskButton,0);
	lay1->insertWidget(2,m_nextDeskButton,0);
	m_prevDeskButton->setMaximumWidth(10);
	m_nextDeskButton->setMaximumWidth(10);
	QSizePolicy butPolicy = QSizePolicy(QSizePolicy::Fixed,QSizePolicy::MinimumExpanding);
	butPolicy.setVerStretch(10);
	m_prevDeskButton->setSizePolicy(butPolicy);
	m_nextDeskButton->setSizePolicy(butPolicy);
	connect( m_prevDeskButton, SIGNAL(clicked()),    this, SLOT(slotActivateLeftDesktop()));
	connect( m_prevDeskButton, SIGNAL(dragSwitch()), this, SLOT(slotActivateLeftDesktop()));
	connect( m_nextDeskButton, SIGNAL(clicked()),    this, SLOT(slotActivateRightDesktop()));
	connect( m_nextDeskButton, SIGNAL(dragSwitch()), this, SLOT(slotActivateRightDesktop()));

	
	// KWin signals
	connect( kwin_module, SIGNAL( desktopNamesChanged() ),
			SLOT( slotDesktopNamesChanged() ) );
	connect( kwin_module, SIGNAL( numberOfDesktopsChanged(int) ),
			SLOT( slotNumberOfDesktopsChanged(int) ) );
	connect(kapp, SIGNAL(backgroundChanged(int)),
			SLOT(slotBackgroundChanged(int)));

	m_taskmanager = new KPagerTaskManager(this); // must be inited after m_desktops
//	m_taskbar = new TaskBar(this, m_taskmanager->taskManager());


	// read those configuration options that are not handled by KPagerConfigDialog
	KConfig *cfg= kpagerglobalconfig;
	cfg->setGroup("Keys");
	m_globalAccel = new KGlobalAccel(this);
	m_globalAccel->insert("LeftDesktop", i18n("Previous desktop"), i18n("Switches to the desktop on the left"), 0, 0, this, SLOT(slotActivateLeftDesktop()));
	m_globalAccel->insert("RightDesktop", i18n("Next desktop"), i18n("Switches to the desktop on the right"), 0, 0, this, SLOT(slotActivateRightDesktop()));
	m_globalAccel->readSettings(cfg);
	m_globalAccel->updateConnections();
	
	// default fonts
	QFont defFont( KGlobalSettings::toolBarFont() );
	defFont = cfg->readFontEntry("Font", &defFont);
	setFont(defFont);

	updateLayout();

	for (int i=0;i<numberOfDesktops;i++)
	{
		m_desktops[i]->redraw();
	}
	redrawSingleDesktop(m_currentDesktop);


	m_anim = 0;
}

KPager2::~KPager2()
{
	delete m_taskmanager;
	//delete m_taskbar;
}

KPagerDesktopList KPager2::desktopList() const
{
	return m_desktops;
}

#include <dcopclient.h>

void KPager2::resizeEvent(QResizeEvent*e)
{
	int cols, rows;
	computeRowsCols(rows,cols);
	if (rows!=m_layout->numRows() || cols!=m_layout->numCols()) updateLayout();
	QFrame::resizeEvent(e);
}

// sends the new desktop layout to KWin (to configure sensitive desktop borders etc.)
void updateDesktopLayout(int o, int x, int y)
{
	QCString replyType;
	QByteArray data, replyData;
	QDataStream arg(data, IO_WriteOnly);
	arg << o << x << y;
	if ( !(kapp->dcopClient()->call( "kwin", "KWinInterface",
				    "setDesktopLayout(int, int, int)",
				    data, replyType, replyData)))
	{
		kdDebug() << "KPager2: Call to KWinInterface::setDesktopLayout(int, int, int) failed" << endl;
		return;
	}
}

// compute the rows/cols from the current configuration (KPagerConfigDialog), correct it following a minimum size
void KPager2::computeRowsCols(int & rows, int & cols)
{
	if (KPagerConfigDialog::m_showOnlyOneDesktop)
	{
		rows = 1;
		cols = 1;
		return;
	}

	bool horiz = orientation() == Horizontal;
	int size;
	if (horiz) size = height();
	else size = width();

	int COMPUTEROWSCOLS_MINSIZE  = 8;
	if ( !horiz && KPagerConfigDialog::m_showName ) COMPUTEROWSCOLS_MINSIZE  = 64; // if vertical and desktop names are displayed

	rows = KPagerConfigDialog::m_numberRows;
	bool small;
	do
	{
		if (rows!=0)
			cols = (int) m_desktops.count()/rows;
		else
		{
			rows = m_desktops.count();
			cols = 1;
		}
		while (m_desktops.count()>(uint)cols*rows) cols++; // correction from rounding

		small = (m_mainwidget->isVisible()) && ( size/rows < COMPUTEROWSCOLS_MINSIZE );
		//if ( !horiz && KPagerConfigDialog::m_showName ) small = TRUE; // if vertical and desktop names are displayed
		if (small && (rows>1) ) rows--;
	}
	while (small);

	if (!horiz)
	{
		int k = rows;
		rows = cols;
		cols = k;
	}
}


void KPager2::updateLayout()
{
	int oldcols = 0, oldrows = 0;
	if (m_layout)
	{
		oldcols = m_layout->numCols();
		oldrows = m_layout->numRows();
	}

	m_mainwidget->setUpdatesEnabled( FALSE );
	m_mainwidget2->setUpdatesEnabled( FALSE );
	setUpdatesEnabled( FALSE );

	this->setPalette(QApplication::palette());
	m_mainwidget->setPalette(QApplication::palette());
	m_mainwidget2->setPalette(QApplication::palette());

	if (parent() && !parent()->parent()) {
		this->setBackgroundMode(X11ParentRelative); // yes, I know it is not the best solution (no background updates if  moved on kicker, for example) but else we would have a solid background (but why?) 
	}
	this->setBackgroundOrigin( AncestorOrigin );

	// get FrameStyle
	//setLineWidth(1);
	switch (KPagerConfigDialog::m_generalFrame)
	{
		case 1:
			m_mainwidget2->setFrameStyle(QFrame::Raised | QFrame::StyledPanel);
			layout()->setMargin(1); // give it some space on kicker
			break;
		case 2:
			m_mainwidget2->setFrameStyle(QFrame::Sunken | QFrame::StyledPanel);
			layout()->setMargin(1); // give it some space on kicker
			break;
		case 3:
			m_mainwidget2->setFrameStyle(QFrame::Plain | QFrame::StyledPanel);
			layout()->setMargin(1); // give it some space on kicker
			break;
		case 0:
		default:
			m_mainwidget2->setFrameStyle(QFrame::NoFrame);
			layout()->setMargin(0);
	}

	QColor col;
	switch (KPagerConfigDialog::m_generalBackgroundColor)
	{
		case 1:
			col = (colorGroup().base());//m_mainwidget->colorGroup().base());
			break;
		case 2:
			col = (colorGroup().background());//m_mainwidget->colorGroup().background());
			break;
		case 3:
			col = (colorGroup().button());//m_mainwidget->colorGroup().button());
			break;
		case 4:
			col = (colorGroup().dark());//m_mainwidget->colorGroup().dark());
			break;
		case 0:
		default:
			col = (Qt::black);
	}
	m_mainwidget->setPaletteBackgroundColor(col);


	// get the layout: calculate how many rows/cols we have
	int rows, cols;
	computeRowsCols(rows,cols);

	// configure layout for kwin (for sensitive desktop borders):
	updateDesktopLayout(Qt::Horizontal, cols, rows);

	if (m_layout!=0) delete m_layout;
	m_emptyWidgetList.clear(); // delete all space holder widgets

	m_mainwidget2->layout()->setMargin(m_mainwidget2->frameWidth());
	if (!KPagerConfigDialog::m_showOnlyOneDesktop)
	{
		m_layout = new MyGridLayout(m_mainwidget, rows, cols, KPagerConfigDialog::m_desktopMargin,
			KPagerConfigDialog::m_desktopDistance);
		QValueList <KPagerDesktop *>::Iterator it;
		int i,j;
		i=j=0;
		int ndesks=0;
		for( it = m_desktops.begin(); it != m_desktops.end(); ++it )
		{
			i = (ndesks) / cols ;
			j = (ndesks) % cols ;
			m_layout->addWidget(*it,i,j);
			ndesks++;
		}
		while (ndesks < cols*rows)
		{
			i = (ndesks) / cols;
			j = (ndesks) % cols;
			QWidget * w = new QWidget(m_mainwidget);
			w->setPaletteBackgroundColor(col);
			m_emptyWidgetList.append(w);
			m_layout->addWidget(w,i,j);
			ndesks++;
		}
	}
	else
	{
		m_layout = new MyGridLayout(m_mainwidget, 1, 1, KPagerConfigDialog::m_desktopMargin,
			KPagerConfigDialog::m_desktopDistance);
		m_layout->addWidget(m_desktops[m_currentDesktop-1],0,0);
	}

    //updateGeometry();
	updateDesktopLayoutIfOnlyOneVisible();
    m_layout->activate(); // else the layout margin/spacign wouldn't be applied!
	m_mainwidget->setUpdatesEnabled( TRUE );
	m_mainwidget2->setUpdatesEnabled( TRUE );
	setUpdatesEnabled(true);
	emit updateLayoutCalled();

	// if the layout has changed we have to redraw our desktops again (refill the pixmaps, since the way how desktops are drawn depends on the layout we use)
	if (oldcols!=m_layout->numCols() || oldrows!=m_layout->numRows())
	{
		redrawDesktops();
	}

	update();

	//resize(w*rows+(rows-1)*KPagerConfigDialog::m_desktopDistance,h*cols+(cols-1)*KPagerConfigDialog::m_desktopDistance);
}



void KPager2::showPopupMenu( WId wid, QPoint pos, int sender_desk, bool onlyTasks)
{
	if (m_menu != 0)
	{
		delete m_menu;
		m_menu = 0;
	}
	Task * t = pagerTaskManager()->findTaskPager(wid);
	if (!onlyTasks)
	{
		QPoint p( mapToGlobal( QPoint( 0, 0 ) ) );
		m_menu = new KPagerRMBMenu (t, sender_desk, this, "kpagerrmbmenu");
		// calc popup menu position
		// right
		if (orientation()==Qt::Vertical && mapToGlobal(rect().center()).x()<=QApplication::desktop()->width()/2)
		{
			// please note: I changed everything from "pos" to "p"
			pos.setX( p.x() + width() );
		}
		else // left
		if (orientation()==Qt::Vertical && mapToGlobal(rect().center()).x()>QApplication::desktop()->width()/2)
		{
			pos.setX( p.x() - m_menu->sizeHint().width() );
		}
		else // Down
		if (orientation()==Qt::Horizontal && mapToGlobal(rect().center()).y()<=QApplication::desktop()->width()/2)
		{
			pos.setY( p.y() + height() );
		}
		else // Up
		if (orientation()==Qt::Horizontal && mapToGlobal(rect().center()).y()>QApplication::desktop()->width()/2)
		{
			pos.setY( p.y() - m_menu->sizeHint().height() );
		}
		//m_menu->popup(pos);
		{
			if (m_anim) m_anim->hide();
			((KPagerRMBMenu*) m_menu)->setPopupPos(pos);
			m_menu->exec(pos);
			delete m_menu;
			m_menu = 0;
		}
	}
	else
	{
		// do nothing
		qDebug("KPager2::showPopupMenu: Shouldn't be reachable!");
	}
}

void KPager2::configureDialog()
{
	KPagerConfigDialog *dialog= new KPagerConfigDialog(this);
	if (dialog->exec())
	{
		slotApplyConfigureFromDialog();
    }
}
void KPager2::configureDesktopsDialog()
{
	kapp->startServiceByDesktopName("desktop");
}

void KPager2::helpDialog()
{
	kapp->invokeHelp("","kpager2");
}

void KPager2::aboutDialog()
{
	if (!m_aboutdata) return;
	KPagerAboutDialog * dlg = new KPagerAboutDialog(this);
	dlg->exec();
}
void KPager2::slotApplyConfigureFromDialog()
{
	KConfig *cfg= kpagerglobalconfig;
//	m_layoutType=static_cast<enum KPager::LayoutTypes>(KPagerConfigDialog::m_layoutType);
	//c_defNumberRows=KPagerConfigDialog::m_numberRows;
	//c_defDesktopDistance=KPagerConfigDialog::m_desktopDistance;
	//int nWd = (parent() ? ((QWidget *)parent())->width() : width());
	//int nHg = (parent() ? ((QWidget *)parent())->height() : height());


//fprintf(stderr,"%d %d %d\n",cfg->getConfigState==KConfig::NoAccess,cfg->getConfigState==KConfig::ReadOnly,cfg->getConfigState==KConfig::ReadWrite);

	cfg->setGroup("KPager2");
	
	cfg->writeEntry("windowDrawMode",KPagerConfigDialog::m_windowDrawMode);
	cfg->writeEntry("showNumber",KPagerConfigDialog::m_showNumber);
	cfg->writeEntry("showName",KPagerConfigDialog::m_showName);
	cfg->writeEntry("showWindows",KPagerConfigDialog::m_showWindows);
	cfg->writeEntry("showBackground",KPagerConfigDialog::m_showBackground);
	cfg->writeEntry("windowDragging", KPagerConfigDialog::m_windowDragging );
	cfg->writeEntry("showDragMoving", KPagerConfigDialog::m_showDragMoving );
//	cfg->writeEntry("repaintIntervall", KPagerConfigDialog::m_repaintIntervall );
	cfg->writeEntry("animation", KPagerConfigDialog::m_animation);
	cfg->writeEntry("showGradient", KPagerConfigDialog::m_showGradient );
//	cfg->writeEntry("showWindowFrame", KPagerConfigDialog::m_showWindowFrame);
	cfg->writeEntry("desktopBackgroundColor", KPagerConfigDialog::m_desktopBackgroundColor);
	cfg->writeEntry("numberRows", KPagerConfigDialog::m_numberRows);
	cfg->writeEntry("showOnlyOneDesktop", KPagerConfigDialog::m_showOnlyOneDesktop);
	cfg->writeEntry("desktopDistance", KPagerConfigDialog::m_desktopDistance);
	cfg->writeEntry("desktopMargin", KPagerConfigDialog::m_desktopMargin);
	cfg->writeEntry("generalBackgroundColor", KPagerConfigDialog::m_generalBackgroundColor);
	cfg->writeEntry("generalFrame", KPagerConfigDialog::m_generalFrame);
	cfg->writeEntry("shadowPixmap", KPagerConfigDialog::m_shadowPixmap);
	cfg->writeEntry("bevelledSelection", KPagerConfigDialog::m_bevelledSelection);
	cfg->writeEntry("generalTheme", KPagerConfigDialog::m_generalTheme);
	cfg->writeEntry("windowTransparentMode", KPagerConfigDialog::m_windowTransparentMode);
	cfg->writeEntry("showTaskMenu", KPagerConfigDialog::m_showTaskMenu);
	cfg->writeEntry("toolTips", KPagerConfigDialog::m_toolTips);
	cfg->writeEntry("showShadowPixmap", KPagerConfigDialog::m_showShadowPixmap);
	cfg->writeEntry("showLauncher", KPagerConfigDialog::m_showLauncher);
	cfg->writeEntry("TaskMenuThumbnailHeight", KPagerConfigDialog::m_menuThumbnailHeight);
	cfg->writeEntry("TaskMenuThumbnailWidth", KPagerConfigDialog::m_menuThumbnailWidth);

	cfg->setGroup("3d");
	cfg->writeEntry("Use3d",  KPagerConfigDialog::m_use3d);			
	cfg->writeEntry("3dMode", KPagerConfigDialog::m_3dMode);
	cfg->writeEntry("3dZoom", KPagerConfigDialog::m_3dZoom);

	cfg->setGroup("Keys");
	m_globalAccel->updateConnections();
	m_globalAccel->writeSettings(cfg);

	
	updateLayout();
	redrawDesktops();
}


void KPager2::slotDesktopNamesChanged()
{
	for ( int i=1; i <= (int) m_desktops.count(); ++i)
	{
		m_desktops[i-1]->updateToolTip();
		if (m_singleDesktop) m_singleDesktop->updateToolTip();
	}
	update();
	emit updateLayout();
}

void KPager2::slotNumberOfDesktopsChanged(int ndesktops)
{
    unsigned int nDesktops=static_cast<unsigned int>(ndesktops);
    // number of desktops decreased
    if (nDesktops<m_desktops.count())
    {
        QValueList <KPagerDesktop *>::Iterator it;
        for ( int i=m_desktops.count()-nDesktops; i > 0; i--)
        {
            it = m_desktops.fromLast();
            delete (*it);
            m_desktops.remove(it);
        }

        emit updateLayout();
    }
    // desktop(s) added
    else if (nDesktops>m_desktops.count())
    {
        for (unsigned int d=m_desktops.count()+1;d<=nDesktops; d++)
        {
            KPagerDesktop *dsk=new KPagerDesktop(d,this,m_mainwidget);
            m_desktops.append(dsk);
            dsk->show();
			slotBackgroundChanged(m_desktops.count());
        }

        emit updateLayout();
    }
}

void KPager2::updateSingleDesktop(int desk)
{
	if (!m_singleDesktop) return;
	m_singleDesktop->setDesktop(desk);
	m_singleDesktop->setAttention(m_desktops[desk-1]->isAttention());
	if (!m_singleDesktop->isHidden()) m_singleDesktop->update();
}
void KPager2::redrawSingleDesktop(int desk)
{
	if (!m_singleDesktop) return;
	m_singleDesktop->setDesktop(desk);
	m_singleDesktop->setAttention(m_desktops[desk-1]->isAttention());
	if (!m_singleDesktop->isHidden()) m_singleDesktop->redraw();
}

void KPager2::updateDesktopLayoutIfOnlyOneVisible()
{
	if (m_anim) 
	{
		delete m_anim;
		m_anim = 0;
	}
	
	// save updatesenabled state
	bool updatesenabled = m_mainwidget->isUpdatesEnabled();
	bool updatesenabled2 = m_mainwidget2->isUpdatesEnabled();
	bool updatesenabledt = isUpdatesEnabled();
	
	m_mainwidget ->setUpdatesEnabled( FALSE );
	m_mainwidget2->setUpdatesEnabled( FALSE );
	setUpdatesEnabled( FALSE );
	if (KPagerConfigDialog::m_showOnlyOneDesktop)
	{
		int i;
		for (i=0;i<(int)m_desktops.count();i++)
		{
			if (i==m_currentDesktop-1) m_desktops[i]->setHidden(false);
		}
		if (m_layout!=0) delete m_layout;
		m_layout = new MyGridLayout(m_mainwidget, 1, 1, KPagerConfigDialog::m_desktopMargin,
			KPagerConfigDialog::m_desktopDistance);
		this->layout()->setMargin(KPagerConfigDialog::m_generalFrame ? 1 : 0);
		if (m_singleDesktop)
		{
			m_layout->addWidget(m_singleDesktop,0,0);
			m_singleDesktop->setHidden(false);
		}
		else
		{
			m_layout->addWidget(m_desktops[m_currentDesktop-1],0,0);
			for (i=0;i<(int)m_desktops.count();i++)
			{
				if (i != m_currentDesktop-1)
					m_desktops[i]->setHidden(true);
			}
		}
		m_prevDeskButton->setHidden(false);
		m_nextDeskButton->setHidden(false);
	}
	else // reset the usual settings: all desks are visible
	{
		uint i;
		if (m_singleDesktop) 
		{
			m_singleDesktop ->setHidden(true);
		}
		for (i=0;i<m_desktops.count();i++)
		{
			m_desktops[i]->setHidden(false);
		}
		m_prevDeskButton->setHidden(true);
		m_nextDeskButton->setHidden(true);
	}
	// restore updatesenabled state 
    m_layout->activate(); // else the layout margin/spacign wouldn't be applied!
	setUpdatesEnabled( updatesenabledt );
	m_mainwidget2->setUpdatesEnabled( updatesenabled2 );
	m_mainwidget ->setUpdatesEnabled( updatesenabled );
}

void KPager2::slotBackgroundChanged(int desk)
{
/*
this is the code how it should be
	if (desk>0)
	{
//printf("changed emitted %d\n",desk);
		m_desktops[desk-1]->loadBgPixmap();
	}
	else if (desk==0) for (uint i=0;i<m_desktops.count();i++) m_desktops[i]->loadBgPixmap();
	return;
*/

/* this is the new code that fixes bugs in KApplication::backgroundChanged(int): */
//	qDebug("KPager: The workaround for a bug in SIGNAL(void KApplication::backgroundChanged(int)) is still active!");
	DCOPClient *client = kapp->dcopClient();
	if (!client->isAttached()) client->attach();
	QByteArray data, data2, replyData;
	QCString replyType;
	bool m_isCommon;
	if (client->call("kdesktop", "KBackgroundIface", "isCommon()", data, replyType, replyData))
	{
		QDataStream reply(replyData, IO_ReadOnly);
		if (replyType == "bool")
		{
			reply >> m_isCommon;
		}
	}
	qDebug(
		QString(
		QString("KPager: received SIGNAL(KApplication::backgroundChanged(int==%1)) with KBackgroundIface->isCommon()==%2").arg(desk).arg(m_isCommon)
		).latin1()
		);
	// the following is a hack: if there is a common background for all desktops desktop 1 will paint the background of the others, too.
	if  ( m_isCommon ) m_desktops[0]->loadBgPixmap();
	else for (uint i=0;i<m_desktops.count();i++) m_desktops[i]->loadBgPixmap();
}



void KPager2::redrawDesktops()
{
    QValueList <KPagerDesktop *>::Iterator it;
    for( it = m_desktops.begin(); it != m_desktops.end(); ++it )
    {
        (*it)->redraw(); //update();
//        (*it)->repaint();
    }
}


void KPager2::wheelEvent(QWheelEvent* e)
{
	// ok, this is copied from the old 'minipager' applet of kicker
	e->accept();
    if( e->delta() < 0 ) {
        // this could be "( curDesk - 1 ) + 1" for clarity, but that seems superfluous
        const int next = m_currentDesktop % KWin::numberOfDesktops() + 1;
        setCurrentDesktop( next );
    }
    else {
        // again, a little ugly -- the addition of the number of desktops makes sure that we are
        // not taking the modulo of a negative number and the "curDesk - 2" could be "curDesk - 1 - 1"
        const int previous = ( KWin::numberOfDesktops() + m_currentDesktop - 2 ) % KWin::numberOfDesktops() + 1;
        setCurrentDesktop( previous );
    }

}

void KPager2::slotActivateRightDesktop()
{
	const int next = m_currentDesktop % KWin::numberOfDesktops() + 1;
	setCurrentDesktop( next );
}
void KPager2::slotActivateLeftDesktop()
{
	const int previous = ( KWin::numberOfDesktops() + m_currentDesktop - 2 ) % KWin::numberOfDesktops() + 1;
	setCurrentDesktop( previous );
}
	
void KPager2::setCurrentDesktop(int desk)
{
	if(KPagerConfigDialog::m_use3d)
	{
		if (desk != KWin::currentDesktop())
		{
			QString mode="";
			switch (KPagerConfigDialog::m_3dMode)
			{
				case 0:
					mode = "carousel";
					break;
				case 1:
					mode = "cylinder";
					break;
				case 2:
					mode = "linear";
					break;
				case 3:
					mode = "viewmaster";
					break;
				case 4:
					mode = "priceisright";
					break;
				case 5:
					mode = "flip";
					break;
				case 6:
					mode = "random";
					break;
				default:
					mode="random";
			}
			QStringList args;
			args << QString("--mode=%1").arg(mode) << QString("--gotocolumn=%1").arg(desk);
			if(!KPagerConfigDialog::m_3dZoom) args<<"--nozoom";
		
			KProcess process;
			process << "3ddesk";
			process << args;
			process.start(KProcess::DontCare);
		}
	}
	else
		KWin::setCurrentDesktop(desk);
	//setCurrentDesktopCounter(desk);
}

void KPager2::setCurrentDesktopCounter(int desk)
{
	m_currentDesktop  = desk;
}








KPager2MainWidget::KPager2MainWidget ( QWidget * parent , const char * name, WFlags f ) : QWidget(parent, name, f)
{
	//setBackgroundMode(NoBackground);
}

void KPager2MainWidget::paintEvent(QPaintEvent *e)
{
	QWidget::paintEvent(e);
	return;
	// let us draw the frames around the desktops here (means we fill the spacing and margin of the layout with a specific color

	// get the background color:
	QColor col;
	switch (KPagerConfigDialog::m_generalBackgroundColor)
	{
		case 1:
			col = (colorGroup().base());//m_mainwidget->colorGroup().base());
			break;
		case 2:
			col = (colorGroup().background());//m_mainwidget->colorGroup().background());
			break;
		case 3:
			col = (colorGroup().button());//m_mainwidget->colorGroup().button());
			break;
		case 4:
			col = (colorGroup().dark());//m_mainwidget->colorGroup().dark());
			break;
		case 0:
		default:
			col = (Qt::black);
	}

/*
	int space = KPagerConfigDialog::m_desktopDistance;
	int frame = frameWidth();
	int margin = KPagerConfigDialog::m_desktopMargin;
	int rows = m_layout->numRows();
	int cols = m_layout->numCols();
*/
	QPainter p;
	p.begin(this);

//col = QColor(Qt::red);
	p.setBrush(col);
	p.fillRect(rect(),col);

	p.end();

	/* not necessary, is done somewhere else

	// paint the child widgets OVER the background!
	QObject * it;
	QObjectList * list = children();
	if (list)
	{ // if there are any children
		for( it = list->first(); it != 0; it = list->next() )
		{
			if (it->inherits ("QWidget")) ((QWidget*)it)->update();
		}
	}
	*/
}

QSize KPager2::sizeHint() const
{
//    int n=m_desktops.count();
    int w=-1,h=-1;

    QSize size=m_desktops[0]->sizeHint();
    int wDsk=size.width();
    int hDsk=size.height();

    // calculate how many rows/cols we have (don't use computeRowsCols() since we don't want to apply the minimum size rule)
    uint rows = KPagerConfigDialog::m_numberRows;
    uint cols;
    if (rows!=0)
		cols = (int) m_desktops.count()/KPagerConfigDialog::m_numberRows;
    else
    {
		rows = m_desktops.count();
		cols = 1;
    }
    if (m_desktops.count()>cols*rows) cols++; // correction from rounding
    if (m_orientation==Qt::Vertical)
    {
		int k = rows;
		rows = cols;
		cols = k;
    }

	if (KPagerConfigDialog::m_showName)
	{
		int max = wDsk;
		for (uint i=0; i<m_desktops.count(); i++)
		{
			QFont f( KGlobalSettings::toolBarFont() );
			QString txt;
			if ( KPagerConfigDialog::m_showName && KPagerConfigDialog::m_showNumber )
			    txt=QString("%1. %2").arg(i+1).arg(kwin()->desktopName( i+1 ));
			else if ( KPagerConfigDialog::m_showName )
			    txt=kwin()->desktopName( i+1 );
			QFontMetrics m = QFontMetrics(f);
			max = QMAX(max,m.width(txt)+4);
		}
		wDsk = max;
	}

    h=hDsk*rows + (rows-1)*KPagerConfigDialog::m_desktopDistance + KPagerConfigDialog::m_desktopMargin + frameWidth();
    w=wDsk*cols + (cols-1)*KPagerConfigDialog::m_desktopDistance + KPagerConfigDialog::m_desktopMargin + frameWidth();
	
	if (KPagerConfigDialog::m_showOnlyOneDesktop) w += 20; // 2x10 Pixel per button 
	
    return QSize(w,h);
}

int KPager2::heightForWidth(int width)
{
    int h=-1;

	if (KPagerConfigDialog::m_showOnlyOneDesktop) width -= 20; // 2x10 Pixel per button 
	
	int rows = m_layout->numRows();
	int cols = m_layout->numCols();

	// now get the size of a desktop
	int wDsk = (width - (cols-1)*KPagerConfigDialog::m_desktopDistance + KPagerConfigDialog::m_desktopMargin + frameWidth()) / cols;
	int hDsk = KApplication::desktop()->height()*wDsk/KApplication::desktop()->width();

	// now translate it back:


	h=hDsk*rows + (rows-1)*KPagerConfigDialog::m_desktopDistance + KPagerConfigDialog::m_desktopMargin + frameWidth();
    return h;
}

#include <qfontmetrics.h>
int KPager2::widthForHeight(int height)
{
    int w=-1;

    // calculate how many rows/cols we have (don't use computeRowsCols() since we don't want to apply the minimum size rule)
	int rows = m_layout->numRows();
	int cols = m_layout->numCols();

	// now get the size of a desktop
	int hDsk = ( height - (rows-1)*KPagerConfigDialog::m_desktopDistance + KPagerConfigDialog::m_desktopMargin + frameWidth()) / rows;
	int wDsk = KApplication::desktop()->width()*hDsk/KApplication::desktop()->height();

	if (KPagerConfigDialog::m_showName)
	{
		int max = wDsk;
		for (uint i=0; i<m_desktops.count(); i++)
		{
			QFont f( KGlobalSettings::toolBarFont() );
			QString txt;
			if ( KPagerConfigDialog::m_showName && KPagerConfigDialog::m_showNumber )
			    txt=QString("%1. %2").arg(i+1).arg(kwin()->desktopName( i+1 ));
			else if ( KPagerConfigDialog::m_showName )
			    txt=kwin()->desktopName( i+1 );
			QFontMetrics m = QFontMetrics(f);
			max = QMAX(max,m.width(txt)+4);
		}
		wDsk = max;
	}

	// now translate it back:
    w=wDsk*cols + (cols-1)*KPagerConfigDialog::m_desktopDistance + KPagerConfigDialog::m_desktopMargin + frameWidth();

	if (KPagerConfigDialog::m_showOnlyOneDesktop) w += 20; // 2x10 Pixel per button 
	
	return w;
}


KGlobalAccel * KPager2::globalAccel(){return m_globalAccel;}

//const KPager2::LayoutTypes KPager2::c_defLayout=KPager2::Horizontal;
const int KPager2::c_defNumberRows=2;
const bool KPager2::c_defShowOnlyOneDesktop=false;
const int KPager2::c_defDesktopDistance=1;
const int KPager2::c_defDesktopMargin=0;
const int KPager2::c_defGeneralBackgroundColor=2;
const int KPager2::c_defGeneralFrame=0;
const int KPager2::c_defGeneralTheme=0;
const bool KPager2::c_defShowTaskMenu=true;
const bool KPager2::c_defShowLauncher=true;

const bool KPager2::c_defUse3d=false;
const bool KPager2::c_def3dZoom=false;
const int  KPager2::c_def3dMode=5;


//#include "kpager.moc"


