/**************************************************************************

    kpagerdesktop.cpp  - KPager's desktop
    Copyright (C) 2000  Antonio Larrosa Jimenez
	 		Matthias Ettrich
			Matthias Elter
    changes (c) 2003 Sebastian Wolff

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Send comments and bug fixes to larrosa@kde.org

***************************************************************************/

#include "kpager.h"

#include <dcopobject.h>
#include <dcopclient.h>
#include <kdatastream.h>
#include <kapplication.h>
#include <kimageeffect.h>
#include <kwinmodule.h>
#include <kwin.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <kdebug.h>
#include <kshadowengine.h>
#include <kshadowsettings.h>
#include <ksharedpixmap.h>
#include <kpixmap.h>
#include <kpixmapio.h>
#include <kpixmapeffect.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kurifilter.h>
#include <kurl.h>
#include <kurldrag.h>
#include <netwm.h>

#include <qapplication.h>
#include <qbitmap.h>
#include <qclipboard.h>
#include <qcstring.h>
#include <qpixmap.h>
#include <qpainter.h>
#include <qdrawutil.h>
#include <qlabel.h>
#include <qpaintdevice.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qpoint.h>
#include <qtimer.h>
#include <qtooltip.h>

#include <math.h>

#include "kpageranimation.h"
#include "kpagerdesktop.h"
#include "kpagerrunwidget.h"
#include "config.h"
#include "windowdrag.h"

//#include "qtpixmap.h" //for our build-in png's
#include "tasklmbmenu.h"
#include "taskpagerman.h"

#define KPAGERDESKTOP_ATTENTION_BLINK_TIMEOUT 12
extern bool guardedContains(TaskPagerMan* man, Task* t );

/*
int round(double f)
// please note: this probably works with positive numbers only
{
	int a = int(f);
	int dir; // direction (positive?)
	if (f < 0) dir = -1; else dir = +1;
	if (abs(a-f)<0.5) return a;
	else
	{
		return (a+dir);
	}
}
*/

QPopupMenu * menu;

KPagerDesktop::KPagerDesktop( int desk, KPager2 * pager, QWidget *parent, const char *name) 	: QWidget(parent,name)
    , discardNextMouseEvent( false )
{

//menu = new QPopupMenu(this, "test");
//menu->insertItem(new QLabel(menu,"TESTTTTTT"));

  m_pixmap = QPixmap();
  m_pager = pager;
  m_desk = desk;
  m_bgSmallPixmap=0L;
  m_bgCommonSmallPixmap=0L;
  m_bgPixmap = 0L;
  m_bgDirty=true;
  setAcceptDrops(TRUE);
  setBackgroundMode(NoBackground);
  m_hadMouse = false;
  m_isCommon = false;
  setMouseTracking(true);
  m_dragtaskmenu = 0;
  m_attentionState = -1;

    justdragging = false;
    setMouseTracking(true);
    mousePos=QPoint(-10,-10);
	m_hadMouse=false;

	QToolTip::setGloballyEnabled(true);
/*
	if (KPagerConfigDialog::m_toolTips) m_tooltip = new KPagerToolTip(this);
	else m_tooltip = 0;
*/  m_tooltip = 0;
	// to ensure that repaintevent is only ONCE called in a single eveentloop
	connect(this, SIGNAL(aboutToRepaint()), this, SLOT(repaintDesktop()));

	// to let dragevents from other apps switch desktops
	connect(&m_dragSwitchTimer, SIGNAL( timeout() ), this, SLOT( slotDragSwitch() ) );
	m_dragTaskMenuTimer = -1;
	//connect(&m_dragTaskMenuTimer, SIGNAL( timeout() ), this, SLOT( slotDragMenu() ) );
	candecode = false;
}

KPagerDesktop::~KPagerDesktop()
{
	slotDeleteDragTaskMenu();
	delete m_bgPixmap;
	delete m_bgSmallPixmap;
	if (m_tooltip) delete m_tooltip;
}

void KPagerDesktop::slotDeleteDragTaskMenu()
{
	if (m_dragtaskmenu)
	{
		delete m_dragtaskmenu;
		m_dragtaskmenu = 0;
	}
}

void KPagerDesktop::testOnAnimationWidget()
{
	// is this desktop responsible?
	bool wrongDesktop = true;
	if ( (!hasMouse()) || (mousePos.x()<0) || (mousePos.x()<0) )
	{
		wrongDesktop = true;
	}
	else
/*
	if ( (!hasMouse()) || (mousePos.x()<0) || (mousePos.x()<0) )
	{
		if (pager()->m_anim)
			if (pager()->m_anim->hasMouse()) wrongDesktop = false;
	}
	else
*/
	{
		wrongDesktop = false;
	}
	if (wrongDesktop) return;
// CHANGE: Close it if it is not on our desktop anymore, but it belongs to our desktop!!!

	// set current state
	bool isAnimActive;
	isAnimActive = false;
	if (pager()->m_anim)
		isAnimActive = pager()->m_anim->isShown();

	// set future state
	QPoint pos;
	QGuardedPtr<Task> t = windowAtPosition(mousePos, &pos);
	// 1st: no window under mouse cursor: hide widget
	if (  ((!t) && (isAnimActive)) || (!guardedContains(pager()->pagerTaskManager()->taskPagerManager(), t)) )
	{
		if (pager()->m_anim) (pager()->m_anim)->hide();
		isAnimActive=false;
	}
	else
	// 2nd: window under mousecursor is the same
	if ((t) && (isAnimActive) && (pager()->m_anim->task() == t) )
		; // do nothing
	else
	// 3rd: window under mousecursor and it is different; hide and reopen animation widget
	if ((t) && (isAnimActive) && (pager()->m_anim->task() != t) )
	{
		pager()->m_anim->hide();
		isAnimActive=false;
	}
	// At last: open new animation widget
	if ((t) && (!isAnimActive))
	{
		if (pager()->m_anim)
		{
			delete pager()->m_anim;
			pager()->m_anim = 0;
		}
		if (!pager()->m_anim)
		{
			if (KPagerConfigDialog::m_animation == 1)
				pager()->m_anim = new KPagerZoomAnimation (m_desk, t, pager());
			else if (KPagerConfigDialog::m_animation == 2)
				pager()->m_anim = new KPagerShiftAnimation (m_desk, t, pager());
			else
			{
				pager()->m_anim = 0;
			}
		}
	}

}
bool KPagerDesktop::event(QEvent *e)
{
	// Let us check for the case that the mouse enters or leaves the widget (for highlighting purposes)
	//if (!justdragging)
	if (e!=0)
	{
		switch (e->type())
		{
		case QEvent::DragEnter:
		case QEvent::Enter:
			m_hadMouse = true;
			if (!justdragging) emit aboutToRepaint(); //paintEvent((QPaintEvent*)0);
			else paintEvent((QPaintEvent*)0);
			if (e->type() != QEvent::DragEnter) return true;
			else return QWidget::event(e);
			break;
		case QEvent::Leave:
		case QEvent::DragLeave:
			mousePos = QPoint(-1,-1);
			m_hadMouse = false;
			slotDeleteDragTaskMenu(); // in case it is open
			if (!justdragging)emit aboutToRepaint(); //paintEvent((QPaintEvent*)0);
			else paintEvent((QPaintEvent*)0);
			// close the animationwidget that might eventually be still open
			if (pager()->m_anim)
			{
				if (KPagerConfigDialog::m_animation==1) //zoom
					;//if (!pager()->m_anim->hasMouse()) pager()->m_anim->unFocus();
				else if (KPagerConfigDialog::m_animation==2) //shift
					//if (pager()->m_anim->hasMouse())
						pager()->m_anim->unFocus();
			}
			if (e->type() != QEvent::DragLeave) return true;
			else return QWidget::event(e);
			// close eventually still opened ToolTips
			if (m_tooltip) m_tooltip->hide();
			break;
		default: ;
			return QWidget::event(e);
		}
	}
	return true;
}

void KPagerDesktop::updateToolTip()
{
	if (m_tooltip)
	{
		if (m_tooltip->loadedText() != m_tooltip->getText(mousePos))
		{
			delete m_tooltip;
			m_tooltip = 0;
			if (KPagerConfigDialog::m_toolTips) m_tooltip = new KPagerToolTip(this);
		}
		else
		{
			if (!KPagerConfigDialog::m_toolTips)
			{
				delete m_tooltip;
				m_tooltip = 0;
			}
		}
	}
	else
		if (KPagerConfigDialog::m_toolTips) m_tooltip = new KPagerToolTip(this);
}

void KPagerDesktop::mouseMoveEvent( QMouseEvent *ev )
{
/*
    if ( m_hadMouse != hasMouse()) // note: does that get the correct mousehover if there is pixmapenlargement
    {
    	m_hadMouse = hasMouse();
		paintEvent((QPaintEvent*)0);
    }
*/
    mousePos = ev->pos();
	testOnAnimationWidget();

	// check if the tooltip must be updated:
	updateToolTip();

	if ( KPagerConfigDialog::m_showWindows ) if ( !KPagerConfigDialog::m_windowDragging )	return;
	if ( (ev->state() & LeftButton) == 0 )	return;
	QPoint p( ev->pos() - pressPos );
	if ( p.manhattanLength() >= qApp->startDragDistance() )
	startDrag( pressPos );


}

void KPagerDesktop::mousePressEvent( QMouseEvent * ev)
{
    if( discardNextMouseEvent ) {
	discardNextMouseEvent = false;
	return;
    }

	justdragging = true; // just a hack to catch this event
	//emit aboutToRepaint(); //paintEvent((QPaintEvent*)0);

	bool showWindows= KPagerConfigDialog::m_showWindows;
	if (ev->button()==LeftButton)
	{
		pressPos = ev->pos();
	}
	else if ((ev->button()==MidButton))
	{
		pressPos = ev->pos();
	}
    else if (ev->button()==RightButton)
	{
		QPoint pos;
		QGuardedPtr<Task> t = windowAtPosition(ev->pos(), &pos);
		if ( t && showWindows )
			pager()->showPopupMenu(t->window(), mapToGlobal(ev->pos()), m_desk);
		else
			pager()->showPopupMenu(0, mapToGlobal(ev->pos()), m_desk);
	}
}

void KPagerDesktop::mouseReleaseEvent( QMouseEvent *ev )
{
	justdragging = false;
	paintEvent((QPaintEvent*)0);
/** Note that mouseReleaseEvent is not called when releasing the mouse
 to drop a window in this desktop */
	killTimer(m_dragTaskMenuTimer);
	m_dragTaskMenuTimer=-1;
	
	// intended behaviour: left click on other desktop activates desktop, left click on current desktop activates window 

	if (ev->button()==LeftButton)
	{
		bool showWindows= KPagerConfigDialog::m_showWindows;
		QPoint pos;
		if (m_desk != KWin::currentDesktop())
		{
			pager()->setCurrentDesktop(m_desk);
		}
		else
		{
			if (showWindows)
			{
				QGuardedPtr<Task> t = windowAtPosition(ev->pos(), &pos);
				if (t)
				{
					KWin::forceActiveWindow(t->window());
					//KWin::setActiveWindow(t->window());
		//	    if ( static_cast<WindowDrawMode>( KPagerConfigDialog::m_windowDrawMode ) == Pixmap )
		//		m_windowPixmapsDirty.replace(info->win,true);
				}
			}
		}
	}
	
	// on Right Button: Insert text from clipboard, interprete it as URL and start it on the desktop 
	else  if(ev->button()==MidButton)
	{
		KURIFilterData* uri = new KURIFilterData();
		bool addToList;
		KPagerRunWidget::run_command(QApplication::clipboard()->text(),m_desk,uri,addToList);
	}
}

void KPagerDesktop::dragEnterEvent(QDragEnterEvent *ev)
{
	candecode = PagerWindowDrag::canDecode( ev );
    if (candecode)
	{
		ev->accept();
		justdragging = true;

		if (KPagerConfigDialog::m_showDragMoving) emit aboutToRepaint();
	}
	else if (KURLDrag::canDecode(ev))
	{
		ev->accept();
		//justdragging = true;
		if (m_pager->m_anim) if (m_pager->m_anim->isVisible()) m_pager->m_anim->hide();
	}
	m_dragSwitchTimer.start( 1000, TRUE );

	QWidget::dragEnterEvent(ev);
}

void KPagerDesktop::dragLeaveEvent(QDragLeaveEvent *ev)
{
    m_dragSwitchTimer.stop();
	killTimer(m_dragTaskMenuTimer);
	m_dragTaskMenuTimer=-1;

	QWidget::dragLeaveEvent(ev);
	if (KPagerConfigDialog::m_showDragMoving)
	if (candecode) // it is a window -> repaint at leaving
	{
		emit aboutToRepaint();
	}
}

void KPagerDesktop::dragMoveEvent(QDragMoveEvent *ev)
{

	// only process if we want to show the windows moving around
	if (KPagerConfigDialog::m_showDragMoving)
	{
		//dropEvent(ev);
	    WId win=0;
	    int deltax,deltay;
	    int origdesk;

		if (PagerWindowDrag::decode(ev,win,deltax,deltay,origdesk))
		{

			int x=ev->pos().x()-deltax;
			int y=ev->pos().y()-deltay;

			/*
			* x and y now contain the position (in local coordinates) which
			* has the origin of the window
			*/
			convertCoordP2S(x,y);

		//  kdDebug() << "moving window " << win << "d from " << origdesk << " to " << m_desk << endl;
		//  NETWinInfo NETinfo( qt_xdisplay(),  win, qt_xrootwin(), NET::Client | NET::WMDesktop);

			//if (origdesk == m_desk)
			//if (isCurrent())
			XMoveWindow(x11Display(), win, x, y );
//!!!!		// We have to find something such that !isCurrent the window disappears on Current Desktop !!!!
			ev->accept();

			if (m_desk==0) // means on all desktops, since M-desk starts at '1'
			{
				KWin::setOnAllDesktops(win, true);
			}
			else
			{
				if (origdesk==0) KWin::setOnAllDesktops(win, false);
				QGuardedPtr<Task> t = pager()->pagerTaskManager()->findTaskPager(win);
				if (!t->isOnAllDesktops())
				KWin::setOnDesktop(win, m_desk);
			}
			emit aboutToRepaint();
		}
	}
	justdragging = true;

}

void KPagerDesktop::dropEvent(QDropEvent *ev)
{
	killTimer(m_dragTaskMenuTimer);
	m_dragTaskMenuTimer=-1;
	m_dragSwitchTimer.stop();

	WId win=0;
	int deltax,deltay;
	int origdesk;
	if (PagerWindowDrag::decode(ev,win,deltax,deltay,origdesk))
	{ // window dragging inside of the pager
//		int x=ev->pos().x()-deltax;
//		int y=ev->pos().y()-deltay;
		/*
		* x and y now contain the position (in local coordinates) which
		* has the origin of the window
		*/
//		convertCoordP2S(x,y);

	//  kdDebug() << "moving window " << win << "d from " << origdesk << " to " << m_desk << endl;
	//  NETWinInfo NETinfo( qt_xdisplay(),  win, qt_xrootwin(), NET::Client | NET::WMDesktop);

		// comment the following because
		// 1. don't show window moving: just set a new desktop, not a new position
		// 2. show window moving: set a new position and desktop, but the position was already set by dragMoveEvent
//		XMoveWindow(x11Display(), win, x, y );

		if (m_desk==0)
		{
			/*
			* The next line moves the window to the active desktop. This is done
			* because in other case, kwm raises the window when it's in a semi
			* changed state and doesn't work well with kpager. Let's see how well
			* KWin behaves.
			* if (activedesktop!=KWM::desktop(w))
			*  KWM::moveToDesktop(w,activedesktop);
			*/
	//    KWin::setState(win, NET::Sticky);
			KWin::setOnAllDesktops(win, true);
		}
		else
		{
			if (origdesk==0) KWin::setOnAllDesktops(win, false);

			QGuardedPtr<Task> t = pager()->pagerTaskManager()->findTaskPager(win);
			if (!t->isOnAllDesktops())
			{
				KWin::setOnAllDesktops(win, false);
				KWin::setOnDesktop(win, m_desk);
				// bring the window to the front:
				if (isCurrent()) KWin::activateWindow(win);
			}
		}
		justdragging = false;
	}
	else
	{
		KURL::List list;
		if (KURLDrag::decode(ev, list))
		{ // URL dragging from outside: open the URL
			justdragging = false;
			
			slotRunURL(list);
		}
	}
	QWidget::dropEvent(ev);
}
void KPagerDesktop::slotRunURL(KURL::List list)
{
	// 1st: HACK - since we can't tell KRun on which desktop we want to open the URL (we can only open it on the current desktop), we have to switch to that desktop first
	if (m_desk >= 1 && m_desk<=KWin::numberOfDesktops()) // quite dangerous!
	{
		while (KWin::currentDesktop()!=m_desk) KWin::setCurrentDesktop(m_desk);
	}

	// now open the URL(s)
	KURL::List::iterator it;
	for (it = list.begin(); it != list.end(); ++it)
	{
		KURL url = *it;
		(void) new KRun(url);
	}
}

void KPagerDesktop::startDrag(const QPoint &p)
{
	QPoint dragpos;
	QGuardedPtr<Task> t=windowAtPosition(p,&dragpos);
	if ( (!t)/* || (info->state & NET::Max)*/ ) return;
	if (!guardedContains(pager()->pagerTaskManager()->taskPagerManager(), t)) return;
	QPixmap *pixmap=paintNewWindow(t);

	if (pixmap)
	{
		if ( pager()->m_anim)
			if ( pager()->m_anim->isVisible()) pager()->m_anim->hide();
		int deltax=dragpos.x();
		int deltay=dragpos.y();
		PagerWindowDrag *wdrag= new PagerWindowDrag( t->window(), deltax, deltay, m_desk, this);
		wdrag->setPixmap( *pixmap, QPoint( deltax, deltay) );
		delete pixmap;
		wdrag->dragCopy();
	}
}


// This is the code that gives us the proper behavior
// when a popup menu is displayed and we are clicked:
// close the menu, and don't reopen it immediately.
// It's copied from QToolButton. Unfortunately Qt is lame
// as usual and makes interesting stuff private or
// non-virtual, so we have to copy code.
bool KPagerDesktop::eventFilter( QObject *o, QEvent *e )
{
    switch ( e->type() ) {
    case QEvent::MouseButtonPress:
    case QEvent::MouseButtonDblClick:
	{
	    QMouseEvent *me = (QMouseEvent*)e;
	    QPoint p = me->globalPos();
	    if ( QApplication::widgetAt( p, TRUE ) == this )
		discardNextMouseEvent = true;
	}
	break;
    default:
	break;
    }
    return QWidget::eventFilter( o, e );
}

void KPagerDesktop::timerEvent(QTimerEvent * e)
{
	if (KPagerConfigDialog::m_showTaskMenu)
	if (e->timerId() == m_dragTaskMenuTimer)
	{
		killTimer(m_dragTaskMenuTimer);
		m_dragTaskMenuTimer=-1;

		// get a TaskList
		TaskList tl = pager()->pagerTaskManager()->taskManager()->tasks();
		TaskList * l = new TaskList();
		l->setAutoDelete(false);
		for ( Task* t = tl.first(); t ; t = tl.next() )
		{
			if (KPagerConfigDialog::m_showOnlyOneDesktop)
			{
				l->append(t); // we show all windows
			}
			else
			{
				if (t->desktop()==m_desk) l->append(t); // on this desktop
				if (t->desktop()==0 || t->isOnAllDesktops()) l->append(t); // on all desktops
			}
		}
		TaskLMBMenu * menu;
		if (l->count())
		{
			menu = new TaskLMBMenu( l );
			// calc popup menu position
			QPoint pos( mapToGlobal( QPoint( 0, 0 ) ) );
			QPoint p( mapToGlobal( QPoint( 0, 0 ) ) );
			if (pager()->orientation()==Qt::Vertical && mapToGlobal(pager()->rect().center()).x()<=QApplication::desktop()->width()/2)
			{
				// please note: I changed everything from "pos" to "p"
				pos.setX( p.x() + pager()->width() );
			}
			else // left
			if (pager()->orientation()==Qt::Vertical && mapToGlobal(pager()->rect().center()).x()>QApplication::desktop()->width()/2)
			{
				pos.setX( p.x() - menu->sizeHint().width() );
			}
			else // Down
			if (pager()->orientation()==Qt::Horizontal && mapToGlobal(pager()->rect().center()).y()<=QApplication::desktop()->width()/2)
			{
				pos.setY( p.y() + height() );
			}
			else // Up
			if (pager()->orientation()==Qt::Horizontal && mapToGlobal(pager()->rect().center()).y()>QApplication::desktop()->width()/2)
			{
				pos.setY( p.y() - menu->sizeHint().height() );
			}

			// the menu
			menu->exec(pos);
			delete menu;
		}
		delete l;
	}
}


void KPagerDesktop::slotDragSwitch()
{
	// since 3ddesktop might behave indefinite from time we don't use 3ddesktop (our own method, but KWin to switch the desktop:
	//pager()->setCurrentDesktop(m_desk);
	KWin::setCurrentDesktop(m_desk);

	if (!candecode)
	if (KPagerConfigDialog::m_showTaskMenu)
		m_dragTaskMenuTimer = startTimer(1000); // don't use a QTimer directly -> that would block the event loop!

}

QGuardedPtr<Task> KPagerDesktop::windowAtPosition(const QPoint &p, QPoint *internalpos)
{
	QRect r;
	const QValueList<WId> &list(pager()->kwin()->stackingOrder());
	if (list.count() <= 0)
		return 0L;

	for (QValueList<WId>::ConstIterator it = list.fromLast(); ; --it)
	{
		QGuardedPtr<Task> t = pager()->pagerTaskManager()->findTaskPager(*it);
		if (shouldPaintWindow(t))
		{
			KWin::WindowInfo info = KWin::windowInfo(t->window());
			r=info.geometry();
			convertRectS2P(r);
			if (r.contains(p))
			{
				if (internalpos)
				{
					internalpos->setX(p.x()-r.x());
					internalpos->setY(p.y()-r.y());
				}
				return t;
			}
		}

		if (it == list.begin())
			break;
	}
	return 0;
}

void KPagerDesktop::drawShadowText(QPainter  &p, QRect tr, int tf, const QString & str, int len, QRect * brect, QTextParag ** internal )
{
	// get the color of the shadow: white for dark text, black for bright text 
	QPen textPen = p.pen();
	QColor shadCol;
	if (textPen.color().red() + textPen.color().green() + textPen.color().blue() <= 3*256/2-1)
		shadCol = QColor(255,255,255);
	else
		shadCol = QColor(0,0,0);
		
	// get a transparent pixmap 
	QPainter pixPainter;
	QPixmap textPixmap(width(), height());
	
	textPixmap.fill(QColor(0,0,0));
	textPixmap.setMask( textPixmap.createHeuristicMask(TRUE) ); // now  its completely transparent! 
	
	// draw text 
	pixPainter.begin(&textPixmap);
	pixPainter.setPen(white/*textPen*/);    // get the pen from the root painter ! interesting: has to be white to work! we will draw the text white here, after that in the rright color 
	pixPainter.setFont(p.font()); // get the font from the root painter
	pixPainter.drawText( tr, tf, str, len, brect, internal );
	pixPainter.end();
	
	// draw shadow
	KShadowSettings * shadset = new KShadowSettings();
	//shadset->fromString("0,0,4.0,192.0,3,0,0,0,0,0"); // Test! 
	shadset->setOffsetX(0);
	shadset->setOffsetY(0);
	shadset->setThickness(1);
	shadset->setMaxOpacity(96);
	KShadowEngine * shadeng = new KShadowEngine(shadset);
	QImage img = shadeng->makeShadow(textPixmap, shadCol);
	delete shadeng;
	
	// return 
	p.drawImage(0,0,img);
	p.drawText( tr, tf, str, len, brect, internal );
}

void KPagerDesktop::convertRectS2P(QRect &r)
{
    QRect tmp(r);
    r.setRect(deskX()+tmp.x()*deskWidth()/kapp->desktop()->width(),
	      deskY()+tmp.y()*deskHeight()/kapp->desktop()->height(),
	      tmp.width()*deskWidth()/kapp->desktop()->width(),
	      tmp.height()*deskHeight()/kapp->desktop()->height());
}

void KPagerDesktop::convertCoordP2S(int &x, int &y)
{
    x=(x-deskX())*(kapp->desktop()->width())/deskWidth();
    y=(y-deskY())*(kapp->desktop()->height())/deskHeight();
}

QPixmap scalePixmap(const QPixmap &pixmap, int width, int height)
{
  if (pixmap.width()>100)
  {
    KPixmapIO io;
    QImage img(io.convertToImage(pixmap));
    return io.convertToPixmap(img.smoothScale(width,height));
  }

  QImage img(pixmap.convertToImage().smoothScale(width,height));
  QPixmap pix;
  pix.convertFromImage(img);

  return pix;
}

QPixmap fastScalePixmap(const QPixmap &pixmap, int width, int height)
{
  if ( width <= 0 || height <= 0 ) return pixmap;
  if ( pixmap.width() <= 0 || pixmap.height() <= 0 ) return pixmap;
  if (pixmap.isNull()) return pixmap;

  QWMatrix m;
  m.scale(width/(double)pixmap.width(),
      height/(double)pixmap.height());
  return pixmap.xForm(m);
}

void KPagerDesktop::loadBgPixmap(void)
// well, here we are grabbing the desktop background for this desktop through a dcop call of KDesktop
{
//printf("start loading %d\n",m_desk);
	DCOPClient *client = kapp->dcopClient();
	if (!client->isAttached())
		client->attach();
	QByteArray data, data2, replyData;
	QCString replyType;
	if (client->call("kdesktop", "KBackgroundIface", "isCommon()",
                  data, replyType, replyData))
	{
		QDataStream reply(replyData, IO_ReadOnly);
		if (replyType == "bool")
		{
			reply >> m_isCommon;
		}
	}
	// the following is a hack: if there is a common background for all desktops desktop 1 will paint the background of the others, too. Hence we will abort and return in case that it is common and not desktop 1
	if  ( m_isCommon && m_desk!=1 ) return;

	QDataStream args( data, IO_WriteOnly );
	args << m_desk;
	client->send( "kdesktop", "KBackgroundIface", "setExport(int)", data );

	if (!m_bgPixmap)
	{
		m_bgPixmap = new KSharedPixmap;
		connect(m_bgPixmap, SIGNAL(done(bool)), SLOT(backgroundLoaded(bool)));
	}

	m_bgPixmap->loadFromShared(QString("DESKTOP%1").arg(m_isCommon?1:m_desk));
}

void KPagerDesktop::backgroundLoaded(bool b)
{
	if (b)
	{
///printf("loaded %d\tisCommon=%d\n",m_desk,m_isCommon);
		if (m_isCommon)
		{
			if (m_bgSmallPixmap)
			{
				delete m_bgSmallPixmap;
				m_bgSmallPixmap=0L ;
			}
			if (!m_bgCommonSmallPixmap)
				m_bgCommonSmallPixmap=new QPixmap;
			*m_bgCommonSmallPixmap = scalePixmap(*m_bgPixmap,(int)(0.2*QApplication::desktop()->width()),(int)(0.2*QApplication::desktop()->height()));
		}
		else
		{
			if (m_bgCommonSmallPixmap)
			{
				delete m_bgCommonSmallPixmap;
				m_bgCommonSmallPixmap=0L ;
			};
			if (!m_bgSmallPixmap)
				m_bgSmallPixmap = new QPixmap;
			*m_bgSmallPixmap = fastScalePixmap(*m_bgPixmap,(int)(0.2*QApplication::desktop()->width()),(int)(0.2*QApplication::desktop()->height()));
		}
		delete m_bgPixmap;
		m_bgPixmap=0L;
		if (m_isCommon)
			pager()->redrawDesktops();
		else
		{
			// update();
			redraw();
		}
	}
	else kdDebug() << "Error getting the background\n";
}

void KPagerDesktop::paintWindow(QPainter &p, const QGuardedPtr<Task> t, double scale, bool onDesktop)
{
	if (!t) return;
	if (!guardedContains(pager()->pagerTaskManager()->taskPagerManager(), t)) return;
	if (t->isMinimized()) return;
    switch (static_cast<WindowDrawMode>(KPagerConfigDialog::m_windowDrawMode ) )
	{
	case (Plain)  : paintWindowPlain (p, t, scale, onDesktop);break;
	case (Icon)   : paintWindowIcon  (p, t, scale, onDesktop);break;
	case (Pixmap) : paintWindowPixmap(p, t, scale, onDesktop);break;
	}
}

QPixmap *KPagerDesktop::paintNewWindow(const QGuardedPtr<Task> t)
{
	if (!t || !guardedContains(pager()->pagerTaskManager()->taskPagerManager(), t)) return 0; // just removed

	KWin::WindowInfo info = KWin::windowInfo(t->window());
	if (!info.valid()) return 0;

    QRect r = info.geometry();
	double centx = r.x()+r.width()/2.;
	double centy = r.y()+r.height()/2.;
    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();
	int left = geometry().x();
	int top  = geometry().y();
	double scale = 1.0;
	r = QRect(
		(int)round(0.+ (centx-r.width() /2.*scale) * width() /dw + left),
		(int)round(0.+ (centy-r.height()/2.*scale) * height()/dh + top),
		(int)round(0.+ (r.width() *scale) * width() /dw),
		(int)round(0.+ (r.height()*scale) * height()/dh ));
    r.moveTopLeft(QPoint(0,0));


    QPixmap *pixmap=new QPixmap(r.width(),r.height());
    QPainter p;

    p.begin(pixmap);
    p.setFont(font());
    p.fillRect( r, colorGroup().dark());
    paintWindow(p, t, 1., false);
    p.end();

    return pixmap;
}

bool KPagerDesktop::shouldPaintWindow( KWin::WindowInfo *info )
{
printf("ERROR\nThis point [bool KPagerDesktop::shouldPaintWindow( KWin::WindowInfo *info )] should never be reached!\n");
  if (!info)
    return false;

//  if (info->mappingState != NET::Visible)
//    return false;

  if (info->windowType(NET::NormalMask) == NET::Dock
      || info->windowType(NET::NormalMask) == NET::Menu)
    return false;

  if (!info->onAllDesktops() && info->desktop() != m_desk)
    return false;

  if (info->state() & NET::SkipPager
      || info->state() & NET::Shaded )
    return false;

  if (info->win() == pager()->winId())
    return false;

  if ( info->isMinimized() )
    return false;

  return true;
}

bool KPagerDesktop::shouldPaintWindow( QGuardedPtr<Task> t)
{
	if (!t)
	{
		return false;
	}

	if (!guardedContains(pager()->pagerTaskManager()->taskPagerManager(), t)) return false; // just removed

	bool ret = true;

	KWin::WindowInfo info = KWin::windowInfo(t->window());

	// is it on this desktop?
	if (!t->isOnAllDesktops() && t->desktop() != m_desk)
		ret= false;

	if (info.windowType(NET::NormalMask) == NET::Dock
	  || info.windowType(NET::NormalMask) == NET::Menu)
		ret= false;

/*
	// is it visible at all? (should already be checked by libTaskManager)
	if ((info.mappingState() != NET::Visible))
		ret= false;
commented since this would affect windows on the current desktop only
*/

	// should it be visible to a pager application?
	if (info.state() & NET::SkipPager)
		ret= false;

	// is it shaded? (should already be checked by libTaskManager)
	if (info.state() & NET::Shaded)
		ret= false;

	// is it our own application? (important)
	if (t->window() == pager()->winId())
		ret= false;

	if ( t->isMinimized() )
		ret= false;

	if ( t->isShaded() )
		ret= false;

	return ret;
}

void KPagerDesktop::paintDesktopHighlighting(QPixmap & pixmap, bool selected, bool mousehover)
// colorizes the desktop depending on mouse hover or if the desktop is the current one
{

	//if (!KPagerConfigDialog::m_bevelledSelection)
	// RIGHT NOW WE ARE NOT ABLE TO DRAW AN INNEr BEVEL EASILY, LETS WAIT FOR QT4
	{
		// let us draw the selection as a pixmap that blends the desktop
		if (selected && !KPagerConfigDialog::m_showOnlyOneDesktop)
		{
			QImage ilow, iup;
			ilow = pixmap;
			iup = KImageEffect::gradient(pixmap.size(),getSelectedDesktopColor(), getSelectedDesktopColor(), KImageEffect::EllipticGradient); // ok, this is a hack, but gives us an image of one color
			KImageEffect::blend(iup,ilow, 0.50);
			pixmap = ilow;
		}

		if (mousehover)
		{
			QImage ilow, iup;
			ilow = pixmap;
			iup = KImageEffect::gradient(pixmap.size(),getHighlightedDesktopColor(), getHighlightedDesktopColor(), KImageEffect::EllipticGradient); // ok, this is a hack, but gives us an image of one color
			KImageEffect::blend(iup,ilow, 0.25);
			pixmap = ilow;
		}
	}
}

void KPagerDesktop::repaintDesktop()
{
	//paintEvent((QPaintEvent *)0);
	update();
}

void KPagerDesktop::update()
{
	if (m_pixmap.isNull()) redraw(m_pixmap);
	QWidget::update();
	if (isCurrent() && m_pager!=0)
		if (m_pager->m_singleDesktop!=this) // else our linux box would be killed
			m_pager->updateSingleDesktop(m_desk); // updates the desktop that is used in showOnlyOneDesktop mode
}

void KPagerDesktop::redraw()
{
	redraw(m_pixmap);
	if (isCurrent() && m_pager!=0)
		if (m_pager->m_singleDesktop!=this) // else our linux box would be killed
			m_pager->redrawSingleDesktop(m_desk); // updates the desktop that is used in showOnlyOneDesktop mode
	update();
}
void KPagerDesktop::redraw(QPixmap & pixmap)
{
	this->setPalette(QApplication::palette());

	// begin drawing operations:
	pixmap = QPixmap(width(),height());
	QPainter p;
	p.begin(&pixmap);

	// draw background
	if (KPagerConfigDialog::m_showBackground )
	{
		if ( ( !m_isCommon && !m_bgSmallPixmap )
		|| (m_isCommon && !m_bgCommonSmallPixmap) )
		loadBgPixmap();

		if ( ( !m_isCommon && m_bgSmallPixmap && !m_bgSmallPixmap->isNull() )
		  || ( m_isCommon &&
			m_bgCommonSmallPixmap && !m_bgCommonSmallPixmap->isNull() ) )
		{
			QPixmap tmp;
			if ( m_isCommon )
				//tmp=fastScalePixmap(*m_bgCommonSmallPixmap, width(),height());
				tmp=scalePixmap(*m_bgCommonSmallPixmap, width(),height());
			else
				//tmp=fastScalePixmap(*m_bgSmallPixmap, width(),height());
				tmp=scalePixmap(*m_bgSmallPixmap, width(),height());

			p.drawPixmap(0,0,tmp);
		}
		else
		{
			// just fill it with a plain color
			p.fillRect(rect(), getPlainDesktopColor());
		}
	}
	else
	{
		p.fillRect(rect(), getPlainDesktopColor()); // colorGroup().brush(QColorGroup::Mid));
	}

	p.end();
	paintDesktopHighlighting(pixmap, isCurrent(),false);


	paintShadowPixmap(pixmap);
	p.begin(&pixmap);

    // set in/active pen for font painting: desktop number or name
  if (isCurrent())
    p.setPen(QColorGroup::HighlightedText);
  else
    p.setPen(QColorGroup::Base);

	// paint windows
    if ( KPagerConfigDialog::m_showWindows ) {
		QValueList<WId>::ConstIterator it;
		for ( it = pager()->kwin()->stackingOrder().begin();
			it != pager()->kwin()->stackingOrder().end(); ++it )
		{
			QGuardedPtr<Task> t = pager()->pagerTaskManager()->findTaskPager( *it );
			if (shouldPaintWindow(t))
			{
				paintWindow(p, t, 1.);
				// if (t->isMaximized()) paintDesktopHighlighting(pixmap, isCurrent(),isCurrent());
				// this is commented since it looks TOO bright (for thumbnails) 
			}
		}
	}

    // apply the gradient to the pixmap
    p.end();
    if (KPagerConfigDialog::m_showGradient)
    {
		QImage ilow, iup;
		ilow = pixmap;
		iup = KImageEffect::gradient(QSize(width(),height()),QColor("Black"), QColor("White"), KImageEffect::EllipticGradient);
		KImageEffect::blend(iup,ilow, 0.40);
		pixmap = ilow;
	}

	// check for 'demands attention' mode
	if (m_attentionState!=-1) // is active
	{
		// do the following here: there are two modes: either just greyscale, or inverted greyscale
		KPixmap kpixmap(pixmap);
		pixmap = KPixmapEffect::toGray(kpixmap,false);
		if( m_attentionState == KPAGERDESKTOP_ATTENTION_BLINK_TIMEOUT || m_attentionState % 2 == 0 )
		{
			QImage im = pixmap.convertToImage();
			im.invertPixels();
			pixmap.convertFromImage(im);
		}
	}

    // paint number & name
    bool sname=KPagerConfigDialog::m_showName;
    bool snumber=KPagerConfigDialog::m_showNumber;
    if ( sname || snumber )
	{
		p.begin(&pixmap);
		QString txt;

		// set font
		if (sname) {
			QFont f( KGlobalSettings::toolBarFont() );
		    p.setFont(f);
		}
		else {
			QFont f( KGlobalSettings::taskbarFont() );
	    	p.setFont(f);
		}

		// draw text
		if ( sname && snumber )
		    txt=QString("%1. %2").arg(m_desk).arg(pager()->kwin()->desktopName( m_desk ));
		else if ( sname )
		    txt=pager()->kwin()->desktopName( m_desk );
		else if ( snumber )
		    txt=QString::number( m_desk );
		if (isCurrent())
		{
			p.setPen(colorGroup().highlightedText());
			drawShadowText(p, QRect(2, 0, width()-4, height()), AlignCenter, txt );
		}
		else
		{
			p.setPen(colorGroup().text());
			drawShadowText(p, QRect(2, 0, width()-4, height()), AlignCenter, txt );
		}
		p.end();
    }

}

void KPagerDesktop::paintEvent( QPaintEvent * )
{
	if (m_pixmap.size() != size())
	{
		redraw();
		return;
	}

	QPixmap pix = m_pixmap;

	// check for selection/mouse hover
    // paintDesktopHighlighting(pixmap, isCurrent(),hasMouse());
	paintDesktopHighlighting(pix, false,hasMouse());

	// blit pixmap to widget
/*	QPainter p;
	p.begin(this);
	p.drawPixmap(0,0,pix);
	p.end();
*/
	bitBlt(this,0,0, &pix,0,0,width(),height());
}

QColor KPagerDesktop::getPlainDesktopColor()
{
	// returns the color that is used to fill the desktop background if not the real desktop is to be displayed
	QColor col;
	switch (KPagerConfigDialog::m_desktopBackgroundColor)
	{
		case 1:
			col = (colorGroup().base());
			break;
		case 2:
			col = (colorGroup().background());
			break;
		case 3:
			col = (colorGroup().button());
			break;
		case 4:
			col = (colorGroup().dark());
			break;
		case 5:
			switch (KPagerConfigDialog::m_generalBackgroundColor)
			{
			case 1:
				col = (colorGroup().base());
				break;
			case 2:
				col = (colorGroup().background());
				break;
			case 3:
				col = (colorGroup().button());
				break;
			case 4:
				col = (colorGroup().dark());
				break;
			case 0:
			default:
				col = Qt::black;
			}
			break;
		case 0:
		default:
			col = Qt::black;
	}
	return col;
}

#include <kiconeffect.h>

void KPagerDesktop::paintWindowPlain(QPainter &p, const QGuardedPtr<Task> t, double scale, bool onDesktop)
{
	if (!t) return;
	if (!guardedContains(pager()->pagerTaskManager()->taskPagerManager(),t)) return;
	if (!KWin::windowInfo(t->window()).valid()) return;
	if (t->isMinimized()) return;
	KWin::WindowInfo info = KWin::windowInfo(t->window());
    QRect r =  info.geometry();
    double dw = QApplication::desktop()->width();
    double dh = QApplication::desktop()->height();
    double centx = r.x()+r.width()/2.;
    double centy = r.y()+r.height()/2.;
    r = QRect(
    	(int)round(0.+ (centx-r.width()/2.*scale) * width()/dw),
    	(int)round(0.+ /*2*/ + (centy-r.height()/2.*scale) * height()/dh),
    	(int)round(0.+ (r.width()*scale) * width()/dw),
    	(int)round(0.+ (r.height()*scale) * height()/dh ));
    if ( !onDesktop )
	r.moveTopLeft(QPoint(0,0));

	bool isActive=t->isActive();


	// get the fill color
	QColor col;
	if ( isActive ) col= colorGroup().highlight();
	else col = colorGroup().button();

	// fill it
	QBrush brush = col;
	WindowTransparentMode m_transparentMode = KPagerConfigDialog::m_windowTransparentMode;
	if ( m_transparentMode==AllWindows || (m_transparentMode==MaximizedWindows && ( t->isMaximized() && !t->isActive())) )
	{
		//brush.setStyle(QBrush::Dense4Pattern);
		//instead all these:
		QPixmap _pm(r.size());
		_pm.fill(col);
		KIconEffect::semiTransparent(_pm); // real transparency instead of a pattern
		p.drawPixmap(r.topLeft(),_pm);
	}
	else p.fillRect(r, brush);

/*
	// inner frame:
	// inner frame color is opposite of the fill color
	if ( !isActive ) col= colorGroup().highlight();
	else col = colorGroup().button();
	p.setPen(col);
	QRect r2 = QRect(r.left()+1, r.top()+1, r.width()-2, r.height()-2);
	if (r2.isValid()) 	p.drawRect(r2);

	// draw a frame around the thumbnail:
	p.setPen(QPen(colorGroup().dark(),0));
	p.drawRect(r);
*/
	// frame:
	// frame color is opposite of the fill color
	if ( !isActive ) col= colorGroup().highlight();
	else col = colorGroup().button();
	p.setPen(col);
	p.drawRect(r);


}


void KPagerDesktop::paintWindowIcon(QPainter &p, const QGuardedPtr<Task> t, double scale, bool onDesktop)
{
	if (!t) return;
	if (!guardedContains(pager()->pagerTaskManager()->taskPagerManager(),t)) return;
	if (!KWin::windowInfo(t->window()).valid()) return;
	if (t->isMinimized()) return;
	KWin::WindowInfo  info = KWin::windowInfo(t->window());
	QRect r =  info.geometry();//info->frameGeometry();
	double dw = QApplication::desktop()->width();
	double dh = QApplication::desktop()->height();
	double centx = r.x()+r.width()/2.;
	double centy = r.y()+r.height()/2.;
	r = QRect(
		(int)round(0.+ (centx-r.width()/2.*scale) * width()/dw),
		(int)round(0.+ /*2*/ + (centy-r.height()/2.*scale) * height()/dh),
		(int)round(0.+ (r.width()*scale) * width()/dw),
		(int)round(0.+ (r.height()*scale) * height()/dh ));


	QPixmap icon = KWin::icon( t->window(), int(r.width()*0.8), int(r.height()*0.8), false);
	// now scale it
	double iscale;
	iscale = ( QMIN(r.width(),r.height())*0.8 ) / (QMAX(icon.width(),icon.height()));
	icon = scalePixmap (icon, int(icon.width()*iscale), int(icon.height()*iscale));
	if (QMIN(icon.width(),icon.height()) < 10) icon.resize(0,0); // setNull(true) if the icon is to small to display it accurately


	if ( icon.isNull() || info.windowType(NET::NormalMask)!=NET::Override )
		paintWindowPlain(p,t,scale,onDesktop);

	if ( !onDesktop )
		r.moveTopLeft(QPoint(0,0));

	p.drawPixmap( r.center()-icon.rect().center(),icon );
}

void KPagerDesktop::paintWindowPixmap(QPainter &p, const QGuardedPtr<Task> t,double scale,
					bool onDesktop)
{
	if (!t) return;
	if (t->isMinimized()) return;
	if (t->isShaded()) return;

	if (!guardedContains(pager()->pagerTaskManager()->taskPagerManager(),t)) return;
	if (!KWin::windowInfo(t->window()).valid()) return;

	QRect rSmall;
	KWin::WindowInfo info = KWin::windowInfo(t->window());
	QRect r =  info.geometry();

	double dw = QApplication::desktop()->width();
	double dh = QApplication::desktop()->height();
	double centx = r.x()+r.width()/2.;
	double centy = r.y()+r.height()/2.;
	rSmall = QRect(
  		(int)round(0.+ (centx-r.width()/2.*scale) * width()/dw),
  		(int)round(0.+ /*2*/ + (centy-r.height()/2.*scale) * height()/dh),
  		(int)round(0.+ (r.width()*scale) * width()/dw),
  		(int)round(0.+ (r.height()*scale) * height()/dh ));

	if (!t->hasThumbnail())
	{
		paintWindowIcon(p, t, scale, onDesktop);
		return;
	}

	QPixmap pixmap = t->thumbnail();

	if (pixmap.isNull())
	{
		paintWindowIcon(p, t, scale, onDesktop);
		return;
	}

	// apply window transparency to the thumbnail (if configured):
	WindowTransparentMode m_transparentMode = KPagerConfigDialog::m_windowTransparentMode;
	if ( m_transparentMode==AllWindows || (m_transparentMode==MaximizedWindows && ( t->isMaximized() && !t->isActive())) )
	{
		KIconEffect::semiTransparent(pixmap);
	}


	if ( !onDesktop )
		rSmall.moveTopLeft(QPoint(0,0));

	if (rSmall.width() != pixmap.width() || rSmall.height() != pixmap.height())
	{
		QPixmap pixmapSmall(fastScalePixmap(pixmap,rSmall.width(),rSmall.height()));
		p.drawPixmap( rSmall.topLeft(), pixmapSmall );
	}
	else
	{
		p.drawPixmap( rSmall.topLeft(), pixmap);
	}

	QColor col;
	if ( t->isActive() ) col= colorGroup().highlight();
	else colorGroup().dark();//col = colorGroup().button();

	// draw a frame around the thumbnail:
	p.setPen(col);
	p.drawRect(rSmall);

}

#include "kpagershadowpng.h"

void KPagerDesktop::paintShadowPixmap(QPixmap& pixmap)
// here we apply a nice shadow to the background of the desktop
// it is not applied to the thumbnails
// either you use no shadow (empty string) or predefined (given by a number) or a user defined pixmap (filename)
// this pixmap's edge sizes should be devidable by 2 and 3.
{
	if (!KPagerConfigDialog::m_showShadowPixmap) return;
	{
		// Let us partition the pixmap. To do that we have to get the col/row infos first

		int rows,cols,row,col;
		if (KPagerConfigDialog::m_showOnlyOneDesktop)
		{
			row=0;
			col=0;
			rows=1;
			cols=1;
			QPixmap p = qtgetPixmap(qtPixmapShadowId);
			QPixmap p2 = scalePixmap(p,width(),height());
			QPainter pr(&pixmap);
			pr.drawPixmap(0,0,p2,0,0,width(),height());
			pr.end();
			return;
		}
		else
		{
			rows=m_pager->m_layout->numRows();
			cols=m_pager->m_layout->numCols();
			if (!m_pager->m_layout->findWidget(this,&row,&col)) return;
			QPixmap p_full = qtgetPixmap(qtPixmapShadowId);
			int scalex,scaley;
			if (rows>2) scaley=3; else scaley = rows;
			if (cols>2) scalex=3; else scalex = cols;
			QPixmap p2 = scalePixmap(p_full,scalex*width(),scaley*height());
			int x1,y1;
			if (col+1==cols) // right
			{
				x1=(int)(width()*(scalex-1));
			}
			else if (col == 0) // left
			{
				x1=(int)(0);
			}
			else if (col == 1) // 2. col (to catch the case that we have only 2 cols)
			{
				x1=(int)(width());
			}
			else // horiz-center
			{
				x1=(int)(width()*1);
			}

			if (row+1==rows) // bottom
			{
				y1=(int)(height()*(scaley-1));
			}
			else if (row == 1) // 2. row (to catch the case that we have only 2 rows)
			{
				y1=height();
			}
			else if (row == 0) // top
			{
				y1=(int)(0);
			}
			else // vert-center
			{
				y1=(int)(height()*1);
			}
			QPainter pr;
			pr.begin(&pixmap);
			pr.drawPixmap(0,0,p2,x1,y1,width(),height());
			pr.end();
			return;
		}
	}
}

KPager2 *KPagerDesktop::pager() const
{
  //return reinterpret_cast<KPager2 *>(parent());
	return m_pager;
}

bool KPagerDesktop::isCurrent() const
{
  return pager()->kwin()->currentDesktop()==m_desk;
}

QColor KPagerDesktop::getHighlightedDesktopColor()
// returns the color to colorize the desktop if the mouse hovers above it
{
	return colorGroup().highlight();
}
QColor KPagerDesktop::getSelectedDesktopColor()
// returns the color to colorize the desktop if it is the current desktop
{
	return colorGroup().light();
}

QSize KPagerDesktop::sizeHint() const
{
  return QSize((QApplication::desktop()->width()*50)/QApplication::desktop()->height(),50);
}





/*
	DEMAND ATTENTION MODE RELATED FUNCTIONS
*/

void KPagerDesktop::slotAttentionTimerFired()
{
	if ( m_attentionState == -1 ) return;

	if( m_attentionState < KPAGERDESKTOP_ATTENTION_BLINK_TIMEOUT )
		++m_attentionState;
	else
	{
		m_attentionTimer.stop();
		disconnect(&m_attentionTimer,0,this,0);
	}
	redraw();
}

void KPagerDesktop::setAttention(bool att)
{
	if (att && m_attentionState == -1)
	{
		// was feshly activated (if not: let it be in old state)
		connect(&m_attentionTimer,SIGNAL(timeout()),this,SLOT(slotAttentionTimerFired()));
		m_attentionTimer.start( 500 );
		m_attentionState = 0;
	}
	else if( !att && m_attentionState >= 0 ) // it was active und was set to inactive
	{
		disconnect(&m_attentionTimer,0,this,0);
		m_attentionTimer.stop();
		m_attentionState = -1;
		redraw();
	}
}

bool KPagerDesktop::isAttention()
{
	return (m_attentionState != -1);
}












QPixmap *KPagerDesktop::m_bgCommonSmallPixmap=0L;
bool KPagerDesktop::m_isCommon=false;

// Default Configuration -------------------------------------------------

const bool KPagerDesktop::c_defShowName=false;
const bool KPagerDesktop::c_defShowNumber=false;
const bool KPagerDesktop::c_defShowWindows=true;
const bool KPagerDesktop::c_defShowBackground=true;
const bool KPagerDesktop::c_defWindowDragging=true;
const bool KPagerDesktop::c_defShowDragMoving=true;
const int KPagerDesktop::c_defAnimation=0;
const bool KPagerDesktop::c_defShowGradient=false;
const int KPagerDesktop::c_defDesktopBackgroundColor=0;
const int KPagerDesktop::c_defShadowPixmap=0;
const KPagerDesktop::WindowDrawMode KPagerDesktop::c_defWindowDrawMode=KPagerDesktop::Icon;
const KPagerDesktop::WindowTransparentMode
		KPagerDesktop::c_defWindowTransparentMode=KPagerDesktop::AllWindows;
const bool KPagerDesktop::c_defBevelledSelection=true;
const bool KPagerDesktop::c_defToolTips=true;
const bool KPagerDesktop::c_defShowShadowPixmap = true;

