//
// C++ Implementation: kpagertaskmanager
//
// Description:
//
//
// Author: Sebastian Wolff <>, (C) 2004
//
// Copyright: See COPYING file that comes with this distribution
//
//


#include "taskmanager.h"
#include <dcopclient.h>
#include <kapplication.h>
#include <kwinmodule.h>


#include "config.h"
#include "kpagertaskmanager.h"
#include "kpager.h"
#include "kpagerdesktop.h"
#include "kpageranimation.h"
#include "taskpagerman.h"



extern KWinModule * kwin_module;

KPagerTaskManager::KPagerTaskManager(KPager2 * pager) :
	QObject()
{
	m_pager = pager;
	m_taskmanager  = new TaskManager(this);
	m_taskpagermanager = new TaskPagerMan(this);
	m_currentDesktop = m_taskmanager->currentDesktop();

	m_grabtimer = new QTimer(this);

	// own signals
	connect(this, SIGNAL(updateThumbnail(QGuardedPtr<Task>)), this,SLOT(slotUpdateThumbnail(QGuardedPtr<Task>)));
	connect(this, SIGNAL(updateActiveThumbnail()), this,SLOT(slotUpdateActiveThumbnail()));
	// connect(this, SIGNAL(repaintDesktop(uint )), this,SLOT(slotRepaintDesktop(uint ))); // never use this, use emitRepaintDesktop instead!, see taskRemoved for details

	// taskmanager listeners:
	// important: if a window was removed, it has to be removed from the taskpagermanager list first. Else KPagerDesktop would try to update the screen, but would cause a segmentation fault because the window is still in the taskpagermanager list. Hence: Connect first taskpagermanager, then taskmanager!

	connect(m_taskpagermanager, SIGNAL(taskAdded(QGuardedPtr<Task>)), this, SLOT(slotTaskAdded(QGuardedPtr<Task> )));
	connect(m_taskpagermanager, SIGNAL(taskRemoved(QGuardedPtr<Task>)), this, SLOT(slotTaskRemoved(QGuardedPtr<Task> )));
	connect(m_taskpagermanager, SIGNAL(startupAdded(Startup*)), this, SLOT(slotStartupAdded(Startup*)));
	connect(m_taskpagermanager, SIGNAL(startupRemoved(Startup*)), this, SLOT(slotStartupRemoved(Startup* )));
	connect(m_taskmanager, SIGNAL(taskAdded(Task*)), this, SLOT(slotTaskAdded(Task* )));
	connect(m_taskmanager, SIGNAL(taskRemoved(Task*)), this, SLOT(slotTaskRemoved(Task* )));
	connect(m_taskmanager, SIGNAL(startupAdded(Startup*)), this, SLOT(slotStartupAdded(Startup*)));
	connect(m_taskmanager, SIGNAL(startupRemoved(Startup*)), this, SLOT(slotStartupRemoved(Startup* )));
	connect(m_taskmanager, SIGNAL(desktopChanged(int)), this, SLOT(slotDesktopChanged(int)));

	/* we have to comment the line

	connect(m_taskmanager, SIGNAL(windowChanged(WId)), this, SLOT(slotWindowChanged(WId)));

	 * since taskManager won't emit this if eg. the window was resized or moved (only textbased taskmanagers)
	 * hence we listen on KWinModule immediately (the taskManager's signal is filtered)
	connect(kwin_module, SIGNAL(windowChanged(WId)), this, SLOT(slotWindowChanged(WId)));
	*/

	/* this is one way, but the internal KWin::WindowInfo object will not be updated, hence the thumbnail will be taken for wrong screen coordinates. alternative: call task->refresh() for KWinModule::windowChanged(). refresh() will emit changed() -> everything will be in order
	*/
	connect(m_taskpagermanager, SIGNAL(windowChanged(WId)), this, SLOT(slotWindowChanged(WId)));
	connect(m_taskmanager, SIGNAL(windowChanged(WId)), this, SLOT(slotWindowChanged(WId)));

	connect(kwin_module, SIGNAL(windowChanged(WId,unsigned int)), this, SLOT(slotRefreshWindow(WId,unsigned int)));
	connect(kwin_module, SIGNAL(stackingOrderChanged()), this, SLOT(slotStackingOrderChanged()));

	connect(m_grabtimer, SIGNAL(timeout()), this, SLOT(slotUpdateActiveThumbnail()));
	m_grabtimer->start(15000, false); // reload the active app thumbnail every 15 seconds

	// we have to connect the necessary slots with the preloaded tasks that means those tasks
	// that are already running when we are starting and that are added to the m_taskmanager object
	// without passing slotTaskAdded
	TaskList tl;
	tl = taskManager()->tasks();
	for (Task* t = tl.first(); t != 0; t = tl.next()) slotTaskAdded(t); // connect all necessary slots
	GuardedTaskList gtl;
	gtl = taskPagerManager()->tasks();
	for (QGuardedPtr<Task>* gt = gtl.first(); gt; gt = gtl.next()) slotTaskAdded(*gt); // connect all necessary slots

	// update the first thumbnail since the first reading of the already existing tasks won't pass taskAdded
	// slotUpdateActiveThumbnail();

	connect(pager->kwin(), SIGNAL(numberOfDesktopsChanged(int)), this, SLOT(slotNumberOfDesktopsChanged(int)));
	slotNumberOfDesktopsChanged(taskManager()->numberOfDesktops());

	slotCheckAttention();
}
KPagerTaskManager::~KPagerTaskManager()
{
	m_grabtimer->stop();
	m_repainttimers.clear();
	delete m_taskmanager;
	delete m_taskpagermanager;
}

QGuardedPtr<Task> KPagerTaskManager::findTaskMan(WId w)
{
//	qDebug("KPagerTaskManager::findTaskMan(WId w)");
	if (KWin::windowInfo(w).valid())
	{
		TaskList tl;
		tl = taskManager()->tasks();
		for (Task* t = tl.first(); t != 0; t = tl.next())
		{
			if (!t) break;
			if (t->window() == w  || t->hasTransient(w))
			{
				//qDebug("KPagerTaskManager::findTaskMan(WId w) returned properly");
				return t;
			}
		}
	}
	//qDebug("KPagerTaskManager::findTaskMan(WId w) returned 0");
	return 0;
}
QGuardedPtr<Task> KPagerTaskManager::findTaskPager(WId w)
{
//	qDebug("KPagerTaskManager::findTaskPager(WId w)");
	if (KWin::windowInfo(w).valid())
	{
		GuardedTaskList tl;
		tl = taskPagerManager()->tasks();
		for (QGuardedPtr<Task>* t = tl.first(); t != 0; t = tl.next())
		{
			if (!t) break;;
			if (!(*t)) break;;

			if ((*t)->window() == w  || (*t)->hasTransient(w))
			{
				//qDebug("KPagerTaskManager::findTaskPager(WId w) returned properly");
				return *t;
			}
		}
	}
	//qDebug("KPagerTaskManager::findTaskPager(WId w) 0");
	return 0;
}

bool guardedContains(TaskPagerMan* man, Task* t )
{
	//qDebug("guardedContains");
	GuardedTaskList tl = man->tasks();
	for (QGuardedPtr<Task>* gt = tl.first(); gt != 0; gt = tl.next())
	{
		if (!gt) break;;
		if ( ((Task*)(*gt)) ==t )
		{
			//qDebug("guardedContains returned true");
			return true;
		}
		if (!(*gt)) break;
	}
	//qDebug("guardedContains returned false");
	return false;
}

void KPagerTaskManager::slotCheckAttention(QGuardedPtr<Task> t)
{
	//qDebug("slotCheckAttention(QGuardedPtr<Task> t)");

	bool attention = false;


	// is it a  task displayed on a desktop?
	// that means it is a child of taskManager()
	// if (!guardedContains(taskPagerManager(),t)) return; // wrong - that could mean that t was just removed
	if (t) if (t->parent()==taskManager())
	{
		//qDebug("slotCheckAttention(QGuardedPtr<Task> t) returned");
		return;
	}

	// check if a task was removed that demanded attention (then attention would be false)
	if (t)

		if ( !guardedContains(taskPagerManager(),t) || !KWin::windowInfo(t->window()).valid() ) attention = false;
		else attention = t->demandsAttention();



	if (attention) // get the desktop(s) of the task and start there attention timers
	{
		if (t->isOnAllDesktops())
		{
			for (uint i=1; i<=pager()->m_desktops.count();i++)
			{
//				if (!pager()->m_desktops[i-1]->isAttention())
					pager()->m_desktops[i-1]->setAttention(true);
			}
		}
		else
		{
//			if (!pager()->m_desktops[t->desktop()-1]->isAttention())
				pager()->m_desktops[t->desktop()-1]->setAttention(true);
		}
	}
	else // set Attention(false), but check if we have to consider all desktops or if there are other tasks that demand attention
	{
		//1st get the desktops that still demand attention
		QMemArray<bool> attentions(pager()->m_desktops.count());
		attentions.fill(false);
		GuardedTaskList tasks = taskPagerManager()->tasks();
		for ( QGuardedPtr<Task>* gt = tasks.first(); gt ; gt = tasks.next() )
		{
			if (!gt) break;
			if (!(*gt)) continue;
			if ( (*gt)->demandsAttention())
			{
				if ((*gt)->isOnAllDesktops() || (*gt)->desktop()==0 )
				{
					attentions.fill(true);
					break;
				}
				else
				{
					attentions[(*gt)->desktop()-1]=true;
				}
			}
		}
		//2nd apply the values to the desktops
		for (uint i=1; i<=pager()->m_desktops.count();i++)
		{
			pager()->m_desktops[i-1]->setAttention(attentions[i-1]);
		}
	}
	//qDebug("slotCheckAttention(QGuardedPtr<Task> t) returned");
}

void KPagerTaskManager::slotRefreshWindow(WId win)
{
	if (!KWin::windowInfo(win).valid()) return;
	QGuardedPtr<Task> t = findTaskMan(win);
	if (t)
	{
		//qDebug("slotRefreshWindow(WId win): man");
		t->refresh();
	}
	t = findTaskPager(win);
	if (t)
	{
		//qDebug("slotRefreshWindow(WId win): pager");
		t->refresh();
	}
	//qDebug("slotRefreshWindow(WId win): returned");
}

void KPagerTaskManager::slotRefreshWindow(WId win, unsigned int prop)
{
	if (!KWin::windowInfo(win).valid()) return;
	//qDebug("slotRefreshWindow(WId win, unsigned int prop)");
	// check here for move and resizeevents and let slotRefreshWindow(WId) decide what to do next
	if (prop & NET::WMMoveResize || prop & NET::WMGeometry)
	// || prop & NET::WMWindowType || prop & NET::WMState || prop & NET::XAWMState )
		slotRefreshWindow(win);
	//qDebug("slotRefreshWindow(WId win, unsigned int prop) returned");
}

void KPagerTaskManager::slotNumberOfDesktopsChanged(int num)
{
	//qDebug("slotNumberOfDesktopsChanged(int num)");
	m_repainttimers.setAutoDelete(true);
	m_repainttimers.clear();
	while ((int) m_repainttimers.count() < num)
	{
		QTimer * t = new QTimer;
		//m_repainttimers.append( QTimer() );
		m_repainttimers.append( t );
		//connect (&m_repainttimers.last(), SIGNAL(timeout()), this, SLOT(slotRepaintDesktop(m_repainttimers.count()));
		connect (t, SIGNAL(timeout()), this, SLOT(slotRepaintDesktop()));
	}
	//qDebug("slotNumberOfDesktopsChanged(int num) returned");
}
void KPagerTaskManager::emitRepaintDesktop(uint desk)
{
	#define repaintDesktopINTERVAL 200
	//qDebug("emitRepaintDesktop(uint desk)");
	if (desk == 0)
	{
		for ( uint i=1; i <= pager()->desktopList().count(); ++i)
		{
			if (i-1 < m_repainttimers.count())
			if (m_repainttimers.at(i-1)) m_repainttimers.at(i-1)->start(repaintDesktopINTERVAL, true); // single shot
		}
	}
	else
	{
		if (desk-1 < m_repainttimers.count())
		if (m_repainttimers.at(desk-1)) m_repainttimers.at(desk-1)->start(repaintDesktopINTERVAL, true); // single shot
	}
	//qDebug("emitRepaintDesktop(uint desk) returned");
}

void KPagerTaskManager::slotTaskAdded(QGuardedPtr<Task> t)
{
	if (t)
	{
		//qDebug("slotTaskAdded(QGuardedPtr<Task> t)");
		slotTaskAdded((Task*) t);
		//qDebug("slotTaskAdded(QGuardedPtr<Task> t) returned");
	}
}
void KPagerTaskManager::slotTaskRemoved(QGuardedPtr<Task> t)
{
	if (t)
	{
		//qDebug("slotTaskRemoved(QGuardedPtr<Task> t)");
		slotTaskRemoved((Task*)t);
		//qDebug("slotTaskRemoved(QGuardedPtr<Task> t) returned");
	}
}
void KPagerTaskManager::slotTaskAdded(Task* t)
{
	if (!t) return;

	//qDebug("slotTaskAdded(Task* t)");

	connect(t, SIGNAL(changed()), this, SLOT(slotChanged()));
// would lead to continuous thumbnail updates (even if I just write something in KWrite)


	connect(t, SIGNAL(iconChanged()), this, SLOT(slotIconChanged()));
	connect(t, SIGNAL(activated()), this, SLOT(slotActivated()));
	connect(t, SIGNAL(deactivated()), this, SLOT(slotDeactivated()));

	connect(t, SIGNAL(thumbnailChanged()), this, SLOT(slotThumbnailChanged()));
	t->setThumbnailSize(0.2); // 20% of window size

	slotCheckAttention(t);

	emit updateThumbnail(t);

	if (KPagerConfigDialog::m_showWindows)
	if (t->parent() != taskManager())
	if (findTaskPager(t->window()))
	if (KWin::windowInfo(t->window()).valid())
		if (t->isOnAllDesktops()) emitRepaintDesktop(0);
		else emitRepaintDesktop(t->desktop());
	//qDebug("slotTaskAdded(Task* t) returned");

}

void KPagerTaskManager::slotTaskRemoved(Task* t)
{
	if (!t) return;

	//qDebug("slotTaskRemoved(Task* t)");

	disconnect(t, SIGNAL(changed()), this, SLOT(slotChanged()));
	disconnect(t, SIGNAL(iconChanged()), this, SLOT(slotIconChanged()));
	disconnect(t, SIGNAL(activated()), this, SLOT(slotActivated()));
	disconnect(t, SIGNAL(deactivated()), this, SLOT(slotDeactivated()));
	disconnect(t, SIGNAL(thumbnailChanged()), this, SLOT(slotThumbnailChanged()));

	// close eventual animation widgets (they would invoke a segmentation fault when it tries to draw a nonexisting thumbnail
	if (t->parent() != taskManager())
	if (findTaskPager(t->window()))
	if (KWin::windowInfo(t->window()).valid())
	if (pager()->m_anim)
	if ((Task*)pager()->m_anim->task() == t)
	{
		pager()->m_anim->unFocus();
		// to be sure:
		delete pager()->m_anim;
		pager()->m_anim = 0;
	}

	slotCheckAttention();

	// repaint
	if (KPagerConfigDialog::m_showWindows)
	if (t->parent() != taskManager())
	if (findTaskPager(t->window()))
	if (KWin::windowInfo(t->window()).valid())
		if (t->isOnAllDesktops()) emitRepaintDesktop(0);
		else emitRepaintDesktop(t->desktop());
	// with Timer to avoid a segmentation fault when more than one task is removed at one time -> that would invoke a repaint of a desktop where repaintDesktop tries to paint thumbnails of nonexisting windows. This is because the repaintDesktop() signal would be invoked several times, once for each removed task -> we want to have one signal for the removed tasks altogether -> We assume that al taskRemoved signals will be emitted within 100 milliseconds and use this time as a treshold for QTimer events, see emitRepaintDesktop for details.
	//qDebug("slotTaskRemoved(Task* t) returned");
}

void KPagerTaskManager::slotStartupAdded(Startup*)
{
	// Right now: nothing
	// TODO: Do something here when we add full taskmanager abilities
}

void KPagerTaskManager::slotStartupRemoved(Startup*)
{
	// Right now: nothing
	// TODO: Do something here when we add full taskmanager abilities
}

void KPagerTaskManager::slotDesktopChanged(int desk)
{
	if (m_currentDesktop==desk) return;

	//qDebug("slotDesktopChanged(int desk)");

	pager()->setCurrentDesktopCounter(desk);

	// update the layout: in 'show only one desktop' mode: change visible desktop
	pager()->updateDesktopLayoutIfOnlyOneVisible();

	if (m_currentDesktop-1<(int)pager()->desktopList().count()) pager()->desktopList()[m_currentDesktop-1]->redraw();
	if (desk-1<(int)pager()->desktopList().count()) pager()->desktopList()[desk-1]->redraw();

	m_currentDesktop=desk;
	//qDebug("slotDesktopChanged(int desk) returned");
}

void KPagerTaskManager::slotWindowChanged(WId win)
{
	if (!KWin::windowInfo(win).valid()) return;
	//qDebug("slotWindowChanged(WId win)");
	QGuardedPtr<Task> gt = findTaskPager(win);
	if (gt)
	{
		if (KPagerConfigDialog::m_showWindows)
		if (gt->isOnAllDesktops())
		{
			if (KPagerConfigDialog::m_showWindows)
			if (0<(int)pager()->desktopList().count()) if (pager()->desktopList()[0]->shouldPaintWindow(gt)) emitRepaintDesktop(0);
		}
		else
			if (KPagerConfigDialog::m_showWindows)
			if (gt->desktop()-1<(int)pager()->desktopList().count())
			if (pager()->desktopList()[gt->desktop()-1]->shouldPaintWindow(gt)) emitRepaintDesktop(gt->desktop());
		emit updateThumbnail(gt);
	}
	else
	{
		Task * t = findTaskMan(win);
		if (t)
		{
			emit updateThumbnail(t);
		}
	}
	//qDebug("slotWindowChanged(WId win) returned");
}


void KPagerTaskManager::slotChanged()
{
	//qDebug("slotChanged()");
	Task * t = identifyTaskSender(sender());
	if (t)
	{
		//qDebug("slotChanged(): t != 0");


		// since the changed() signal is even emitted whenever the window name is change and editors like kwrite do change it very often, it would lead to a continuous update of the thumbnail and repaint of the desktops. hence we will forget these kinds of changed and do only check on attention here.
		// NOTE #2: At least repaint the current desktop since window shading has to be checked!


//		emit updateThumbnail(t);

		if (t->parent() != taskManager())
		if (findTaskPager(t->window()))
		if (KWin::windowInfo(t->window()).valid())
		{
			slotCheckAttention(t);


			if (KPagerConfigDialog::m_showWindows)
			if (t->isMinimized() || t->isShaded()) emitRepaintDesktop(t->desktop());

			// To catch the likely event that a window was set to 'sticky' or from sticky to not sticky we have to be sure and draw all desktops in any case -> simply check if it is paintable (not minimized...)

			if (KPagerConfigDialog::m_showWindows)
			if (t->isOnAllDesktops())
			{
				if (0<(int)pager()->desktopList().count())
				if (pager()->desktopList()[0]->shouldPaintWindow(t)) emitRepaintDesktop(0);
			}
			else
				if (t->desktop()-1<(int)pager()->desktopList().count())
					if (pager()->desktopList()[t->desktop()-1]->shouldPaintWindow(t)) emitRepaintDesktop(0);
		}
	}
	//qDebug("slotChanged() returned");
}

void KPagerTaskManager::slotIconChanged()
{
	//qDebug("KPagerTaskManager::slotIconChanged()");
	Task * t = identifyTaskSender(sender());
	if (t)
	if (KWin::windowInfo(t->window()).valid())
	if (t->parent() != taskManager())
	if (findTaskPager(t->window()))
	//if (guardedContains(taskPagerManager(),t))
	//if (findTaskPager(t->window()))
	{
		// repaint
		if (KPagerConfigDialog::m_showWindows)
		if (t->isOnAllDesktops())
		{
			if (0<(int)pager()->desktopList().count())
			if (pager()->desktopList()[0]->shouldPaintWindow(t)) emitRepaintDesktop(0);
		}
		else
		if (t->desktop()-1<(int)pager()->desktopList().count())
			if (pager()->desktopList()[t->desktop()-1]->shouldPaintWindow(t)) emitRepaintDesktop(t->desktop());
	}
	//qDebug("KPagerTaskManager::slotIconChanged() returned");
}

void KPagerTaskManager::slotActivated()
{
	//qDebug("KPagerTaskManager::slotActivated()");
	Task * t = identifyTaskSender(sender());
	if (t)
	{
		if (KPagerConfigDialog::m_showWindows)
		if (findTaskPager(t->window()))
		if (t->parent() != taskManager())
		if (KWin::windowInfo(t->window()).valid())
		{
			if (t->isOnAllDesktops())
			{
				if (0<(int)pager()->desktopList().count())
				if (pager()->desktopList()[0]->shouldPaintWindow(t)) emitRepaintDesktop(0);
			}
			else
			if (t->desktop()-1<(int)pager()->desktopList().count())
			if (pager()->desktopList()[t->desktop()-1]->shouldPaintWindow(t)) emitRepaintDesktop(t->desktop());
//			emitRepaintDesktop(t->desktop());
		}
		emit updateThumbnail(t);
	}
	//qDebug("KPagerTaskManager::slotActivated() returned");
}

void KPagerTaskManager::slotDeactivated()
{
	//qDebug("KPagerTaskManager::slotDeactivated()");
	Task * t = identifyTaskSender(sender());
	if (t)
	{
		if (KPagerConfigDialog::m_showWindows)
		if (findTaskPager(t->window()))
		if (t->parent() != taskManager())
		if (KWin::windowInfo(t->window()).valid())
		{
			// repaint
			if (t->isOnAllDesktops())
			{
				if (0<pager()->desktopList().count())
				if (pager()->desktopList()[0]->shouldPaintWindow(t)) emitRepaintDesktop(0);
			}
			else
			if (t->desktop()-1<(int)pager()->desktopList().count())
				if (pager()->desktopList()[t->desktop()-1]->shouldPaintWindow(t)) emitRepaintDesktop(t->desktop());
		}
//		emit updateActiveThumbnail();
// check activated only! 
	}
	//qDebug("KPagerTaskManager::slotDeactivated() returned");
}

void KPagerTaskManager::slotThumbnailChanged()
{
	//qDebug("KPagerTaskManager::slotThumbnailChanged()");
	Task * t = identifyTaskSender(sender());
	if (t)
	{
		if (KPagerConfigDialog::m_showWindows)
		if (findTaskPager(t->window()))
		if (KWin::windowInfo(t->window()).valid())
		if (t->parent() != taskManager())
		{
			// repaint
			if (t->isOnAllDesktops())
			{
				if (0<pager()->desktopList().count())
				if (pager()->desktopList()[0]->shouldPaintWindow(t)) emitRepaintDesktop(0);
			}
			else
			if (t->desktop()-1<(int)pager()->desktopList().count())
			if (pager()->desktopList()[t->desktop()-1]->shouldPaintWindow(t)) emitRepaintDesktop(t->desktop());
			//emitRepaintDesktop(t->desktop());
		}
	}
	//qDebug("KPagerTaskManager::slotThumbnailChanged() returned");
}

/*
void KPagerTaskManager::slotUpdateThumbnail()
{
	QGuardedPtr<Task> t = identifyTaskSender(sender());
	if (t)
	{
		if (KWin::windowInfo(t->window()).valid())
			emit updateThumbnail(t);
	}
}

void KPagerTaskManager::slotUpdateThumbnail(WId win)
{
	QGuardedPtr<Task> t = findTaskPager(win);
	if (t)
	{
		if (KWin::windowInfo(t->window()).valid())
			emit updateThumbnail(t);
	}
	t = findTaskMan(win);
	if (t)
	{
		if (KWin::windowInfo(t->window()).valid())
			emit updateThumbnail(t);
	}
}
*/

void KPagerTaskManager::slotUpdateThumbnail(QGuardedPtr<Task> t)
{
	if (!t) return;
	//qDebug("slotUpdateThumbnail(QGuardedPtr<Task> t)");

	// use a timer since we want to get the thumbnail in 0.5 seconds (we have to draw the window in that time first)
	// we don't want thumbnails on slow machines when thumbnails are turned off

	QTimer * timer = new QTimer(this);
	connect (timer, SIGNAL(timeout()), this, SLOT(slotGrabTimersFired()));
	m_grabtimers.append(timer);
	m_grabwindows.append(t);
	timer->start(750, true);
	//qDebug("slotUpdateThumbnail(QGuardedPtr<Task> t) returned");
}

void KPagerTaskManager::slotGrabTimersFired()
{
	//qDebug("KPagerTaskManager::slotGrabTimersFired()");
	// get the associated Task
	const QObject * sender = this->sender();
	if (!sender) return;
	if (!sender->inherits("QTimer")) return;
	if (m_grabtimers.count()!=m_grabwindows.count())
	{
		qWarning("KPagerTaskManager::slotGrabTimersFired(): INTERNAL ERROR: m_grabtimers.count()!=m_grabwindows.count()");
		m_grabtimers.setAutoDelete(true);
		m_grabwindows.setAutoDelete(false);
		m_grabtimers.clear();
		m_grabwindows.clear();
		return;
	}
	QTimer * timer = (QTimer*) sender;
	int index = m_grabtimers.find(timer);
	if (index < 0) // not in list, another timer
	{
		qWarning("KPagerTaskManager::slotGrabTimersFired(): INTERNAL ERROR: sender() not in m_grabtimers list.");
		return;
	}
	Task * t = m_grabwindows.at(index);
	if (!t)
	{
		qWarning("KPagerTaskManager::slotGrabTimersFired(): INTERNAL ERROR: task not window list.");
		return;
	}
	// delete the items out of the lists:
	m_grabtimers.setAutoDelete(true);
	m_grabwindows.setAutoDelete(false);
	m_grabtimers.remove(index);
	m_grabwindows.remove(index);

	//Q_ASSERT(t);
	if (!t) return;
	// is the configuration saying that we can grab the thumbnail?
	if ( KPagerConfigDialog::m_windowDrawMode==KPagerDesktop::Pixmap)
	{
		// if thumbnails at all disabled -> generally return;
		// if thumbnails enabled, but 'show windows' disabled: take thumbnail only for the TaskMenu items
		if ( ( !KPagerConfigDialog::m_showWindows && guardedContains(taskPagerManager(),t) ) ) return;
	}
	else
	{
		//Q_ASSERT((KPagerConfigDialog::m_windowDrawMode==KPagerDesktop::Pixmap));
		return;
	}

	// no check if the window is still valid and then take the thumbnail.

	// is it still valid?
	//Q_ASSERT(KWin::windowInfo(t->window()).valid());
	if (!KWin::windowInfo(t->window()).valid()) return;
	// do nothing if it is not the active task:
	//Q_ASSERT(t->isActive());
	if (!findTaskPager(t->window()) && !findTaskMan(t->window())) return; // not exists anymore 
	if (!t->isActive()) return;
	// is it visible at all?  (this works only if it really is the active window)
	//Q_ASSERT((KWin::windowInfo(t->window()).mappingState() == NET::Visible));
	if (!(KWin::windowInfo(t->window()).mappingState() == NET::Visible)) return;
	if (t->isShaded() || t->isMinimized()) return;
	// check screen saver activity, that means it must be a KDE screen saver to work
	bool screensaverActive = false;
	DCOPClient *client = kapp->dcopClient();
	if (!client->isAttached()) client->attach();
	QByteArray data, data2, replyData;
	QCString replyType;
	if (client->call("kdesktop", "KScreensaverIface", "isBlanked()",
                  data, replyType, replyData))
	{
		QDataStream reply(replyData, IO_ReadOnly);
		if (replyType == "bool")
		{
			reply >> screensaverActive;
		}
	}
	//Q_ASSERT(!screensaverActive);
	if (screensaverActive) return;

	// get the thumbnail
	t->updateThumbnail();
	//qDebug("KPagerTaskManager::slotGrabTimersFired() returned properly");
}

void KPagerTaskManager::slotUpdateActiveThumbnail()
{
	//qDebug("slotUpdateActiveThumbnail()");
	TaskList tl;
	tl = taskManager()->tasks();
	for (Task* t = tl.first(); t != 0; t = tl.next())
	{
		if (!t) continue;
		if (t->isActive())
		{
			emit updateThumbnail(t);
			break;
		}
	}
	GuardedTaskList gtl;
	gtl = taskPagerManager()->tasks();
	for (QGuardedPtr<Task>* gt = gtl.first(); gt != 0; gt = gtl.next())
	{
		if (!gt) continue;
		if (!(*gt)) continue;
		if ((*gt)->isActive())
		{
			emit updateThumbnail(*gt);
			break;
		}
	}
	//qDebug("slotUpdateActiveThumbnail() returned");
}

void KPagerTaskManager::slotRepaintDesktop()
{
	if (!sender()) return;
	//qDebug("KPagerTaskManager::slotRepaintDesktop()");
	QTimer * t;
	if (sender()->inherits("QTimer") || sender()->isA("QTimer")) t = (QTimer*) sender();
	else return;
	int f (m_repainttimers.find(t));
	if ( (f<0) || (f>=(int)m_repainttimers.count())) return; // not one of our timers!

	slotRepaintDesktop(f+1);
	//qDebug("KPagerTaskManager::slotRepaintDesktop() returned");
}

void KPagerTaskManager::slotRepaintDesktop(uint desk)
{
	// at the beginning there is no desktop visible -> do not repaint or there is a segmentation fault
	// (the constructor is calling slottasklAdded which invokes slotRepaintDesktop
	if (!(int)pager()->desktopList().count()) return;
	if (desk > pager()->desktopList().count()) return;

	//qDebug("slotRepaintDesktop(uint desk)");

	if (desk == 0)
	{
		for ( int i=1; i <= (int) pager()->desktopList().count(); i++)
		{
			//if (pager()->desktopList()[i-1]->isVisible())
			pager()->desktopList()[i-1]->redraw();
		}
	}
	else
	{
		if (desk-1<pager()->desktopList().count())
		//if (pager()->desktopList()[desk-1]->isVisible())
		pager()->desktopList()[desk-1]->redraw();
	}
	if (pager()->m_anim)
	if (pager()->m_anim->isVisible()) pager()->m_anim->update();
	//qDebug("slotRepaintDesktop(uint desk) returned");
}

void KPagerTaskManager::slotStackingOrderChanged()
{
	//qDebug("KPagerTaskManager::slotStackingOrderChanged()");
    // m_desktops[m_currentDesktop-1]->m_grabWindows=true;
	// TODO: Eventually regrab windows on the current desktop

	if (KPagerConfigDialog::m_showWindows)
		emitRepaintDesktop(0); // repaint all desktops
	//qDebug("KPagerTaskManager::slotStackingOrderChanged() returned");
}

QGuardedPtr<Task> KPagerTaskManager::identifyTaskSender(const QObject* sender)
{
	if (!sender) return 0;
	if (sender->inherits("Task") || sender->isA("Task")) return (Task*) sender;
	return 0;
}

