/*****************************************************************

Copyright (c) 2004 Sebastian Wolff
Copyright (c) 2001 Matthias Elter <elter@kde.org>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <math.h>

#include <qlayout.h>
#include <qpopupmenu.h>
#include <qstringlist.h>
#include <qtoolbutton.h>

#include <kdebug.h>
#include <kglobal.h>
#include <kconfig.h>
#include <kstandarddirs.h>
#include <klocale.h>

#include "taskbar.h"
//#include "taskbar.moc"
#include "taskcontainer.h"
#include "kpagerrmbmenu.h"
#include "kpagertaskcontainer.h"
#include "kpager.h"
#include "config.h"

//TaskManager *TaskBar::manager = 0;

#define WINDOWLISTBUTTON_SIZE 12

#define BUTTON_MIN_HEIGHT (THUMBNAIL_HEIGHT+2)
#define BUTTON_HEIGHT (THUMBNAIL_HEIGHT+4)
#define BUTTON_MAX_WIDTH 300
#define BUTTON_MIN_WIDTH (THUMBNAIL_WIDTH+4 - 50)
//#define BUTTON_MIN_WIDTH BUTTON_MAX_WIDTH

TaskBar::TaskBar( KPager2 * pager, QPopupMenu * menu, int desktop, QWidget *parent, const char *name )
    : QObject( parent, name )
{

	m_pager = pager;
	m_menu = menu;
	m_menuidx = m_menu->count(); // this is the index of the first container! (if nothing above will change in m_menu)
	m_desktop = desktop;
//	manager = tm;
	arrowType = LeftArrow;
	blocklayout = TRUE;

    // init
	//containers = new TaskContainerList();
	containers.setAutoDelete( FALSE );

    // setup animation frames
	frames = new PixmapList();
	frames->setAutoDelete( true );

	for ( int i=1; i<11; i++ )
		frames->append( new QPixmap( locate("data", "kicker/pics/disk" + QString::number(i) + ".png") ) );

    // configure
	configure();

    // connect manager
	connect( taskManager(), SIGNAL( taskAdded( Task* ) ), SLOT( add( Task* ) ) );
	connect( taskManager(), SIGNAL( taskRemoved( Task* ) ), SLOT( remove( Task* ) ) );
	connect( taskManager(), SIGNAL( startupAdded( Startup* ) ), SLOT( add( Startup* ) ) );
	connect( taskManager(), SIGNAL( startupRemoved( Startup* ) ), SLOT( remove( Startup* ) ) );
//    connect( taskManager(), SIGNAL( desktopChanged( int ) ), SLOT( desktopChanged( int ) ) );
	connect( taskManager(), SIGNAL( windowChanged( WId ) ), SLOT( windowChanged( WId ) ) );

	isGrouping = shouldGroup();

    // register existant tasks
	TaskList tasks = taskManager()->tasks();
	for (Task* t = tasks.first(); t!=0; t = tasks.next())
		add( t );

    // register existant startups
	StartupList startups = taskManager()->startups();
	for (Startup* s = startups.first(); s!=0; s = startups.next())
		add( s );

	blocklayout = FALSE;

	connect (m_menu, SIGNAL(aboutToShow()), this, SLOT(menuAboutToShow()));
}

TaskBar::~TaskBar()
{
	delete frames;
//	delete taskManager();
//	disconnect(0,0,this,0);
}

/*
int _activeTaskContainer(TaskContainerList & l) // returns the index
{
	for ( KPagerTaskContainer* t = l.first(); t ; t = l.next() )
	{
		if (t->isOn()) return l.find(t);
	}
	return -1;
}
*/

// These are the strings that are actually stored in the config file.
// They must be synchronized with kcmtaskbar. They are not translated.
const QStringList& TaskBar::actionList()
{
	static QStringList list(
            QStringList() << "Show Task List" << "Show Operations Menu"
            << "Activate, Raise or Minimize Task"
            << "Activate Task" << "Raise Task"
            << "Lower Task" << "Minimize Task" );
	return list;
}

// Translate from config entry name to enumeration
enum TaskBar::Action TaskBar::buttonAction( ButtonState button, const QString& actionName )
{
	int index = actionList().findIndex( actionName );
	if( index != -1 ) return static_cast<Action>(index);

   // Otherwise return the default.
	switch( button )
	{
		case MidButton: return ActivateRaiseOrIconify;
		case RightButton: return ShowOperationsMenu;
		case LeftButton: // fall through
		default: return ShowTaskList;
	}
}

// These are the strings that are actually stored in the config file.
const QStringList& TaskBar::groupModeList()
{
	static QStringList list(
            QStringList() << I18N_NOOP("Never") << I18N_NOOP("When Taskbar Full")
            << I18N_NOOP("Always"));
	return list;
}

// Translate from config entry name to enumeration
enum TaskBar::GroupMode TaskBar::groupMode(const QString& groupModeName )
{
	int index = groupModeList().findIndex( groupModeName );
	if( index != -1 ) return static_cast<GroupMode>(index);

	// Translate old entries
	if( groupModeName == "true" ) return GroupAlways;
	if( groupModeName == "false" ) return GroupNever;

	// Otherwise return the default.
	return GroupWhenFull;
}

void TaskBar::configure()
{
//	setAutoDelete(false);

	showAllWindows = (bool) (KPagerConfigDialog::m_showOnlyOneDesktop);
	showOnlyIconified = false;
	currentGroupMode = GroupWhenFull;
	// groupWhenFull doesn't work. Since we compute the size before the widget is shown it would be grouped in any case
	sortByDesktop = true;
	showIcon = true;
	maxButtonWidth = BUTTON_MAX_WIDTH;

	for ( TaskContainerIterator it(containers); it.current(); ++it )
	{
		KPagerTaskContainer* container = it.current();
		container->setShowAll( showAllWindows );
		container->setSortByDesktop( sortByDesktop );
	}

	Action leftButtonAction = ShowTaskList;
	Action middleButtonAction = ActivateRaiseOrIconify;
	Action rightButtonAction = ShowOperationsMenu;

	KPagerTaskContainer::setLeftButtonAction( leftButtonAction );
	KPagerTaskContainer::setMiddleButtonAction( middleButtonAction );
	KPagerTaskContainer::setRightButtonAction( rightButtonAction );

	if ( !blocklayout ) reLayout();
}

TaskManager* TaskBar::taskManager()
{
/*
	if ( !manager )
		manager = new TaskManager( this );
	return manager;
*/
	return m_pager->pagerTaskManager()->taskManager();
}

void TaskBar::add( Task* task )
{
	if ( !task ) return;

//	if ( !showAllWindows   &&   !(task->desktop() == 0 || task->isOnAllDesktops())  &&   task->desktop() != m_desktop) return;

	// try to group
	// TODO: grouping is currently not supported
	if ( isGrouping )
	{
		for ( TaskContainerIterator it(containers); it.current(); ++it )
		{
			KPagerTaskContainer* c = it.current();

			if ( idMatch( task->classClass().lower(), ((TaskContainer*)c)->id().lower() ) )
			{
				c->add( task );
				if ( !blocklayout ) reLayout();
				return;
			}
		}
	}

	// create new container
	KPagerTaskContainer *c = new KPagerTaskContainer( task, taskManager(), showAllWindows, sortByDesktop, showIcon, showOnlyIconified, m_menu);
//    addChild( c );
	if (c)
	{
		int id = m_menu->insertItem((QCustomMenuItem*)c, -1, m_menuidx+containers.count());
		c->setId(id);
		m_menu->connectItem(id, c, SLOT(slotActivated()));
	}
	containers.append( c );

	if ( !blocklayout )
	{
		emit containerCountChanged();
		reLayout();
	}
}

void TaskBar::add( Startup* startup )
{
	if ( !startup ) return;

	// try to group
	// TODO: grouping is currently not supported
	if ( isGrouping )
	{
		for ( TaskContainerIterator it(containers); it.current(); ++it )
		{
			KPagerTaskContainer* c = it.current();
			if ( idMatch( startup->text().lower(), ((TaskContainer*)c)->id().lower() ) )
			{
				c->add( startup );
				if ( !blocklayout ) reLayout();
				return;
			}
		}
	}

	// create new container
	KPagerTaskContainer *c = new KPagerTaskContainer( startup, frames, taskManager(), showAllWindows, sortByDesktop, showIcon, showOnlyIconified, m_menu);
//    addChild( c );
	if (c)
	{
		int id = m_menu->insertItem((QCustomMenuItem*)c, -1, m_menuidx+containers.count());
		c->setId(id);
	}
	containers.append( c );

	if ( !blocklayout )
	{
		emit containerCountChanged();
		reLayout();
	}
}

void TaskBar::remove( Task *task )
{
	TaskContainerList killList;
	for ( TaskContainerIterator it(containers); it.current(); ++it )
	{
		KPagerTaskContainer* c = it.current();
		if ( c->contains( task ) )
		{
			c->remove( task );
			if ( c->isEmpty() )
				killList.append( c );
			break;
		}
	}

	killList.setAutoDelete(false);
	containers.setAutoDelete(false);

	for ( TaskContainerIterator it(killList); it.current(); ++it )
	{
		KPagerTaskContainer* c = it.current();
		//containers.removeRef( c );
		containers.remove( c );
		//c->deleteLater();
		m_menu->removeItem(c->id());
	}
	if ( !blocklayout )
	{
		emit containerCountChanged();
		reLayout();
	}
}

void TaskBar::remove( Startup *startup )
{
	bool found = false;
	TaskContainerList killList;
	for ( TaskContainerIterator it(containers); it.current(); ++it )
	{
		KPagerTaskContainer* c = it.current();
		if ( c->contains( startup ) )
		{
			c->remove( startup );
			found = true;
			if ( c->isEmpty() )
				killList.append( c );
			break;
		}
	}

	killList.setAutoDelete(false);
	containers.setAutoDelete(false);

    for ( TaskContainerIterator it(killList); it.current(); ++it )
	{
		KPagerTaskContainer* c = it.current();
		containers.remove(c);
		m_menu->removeItem(c->id());
	}

    if (found && !blocklayout )
    {
        emit containerCountChanged();
		reLayout();
    }
}

void TaskBar::desktopChanged( int desktop )
{
	m_desktop = desktop;
	if ( !showAllWindows )
	{
		for ( TaskContainerIterator it(containers); it.current(); ++it )
		{
			KPagerTaskContainer* c = it.current();
			c->desktopChanged( desktop );
		}
		emit containerCountChanged();
		reLayout();
	}
	// TODO: remove and add now hidden/visible containers in popupmenu
}

void TaskBar::windowChanged( WId win )
{
	for ( TaskContainerIterator it(containers); it.current(); ++it )
	{
		KPagerTaskContainer* c = it.current();
		if ( c->contains( win ) )
			c->windowChanged( win );
    }

	if ( !showAllWindows )
		emit containerCountChanged();
	reLayout();
}

void TaskBar::reLayout()
{
// printf("relayout\n");
    // filter task container list
	TaskContainerList list = filteredContainers();

	if ( isGrouping != shouldGroup() )
	{
		reGroup();
		return;
	}

    // sort container list by desktop
	if ( sortByDesktop )
	if ( showAllWindows ) // for performance (note: now in singleDesktopMode there sticky windows aren't sorted)
	{
		blocklayout = true;
		// 1st: get a list with the sorted items
		TaskContainerList sorted;
		TaskContainerList residue = list;
		for ( int desktop = -1; desktop <= taskManager()->numberOfDesktops(); desktop++ )
		{
			for ( TaskContainerIterator it(list); it.current(); ++it )
			{
				KPagerTaskContainer* c = it.current();
				if ( c->desktop() == desktop )
				{
					sorted.append( c );
					residue.remove( c );
				}
			}
		}
		list = sorted;
		// 1st (b): add those items that couldn't be sorted (just for sanity, shouldn't be possible)
		for ( KPagerTaskContainer *c = residue.first(); c; c = residue.next() )
			list.append( c );

		// 2nd reorder the list
/*
		// 2nd (a) find the index in the menu for the first container item
		int ind = m_menu->count();
		for ( TaskContainerIterator it(list); it.current(); ++it )
		{
			KPagerTaskContainer* c = it.current();
			if (m_menu->isItemVisible(c->id()))
			if (m_menu->indexOf(c->id()) != -1)
				ind = QMIN(ind,m_menu->indexOf(c->id()));
		}
		// at this index (ind) we have to insert all our containers
		// 2nd (b) change the indices of the menu items:
*/
		int i = 0;
		for ( TaskContainerIterator it(list); it.current(); ++it)
		{
			KPagerTaskContainer* c = it.current();
			if (m_menu->isItemVisible(c->id()))
			{
				// save the tasks:
				TaskList tasks = c->getTasks();
				StartupList startups = c->getStartups();
				int id = c->id();
				// remove item:
				containers.remove(c);
				m_menu->removeItem(id);
				// reinsert item (also check grouped tasks) ! case: task item
				for ( Task* task = tasks.first(); task ; task = tasks.next() )
				{
					add(task);
				}
				// reinsert item (also check grouped tasks) ! case: startup item
				for ( Startup* startup = startups.first(); startup ; startup = startups.next() )
				{
					add(startup);
				}
				i++;
			}
		}
		blocklayout = false;
	}
	//	eventually TODO: publishIconGeometry (that means tell KWin where to do the animation if a window was minimized)
	//	QTimer::singleShot( 100, this, SLOT( publishIconGeometry() ) );
// printf("return reLayout();\n");
}

void TaskBar::setArrowType( Qt::ArrowType at )
{
	arrowType = at;
	for ( TaskContainerIterator it(containers); it.current(); ++it )
	{
		KPagerTaskContainer* c = it.current();
		c->setArrowType( arrowType );
	}
}



bool TaskBar::idMatch( const QString& id1, const QString& id2 )
{
	if ( id1.isEmpty() || id2.isEmpty() )
		return FALSE;

	return id1.lower() == id2.lower();
}

int TaskBar::containerCount() const
{
	int i = 0;
	TaskContainerIterator it(containers);
	for ( ; it.current(); ++it )
	{
		if ( showAllWindows ||  (*it)->onCurrentDesktop() )
		{
			i++;
		}
	}
	return i;
}

int TaskBar::taskCount() const
{
	int i = 0;
	TaskContainerIterator it(containers);
	for ( ; it.current(); ++it )
	{
		if ( showAllWindows || (*it)->onCurrentDesktop() )
		{
			i += (*it)->filteredTaskCount();
		}
	}
	return i;
}


bool TaskBar::shouldGroup() const
{
	bool b = currentGroupMode == GroupAlways;
	if (currentGroupMode == GroupWhenFull)
	{
// NOTE TODO: this hack means that once groupWhenFull is activated, it can't be made undone.
		if (isGrouping) b = true; else

		if (m_menu->inherits("KPagerRMBMenu") || m_menu->className()=="KPagerRMBMenu")
			if (((KPagerRMBMenu*)m_menu)->columns()>1) b =true;
//printf("shouldGroup() %d\n",b);
	}
	return b;
}

void TaskBar::reGroup()
{
//printf("regroup\n");
    isGrouping = shouldGroup();
    blocklayout = TRUE;

    TaskList tasks = taskManager()->tasks();
	for (Task* t = tasks.first(); t!=0; t = tasks.next())
		remove( t );

    StartupList startups = taskManager()->startups();
	for (Startup* s = startups.first(); s!=0; s = startups.next())
		remove( s );

	for (Task* t = tasks.first(); t!=0; t = tasks.next())
		add( t );

	for (Startup* s = startups.first(); s!=0; s = startups.next())
		add( s );

    blocklayout = FALSE;
    reLayout();
//printf("return regroup\n");
}

TaskContainerList TaskBar::filteredContainers()
{
	// filter task container list
	TaskContainerList list;
	for ( TaskContainerIterator it(containers); it.current(); ++it )
	{
		KPagerTaskContainer* c = it.current();
		if ( (showAllWindows || c->onCurrentDesktop())
		  && (!showOnlyIconified || c->isIconified()) )
		{
			list.append( c );
			m_menu->setItemVisible(c->id(), true);
		}
		else m_menu->setItemVisible(c->id(), false);
	}
	return list;
}

void TaskBar::menuAboutToShow()
{
//printf("1 menuAboutToShow(): count %d\tcols %d\theight %d\n", m_menu->count(), ((KPagerRMBMenu*)m_menu)->columns(), m_menu->height());
	if ( isGrouping != shouldGroup() )
	{
//printf("2 menuAboutToShow(): count %d\tcols %d\theight %d\n", m_menu->count(), ((KPagerRMBMenu*)m_menu)->columns(), m_menu->height());
		reGroup();
	}
}
TaskContainerList TaskBar::containerList(){return containers;}

	
