/*****************************************************************

Copyright (c) 2004 Sebastian Wolff

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kiconloader.h>
#include <kwinmodule.h>
#include <netwm.h>
#include <qtimer.h>
#include <qimage.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "taskpagerman.h"
#include "taskmanager.h"

extern KWinModule* kwin_module;

TaskPagerMan::TaskPagerMan(QObject *parent, const char *name)
    : QObject(parent, name), _active(0), _startup_info( NULL )
{
    if ( kwin_module == NULL )
        kwin_module = new KWinModule();

//    KGlobal::locale()->insertCatalogue("libTaskPagerMan");
    connect(kwin_module, SIGNAL(windowAdded(WId)), SLOT(windowAdded(WId)));
    connect(kwin_module, SIGNAL(windowRemoved(WId)), SLOT(windowRemoved(WId)));
    connect(kwin_module, SIGNAL(activeWindowChanged(WId)), SLOT(activeWindowChanged(WId)));
    connect(kwin_module, SIGNAL(currentDesktopChanged(int)), SLOT(currentDesktopChanged(int)));
    connect(kwin_module, SIGNAL(windowChanged(WId,unsigned int)), SLOT(windowChanged(WId,unsigned int)));

    // register existing windows
    const QValueList<WId> windows = kwin_module->windows();
    for (QValueList<WId>::ConstIterator it = windows.begin(); it != windows.end(); ++it )
	windowAdded(*it);

    // set active window
    WId win = kwin_module->activeWindow();
    activeWindowChanged(win);

    configure_startup();
	_tasks.setAutoDelete(false);
}

TaskPagerMan::~TaskPagerMan()
{
}

void TaskPagerMan::configure_startup()
{
    KConfig c("klaunchrc", true);
    c.setGroup("FeedbackStyle");
    if (!c.readBoolEntry("TaskbarButton", true))
        return;
    _startup_info = new KStartupInfo( KStartupInfo::CleanOnCantDetect, this );
    connect( _startup_info,
        SIGNAL( gotNewStartup( const KStartupInfoId&, const KStartupInfoData& )),
        SLOT( gotNewStartup( const KStartupInfoId&, const KStartupInfoData& )));
    connect( _startup_info,
        SIGNAL( gotStartupChange( const KStartupInfoId&, const KStartupInfoData& )),
        SLOT( gotStartupChange( const KStartupInfoId&, const KStartupInfoData& )));
    connect( _startup_info,
        SIGNAL( gotRemoveStartup( const KStartupInfoId&, const KStartupInfoData& )),
        SLOT( gotRemoveStartup( const KStartupInfoId& )));
    c.setGroup( "TaskbarButtonSettings" );
    _startup_info->setTimeout( c.readUnsignedNumEntry( "Timeout", 30 ));
}

QGuardedPtr<Task> * TaskPagerMan::findTask(WId w)
{
	for (QGuardedPtr<Task>* t = _tasks.first(); t != 0; t = _tasks.next())
	{
		if (!t) return 0;
		if (!(*t)) return 0;
        if ((*t)->window() == w  || (*t)->hasTransient(w))
            return t;
	}
    return 0;
}

void TaskPagerMan::windowAdded(WId w )
{
    NETWinInfo info (qt_xdisplay(),  w, qt_xrootwin(),
		     NET::WMWindowType | NET::WMPid | NET::WMState );

    // ignore NET::Tool and other special window types
    NET::WindowType wType = info.windowType( NET::NormalMask | NET::DesktopMask | NET::DockMask
	| NET::ToolbarMask | NET::MenuMask | NET::DialogMask | NET::OverrideMask
	| NET::TopMenuMask | NET::UtilityMask | NET::SplashMask );
    if ( wType != NET::Normal
        && wType != NET::Override
        && wType != NET::Unknown
        && wType != NET::Dialog
	&& wType != NET::Utility)
	return;

    // ignore windows that want to be ignored by the taskbar
    if ((info.state() & NET::SkipPager) != 0) {
        _skiptaskbar_windows.push_front( w ); // remember them though
	return;
    }

    //Task* t = new Task(w, this);
	QGuardedPtr<Task> * gt = new QGuardedPtr<Task>();
	if (gt)
	{
		*gt = new Task(w,0);
		if (*gt)
		{
			//insertChild(gt);
			_tasks.append(gt);
			emit taskAdded(*gt);
		} else delete gt;
	}

    // kdDebug() << "TM: Task added for WId: " << w << endl;

}

void TaskPagerMan::windowRemoved(WId w )
{
	/*while*/ (_skiptaskbar_windows.remove( w ));
	// find task
	QGuardedPtr<Task>* t=findTask(w);
	if (!t) return;
	if (!(*t)) { return; }

	if ((*t)->window() == w) 
	{
		while (_tasks.removeRef(t)) ;
		emit taskRemoved(*t); // means that we have to check if t is still in the list, if not, it is removed (not, if it is 0)

		if(*t == _active) _active = 0;
		delete (*t);
		delete t;
		//kdDebug() << "TM: Task for WId " << w << " removed." << endl;
	}
	else {
		(*t)->removeTransient( w );
        //kdDebug() << "TM: Transient " << w << " for Task " << t->window() << " removed." << endl;
	}
}

void TaskPagerMan::windowChanged(WId w, unsigned int dirty)
{
    if( dirty & NET::WMState ) {
        NETWinInfo info ( qt_xdisplay(),  w, qt_xrootwin(), NET::WMState );
        if ( (info.state() & NET::SkipTaskbar) != 0 ) {
            windowRemoved( w );
            _skiptaskbar_windows.push_front( w );
            return;
        }
        else {
            _skiptaskbar_windows.remove( w );
            if( !findTask( w ))
                windowAdded( w ); // skipTaskBar state was removed, so add this window
        }
    }

    // check if any state we are interested in is marked dirty
    if(!(dirty & (NET::WMVisibleName|NET::WMName|NET::WMVisibleIconName|NET::WMIconName|NET::WMState
                    |NET::WMIcon|NET::XAWMState|NET::WMDesktop)) )
        return;

    // find task
    QGuardedPtr<Task> * t = findTask( w );
    if (!t) return;
    if (!(*t)) return;

    //kdDebug() << "TaskPagerMan::windowChanged " << w << " " << dirty << endl;

    if( dirty & NET::WMState )
        (*t)->updateDemandsAttentionState( w );

    // refresh icon pixmap if necessary
    if (dirty & NET::WMIcon)
        (*t)->refresh(true);
    else
        (*t)->refresh();

    if(dirty & (NET::WMDesktop|NET::WMState|NET::XAWMState))
        emit windowChanged(w); // moved to different desktop or is on all or change in iconification/withdrawnnes
}

void TaskPagerMan::activeWindowChanged(WId w )
{
    //kdDebug() << "TaskPagerMan::activeWindowChanged" << endl;

    QGuardedPtr<Task> * t = findTask( w );
	if (t) if (!(*t)) t=0;
    if (!(t)) {
        if (_active) {
            _active->setActive(false);
            _active = 0;
        }
    }
    else {
        if (_active)
            _active->setActive(false);

        _active = *t;
        _active->setActive(true);
    }
}

void TaskPagerMan::currentDesktopChanged(int desktop)
{
    emit desktopChanged(desktop);
}

void TaskPagerMan::gotNewStartup( const KStartupInfoId& id, const KStartupInfoData& data )
{
    Startup* s = new Startup( id, data, this );
    _startups.append(s);

    emit startupAdded(s);
}

void TaskPagerMan::gotStartupChange( const KStartupInfoId& id, const KStartupInfoData& data )
{
    for( Startup* s = _startups.first(); s != 0; s = _startups.next()) {
        if ( s->id() == id ) {
            s->update( data );
            return;
        }
    }
}

void TaskPagerMan::gotRemoveStartup( const KStartupInfoId& id )
{
    killStartup( id );
}

void TaskPagerMan::killStartup( const KStartupInfoId& id )
{
    Startup* s = 0;
    for(s = _startups.first(); s != 0; s = _startups.next()) {
        if (s->id() == id)
            break;
    }
    if (s == 0) return;

    _startups.removeRef(s);
    emit startupRemoved(s);
    delete s;
}

void TaskPagerMan::killStartup(Startup* s)
{
    if (s == 0) return;

    _startups.removeRef(s);
    emit startupRemoved(s);
    delete s;
}

QString TaskPagerMan::desktopName(int desk) const
{
    return kwin_module->desktopName(desk);
}

int TaskPagerMan::numberOfDesktops() const
{
    return kwin_module->numberOfDesktops();
}

bool TaskPagerMan::isOnTop(const QGuardedPtr<Task> task)
{
    if(!task) return false;

    for (QValueList<WId>::ConstIterator it = kwin_module->stackingOrder().fromLast();
         it != kwin_module->stackingOrder().end(); --it ) {
	for (QGuardedPtr<Task>* t = _tasks.first(); t != 0; t = _tasks.next() ) {
	    if ( (*it) == (*t)->window() ) {
		if ( *t == task )
		    return true;
		if ( !(*t)->isIconified() && ((*t)->isAlwaysOnTop() == task->isAlwaysOnTop()) )
		    return false;
		break;
	    }
	}
    }
    return false;
}

int TaskPagerMan::currentDesktop() const
{
    return kwin_module->currentDesktop();
}


