/*****************************************************************

Copyright (c) 2004 Sebastian Wolff

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#ifndef __taskpagerman_h__
#define __taskpagerman_h__

#include "version.h"

#include <sys/types.h>

#include <qguardedptr.h>
#include <qpoint.h>
#include <qobject.h>
#include <qvaluelist.h>
#include <qptrlist.h>
#include <qpixmap.h>

#include <dcopobject.h>
#include <kwin.h>
#include <kstartupinfo.h>

#include "taskmanager.h"

typedef QPtrList < QGuardedPtr < Task > > GuardedTaskList;
//typedef QPtrList<Startup> StartupList;

/**
 * A generic API for task managers. This class provides an easy way to
 * build NET compliant task managers. It provides support for startup
 * notification, virtual desktops and the full range of WM properties.
 *
 * In difference to TaskManager the class TaskPagerMan offers a task
 * manager for pagers. That means that windows with the flag NET::SkipPager
 * instead of NET::SkipTaskManager are ignored as well as transients 
 * are not supported. The parent of the contain Task objects will be '0'
 * (the task manager class in TaskManager). All tasks are saved as QGuardedPtr 
 *
 * @see TaskManager
 * @see Task
 * @see Startup
 * @see KWinModule
 * @version $Id: taskmanager.h,v 1.46 2003/09/29 11:24:01 lunakl Exp $
 */
class TaskPagerMan : public QObject  
{
    Q_OBJECT
    Q_PROPERTY( int currentDesktop READ currentDesktop )
    Q_PROPERTY( int numberOfDesktops READ numberOfDesktops )

public:
    TaskPagerMan( QObject *parent = 0, const char *name = 0 );
    virtual ~TaskPagerMan();

    /**
     * Returns a list of all current tasks. Return type changed to
     * QPtrList in KDE 3.
     */
    GuardedTaskList tasks() const { return _tasks; }

    /**
     * Returns a list of all current startups. Return type changed to
     * QPtrList in KDE 3.
     */
    StartupList startups() const { return _startups; }

    /**
     * Returns the name of the nth desktop.
     */
    QString desktopName(int n) const;

    /**
     * Returns the number of virtual desktops.
     */
    int numberOfDesktops() const;

    /**
     * Returns the number of the current desktop.
     */
    int currentDesktop() const;

    /**
     * Returns true if the specified task is on top.
     */
    bool isOnTop( const QGuardedPtr<Task>);
signals:
    /**
     * Emitted when a new task has started.
     */
    void taskAdded(QGuardedPtr<Task>);

    /**
     * Emitted when a task has terminated.
     */
    void taskRemoved(QGuardedPtr<Task>);

    /**
     * Emitted when a new task is expected.
     */
    void startupAdded(Startup*);

    /**
     * Emitted when a startup item should be removed. This could be because
     * the task has started, because it is known to have died, or simply
     * as a result of a timeout.
     */
    void startupRemoved(Startup*);

    /**
     * Emitted when the current desktop changes.
     */
    void desktopChanged(int desktop);

    /**
     * Emitted when a window changes desktop.
     */
    void windowChanged(WId);

protected slots:
    //* @internal
    void windowAdded(WId);
    //* @internal
    void windowRemoved(WId);
    //* @internal
    void windowChanged(WId, unsigned int);

    //* @internal
    void activeWindowChanged(WId);
    //* @internal
    void currentDesktopChanged(int);
    //* @internal
    void killStartup( const KStartupInfoId& );
    //* @internal
    void killStartup(Startup*);

    //* @internal
    void gotNewStartup( const KStartupInfoId&, const KStartupInfoData& );
    //* @internal
    void gotStartupChange( const KStartupInfoId&, const KStartupInfoData& );
    //* @internal
    void gotRemoveStartup( const KStartupInfoId& );

protected:
    /**
     * Returns the task for a given WId, or 0 if there is no such task.
     */
    QGuardedPtr<Task>* findTask(WId w);
    void configure_startup();

private:
    QGuardedPtr<Task>   _active;
    GuardedTaskList     _tasks;
    QValueList< WId >   _skiptaskbar_windows;
    StartupList         _startups;
    KStartupInfo*       _startup_info;

    class TaskManagerPrivate *d;
};

#endif
