/***************************************************************************
 *   Copyright (C) 2006 by the KSmoothDock team   *
 *   dangvd@yahoo.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "iconbaseddockitem.h"

#include <iostream>

#include <qimage.h>

#include <kglobal.h>
#include <kiconloader.h>

/// PUBLIC ///

/** 
 * Constructor
 */

IconBasedDockItem::IconBasedDockItem() {}

/** 
 * Constructor
 * @param parent the parent dock
 * @param itemId id of the item
 * @param desc description of the dock item
 * @param desktop the desktop the dock item is on
 * @param icon the icon represents the dock item
 * @param minSize the minimum size (height or width depending on the position of the dock) of the dock item's icon
 * @param maxSize the maximum size (height or width depending on the position of the dock) of the dock item's icon
 * @param orientation orienation of the dock
 */

IconBasedDockItem::IconBasedDockItem(KSmoothDock* parent, int itemId, QString desc, int desktop, QPixmap& icon, int minSize, int maxSize, Qt::Orientation orientation) 
 : DockItem(parent, itemId, desc, desktop, orientation), m_minSize(minSize), m_maxSize(maxSize) {
    generateIcons(icon);
}

/** 
 * Constructor
 * @param parent the parent dock
 * @param itemId id of the item
 * @param desc description of the dock item
 * @param desktop the desktop the dock item is on
 * @param iconName the name of the icon that represents the dock item
 * @param baseSize the original size of the icon to load
 * @param minSize the minimum size of the dock item's icon
 * @param maxSize the maximum size of the dock item's icon
 * @param orientation orienation of the dock
 */

IconBasedDockItem::IconBasedDockItem(KSmoothDock* parent, int itemId, QString desc, int desktop, QString iconName, int baseSize, int minSize, int maxSize, Qt::Orientation orientation)
 : DockItem(parent, itemId, desc, desktop, orientation), m_minSize(minSize), m_maxSize(maxSize) {
    KIconLoader* iLoader = KGlobal::iconLoader();
    QPixmap icon = iLoader->loadIcon(iconName, KIcon::NoGroup, baseSize);
    generateIcons(icon);
}

/**
 * Get the icon with a specific height
 * @param height the height of the icon to return
 */

QPixmap& IconBasedDockItem::getIcon(int size) {
    int s = size;

    // if height is out of range, adjust it
    if (s < m_minSize) {
        s = m_minSize;
    } else if (s > m_maxSize) {
        s = m_maxSize;
    }

    return m_icons.at(s - m_minSize);
}

/**
 * Set the base icon
 * @param icon the base icon
 */

void IconBasedDockItem::setIcon(QPixmap& icon) {
    m_icons.clear(); // clear the vector
    generateIcons(icon); // re-generate the icons
}

/** 
 * Draw itself into the offscreen buffer
 * @param buffer the offscreen buffer
 * @param x x-coordinate to start drawing
 * @param y y-coordinate to start drawing
 * @param size size of the dock item icon (width or height depending on the orientation)
 */

void IconBasedDockItem::draw(QPixmap& buffer, int x, int y, int size) {
    if (size >= m_minSize && size <= m_maxSize)
        bitBlt(&buffer, x, y, &m_icons.at(size - m_minSize));
}

/** 
 * Get max width
 */

int IconBasedDockItem::getMaxWidth() {
    return getIcon(m_maxSize).width();        
}

/** 
 * Get max height
 */

int IconBasedDockItem::getMaxHeight() {
    return getIcon(m_maxSize).height();
}

/** 
 * Get min width
 */

int IconBasedDockItem::getMinWidth() {
    return getIcon(m_minSize).width();        
}

/** 
 * Get min height
 */

int IconBasedDockItem::getMinHeight() {
    return getIcon(m_minSize).height();
}

/**
 * Get width for a specific size
 */

int IconBasedDockItem::getWidth(int size) {
    return getIcon(size).width();
}
    
/**
 * Get height for a specific size
 */

int IconBasedDockItem::getHeight(int size) {
    return getIcon(size).height();
}

/// PRIVATE ///

/** 
 * Generate icons with different sizes
 * @param icon the base icon
 */

void IconBasedDockItem::generateIcons(QPixmap& icon) {
    // generate copies of the icon with different sizes and put them into m_icons vector
    int original_width = icon.width();
    int original_height = icon.height();

    QImage img = icon.convertToImage(); // convert to QImage to enable scaling

    if (m_orientation == Qt::Horizontal) {
        for (int h = m_minSize; h <= m_maxSize; h++) {
            QPixmap pix = img.smoothScale(h * original_width / original_height, h); // scale the icon
            m_icons.push_back(pix); // put the icon into the vector
        }
    } else { // Vertical
        for (int w = m_minSize; w <= m_maxSize; w++) {
            QPixmap pix = img.smoothScale(w, w * original_height / original_width); // scale the icon
            m_icons.push_back(pix); // put the icon into the vector
        }
    }
}

