/***************************************************************************
 *   Copyright (C) 2004 by Michał Kosmulski                                *
 *   mkosmul <at> users <dot> sourceforge <dot> net                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>
#include <kapplication.h>
#include <kfileitem.h>
#include <kmessagebox.h>
#include <kaboutapplication.h>
#include <qimage.h>
#include <qtooltip.h>
#include <qgrid.h>
#include <qlayout.h>
#include <kiconloader.h>
#include <kstandarddirs.h>
#include <kurlrequester.h>
#include <kurl.h>
#include <kglobal.h>
#include <kprocess.h>
#include <kio/netaccess.h>

#include "kuote.h"
#include "prefdialog.h"
#include "prefanimation.h"
#include "prefprogram.h"
#include "preffortunes.h"
#include "fortunedialog.h"
#include "clickablelabel.h"
#include "configuration.h"
#include "config.h"

kuote::kuote(const QString& configFile, Type type, int actions, QWidget *parent, const char *name)
    : KPanelApplet(configFile, type, actions, parent, name)
{
    // Get the current application configuration handle
    ksConfig = config();
    readConfig();
    
    aboutData = new KAboutData("kuote", I18N_NOOP("Kuote"), VERSION,
    	I18N_NOOP("An applet for displaying quotes from fortune(6)."),
    	KAboutData::License_GPL, "Copyright (c) 2004, Michal Kosmulski",
	I18N_NOOP("This kicker applet was inspired by Gnome's Wanda the Fish.\n"
	"It displays quotes generated by fortune(6) when the animated\n"
	"fish (or other character you choose) is clicked."),
	"http://hektor.umcs.lublin.pl/~mikosmul/",
	"mkosmul <at> users <dot> sourceforge <dot> net");
    aboutData->addAuthor("Michal Kosmulski",I18N_NOOP("Developer"),
    	"mkosmul <at> users <dot> sourceforge <dot> net",
	"http://hektor.umcs.lublin.pl/~mikosmul/");
    aboutData->addCredit(I18N_NOOP("The Gnome team"),I18N_NOOP("Creators of the original Fish Applet and animations."),
    	0,"http://www.gnome.org/");
    aboutData->addCredit("Mark Riedesel [Klowner]",I18N_NOOP("Author of the \"See, Hear, Speak\" penguin animations.\n"
    	"These animations as well as the original image are available under\nthe CC-BY-NC-SA license (see http://creativecommons.org/licenses/by-nc-sa/2.0/)."),
    	"klown <at> revealed <dot> net","http://www.klowner.com/");

    refreshTimerId = 0;
    frameWidth = 1;
    frameHeight = 1;
    animationFrames = new QValueList<QPixmap>;
    
    QGridLayout *grid = new QGridLayout(this);
    
    image = new ClickableLabel(this);
    image->setSizePolicy(QSizePolicy::Ignored,QSizePolicy::Ignored);
    image->setScaledContents(false);

    connect(image,SIGNAL(clicked()),this,SLOT(showFortune()));
    connect(image,SIGNAL(rightClicked()),this,SLOT(askFortune()));
    
    //askFortune();
    restartAnimation();
    
    grid->addWidget(image,0,0);
    mainView = image;
}


kuote::~kuote()
{
	killTimer(refreshTimerId);
	writeConfig();
	KGlobal::locale()->removeCatalogue("kuote");
	delete aboutData;
	delete animationFrames;
}


void kuote::about()
{
    KAboutApplication dlg(aboutData,this,"AboutBox",true);
    //dlg->setImage(KGlobal::dirs()->findResource("icon","kuote"));
    dlg.exec();
}


void kuote::help()
{
    KMessageBox::information(0, i18n("This is a help box"));
}


void kuote::preferences()
{
    PrefDialog dlg(this);
    connect(&dlg, SIGNAL(settingsChanged()), this, SLOT(restartAnimation()));
    dlg.exec();
}

int kuote::widthForHeight(int height) const
{
    return height*frameWidth/frameHeight;
}

int kuote::heightForWidth(int width) const
{
    return width*frameHeight/frameWidth;
}

void kuote::resizeEvent(QResizeEvent *e)
{
	animationFromFile(Config().animationFile);
	image->setPixmap((*animationFrames)[currentFrame]);
}

bool kuote::animationFromFile(QString fileName)
{
	QString tempFile;
	if (!KIO::NetAccess::download(fileName,tempFile))
		return false;
	QImage completeImage(tempFile);
	KIO::NetAccess::removeTempFile(tempFile);
	if (completeImage.isNull())
		return false;
	int w = completeImage.width();
	int h = completeImage.height();
	frameWidth = w/Config().numFrames;
	frameHeight = h;
	animationFrames->clear();
	for (int i = 0; i < Config().numFrames; i++) {
		QImage image = completeImage.copy(i*frameWidth,0,frameWidth,frameHeight);
		image = image.smoothScale(width(),height());
		QPixmap pixmap(image);
		pixmap.setOptimization(QPixmap::BestOptim);
		animationFrames->push_back(pixmap);
	}
	image->setEffectsEnabled(Config().applyEffects);
	emit updateLayout();
	return true;
}

void kuote::restartAnimation()
{
	animationFromFile(Config().animationFile);
	maybeUpdateTooltip();
	//load first frame
	image->setPixmap(animationFrames->front());
	currentFrame = 0;
	//start the refresh timer
	if (refreshTimerId!=0)
		killTimer(refreshTimerId);
	refreshTimerId = startTimer(Config().animDelay);
}

void kuote::timerEvent(QTimerEvent *event)
{
	if (event->timerId()==refreshTimerId) {
		currentFrame = (currentFrame+1)%Config().numFrames;
		((ClickableLabel *)mainView)->setPixmap((*animationFrames)[currentFrame]);
	} else KPanelApplet::timerEvent(event);
}

void kuote::askFortune()
{
	currentFortune.truncate(0);
	QStringList params;
	QStringList temps; //temporary file names, created and removed by KIO:NetAccess
	QStringList dattemps; //temporary file names, created and removed manually (based on names from temps)
	if(Config().useCustomProgram) {
		params.push_back("sh");
		params.push_back("-c");
		params.push_back(Config().customProgram);
	} else {
		params.push_back("fortune");
		//options
		if (!Config().considerSizes)
			params.push_back("-e");
		if (Config().length==Configuration::lenLong)
			params.push_back("-l");
		else if (Config().length==Configuration::lenShort)
			params.push_back("-s");
		if (Config().type==Configuration::typeOffensive)
			params.push_back("-o");
		else if (Config().type==Configuration::typeAny)
			params.push_back("-a");
		//fortune files and probabilities
		QStringList::iterator it = Config().selectedFortunes.begin();
		QValueList<int>::iterator it2 = Config().selectedFortunesProbs.begin();
		for (/*nothing*/; it!=Config().selectedFortunes.end() && it2!=Config().selectedFortunesProbs.end(); it++, it2++) {
			//TODO: are there potential problems with URL encoding vs local file names in here ?
			KFileItem fileItem(KFileItem::Unknown,KFileItem::Unknown,*it);
			//If the URL doesn't contain any slashes, it is considered a local file name relative
			//to fortune's data directory, otherwise it is treated as a full URL (local or remote)
			QString tempFile;
			bool hasPath = ((*it).contains('/')!=0);
			if (hasPath) {
				if (!KIO::NetAccess::download(*it,tempFile)) {
					currentFortune+=i18n("Couldn't download fortune file %1\n").arg(*it);
					continue;
				}
				temps.push_back(tempFile);
			} else {
				tempFile = *it;
			}
			//If the fortune file is remote, we also need to download the index file
			//unless it already exists
			if (hasPath && !fileItem.isDir()) {
				QString datTempFile = tempFile+".dat";
				if (!KIO::NetAccess::exists(datTempFile)) {
					if (!KIO::NetAccess::download((*it)+".dat",datTempFile)) {
						currentFortune+=i18n("Couldn't download fortune index file %1\n").arg((*it)+".dat");
						continue;
					}
					dattemps.push_back(datTempFile);
				}
			}
			if (*it2!=0)
				params.push_back(QString::number(*it2).append("%"));
			params.push_back(tempFile);
		}
	}
	KProcess proc;
	proc << params;
	connect(&proc,SIGNAL(receivedStdout(KProcess *,char *,int)),this,SLOT(receivedStdout(KProcess *,char *,int)));
	connect(&proc,SIGNAL(receivedStderr(KProcess *,char *,int)),this,SLOT(receivedStderr(KProcess *,char *,int)));
	//Run fortune and wait at most 1 second for the program to finish.
	//Sleep & Co. are used instead of wait() in order for kuote to compile on KDE 3.0
	if (!proc.start(KProcess::DontCare,KProcess::AllOutput)) {
		currentFortune = i18n("Error executing external program.");
	}
	const int maxRunTime = 1000; //max wait time [ms]
	const int timeSlice = 50; //how often to check if program has finished [ms]
	for (int i = 0; i < maxRunTime/timeSlice; i++) {
		usleep(1000*timeSlice);
		qApp->processEvents();
		if (!proc.isRunning())
			break;
	}
	if (proc.isRunning()) {
		proc.kill(SIGKILL);
		currentFortune+=i18n("\n\nProgram did not finish within its time limit and was killed.");
	}
	for (QStringList::iterator dit = temps.begin(); dit!=temps.end(); dit++)
		KIO::NetAccess::removeTempFile(*dit);
	for (QStringList::iterator ddit = dattemps.begin(); ddit!=dattemps.end(); ddit++)
		KIO::NetAccess::del(*ddit);
	maybeUpdateTooltip();
}

void kuote::maybeUpdateTooltip()
{
	QToolTip::remove(image);
	if (Config().showFortuneInTT) {
		QString tip = currentFortune;
		if (Config().truncateLongTT && tip.length() > longTooltipThreshold) {
			tip.truncate(longTooltipThreshold);
			tip+=i18n(" ... more ...");
		}
		QToolTip::add(image,tip);
	} else {
		QToolTip::add(image,i18n("Fish Applet"));
	}
}

void kuote::showFortune()
{
	FortuneDialog *dlg = new FortuneDialog(this);
	dlg->setText(currentFortune);
	dlg->show();
	askFortune(); //prepare fortune for next time
}

void kuote::receivedStdout (KProcess *proc, char *buffer, int buflen)
{
	currentFortune+=QString::fromLocal8Bit(buffer);
}

void kuote::receivedStderr (KProcess *proc, char *buffer, int buflen)
{
	currentFortune+=QString::fromLocal8Bit(buffer);
}

QString kuote::getCurrentFortune()
{
	return currentFortune;
}

void kuote::readConfig()
{
	Config().read(ksConfig);
	//currentFortune is not a part of Configuration, since it's generated and not set by the user
	ksConfig->setGroup("");
	currentFortune = ksConfig->readEntry("currentFortune",
		i18n("Welcome to Kuote!\nClick the animation to display fortune.\nRight click to generate next one without viewing."));
}

void kuote::writeConfig()
{
	Config().write(ksConfig);
	//currentFortune is not a part of Configuration, since it's generated and not set by the user
	ksConfig->setGroup("");
	ksConfig->writeEntry("currentFortune",currentFortune);
	ksConfig->sync();
}

extern "C"
{
    KPanelApplet* init( QWidget *parent, const QString& configFile)
    {
        KGlobal::locale()->insertCatalogue("kuote");
        return new kuote(configFile, KPanelApplet::Normal,
                             KPanelApplet::About /*| KPanelApplet::Help*/ | KPanelApplet::Preferences,
                             parent, "kuote");
    }
}
