#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "../include/string.h"
#include "../include/disk.h"
#include "guiutils.h"
#include "imgfio.h"
#include "pulist.h"
#include "tlist.h"
#include "iconsel.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_folder_closed_20x20.xpm"


typedef struct _IconSel		IconSel;
#define ICON_SEL(p)		((IconSel *)(p))


/*
 *	Icon Selector:
 */
struct _IconSel {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;
	gpointer	imlib_handle;

	GtkWidget	*main_vbox;
	pulistbox_struct	*location_pulistbox;
	tlist_struct	*tlist;
	GtkWidget	*ok_btn,
			*ok_btn_label,
			*cancel_btn,
			*cancel_btn_label,
			*status_bar_parent,
			*status_bar_progress,
			*status_bar_label;
	gfloat		status_bar_progress_last;

	gint		block_loop_level;
	guint		reload_idleid;
	gint		next_load_thumb_num;
	gchar		*location;
	gint		strc;
	gchar		**strv;

};


static IconSel	*iconsel = NULL;


static void IconSelTListThumbDestroyCB(gpointer data);

static void IconSelLoadThumb(
	IconSel *is, gint thumb_num, gboolean force
);
static void IconSelQueueReload(IconSel *is, const gchar *path);
static gint IconSelLoadThumbIdleCB(gpointer data);

static gint IconSelEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data 
);
static gint IconSelTListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void IconSelOKCB(GtkWidget *widget, gpointer data);
static void IconSelCancelCB(GtkWidget *widget, gpointer data);
static void IconSelLocationChangedCB(
	pulistbox_struct *box, gint i, gpointer data
);
static void IconSelTListSelectCB(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
);
static void IconSelTListUnselectCB(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
);

gint IconSelInit(gpointer imlib_h);
void IconSelSetStyle(GtkRcStyle *rc_style);
void IconSelSetTransientFor(GtkWidget *w);
gboolean IconSelIsQuery(void);
void IconSelBreakQuery(void);
GtkWidget *IconSelGetToplevel(void);

gchar **IconSelGetResponse(
	const gchar *title,
	const gchar *location,			/* Initial location */
	GList *locations_list,			/* List of gchar * locations */
	const gchar *ok_label,
	const gchar *cancel_label,
	icon_sel_btn_flags show_buttons,	/* Any of ICON_SEL_BTNFLAG_* */
	icon_sel_btn_flags default_button,	/* One of ICON_SEL_BTNFLAG_* */
	gint list_width, gint list_height,
	gint thumb_width, gint thumb_height, gint thumb_border,
	gboolean horizontal,
	gboolean show_frames, 
	gboolean show_labels,
	gint *nvalues
);

static void IconSelStatusProgress(
	IconSel *is, gfloat v, gboolean allow_gtk_iteration
);
static void IconSelStatusMessage(
	IconSel *is, const gchar *s, gboolean allow_gtk_iteration
);
void IconSelShutdown(void);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#define ICON_SEL_WIDTH			-1
#define ICON_SEL_HEIGHT			-1

#define ICON_SEL_STATUS_BAR_WIDTH	-1
#define ICON_SEL_STATUS_BAR_HEIGHT	26


/*
 *	Icon Selector Thumbs List thumb destroy signal callback.
 */
static void IconSelTListThumbDestroyCB(gpointer data)
{
	gchar *path = (gchar *)data;
	if(path == NULL)
	    return;

	g_free(path);
}


/*
 *	Loads the image for the Icon Selector Thumb List thumb.
 *
 *	If force is FALSE then the image will only be loaded for the
 *	thumb if the thumb does not already have an image loaded.
 *
 *	If the thumb's load state will be updated.
 */
static void IconSelLoadThumb(
	IconSel *is, gint thumb_num, gboolean force
)
{
	gint width, height;
	const gchar *full_path, *ext;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	tlist_struct *tlist = (is != NULL) ? is->tlist : NULL;
	if(tlist == NULL)
	    return;

	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
	    return;

	/* Thumb already loaded (or failed to load) and not forcing
	 * to load?
	 */
	if((TListGetLoadState(tlist, thumb_num) != TLIST_LOAD_STATE_NOT_LOADED) &&
	   !force
	)
	    return;

	TListFreeze(tlist);

	/* Get thumb data as the full path to the image to be opened */
	full_path = (gchar *)TListGetThumbData(tlist, thumb_num);
	if(STRISEMPTY(full_path))
	{
	    TListSetLoadState(
		tlist, thumb_num, TLIST_LOAD_STATE_FAILED
	    );
	    TListThaw(tlist);
	    return;
	}

	/* Get extension from the full path */
	ext = strrchr(full_path, '.');
	if(ext == NULL)
	{
	    TListSetLoadState(
		tlist, thumb_num, TLIST_LOAD_STATE_FAILED
	    );
	    TListThaw(tlist);
	    return;
	}

	/* Open image */
	pixmap = ImgOpenPixmap(
	    full_path,
	    &width, &height,
	    &mask,
	    is->imlib_handle
	);
	/* Failed to open image? */
	if(pixmap == NULL)
	{
	    TListSetLoadState(
		tlist, thumb_num, TLIST_LOAD_STATE_FAILED
	    );
	    TListThaw(tlist);
	    return;
	}          

	/* Set the image to the Thumb */
	TListSetPixmap(tlist, thumb_num, pixmap, mask);
	GDK_PIXMAP_UNREF(pixmap)
	GDK_BITMAP_UNREF(mask)

	/* Mark thumb as loaded successfully */
	TListSetLoadState(
	    tlist, thumb_num, TLIST_LOAD_STATE_LOADED
	);

	TListThaw(tlist);
}

/*
 *	Queues a reload thumbs list for the Icon Selector.
 */
static void IconSelQueueReload(IconSel *is, const gchar *path)
{
	tlist_struct *tlist;

	if(is == NULL)
	    return;

	/* New location specified? */
	if((path != NULL) && (path != is->location))
	{
	    /* No change in location? */
	    if((is->location != NULL) ?
	       !strcmp(is->location, path) : FALSE
	    )
		return;

	    g_free(is->location);
	    is->location = STRDUP(path);
	}

	/* Cancel existing reload idle callback and add new reload
	 * idle callback
	 */
	GTK_IDLE_REMOVE(is->reload_idleid)
	is->reload_idleid = gtk_idle_add_priority(
	    G_PRIORITY_LOW,
	    IconSelLoadThumbIdleCB, is
	);

	/* Clear thumbs list so the reload idle callback will know to
	 * get the listing of the new location
	 */
	tlist = is->tlist;
	TListFreeze(tlist);
	TListClear(tlist);
	TListThaw(tlist);
}

/*
 *	Icon Selector reload thumbs list idle callback.
 */
static gint IconSelLoadThumbIdleCB(gpointer data)
{
	tlist_struct *tlist;
	IconSel *is = ICON_SEL(data);
	if(is == NULL)
	    return(FALSE);

	if(is->freeze_count > 0)
	    return(TRUE);

	tlist = is->tlist;

	/* If the list is empty then get listing of the new
	 * location
	 */
	if(tlist->total_thumbs <= 0)
	{
	    gint i, strc, thumb_num;
	    const gchar *name, *ext, *location = is->location;
	    gchar *full_path, **strv;

	    TListFreeze(tlist);

	    IconSelStatusProgress(is, 0.0f, FALSE);
	    IconSelStatusMessage(is, "Loading icons...", FALSE);

	    /* Get directory listing of the new location and sort it */
	    strv = GetDirEntNames2(location, &strc);
	    strv = StringQSort(strv, strc);

	    /* Handle each directory entry */
	    for(i = 0; i < strc; i++)
	    {
		name = strv[i];
		if(name == NULL)
		{
		    g_free(strv[i]);
		    continue;
		}

		/* Skip special directory entry notations */
		if(!g_strcasecmp(name, ".") ||
		   !g_strcasecmp(name, "..")
		)
		{
		    g_free(strv[i]);
		    continue;
		}

		/* Get extension of thedirectory entry */
		ext = strrchr(name, '.');
		if(ext == NULL)
		{
		    g_free(strv[i]);
		    continue;
		}

		/* Format full path to the directory entry */
		full_path = STRDUP(PrefixPaths(location, name));
		if(full_path == NULL)
		{
		    g_free(strv[i]);
		    continue;
		}
		 
		/* Check the extension to see if it is not an image
		 * that should be included in the Thumbs List
		 */
		if(g_strcasecmp(ext, ".bmp") &&
		   g_strcasecmp(ext, ".gif") &&
		   g_strcasecmp(ext, ".jpg") &&
		   g_strcasecmp(ext, ".jpeg") &&
		   g_strcasecmp(ext, ".mng") &&
		   g_strcasecmp(ext, ".pcx") &&
		   g_strcasecmp(ext, ".png") &&
		   g_strcasecmp(ext, ".tga") &&
		   g_strcasecmp(ext, ".xbm") &&
		   g_strcasecmp(ext, ".xpm")
		)
		{
		    g_free(full_path);
		    g_free(strv[i]);
		    continue;
		}

		/* Append a new Thumb to the Thumbs List */
		thumb_num = TListAppend(tlist, name);
		if(thumb_num < 0)
		{                                         
		    g_free(full_path);
		    g_free(strv[i]);
		    continue;
		}  
		   
		/* Set the full path as the Thumbs data */
		TListSetThumbDataFull(
		    tlist, thumb_num,
		    STRDUP(full_path),  
		    IconSelTListThumbDestroyCB
		);

		g_free(full_path);
		g_free(strv[i]);
	    }

	    g_free(strv);

	    TListThaw(tlist);

	    /* If there are any thumbs to load images for then set new
	     * idle callback to this function to load the images for
	     * each thumb
	     */

	    if(tlist->total_thumbs > 0)
	    {
		is->next_load_thumb_num = 0;
		is->reload_idleid = gtk_idle_add_priority(
		    G_PRIORITY_LOW,
		    IconSelLoadThumbIdleCB, is
		);
	    }
	    else
	    {
		IconSelStatusProgress(is, 0.0f, FALSE);
		IconSelStatusMessage(is, "0 icons loaded", FALSE);
		is->reload_idleid = 0;
		is->next_load_thumb_num = -1;
	    }
	}
	/* Load next thumb? */
	else if((is->next_load_thumb_num >= 0) && 
	        (is->next_load_thumb_num < tlist->total_thumbs)
	)
	{
	    gint thumb_num = is->next_load_thumb_num;

	    /* Load the next thumb as needed and update its load
	     * state
	     */
	    IconSelLoadThumb(is, thumb_num, FALSE);

	    /* Go on to the next thumb and set idle callback to this
	     * function if more thumbs need to be loaded
	     */
	    thumb_num++;
	    if(thumb_num < tlist->total_thumbs)
	    {
		IconSelStatusProgress(
		    is,
		    (tlist->total_thumbs > 0) ?
			((gfloat)thumb_num / (gfloat)tlist->total_thumbs) : 0.0f,
		    FALSE
		);
		is->next_load_thumb_num = thumb_num;
		is->reload_idleid = gtk_idle_add_priority(
		    G_PRIORITY_LOW,
		    IconSelLoadThumbIdleCB, is 
		);
	    }
	    else
	    {
		/* No more thumbs need to be loaded */
		gchar *s = g_strdup_printf(
		    "%i icon%s loaded",
		    thumb_num,
		    (thumb_num == 1) ? "" : "s"
		);
		IconSelStatusProgress(is, 0.0f, FALSE);
		IconSelStatusMessage(is, s, FALSE);
		g_free(s);
		is->next_load_thumb_num = -1;
		is->reload_idleid = 0;
	    }
	}

	return(FALSE);
}

/*
 *	Icon Selector toplevel GtkWindow event signal callback.
 */
static gint IconSelEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	IconSel *is = ICON_SEL(data);
	if((widget == NULL) || (event == NULL) || (is == NULL))
	    return(status);

	if(is->freeze_count > 0)
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_DELETE:
	    IconSelCancelCB(is->cancel_btn, is);
	    status = TRUE;
	    break;
	}

	return(status);
}

/*
 *	Icon Selector Thumbs List event signal callback.
 */
static gint IconSelTListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean press;
	GdkEventKey *key;
	GdkEventButton *button;
	tlist_struct *tlist;
	IconSel *is = ICON_SEL(data);
	if((widget == NULL) || (event == NULL) || (is == NULL))
	    return(status);

	if(is->freeze_count > 0)
	    return(status);

	tlist = is->tlist;

	switch((gint)event->type)
	{
	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
#define SIGNAL_EMIT_STOP        {               \
 gtk_signal_emit_stop_by_name(                  \
  GTK_OBJECT(widget),                           \
  press ?                                       \
   "key_press_event" : "key_release_event"      \
 );                                             \
}
	    key = (GdkEventKey *)event;
	    press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    switch(key->keyval)
	    {
	      case GDK_Return:
	      case GDK_3270_Enter:
	      case GDK_KP_Enter:
	      case GDK_ISO_Enter:
		if(press)
		    IconSelOKCB(is->ok_btn, is);
		status = TRUE;
		break;

	      case GDK_Up:
	      case GDK_KP_Up:
		if(press)
		{
#if 0
		    GtkCombo *combo = GTK_COMBO(is->location_combo);
		    const gchar *location = gtk_entry_get_text(
			GTK_ENTRY(combo->entry)
		    );
		    GList *glist = GUIComboGetList(GTK_WIDGET(combo));
		    while(glist != NULL)
		    {
			if((glist->data != NULL) && (location != NULL))
			{
			    if(!strcmp((const gchar *)glist->data, location))
				break;
			}
			glist = g_list_next(glist);
		    }
		    glist = (glist != NULL) ? g_list_previous(glist) : NULL;
		    if((glist != NULL) ? (glist->data != NULL) : FALSE)
			gtk_entry_set_text(
			    GTK_ENTRY(combo->entry),
			    (const gchar *)glist->data
			);
#endif
		}
		SIGNAL_EMIT_STOP
		status = TRUE;
		break;

	      case GDK_Down:
	      case GDK_KP_Down:
		if(press)
		{
#if 0
		    GtkCombo *combo = GTK_COMBO(is->location_combo);
		    const gchar *location = gtk_entry_get_text(
			GTK_ENTRY(combo->entry)
		    );
		    GList *glist = GUIComboGetList(GTK_WIDGET(combo));
		    while(glist != NULL)
		    {
			if((glist->data != NULL) && (location != NULL))
			{
			    if(!strcmp((const gchar *)glist->data, location))
				break;
			}
			glist = g_list_next(glist);
		    }
		    glist = (glist != NULL) ? g_list_next(glist) : NULL;
		    if((glist != NULL) ? (glist->data != NULL) : FALSE)
			gtk_entry_set_text(
			    GTK_ENTRY(combo->entry), 
			    (const gchar *)glist->data
			);
#endif
		}
		SIGNAL_EMIT_STOP
		status = TRUE;
		break;
	    }
#undef SIGNAL_EMIT_STOP
	    break;

	  case GDK_2BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON1:
		IconSelOKCB(is->ok_btn, is);
		status = TRUE;
		break;
	    }
	    break;
	}

	return(status);
}

/*
 *	Icon Selector OK callback.
 */
static void IconSelOKCB(GtkWidget *widget, gpointer data)
{
	gint i, thumb_num;
	const gchar *path;
	GList *glist;
	tlist_struct *tlist;
	IconSel *is = ICON_SEL(data);
	if(is == NULL)
	    return;

	if(is->freeze_count > 0)
	    return;

	/* Get list of selected thumbs and copy their paths to the
	 * Icon Selector's return strings
	 */
	tlist = is->tlist;
	for(glist = tlist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    thumb_num = (gint)glist->data;
	    path = (const gchar *)TListGetThumbData(tlist, thumb_num);
	    if(path == NULL)
		continue;

	    i = MAX(is->strc, 0);
	    is->strc = i + 1;
	    is->strv = (gchar **)g_realloc(
		is->strv, is->strc * sizeof(gchar *)
	    );
	    if(is->strv == NULL)
	    {
		is->strc = 0;
		break;
	    }

	    is->strv[i] = STRDUP(path);
	}

	/* Pop out of the main loop */
	if(is->block_loop_level > 0)
	{
	    is->block_loop_level--;
	    gtk_main_quit();
	}
}

/*
 *	Icon Selector Cancel callback.
 */
static void IconSelCancelCB(GtkWidget *widget, gpointer data)
{
	IconSel *is = ICON_SEL(data);
	if(is == NULL)
	    return;

	if(is->freeze_count > 0)
	    return;

	/* Pop out of the main loop */
	if(is->block_loop_level > 0)
	{
	    is->block_loop_level--;
	    gtk_main_quit();
	}
}

/*
 *	Icon Selector Location changed callback.
 */
static void IconSelLocationChangedCB(
	pulistbox_struct *box, gint i, gpointer data
)
{
	const gchar *location;
	IconSel *is = ICON_SEL(data);
	if((box == NULL) || (i < 0) || (is == NULL))
	    return;

	if(is->freeze_count > 0)
	    return;

	location = (gchar *)PUListGetItemData(
	    PUListBoxGetPUList(box), i
	);
	if(location != NULL)
	    IconSelQueueReload(is, location);
}

/*
 *	Icon Selector Thumbs List select thumb signal callback.
 */
static void IconSelTListSelectCB(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
)
{
	const gchar *full_path;
	IconSel *is = ICON_SEL(data);
	if((tlist == NULL) || (is == NULL))
	    return;

	if(is->freeze_count > 0)
	    return;

	/* Load thumb as needed and update its load state */
	IconSelLoadThumb(is, thumb_num, FALSE);

	/* Get thumb data and update status message */
	full_path = (const gchar *)TListGetThumbData(tlist, thumb_num);
	if(full_path != NULL)
	{
	    gchar *s;
	    const gchar *name = strrchr(full_path, DIR_DELIMINATOR);
	    if(name != NULL)
		name++;
	    else
		name = full_path;
	    s = g_strdup_printf(
		"Icon \"%s\" selected",
		name
	    );
	    IconSelStatusMessage(is, s, FALSE);
	    g_free(s);
	}

	/* Scroll if the selected thumb is not fully visible */
	if(TListIsThumbVisible(tlist, thumb_num) !=
	    GTK_VISIBILITY_FULL
	)
	    TListMoveTo(tlist, thumb_num, 0.5f);
}

/*
 *      Unselect thumb callback.
 */
static void IconSelTListUnselectCB(
	tlist_struct *tlist, GdkEventButton *button, gint thumb_num,
	gpointer data
)
{
	IconSel *is = ICON_SEL(data);   
	if((tlist == NULL) || (is == NULL))
	    return;

	if(is->freeze_count > 0)
	    return;

}


/* 
 *	Initializes the Icon Selector.
 */
gint IconSelInit(gpointer imlib_h)
{
	const gint	border_major = 5,
			border_minor = 2;
	GdkWindow *window;
	GtkAdjustment *adj;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2, *parent3;
	pulistbox_struct *pulistbox;
	tlist_struct *tlist;
	IconSel *is;

	/* Already initialized? */
	if(iconsel != NULL)
	    return(0);

	/* Create new Icon Selector */
	is = ICON_SEL(g_malloc0(sizeof(IconSel)));
	if(is == NULL)
	    return(-3);

	is->accelgrp = accelgrp = gtk_accel_group_new();
	is->freeze_count = 0;
	is->imlib_handle = imlib_h;
	is->status_bar_progress_last = 0.0f;
	is->block_loop_level = 0;
	is->reload_idleid = 0;
	is->location = NULL;
	is->strc = 0;
	is->strv = NULL;

	is->freeze_count++;

	/* Toplevel */
	is->toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(
	    GTK_WINDOW(w),
	    TRUE, TRUE, TRUE
	);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	}
        gtk_widget_add_events(
            w,
            GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
            GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
        );
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(IconSelEventCB), is
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Main vbox */
	is->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);

	/* Vbox for Location and Thumbs List */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(is->main_vbox), w, FALSE, FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent = w;

	/* Location hbox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;
	w = gtk_label_new("Location:");
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Location Popup List Box */
	is->location_pulistbox = pulistbox = PUListBoxNew(
	    parent2,
	    -1,
	    20 + (2 * 3)
	);
	PUListBoxSetChangedCB(
	    pulistbox,
	    IconSelLocationChangedCB, is
	);
	PUListBoxMap(pulistbox);

	/* Thumbs list */
	is->tlist = tlist = TListNew(
	    FALSE, 100, 90, 2,
	    is,
	    IconSelTListSelectCB,
	    IconSelTListUnselectCB
	);
	w = tlist->toplevel;
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	w = tlist->list_da;
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(IconSelTListEventCB), is
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(IconSelTListEventCB), is
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(IconSelTListEventCB), is
	);
	TListDoubleBuffer(tlist, TRUE);
	TListSelectionMode(tlist, GTK_SELECTION_EXTENDED);
	TListShowThumbFrames(tlist, TRUE);
	TListShowThumbLabels(tlist, TRUE);
	TListShowTextTips(tlist, FALSE);
	TListMap(tlist);


	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(is->main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons hbox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(is->main_vbox), w, FALSE, FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent = w;

	/* OK button */
	is->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK",
	    &is->ok_btn_label
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(IconSelOKCB), is
	);
	gtk_widget_show(w);

	/* Cancel button */
	is->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancela"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annulla"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
	    , &is->cancel_btn_label
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(IconSelCancelCB), is
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);                          
	gtk_widget_show(w);


	/* Status Bar Frame */
	is->status_bar_parent = w = gtk_frame_new(NULL);
	gtk_widget_set_usize(
	    w,
	    ICON_SEL_STATUS_BAR_WIDTH,
	    ICON_SEL_STATUS_BAR_HEIGHT
	);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_box_pack_end(GTK_BOX(is->main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;

	/* Status Bar table */
	w = gtk_table_new(1, 2, FALSE);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Progress Bar */
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f, 1.0f, 100.0f,
	    0.0f, 0.0f, 0.0f
	);
	is->status_bar_progress = w = gtk_progress_bar_new_with_adjustment(adj);
	gtk_widget_set_usize(w, 100, 20);
	gtk_progress_bar_set_orientation(
	    GTK_PROGRESS_BAR(w), GTK_PROGRESS_LEFT_TO_RIGHT
	);                                                 
	gtk_progress_bar_set_bar_style(
	    GTK_PROGRESS_BAR(w), GTK_PROGRESS_CONTINUOUS
	);
	gtk_progress_set_activity_mode(
	    GTK_PROGRESS(w), FALSE
	);
	gtk_table_attach(
	    GTK_TABLE(parent), w,
	    0, 1, 0, 1,
	    0,
	    GTK_SHRINK,
	    0, 0
	);
	gtk_widget_show(w);

	/* Label */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_container_border_width(GTK_CONTAINER(w), 1);
	gtk_table_attach(
	    GTK_TABLE(parent), w,
	    1, 2, 0, 1,
	    GTK_SHRINK | GTK_EXPAND | GTK_FILL,
	    GTK_FILL,
	    0, 0
	);
	gtk_widget_show(w);
	parent2 = w;                                                 

	w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent3 = w;

	is->status_bar_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, border_minor);
	gtk_widget_show(w);




	is->freeze_count--;

	iconsel = is;

	return(0);
}

/*
 *	Sets the Icon Selector's style. 
 */
void IconSelSetStyle(GtkRcStyle *rc_style)
{
	GtkWidget *w;
	IconSel *is = iconsel;

	if(is == NULL)
	    return;

	w = is->toplevel;
	if(w != NULL)
	{
	    if(rc_style != NULL)
	    {
		gtk_widget_modify_style_recursive(w, rc_style);
	    }
	    else
	    {   
		rc_style = gtk_rc_style_new();
		gtk_widget_modify_style_recursive(w, rc_style);
		GTK_RC_STYLE_UNREF(rc_style)
	    }
	}
}

/*
 *	Sets the Icon Selector to be a transient for the specified
 *	GtkWindow.
 *
 *	If w is NULL then transient for will be unset.
 */
void IconSelSetTransientFor(GtkWidget *w)
{
	IconSel *is = iconsel;

	if(is == NULL)
	    return;

	if(is->toplevel != NULL)
	{
	    if(w != NULL)
	    {
		if(!GTK_IS_WINDOW(GTK_OBJECT(w)))
		    return;

		if(GTK_WINDOW(w)->modal)
		    gtk_window_set_modal(GTK_WINDOW(w), FALSE);

		gtk_window_set_modal(
		    GTK_WINDOW(is->toplevel), TRUE
		);
		gtk_window_set_transient_for(
		    GTK_WINDOW(is->toplevel), GTK_WINDOW(w)
		);
	    }
	    else
	    {
		gtk_window_set_modal(
		    GTK_WINDOW(is->toplevel), FALSE
		);
		gtk_window_set_transient_for(
		    GTK_WINDOW(is->toplevel), NULL
		);
	    }
	}
}

/*
 *	Checks if the Icon Selector is mapped and waiting for user
 *	input.
 */
gboolean IconSelIsQuery(void)
{
	IconSel *is = iconsel;
	if(is != NULL)
	    return((is->block_loop_level > 0) ? TRUE : FALSE);
	else
	    return(FALSE);
}

/*
 *	Breaks the Icon Selector's user input query (if any) and
 *	causes the querying function to return NULL.
 */
void IconSelBreakQuery(void)
{
	IconSel *is = iconsel;
	if(is == NULL)
	    return;

	/* Break out of all our main loops */
	while(is->block_loop_level > 0)
	{
	    gtk_main_quit();
	    is->block_loop_level--;
	}
	is->block_loop_level = 0;
}

/*
 *	Returns the Icon Selector's toplevel widget.
 */
GtkWidget *IconSelGetToplevel(void)
{
	IconSel *is = iconsel;
	return((is != NULL) ? is->toplevel : NULL);
}

/*
 *	Block input and wait for a response.
 *
 *	Returns a list of selected icon paths or NULL if the user
 *	canceled.
 *
 *	The returned values must not be modified or deleted.
 */
gchar **IconSelGetResponse(
	const gchar *title,
	const gchar *location,			/* Initial location */
	GList *locations_list,			/* List of gchar * locations */
	const gchar *ok_label,
	const gchar *cancel_label,
	icon_sel_btn_flags show_buttons,	/* Any of ICON_SEL_BTNFLAG_* */
	icon_sel_btn_flags default_button,	/* One of ICON_SEL_BTNFLAG_* */
	gint list_width, gint list_height,
	gint thumb_width, gint thumb_height, gint thumb_border,
	gboolean horizontal,
	gboolean show_frames, 
	gboolean show_labels,
	gint *nvalues
)
{
	gint i;
	GtkWidget *w;
	pulistbox_struct *pulistbox;
	tlist_struct *tlist;
	IconSel *is = iconsel;

	if(nvalues != NULL)
	    *nvalues = 0;

	if(is == NULL)
	    return(NULL);

	/* Already waiting for user response? */
	if(is->block_loop_level > 0)
	    return(NULL);

	/* Reset responses */
	for(i = 0; i < is->strc; i++)
	    g_free(is->strv[i]);
	g_free(is->strv);
	is->strv = NULL;
	is->strc = 0;


	is->freeze_count++;

	/* Set title */
	w = is->toplevel;
	if(title != NULL)
	    gtk_window_set_title(GTK_WINDOW(w), title);

	/* Set thumbs list options */
	tlist = is->tlist;
	w = tlist->list_da;
	gtk_widget_set_usize(w, list_width, list_height);

	TListThumbGeometry(tlist, thumb_width, thumb_height, thumb_border);
	TListOrientation(tlist, horizontal);
	TListShowThumbFrames(tlist, show_frames);
	TListShowThumbLabels(tlist, show_labels);
	TListShowTextTips(tlist, show_labels);


	/* Set the locations list */
	pulistbox = is->location_pulistbox;
	if(pulistbox != NULL)
	{
	    gint i;
	    const gchar *full_path;
	    GList *glist;
	    GdkBitmap *mask;
	    GdkPixmap *pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		&mask, (guint8 **)icon_folder_closed_20x20_xpm
	    );
	    pulist_struct *pulist = PUListBoxGetPUList(pulistbox);

	    PUListClear(pulist);
	    for(glist = locations_list; glist != NULL; glist = g_list_next(glist))
	    {
		full_path = (gchar *)glist->data;
		if(full_path == NULL)
		    continue;

		i = PUListAddItemPixText(
		    pulist, full_path,
		    pixmap, mask
		);
		if(i < 0)
		    continue;

		PUListSetItemDataFull(
		    pulist, i,
		    STRDUP(full_path), (GtkDestroyNotify)g_free
		);
	    }

	    PUListBoxSetLinesVisible(
		pulistbox,
		MIN(PUListGetTotalItems(pulist), 10)
	    );

	    GDK_PIXMAP_UNREF(pixmap);
	    GDK_BITMAP_UNREF(mask);
	}

	/* Set button labels */
	if(!STRISEMPTY(ok_label))
	    gtk_label_set_text(
		GTK_LABEL(is->ok_btn_label), ok_label
	    );
	if(!STRISEMPTY(cancel_label))
	    gtk_label_set_text(
		GTK_LABEL(is->cancel_btn_label), cancel_label
	    );


	is->freeze_count--;


	/* Set initial location and get listing */
	if(location != NULL)
	    IconSelQueueReload(is, location);

	/* Map the Icon Selector */
	w = is->toplevel;
	gtk_widget_show_raise(w);

	/* Push main loop and wait for user response */
	is->block_loop_level++;
	gtk_main();

	/* Unmap the Icon Selector */
	w = is->toplevel;
	gtk_widget_hide(w);

	/* Stop reload idle callback as needed */
	GTK_IDLE_REMOVE(is->reload_idleid)
	is->reload_idleid = 0;
	  


	if(nvalues != NULL)
	    *nvalues = is->strc;

	return(is->strv);
}


/*
 *	Sets the Icon Selector Status Bar's progress.
 */
static void IconSelStatusProgress(
	IconSel *is, gfloat v, gboolean allow_gtk_iteration
)
{
	GtkProgress *progress;
	GtkWidget *w = (is != NULL) ? is->status_bar_progress : NULL;
	if(w == NULL)
	    return;

	progress = GTK_PROGRESS(w);

	/* Do activity? */
	if(v < 0.0f)
	{             
	    GtkAdjustment *adj = progress->adjustment;

	    /* Get new value based on unknown position */
	    v = gtk_progress_get_value(progress) + 1.0f;
	    if(v > adj->upper)
		v = adj->lower;

	    /* Update progress bar for `activity mode' (unknown limit)
	     * and set new value            
	     */
	    gtk_progress_set_activity_mode(progress, TRUE);
	    gtk_progress_set_show_text(progress, FALSE);
	    gtk_progress_set_value(progress, v);

	    /* Reset last progress position to -1.0, indicating that
	     * there is no known last position since we are doing
	     * activity mode
	     */
	    is->status_bar_progress_last = -1.0f;
	}
	else
	{   
	    /* Clip value to 0.0, 1.0 */
	    if(v > 1.0f)
		v = 1.0f;

	    /* Reset last progress position if it is greater than the
	     * given value, implying the progress has warped back to
	     * the beginning
	     */
	    if(is->status_bar_progress_last > v)
		is->status_bar_progress_last = -1.0f;

	    /* Has value not sufficiently changed? (less than 0.01
	     * change in value)
	     */
	    if((is->status_bar_progress_last > 0.0f) &&   
	       ((v - is->status_bar_progress_last) < 0.01f)
	    )
		return;

	    /* Update progress */           
	    gtk_progress_set_activity_mode(progress, FALSE);
	    gtk_progress_set_format_string(
		progress, "%p%%"
	    );
	    /* Display text only if value is positive */
	    gtk_progress_set_show_text(
		progress, (v > 0.0f) ? TRUE : FALSE
	    );
	    gtk_progress_bar_update(GTK_PROGRESS_BAR(w), v);

	    /* Record progress value */
	    is->status_bar_progress_last = v;
	}

	/* Flush events? */
	if(allow_gtk_iteration)
	{   
	    while(gtk_events_pending() > 0)
		gtk_main_iteration();
	}
}

/*
 *	Sets the Icon Selector Status Bar's message.
 */
static void IconSelStatusMessage(
	IconSel *is, const gchar *s, gboolean allow_gtk_iteration
)
{
	GtkWidget *w = (is != NULL) ? is->status_bar_label : NULL;
	if(w == NULL)
	    return;

	/* Set status message */
	gtk_label_set_text(
	    GTK_LABEL(w), (s != NULL) ? s : ""
	);

	/* Flush events */
	if(allow_gtk_iteration)
	{
	    while(gtk_events_pending() > 0)
		gtk_main_iteration();
	}
}

/*
 *	Shuts down the Icon Selector.
 */
void IconSelShutdown(void)
{
	gint i;
	IconSel *is = iconsel;
	if(is == NULL)
	    return;

	/* Break out of any main loops */
	while(is->block_loop_level > 0)
	{
	    gtk_main_quit();
	    is->block_loop_level--;
	}
	is->block_loop_level = 0;

	GTK_IDLE_REMOVE(is->reload_idleid);
	is->reload_idleid = 0;

	PUListBoxDelete(is->location_pulistbox);
	TListDelete(is->tlist);
	GTK_WIDGET_DESTROY(is->ok_btn)
	GTK_WIDGET_DESTROY(is->cancel_btn)
	GTK_WIDGET_DESTROY(is->status_bar_label)
	GTK_WIDGET_DESTROY(is->status_bar_progress)
	GTK_WIDGET_DESTROY(is->status_bar_parent)
	GTK_WIDGET_DESTROY(is->main_vbox)
	GTK_WIDGET_DESTROY(is->toplevel)
	GTK_ACCEL_GROUP_UNREF(is->accelgrp)

	for(i = 0; i < is->strc; i++)
	    g_free(is->strv[i]);
	g_free(is->strv);

	g_free(is->location);

	g_free(is);
	iconsel = is = NULL;
}
