#include <string.h>
#include <gtk/gtk.h>
#include "guiutils.h"
#include "obj.h"


static GdkBitmap *GDK_BITMAP_REF_RTN(GdkBitmap *bitmap);
static GdkPixmap *GDK_PIXMAP_REF_RTN(GdkPixmap *pixmap);


obj_struct *ObjNew(void);
obj_struct *ObjNewWithValues(
	const obj_type type,
	const obj_flags flags,
	obj_struct *parent,		/* Transfered */
	GList *children,		/* Transfered */
	const gchar *name,
	const gchar *icon_name,
	const gchar *bg_name,
	const gchar *value,
	const gchar *tip,
	const GtkOrientation name_orientation,
	const GtkJustification name_justify,
	GdkPixmap *pixmap,
	GdkBitmap *mask,
	GdkPixmap *pixmap_opened,
	GdkBitmap *mask_opened,
	const guint accel_key,
	const guint accel_mods,
	const gulong last_modified
);
obj_struct *ObjCopy(obj_struct *obj);
void ObjDelete(obj_struct *obj);

obj_struct *ObjGetSiblingYounger(obj_struct *obj);
obj_struct *ObjGetSiblingOlder(obj_struct *obj);
obj_struct *ObjGetSiblingOldest(obj_struct *obj);
obj_struct *ObjGetToplevel(obj_struct *obj);

gboolean ObjIsDescendent(obj_struct *obj, obj_struct *obj_descendent);
gboolean ObjIsAncestor(obj_struct *obj, obj_struct *obj_ancestor);

obj_struct *ObjGetObjFromPath(
	obj_struct *obj_toplevel, const gchar *path
);
gchar *ObjGetPathFromObj(obj_struct *obj);

gulong ObjGetMemorySize(obj_struct *obj);

GList *ObjDDEBufferParse(const guint8 *buf, const gint buf_len);
guint8 *ObjDDEBufferAppend(
	guint8 *buf, gint *buf_len,
	const obj_struct *obj  
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define PATH_DELIM	'/'


/*
 *	Adds a ref count to the GdkBitmap and returns it.
 */
static GdkBitmap *GDK_BITMAP_REF_RTN(GdkBitmap *bitmap)
{
	if(bitmap != NULL)
	    gdk_bitmap_ref(bitmap);
	return(bitmap);
}

/*
 *	Adds a ref count to the GdkPixmap and returns it.
 */
static GdkPixmap *GDK_PIXMAP_REF_RTN(GdkPixmap *pixmap)
{
	if(pixmap != NULL)
	    gdk_pixmap_ref(pixmap);
	return(pixmap);
}


/*
 *	Creates a new Object.
 */
obj_struct *ObjNew(void)
{
	return(
	    OBJ(g_malloc0(sizeof(obj_struct)))
	);

}

/*
 *	Creates a new Object with the specified values.
 */
obj_struct *ObjNewWithValues(
	const obj_type type,
	const obj_flags flags,
	obj_struct *parent, 		/* Transfered */
	GList *children,		/* Transfered */
	const gchar *name,
	const gchar *icon_name,
	const gchar *bg_name,  
	const gchar *value,
	const gchar *tip,
	const GtkOrientation name_orientation,
	const GtkJustification name_justify,
	GdkPixmap *pixmap,
	GdkBitmap *mask,
	GdkPixmap *pixmap_opened,
	GdkBitmap *mask_opened,
	const guint accel_key,
	const guint accel_mods,
	const gulong last_modified
)
{
	obj_struct *obj = ObjNew();
	if(obj == NULL)
	    return(NULL);

	obj->type = type;
	obj->flags = flags;
	obj->parent = parent;
	obj->children = children;
	obj->name = STRDUP(name);
	obj->icon_name = STRDUP(icon_name);
	obj->bg_name = STRDUP(bg_name);
	obj->value = STRDUP(value);
	obj->tip = STRDUP(tip);
	obj->name_orientation = name_orientation;
	obj->name_justify = name_justify;
	obj->pixmap = GDK_PIXMAP_REF_RTN(pixmap);
	obj->mask = GDK_BITMAP_REF_RTN(mask);
	obj->pixmap_opened = GDK_PIXMAP_REF_RTN(pixmap_opened);
	obj->mask_opened = GDK_BITMAP_REF_RTN(mask_opened);
	obj->accel_key = accel_key;
	obj->accel_mods = accel_mods;
	obj->last_modified = last_modified;

	return(obj);
}

/*
 *	Coppies the Object.
 */
obj_struct *ObjCopy(obj_struct *obj)
{
	GList *glist;
	const obj_struct *src = obj;
	obj_struct *tar_child, *tar = ObjNew();
	if(tar == NULL)
	    return(NULL);

	tar->type = src->type;
	tar->flags = src->flags;
	tar->name = STRDUP(src->name);
	tar->icon_name = STRDUP(src->icon_name);
	tar->bg_name = STRDUP(src->bg_name);
	tar->value = STRDUP(src->value);
	tar->tip = STRDUP(src->tip);

	tar->name_orientation = src->name_orientation;
	tar->name_justify = tar->name_justify;

	/* Copy pixmaps & masks */
	if(src->pixmap != NULL)
	{
	    gdk_pixmap_ref(src->pixmap);
	    tar->pixmap = src->pixmap;
	}
	if(src->mask != NULL)
	{
	    gdk_bitmap_ref(src->mask);
	    tar->mask = src->mask;
	}
	if(src->pixmap_opened != NULL)
	{
	    gdk_pixmap_ref(src->pixmap_opened);
	    tar->pixmap_opened = src->pixmap_opened;
	}
	if(src->mask_opened != NULL)
	{
	    gdk_bitmap_ref(src->mask_opened);
	    tar->mask_opened = src->mask_opened;
	}

	tar->accel_key = src->accel_key;
	tar->accel_mods = src->accel_mods;

	tar->last_modified = src->last_modified;

	tar->parent = NULL;	/* Let calling function set parent object */

	/* Copy all child objects */
	for(glist = src->children; glist != NULL; glist = g_list_next(glist))
	{
	    tar_child = ObjCopy(OBJ(glist->data));
	    if(tar_child == NULL)
		continue;

	    tar->children = g_list_append(
		tar->children, tar_child
	    );
	    tar_child->parent = tar;
	}

	return(tar);
}

/*
 *	Deletes the Object and all its child Objects.
 *
 *	The calling fnuction must update the parent Object's list of
 *	child objects.
 */
void ObjDelete(obj_struct *obj)
{
	GList *glist;

	if(obj == NULL)
	    return;

	/* Delete all child Objects */
	for(glist = obj->children; glist != NULL; glist = g_list_next(glist))
	    ObjDelete(OBJ(glist->data));
	g_list_free(obj->children);

	/* Unref pixmaps & masks */
	GDK_PIXMAP_UNREF(obj->pixmap)
	GDK_BITMAP_UNREF(obj->mask)
	GDK_PIXMAP_UNREF(obj->pixmap_opened)
	GDK_BITMAP_UNREF(obj->mask_opened)

	g_free(obj->name);
	g_free(obj->icon_name);
	g_free(obj->bg_name);
	g_free(obj->value);
	g_free(obj->tip);
	g_free(obj);
}


/*
 *	Returns the Object's younger (previous) sibling Object.
 *
 *	Can return NULL if the specified Object does not have a
 *	younger sibling or the specified Object is a toplevel Object.
 */
obj_struct *ObjGetSiblingYounger(obj_struct *obj)
{
	GList *glist;
	obj_struct *parent = OBJ_PARENT(obj);
	if(parent == NULL)
	    return(NULL);

	for(glist = OBJ_CHILDREN(parent);
	    glist != NULL;
	    glist = g_list_next(glist)
	)                             
	{
	    if(OBJ(glist->data) == obj)
	    {
		glist = g_list_previous(glist);
		return(OBJ((glist != NULL) ? glist->data : NULL));
	    }
	}
	 
	return(NULL);
}

/*
 *	Returns the Object's older (next) sibling Object.
 *
 *	Can return NULL if the specified Object does not have an
 *	older sibling or the specified Object is a toplevel Object.
 */
obj_struct *ObjGetSiblingOlder(obj_struct *obj)
{
	GList *glist;
	obj_struct *parent = OBJ_PARENT(obj);
	if(parent == NULL)
	    return(NULL);

	for(glist = OBJ_CHILDREN(parent);
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    if(OBJ(glist->data) == obj)
	    {
		glist = g_list_next(glist);
		return(OBJ((glist != NULL) ? glist->data : NULL));
	    }
	}

	return(NULL);
}

/*
 *	Returns the Object's oldest (first) sibling.
 *
 *	Can return NULL if the specified Object is a toplevel Object.
 */
obj_struct *ObjGetSiblingOldest(obj_struct *obj)
{
	GList *glist;
	obj_struct *parent = OBJ_PARENT(obj);
	if(parent == NULL)
	    return(NULL);

	glist = OBJ_CHILDREN(parent);
	return(OBJ((glist != NULL) ? glist->data : NULL));
}


/*
 *	Returns the Object's toplevel Object or the Object itself if
 *	it has no parent.
 */
obj_struct *ObjGetToplevel(obj_struct *obj)
{
	obj_struct *parent = OBJ_PARENT(obj);
	while(parent != NULL)
	{
	    obj = parent;
	    parent = OBJ_PARENT(obj);
	}

	return(obj);
}

/*
 *	Checks if obj_descendent is a descendent (grandchild) of obj or
 *	is obj itself.
 */
gboolean ObjIsDescendent(obj_struct *obj, obj_struct *obj_descendent)
{
	GList *glist;

	if((obj == NULL) || (obj_descendent == NULL))
	    return(FALSE);

	if(obj == obj_descendent)
	    return(TRUE);

	for(glist = obj->children; glist != NULL; glist = g_list_next(glist))
	{
	    if(ObjIsDescendent(OBJ(glist->data), obj_descendent))
		return(TRUE);
	}

	return(FALSE);
}

/*
 *	Checks if obj_ancestor is an ancestor (grandparent) of obj or
 *	is obj itself.
 */
gboolean ObjIsAncestor(obj_struct *obj, obj_struct *obj_ancestor)
{
	if((obj == NULL) || (obj_ancestor == NULL))
	    return(FALSE);

	if(obj == obj_ancestor)
	    return(TRUE);

	while(obj->parent != NULL)
	{
	    obj = obj->parent;
	    if(obj == obj_ancestor)
		return(TRUE);
	}

	return(FALSE);
}

/*
 *	Returns the Object specified by the path.
 */
obj_struct *ObjGetObjFromPath(
	obj_struct *obj_toplevel, const gchar *path
)
{
	gint len;
	const gchar *s, *s_end;
	GList *glist;
	obj_struct *obj;

	if((obj_toplevel == NULL) || STRISEMPTY(path))
	    return(NULL);

	/* Start at the specified toplevel Object and seek s to the
	 * first path segment name
	 */
	obj = obj_toplevel;
	for(s = path; *s == PATH_DELIM; s++);

	/* Check if the first path segment name matches the toplevel
	 * Object's name, if it does not then the rest of the path is
	 * invalid
	 */
	for(s_end = s; *s_end != '\0'; s_end++)
	{
	    if(*s_end == PATH_DELIM)
		break;
	}
	len = (gint)(s_end - s);
	if(len <= 0)
	    return(NULL);
	if(g_strncasecmp(obj->name, s, len))
	    return(NULL);

	/* Seek s to next path segment name, if there isn't any then
	 * return the toplevel Object
	 */
	if(*s_end != '\0')   
	    for(s = s_end; *s == PATH_DELIM; s++);
	else
	    return(obj);

	while(TRUE)
	{
	    /* At this point s is at the start of this path segment's
	     * name
	     */

	    /* Seek s_end to the next deliminator or end of string */
	    for(s_end = s; *s_end != '\0'; s_end++)
	    {
		if(*s_end == PATH_DELIM)
		    break;
	    }
	    len = (gint)(s_end - s);
	    if(len <= 0)
	    {
		if(*s_end != '\0')
		{
		    s = s_end + 1;
		    continue;
		}
		else
		    return(obj);
	    }

	    /* Iterate through child objects to find an Object who's
	     * name matches this path segment
	     */
	    for(glist = OBJ_CHILDREN(obj); glist != NULL; glist = g_list_next(glist))
	    {
		obj = OBJ(glist->data);
		if((obj != NULL) ?
		    (STRLEN(obj->name) != len) : TRUE
		)
		    continue;

		if(!g_strncasecmp(obj->name, s, len))
		{
		    if(*s_end != '\0')
			break;
		    else
			return(obj);
		}
	    }
	    /* If Object was not matched then the path is invalid */
	    if((glist == NULL) || (obj == NULL))
		break;

	    /* Matched Object, now seek to next path segment */
	    if(*s_end != '\0')
		for(s = s_end; *s == PATH_DELIM; s++);
	    else
		return(obj);
	}

	return(NULL);
}

/*
 *	Returns a string describing the path to the specified Object.
 *
 *	The calling function. must delete the returned string.
 */
gchar *ObjGetPathFromObj(obj_struct *obj)
{
	gchar *path, *s;

	if(obj == NULL)
	    return(NULL);

	path = STRDUP(obj->name);
	obj = OBJ_PARENT(obj);
	while(obj != NULL)
	{
	    s = g_strdup_printf("%s%c%s", obj->name, PATH_DELIM, path);
	    g_free(path);
	    path = s;

	    obj = OBJ_PARENT(obj);
	}

	s = g_strdup_printf("%c%s", PATH_DELIM, path);
	g_free(path);
	path = s;

	return(path);
}


/*
 *	Returns the size of memory used by the Object.
 */
gulong ObjGetMemorySize(obj_struct *obj)
{
	gulong x = 0l;

	if(obj == NULL)
	    return(x);

	x += sizeof(obj_struct);
	x += g_list_length(obj->children) * sizeof(GList);
	x += STRLEN(obj->name);
	x += STRLEN(obj->icon_name);
	x += STRLEN(obj->bg_name);
	x += STRLEN(obj->value);
	x += STRLEN(obj->tip);

	return(x);
}

/*
 *	Parses the DDE buffer into a list of Object pointers.
 *
 *	Each pointer to an Object may or may not exist and should be
 *	checked.
 *
 *	The returned list must be deleted but not each object.
 */
GList *ObjDDEBufferParse(const guint8 *buf, const gint buf_len)
{
	GList *glist = NULL;
	const gint nobjs = buf_len / sizeof(obj_struct *);
	const obj_struct	*obj,
				**ptr = (const obj_struct **)buf,
				**end = ptr + nobjs;

	if(buf == NULL)
	    return(glist);

	/* Iterate through the Object pointers list */
	while(ptr < end)
	{
	    obj = *ptr++;
	    if(obj != NULL)
		glist = g_list_append(glist, (gpointer)obj);
	}

	return(glist);
}

/*
 *	Appends the Object's pointer value to the DDE buffer.
 */
guint8 *ObjDDEBufferAppend(
	guint8 *buf, gint *buf_len,
	const obj_struct *obj
)
{
	gint i, nobjs;
	obj_struct **ptr = (obj_struct **)buf;

	if((obj == NULL) || (buf_len == NULL))
	    return(buf);

	nobjs = (*buf_len) / sizeof(obj_struct *);


	/* Increase allocation of Object pointers list */
	i = MAX(nobjs, 0);
	nobjs = i + 1;
	ptr = (obj_struct **)g_realloc(
	    ptr, nobjs * sizeof(obj_struct *)
	);
	if(ptr == NULL)
	{
	    *buf_len = 0;
	    return(NULL);
	}

	/* Add the specified object to the Object pointers list */
	ptr[i] = (obj_struct *)obj;

	/* Update returns */
	buf = (guint8 *)ptr;
	*buf_len = nobjs * sizeof(obj_struct *);

	return(buf);
}
