#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "keyscan.h"
#include "iconsel.h"

#include "obj.h"
#include "propdlg.h"
#include "win.h"
#include "winfio.h"
#include "core.h"
#include "config.h"

#include "images/icon_executable_20x20.xpm"
#include "images/icon_link2_20x20.xpm"
#include "images/icon_hsep_20x20.xpm"
#include "images/icon_folder_closed_20x20.xpm"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_browse_20x20.xpm"
#include "images/icon_run_20x20.xpm"

#include "images/icon_no_icon_32x32.xpm"

#include "images/icon_properties2_48x48.xpm"


/*
 *	Properties Dialog:
 */
typedef struct {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	win_struct	*win;

	GtkWidget	*main_vbox,

			*name_parent,		/* For all Objects */
			*name_icon_pm,
			*name_label,
			*type_icon_pm,
			*type_label,
			*read_only_label,
			*parent_icon_pm,
			*parent_label,
			*size_label,
			*last_modified_label,
			*change_icon_btn,

			*command_parent,	/* For Item Objects */
			*program_entry,
			*browse_program_btn,
			*arguments_entry,
			*run_btn,

			*accelkey_parent,
			*accelkey_keyscan,

			*contents_parent,	/* For Group Objects */
			*contents_label,

			*link_parent, 		/* For Link Objects */
			*link_entry,

			*special_parent,	/* For Special Objects */
			*special_value_entry,

			*ok_btn,
			*cancel_btn,
			*close_btn;

	gint		block_loop_level;
	obj_struct	*obj;
	menu_format	format;
	gboolean	user_response;
	gboolean	has_changes;
	gchar		*icon_name;

} PropDlg;
#define PROP_DLG(p)	((PropDlg *)(p))

static PropDlg		*propdlg = NULL;


static gchar *PropDlgSizeString(gulong i);

static gint PropDlgEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data 
);
static void PropDlgChangedCB(GtkWidget *widget, gpointer data);
static void PropDlgOKCB(GtkWidget *widget, gpointer data);
static void PropDlgCancelCB(GtkWidget *widget, gpointer data);
static void PropDlgChangeIconCB(GtkWidget *widget, gpointer data);
static void PropDlgBrowseCommandCB(GtkWidget *widget, gpointer data);
static void PropDlgRunCommandCB(GtkWidget *widget, gpointer data);

gint PropDlgInit(void);
void PropDlgSetStyle(GtkRcStyle *rc_style);
void PropDlgSetTransientFor(GtkWidget *w);
gboolean PropDlgIsQuery(void);
void PropDlgBreakQuery(void);
GtkWidget *PropDlgGetToplevel(void);

gboolean PropDlgGetResponse(win_struct *win, obj_struct *obj);

void PropDlgShutdown(void);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#define PROP_DLG_TITLE		"Properties"
#define PROP_DLG_WIDTH		380
#define PROP_DLG_HEIGHT		-1


/*
 *	Returns a statically allocated string describing the specified
 *	size.
 */
static gchar *PropDlgSizeString(gulong i)
{
	gint comma_countdown, slen;
	gchar ss[80], *ss_ptr;
	static gchar ts[80], *ts_ptr;


	g_snprintf(ss, sizeof(ss), "%ld", i);
	slen = STRLEN(ss);

	/* 3 digits or less? (no commas needed) */
	if(slen <= 3)
	{
	    strcpy(ts, ss);
	    return(ts);
	}
	 
	ts_ptr = ts;
	ss_ptr = ss;

	/* Initialize comma counter */
	comma_countdown = slen % 3;
	if(comma_countdown <= 0)
	    comma_countdown = 3;

	/* Iterate through size string until end is reached */
	while(*ss_ptr != '\0')
	{
	    /* Reset comma counter and put in a comma? */
	    if(comma_countdown <= 0)
	    {
		*ts_ptr++ = ',';
		comma_countdown = 3;
	    }
	     
	    *ts_ptr++ = *ss_ptr++;
	    comma_countdown--;
	}
	 
	/* Null terminate return string */
	*ts_ptr = '\0';

	return(ts);
}


/*
 *	Properties Dialog toplevel GtkWindow event signal callback.
 */
static gint PropDlgEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	PropDlg *d = PROP_DLG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_DELETE:
	    PropDlgCancelCB(d->cancel_btn, d);
	    status = TRUE;
	    break;
	}

	return(status);
}

/*
 *	Properties Dialog Change callback.
 */
static void PropDlgChangedCB(GtkWidget *widget, gpointer data)
{
	const gchar *name;
	gchar *buf;
	obj_struct *obj;
	PropDlg *d = PROP_DLG(data);
	if(d == NULL)
	    return;

	if(d->has_changes)
	    return;

	d->has_changes = TRUE;

	obj = d->obj;
	if(obj == NULL)
	    return;

	name = obj->name;
	if(STRISEMPTY(name))
	    name = "(unnamed)";

	/* Update Title */
	buf = g_strdup_printf(
	    "%s: %s (*)",
	    PROP_DLG_TITLE, name
	);
	gtk_window_set_title(GTK_WINDOW(d->toplevel), buf);
	g_free(buf);

	/* Update widget sensitivity and show/hide */
	GTK_WIDGET_SET_SENSITIVE(
	    d->ok_btn,
	    OBJ_READ_ONLY(obj) ? FALSE : TRUE
	);
	GTK_WIDGET_SET_SENSITIVE(d->cancel_btn, TRUE);
	GTK_WIDGET_SET_SENSITIVE(d->close_btn, TRUE);
	gtk_widget_show(d->ok_btn);
	gtk_widget_show(d->cancel_btn);
	gtk_widget_hide(d->close_btn);
}

/*
 *	Properties Dialog OK callback.
 */
static void PropDlgOKCB(GtkWidget *widget, gpointer data)
{
	const gchar *prog, *args;
	gchar *buf;
	GtkWidget *toplevel;
	obj_struct *obj, *dest_obj;
	win_struct *win;
	core_struct *core;
	PropDlg *d = PROP_DLG(data);
	if(d == NULL)
	    return;

	toplevel = d->toplevel;
	obj = d->obj;
	win = d->win;

	if((obj == NULL) || (win == NULL))
	    return;

	core = CORE(win->core);

	/* Do not proceed if object is read only */
	if(OBJ_READ_ONLY(obj))
	    return;

#define GET_ENTRY(_w_)	(			\
 ((_w_) != NULL) ?				\
  gtk_entry_get_text((GtkEntry *)(_w_)) : NULL	\
)
	/* Apply values to object */
	switch(obj->type)
	{
	  case OBJ_TYPE_ITEM:
	    if(d->icon_name != NULL)
	    {
		g_free(obj->icon_name);
		obj->icon_name = STRDUP(d->icon_name);
		WinFIOObjLoadIcons(win, obj, d->icon_name);
	    }
	    /* Get program & arguments to format the command string */
	    prog = GET_ENTRY(d->program_entry);
	    args = GET_ENTRY(d->arguments_entry);
	    if(prog != NULL)
	    {
		if(!STRISEMPTY(args))
		    buf = g_strdup_printf("%s %s", prog, args);
		else
		    buf = STRDUP(prog);

		/* Set new command */
		g_free(obj->value);
		obj->value = STRDUP(buf);
		g_free(buf);

		/* If the program is specified as an absolute path
		 * then check if it exists and is set executable
		 */
		if(ISPATHABSOLUTE(prog))
		{
		    gint euid = core->euid;
		    struct stat stat_buf;

		    if(stat(prog, &stat_buf))
		    {
			gchar *s = STRDUP(strerror(errno));
			if(s == NULL)
			    s = STRDUP("No such file");
			*s = toupper(*s);
			buf = g_strdup_printf(
"%s:\n\
\n\
    %s\n",
			    s, prog
			);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
			    "Warning", buf, NULL,
			    CDIALOG_ICON_WARNING,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
		        CDialogSetTransientFor(NULL);
		        g_free(buf);
		        g_free(s);
		    }
		    else if(!S_ISREG(stat_buf.st_mode))
		    {
			buf = g_strdup_printf(
"Not a file:\n\
\n\
    %s\n\
\n\
The specified program must be a file.\n",
			    prog
			);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
			    "Warning", buf, NULL,
			    CDIALOG_ICON_WARNING,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(buf);
		    }
		    else if(((euid == stat_buf.st_uid) || (euid == 0)) ?
			!(stat_buf.st_mode & S_IXUSR) :
			!(stat_buf.st_mode & S_IXOTH)
		    )
		    {
			buf = g_strdup_printf(
"Not executable:\n\
\n\
    %s\n\
\n\
The specified program must be set executable.\n",
			    prog
			);
			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
			    "Warning", buf, NULL,
			    CDIALOG_ICON_WARNING,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(buf);
		    }
		}
	    }
	    /* Get Accelerator Key */
	    obj->accel_key = KeyScanGetKey(d->accelkey_keyscan);
	    obj->accel_mods = KeyScanGetModifiers(d->accelkey_keyscan);
	    break;

	  case OBJ_TYPE_ITEM_LINK:
	    if(d->icon_name != NULL)
	    {
		g_free(obj->icon_name);
		obj->icon_name = STRDUP(d->icon_name);
		WinFIOObjLoadIcons(win, obj, d->icon_name);
	    }
	    g_free(obj->value);
	    obj->value = STRDUP(GET_ENTRY(d->link_entry));
	    /* Check if the link destination is valid */
	    dest_obj = WinObjGetLinkObj(win, obj);
	    if(dest_obj == NULL)
	    {
		buf = g_strdup_printf(
"Invalid link destination:\n\
\n\
    %s\n\
\n\
The specified link destination does not exist.\n",
		    obj->value
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Warning", buf, NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
	    }
	    else if(dest_obj == obj)
	    {
		buf = g_strdup_printf(
"Invalid link destination:\n\
\n\
    %s\n\
\n\
The specified link destination refers to the link itself.\n",
		    obj->value
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Warning", buf, NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL); 
		g_free(buf);
	    }
	    break;

	  case OBJ_TYPE_ITEM_SEPARATOR:
	    break;

	  case OBJ_TYPE_ITEM_SPECIAL:
#warning Check if the name for the Special Object is supported by the format
	    if(d->icon_name != NULL)
	    {
		g_free(obj->icon_name);
		obj->icon_name = STRDUP(d->icon_name);
		WinFIOObjLoadIcons(win, obj, d->icon_name);
	    }
	    g_free(obj->value);
	    obj->value = STRDUP(GET_ENTRY(d->special_value_entry));
	    /* Get Accelerator Key */
	    obj->accel_key = KeyScanGetKey(d->accelkey_keyscan);
	    obj->accel_mods = KeyScanGetModifiers(d->accelkey_keyscan);
	    break;

	  case OBJ_TYPE_GROUP:
	    if(d->icon_name != NULL)
	    {
		g_free(obj->icon_name);
		obj->icon_name = STRDUP(d->icon_name);
		WinFIOObjLoadIcons(win, obj, d->icon_name);
	    }
	    break;
	}
#undef GET_ENTRY

	/* Update last modified time */
	obj->last_modified = time(NULL);


	/* Mark that we got a positive user response */
	d->user_response = TRUE;

	/* Pop out of the main loop */
	if(d->block_loop_level > 0)
	{
	    d->block_loop_level--;
	    gtk_main_quit();
	}
}

/*
 *	Properties Dialog Cancel callback.
 */
static void PropDlgCancelCB(GtkWidget *widget, gpointer data)
{
	PropDlg *d = PROP_DLG(data);
	if(d == NULL)
	    return;

	/* Pop out of the main loop */
	if(d->block_loop_level > 0)
	{
	    d->block_loop_level--;
	    gtk_main_quit();
	}
}

/*
 *	Properties Dialog Change Icon callback.
 */
static void PropDlgChangeIconCB(GtkWidget *widget, gpointer data)
{
	gchar **strv, *location;
	gint strc;
	GList *locations_list, *glist;
	GtkWidget *toplevel;
	obj_struct *obj;
	win_struct *win;
	core_struct *core;
	PropDlg *d = PROP_DLG(data);
	if((d == NULL) || IconSelIsQuery())
	    return;

	toplevel = d->toplevel;
	obj = d->obj;
	win = d->win;

	if((win == NULL) || (obj == NULL))
	    return;

	core = CORE(win->core);

	/* Get locations list as the icon paths list */
	locations_list = glist = win->icon_paths;

	/* Get initial location, the first path in the list */
	location = (glist != NULL) ? (gchar *)glist->data : NULL;

	/* Query user for icon */
	IconSelSetTransientFor(toplevel);
	strv = IconSelGetResponse(
	    "Select Icon",
	    location,
	    locations_list,
	    "Select", "Cancel",
	    ICON_SEL_BTNFLAG_OK | ICON_SEL_BTNFLAG_CANCEL,
	    ICON_SEL_BTNFLAG_OK, 
	    400, 56,
	    56, 56, 0,
	    TRUE, FALSE, FALSE,
	    &strc
	);
	IconSelSetTransientFor(NULL);

	/* Get response? */
	if((strv != NULL) && (strc > 0))
	{
	    /* Get full path to icon file and the icon name */
	    const gchar *full_path = strv[strc - 1];
	    gchar *icon_name = WinFIOObjGetIconNameFromFullPath(
		win, full_path
	    );
	    if(icon_name != NULL)
	    {
		/* Change in icon name? */
		if((d->icon_name != NULL) ?
		    strcmp(d->icon_name, icon_name) : TRUE
		)
		{
		    /* Load icon */
		    GdkBitmap *mask;
		    GdkPixmap *pixmap = WinFIOOpenIcon(
			win, icon_name, MENU_ICON_SIZE_MEDIUM, &mask
		    );

		    /* Set new icon being displayed */
		    if(pixmap != NULL)
			gtk_pixmap_set(
			    GTK_PIXMAP(d->name_icon_pm), pixmap, mask
			);

		    /* Set new icon name */
		    g_free(d->icon_name);
		    d->icon_name = STRDUP(icon_name);

		    PropDlgChangedCB(d->name_icon_pm, d);

		    GDK_PIXMAP_UNREF(pixmap);
		    GDK_BITMAP_UNREF(mask);
		}

		g_free(icon_name);
	    }
	}
}

/*
 *	Properties Dialog Browse Command callback.
 */
static void PropDlgBrowseCommandCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	gchar *prev_path;
	GtkWidget *w, *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0; 
	PropDlg *d = PROP_DLG(data);
	if((d == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = d->toplevel;
	w = d->program_entry;
	prev_path = STRDUP(gtk_entry_get_text(GTK_ENTRY(w)));

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);                   

	/* Query user for command */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Select Program",
	    "Select", "Cancel",
	    prev_path,
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *new_path = (total_path_rtns > 0) ?
		path_rtn[total_path_rtns - 1] : NULL;
	    if(!STRISEMPTY(new_path))
	    {
		gtk_entry_set_text(GTK_ENTRY(w), new_path);
	    }
	}    
	     
	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	g_free(prev_path);
}

/*
 *	Properties Dialog Run Command callback.
 */
static void PropDlgRunCommandCB(GtkWidget *widget, gpointer data)
{
	gint p;
	const gchar *prog, *args;
	gchar *cmd;
	GtkWidget *toplevel;
	PropDlg *d = PROP_DLG(data);
	if(d == NULL)
	    return;

	toplevel = d->toplevel;
	prog = gtk_entry_get_text(GTK_ENTRY(d->program_entry));
	args = gtk_entry_get_text(GTK_ENTRY(d->arguments_entry));

	if(STRISEMPTY(prog))
	{
            gchar *buf = STRDUP(
"There is no program specified to be runned."
	    );
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(  
                "Run Failed", buf, NULL,
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK, 
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
            g_free(buf);
	    return;
	}
	if(!ISPATHABSOLUTE(prog))
	{
            gchar *buf = STRDUP(
"The specified path to the program is not an\n\
absolute path."
            );
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(  
                "Run Failed", buf, NULL,
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK, 
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
            g_free(buf);
            return;
	}

	/* Format command */
	if(STRISEMPTY(args))
	    cmd = STRDUP(prog);
	else
	    cmd = g_strdup_printf("%s %s", prog, args);
	if(cmd == NULL)
	    return;

	/* Run the command */
	p = Exec(cmd);
	if(p <= 0)
	{
	    gchar *buf = g_strdup_printf(
"Unable to execute:\n\
\n\
    %s\n",
		cmd
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Run Failed", buf, NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	}

	g_free(cmd);
}


/* 
 *	Initializes the Properties Dialog.
 */
gint PropDlgInit(void)
{
	const gint	border_major = 5,
			border_minor = 2;
	GdkWindow *window;
	GtkRcStyle *rcstyle;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2, *parent3, *parent4;
	PropDlg *d;

	/* Already initialized? */
	if(propdlg != NULL)
	    return(0);

	/* Create new Properties Dialog */
	d = PROP_DLG(g_malloc0(sizeof(PropDlg)));
	if(d == NULL)
	    return(-3);

	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->block_loop_level = 0;
	d->obj = NULL;
	d->format = MENU_FORMAT_ENDEAVOUR2;
	d->user_response = FALSE;
	d->has_changes = FALSE;
	d->icon_name = NULL;

	/* Toplevel */
	d->toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_usize(w, PROP_DLG_WIDTH, PROP_DLG_HEIGHT);
	gtk_window_set_policy(
	    GTK_WINDOW(w),
	    TRUE, TRUE, TRUE
	);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_properties2_48x48_xpm);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(PropDlgEventCB), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Main vbox */
	d->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);


	/* Vbox for frames */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(d->main_vbox), w, FALSE, FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent = w;

	/* Name Frame */
	d->name_parent = w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Identificacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Identification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Identifikation"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Identificazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Identificatie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Identificao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Identification"
#else
"Identification"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w); 
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Hbox for Icon, Name, and Change Icon button */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Icon */
	d->name_icon_pm = w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_no_icon_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Name */
	d->name_label = w = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(
"-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-1"
	);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle)
	gtk_widget_show(w);
	/* Vbox for the Change Icon Button */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Change Icon Button */
	d->change_icon_btn = w = gtk_button_new_with_label(
	    "Change Icon"
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(                                        
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PropDlgChangeIconCB), d
	);
	GUISetWidgetTip(w, "Click this to change the icon");
	gtk_widget_show(w);

	/* Hbox for Type */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Type */
	w = gtk_label_new("Type:");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Icon */
	d->type_icon_pm = w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,  
	    (guint8 **)icon_executable_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Label */
	d->type_label = w = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Read Only Label */
	d->read_only_label = w = gtk_label_new("(read-only)");
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Hbox for Parent */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Parent */
	w = gtk_label_new("Parent:");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Icon */
	d->parent_icon_pm = w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_no_icon_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Label */
	d->parent_label = w = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Hbox for Size */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);     
	gtk_widget_show(w);    
	parent3 = w;    
	/* Parent */
	w = gtk_label_new("Size:");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);    
	d->size_label = w = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Hbox for Last Modified */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Parent */
	w = gtk_label_new("Last Modified:"); 
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	d->last_modified_label = w = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Command Frame */
	d->command_parent = w = gtk_frame_new(
	    "Command"
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Program */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Label */
	w = gtk_label_new("Program:");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Entry */
	d->program_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PropDlgChangedCB), d
	);
	gtk_widget_show(w);
	/* Browse Button */
	d->browse_program_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_browse_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PropDlgBrowseCommandCB), d
	);
	GUISetWidgetTip(w, "Browse");
	gtk_widget_show(w);

	/* Arguments */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Label */
	w = gtk_label_new("Arguments:");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Entry */
	d->arguments_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PropDlgChangedCB), d
	);
	gtk_widget_show(w);
	/* Run Button */
	d->run_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_run_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PropDlgRunCommandCB), d
	);
	GUISetWidgetTip(w, "Run");
	gtk_widget_show(w);


	/* Contents Frame */
	d->contents_parent = w = gtk_frame_new(
	    "Contents"
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Contents */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Label */
	d->contents_label = w = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Link Frame */
	d->link_parent = w = gtk_frame_new(
	    "Link"
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Link Destination */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Entry */
	d->link_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PropDlgChangedCB), d
	);
	gtk_widget_show(w);


	/* Special Frame */
	d->special_parent = w = gtk_frame_new(
	    "Special"
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);   
	gtk_widget_show(w);
	parent2 = w;

	/* Special Value */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Label */
	w = gtk_label_new("Value:");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Entry */
	d->special_value_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PropDlgChangedCB), d
	);
	gtk_widget_show(w);


	/* Accelerator Key Frame */
	d->accelkey_parent = w = gtk_frame_new(
	    "Accelerator Key"
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Accelerator Key Hbox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* KeyScan */
	d->accelkey_keyscan = w = KeyScanNew();
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	KeyScanSetChangedCB(w, PropDlgChangedCB, d);
	gtk_widget_show(w);


	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(d->main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons hbox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(d->main_vbox), w, FALSE, FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PropDlgOKCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancela"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annulla"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PropDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);                          

	/* Close button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fin"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PropDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);


	propdlg = d;

	return(0);
}

/*
 *	Sets the Properties Dialog's style. 
 */
void PropDlgSetStyle(GtkRcStyle *rc_style)
{
	GtkWidget *w;
	PropDlg *d = propdlg;

	if(d == NULL)
	    return;

	w = d->toplevel;
	if(w != NULL)
	{
	    if(rc_style != NULL)
	    {
		gtk_widget_modify_style_recursive(w, rc_style);
	    }
	    else
	    {   
		rc_style = gtk_rc_style_new();
		gtk_widget_modify_style_recursive(w, rc_style);
		GTK_RC_STYLE_UNREF(rc_style)
	    }
	}
}

/*
 *	Sets the Properties Dialog to be a transient for the specified
 *	GtkWindow.
 *
 *	If w is NULL then transient for will be unset.
 */
void PropDlgSetTransientFor(GtkWidget *w)
{
	PropDlg *d = propdlg;

	if(d == NULL)
	    return;

	if(d->toplevel != NULL)
	{
	    if(w != NULL)
	    {
		if(!GTK_IS_WINDOW(w))
		    return;

		gtk_window_set_transient_for(
		    GTK_WINDOW(d->toplevel), GTK_WINDOW(w)
		);
	    }
	    else
	    {
		gtk_window_set_transient_for(
		    GTK_WINDOW(d->toplevel), NULL
		);
	    }
	}
}

/*
 *	Checks if the Properties Dialog is mapped and waiting for user
 *	input.
 */
gboolean PropDlgIsQuery(void)
{
	PropDlg *d = propdlg;
	if(d != NULL)
	    return((d->block_loop_level > 0) ? TRUE : FALSE);
	else
	    return(FALSE);
}

/*
 *	Breaks the Properties Dialog's user input query (if any) and
 *	causes the querying function to return NULL.
 */
void PropDlgBreakQuery(void)
{
	PropDlg *d = propdlg;
	if(d == NULL)
	    return;

	/* Break out of all our main loops */
	while(d->block_loop_level > 0)
	{
	    gtk_main_quit();
	    d->block_loop_level--;
	}
	d->block_loop_level = 0;
}

/*
 *	Returns the Properties Dialog's toplevel widget.
 */
GtkWidget *PropDlgGetToplevel(void)
{
	PropDlg *d = propdlg;
	return((d != NULL) ? d->toplevel : NULL);
}


/*
 *	Blocks input, displays the Properties Dialog with the values
 *	for the specified Object, and waits for user response.
 *
 *	Returns TRUE if the Object was modified or FALSE if the user
 *	canceled.
 */
gboolean PropDlgGetResponse(win_struct *win, obj_struct *obj)
{
	const gchar *name;
	gchar *buf;
	GtkWidget *w;
	obj_struct *parent_obj;
	PropDlg *d = propdlg;
	core_struct *core;

	if((d == NULL) || (win == NULL) || (obj == NULL))
	    return(FALSE);

	core = CORE(win->core);

	/* Already waiting for user response? */
	if(d->block_loop_level > 0)
	    return(FALSE);

	/* Reset responses */
	d->win = win;
	d->obj = obj;
	d->format = win->format;
	d->user_response = FALSE;
	g_free(d->icon_name);
	d->icon_name = STRDUP(obj->icon_name);

	/* Get Object's name */
	name = obj->name;
	if(STRISEMPTY(name))
	    name = "(unnamed)";

	/* Get parent Object (if any) */
	parent_obj = obj->parent;

#define SET_LABEL(_w_,_s_)	{	\
 if((_w_) != NULL)			\
  gtk_label_set_text(			\
   (GtkLabel *)(_w_),			\
   ((_s_) != NULL) ? (_s_) : ""		\
  );					\
}
#define SET_ENTRY(_w_,_s_)	{	\
 if((_w_) != NULL)			\
  gtk_entry_set_text(			\
   (GtkEntry *)(_w_),			\
   ((_s_) != NULL) ? (_s_) : ""		\
  );					\
}
#define SET_ICON(_w_,_p_,_m_)	{	\
 if(((_w_) != NULL) && ((_p_) != NULL))	\
  gtk_pixmap_set(			\
   (GtkPixmap *)(_w_),			\
   (GdkPixmap *)(_p_),			\
   (GdkBitmap *)(_m_)			\
  );					\
}
#define SET_ICON_FROM_DATA(_w_,_d_)	{	\
 GdkBitmap *mask;				\
 GdkPixmap *pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA( \
  &mask, (guint8 **)(_d_)			\
 );						\
 SET_ICON((_w_), pixmap, mask);			\
 GDK_PIXMAP_UNREF(pixmap);			\
 GDK_BITMAP_UNREF(mask);			\
}
#define SET_ICON_BY_NAME(_w_,_n_)	{	\
 GdkBitmap *mask = NULL;			\
 GdkPixmap *pixmap = NULL;			\
						\
 /* Open icon from file */			\
 if(!STRISEMPTY(_n_))				\
  pixmap = WinFIOOpenIcon(			\
   win, (_n_), MENU_ICON_SIZE_MEDIUM, &mask	\
  );						\
						\
 /* If unable to open icon from file then load	\
  * default icon				\
  */						\
 if(pixmap == NULL)				\
  pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(	\
   &mask, (guint8 **)icon_no_icon_32x32_xpm	\
  );						\
						\
 SET_ICON((_w_), pixmap, mask);			\
						\
 GDK_PIXMAP_UNREF(pixmap);			\
 GDK_BITMAP_UNREF(mask);			\
}
#define SET_DATE_LABEL(_w_,_t_)	{		\
 time_t t = (_t_);				\
 gchar *s = STRDUP((t > 0) ?			\
  ctime(&t) : "(unknown)"			\
 );						\
 if(s != NULL) {				\
  gchar *s2 = strchr(s, '\n');			\
  if(s2 != NULL)				\
   *s2 = '\0';					\
  SET_LABEL((_w_), s);				\
  g_free(s);					\
 }						\
}

	/* Hide all parents initially, show select parents later
	 * depending on the Object's type
	 */
	gtk_widget_hide(d->name_parent);
	gtk_widget_hide(d->command_parent);
	gtk_widget_hide(d->accelkey_parent);
	gtk_widget_hide(d->contents_parent);
	gtk_widget_hide(d->link_parent);
	gtk_widget_hide(d->special_parent);

	/* Set values from the specified object */
	switch(obj->type)
	{
	  case OBJ_TYPE_ITEM:
	    /* Name */
	    gtk_widget_show(d->name_parent);
	    gtk_widget_show(d->name_icon_pm);
	    SET_ICON_BY_NAME(d->name_icon_pm, obj->icon_name);
	    SET_LABEL(d->name_label, name);
	    gtk_widget_show(d->change_icon_btn);
	    SET_ICON_FROM_DATA(
		d->type_icon_pm, icon_executable_20x20_xpm
	    );
	    SET_LABEL(d->type_label, "Item");
	    /* Command */
	    gtk_widget_show(d->command_parent);
	    if(obj->value != NULL)
	    {
		gchar *s, *buf = STRDUP(obj->value);
		for(s = buf; *s != '\0'; s++)
		{
		    if(ISBLANK(*s))
			break;
		}
		if(*s != '\0')
		{
		    *s = '\0';
		    SET_ENTRY(d->program_entry, buf);
		    *s = ' ';
		    while(ISBLANK(*s))
			s++;
		    SET_ENTRY(d->arguments_entry, s);
		}
		else
		{
		    SET_ENTRY(d->program_entry, buf);
		    SET_ENTRY(d->arguments_entry, "");
		}
		g_free(buf);
	    }
	    else
	    {
		SET_ENTRY(d->program_entry, "");
		SET_ENTRY(d->arguments_entry, "");
	    }
	    /* Accelerator Key */
	    gtk_widget_show(d->accelkey_parent);
	    KeyScanSetKey(d->accelkey_keyscan, obj->accel_key);
	    KeyScanSetModifiers(d->accelkey_keyscan, obj->accel_mods);
	    break;

	  case OBJ_TYPE_ITEM_LINK:
	    /* Name */
	    gtk_widget_show(d->name_parent);
	    gtk_widget_show(d->name_icon_pm);
	    SET_ICON_BY_NAME(d->name_icon_pm, obj->icon_name);
	    SET_LABEL(d->name_label, name);
	    gtk_widget_show(d->change_icon_btn);
	    SET_ICON_FROM_DATA(
		d->type_icon_pm, icon_link2_20x20_xpm
	    );
	    SET_LABEL(d->type_label, "Link");
	    /* Link */
	    gtk_widget_show(d->link_parent);
	    SET_ENTRY(d->link_entry, obj->value);
	    break;

	  case OBJ_TYPE_ITEM_SEPARATOR:
	    /* Name */
	    gtk_widget_show(d->name_parent);
	    gtk_widget_hide(d->name_icon_pm);
	    SET_LABEL(d->name_label, name);
	    gtk_widget_hide(d->change_icon_btn);
	    SET_ICON_FROM_DATA(
		d->type_icon_pm, icon_hsep_20x20_xpm
	    );
	    SET_LABEL(d->type_label, "Separator");
	    break;

	  case OBJ_TYPE_ITEM_SPECIAL:
	    /* Name */
	    gtk_widget_show(d->name_parent);
	    gtk_widget_show(d->name_icon_pm);
	    SET_ICON_BY_NAME(d->name_icon_pm, obj->icon_name);
	    SET_LABEL(d->name_label, name);
	    gtk_widget_show(d->change_icon_btn);
	    SET_ICON_FROM_DATA(
		d->type_icon_pm, icon_executable_20x20_xpm
	    );
	    SET_LABEL(d->type_label, "Special Item");
	    /* Special */
	    gtk_widget_show(d->special_parent);
	    SET_ENTRY(d->special_value_entry, obj->value);
	    /* Accelerator Key */
	    gtk_widget_show(d->accelkey_parent);
	    KeyScanSetKey(d->accelkey_keyscan, obj->accel_key);
	    KeyScanSetModifiers(d->accelkey_keyscan, obj->accel_mods);
	    break;

	  case OBJ_TYPE_GROUP:
	    /* Name */
	    gtk_widget_show(d->name_parent);
	    gtk_widget_show(d->name_icon_pm);
	    SET_ICON_BY_NAME(d->name_icon_pm, obj->icon_name);
	    SET_LABEL(d->name_label, name);
	    gtk_widget_show(d->change_icon_btn);
	    SET_ICON_FROM_DATA(
		d->type_icon_pm, icon_folder_closed_20x20_xpm
	    );
	    SET_LABEL(d->type_label, "Group");
	    /* Contents */
	    gtk_widget_show(d->contents_parent);
	    if(d->contents_label != NULL)
	    {
		gchar *s;
		GList *glist = obj->children;
		const gint nchildren = g_list_length(glist);
		if(nchildren > 0)
		{
		    gint nsubgroups = 0;
		    for(glist = obj->children; glist != NULL; glist = g_list_next(glist))
		    {
			if(OBJ_IS_GROUP(glist->data))
			    nsubgroups++;
		    }
		    if(nsubgroups > 0)
			s = g_strdup_printf(
			    "%i %s, %i %s",
			    nchildren,
			    (nchildren == 1) ? "child" : "children",
			    nsubgroups,
			    (nsubgroups == 1) ? "subgroup" : "subgroups"
			);
		    else
			s = g_strdup_printf(
			    "%i %s",
			    nchildren,   
			    (nchildren == 1) ? "child" : "children"
			);
		}
		else
		    s = STRDUP("Empty");
		SET_LABEL(d->contents_label, s);
		g_free(s);
	    }
	    break;
	}

	/* Parent Label */
	if(parent_obj != NULL)
	{
	    GdkPixmap *pixmap = parent_obj->pixmap;
	    GdkBitmap *mask = parent_obj->mask;
	    if(pixmap != NULL)
	    {
		gtk_widget_show(d->parent_icon_pm);
		SET_ICON(d->parent_icon_pm, pixmap, mask);
	    }
	    else
		gtk_widget_hide(d->parent_icon_pm);
	    SET_LABEL(d->parent_label, parent_obj->name);
	}
	else
	{
	    gtk_widget_hide(d->parent_icon_pm);
	    SET_LABEL(d->parent_label, "(none)");
	}

	/* Read Only Label */
	w = d->read_only_label;
	if(OBJ_READ_ONLY(obj))
	    gtk_widget_show(w);
	else
	    gtk_widget_hide(w);

	/* Size Label */
	w = d->size_label;
	if(w != NULL)
	{
	    gulong size = ObjGetMemorySize(obj);
	    gchar *s = g_strdup_printf(
		"%s byte%s",
		PropDlgSizeString(size), (size == 1) ? "" : "s"
	    );
	    SET_LABEL(w, s);
	    g_free(s);
	}

	/* Last Modified Label */
	SET_DATE_LABEL(d->last_modified_label, obj->last_modified);

#undef SET_LABEL
#undef SET_ENTRY
#undef SET_ICON
#undef SET_ICON_FROM_DATA
#undef SET_ICON_BY_NAME
#undef SET_DATE_LABEL


	/* Reset has_changes after setting the values */
	d->has_changes = FALSE;

	/* Set title */
	buf = g_strdup_printf(
	    "%s: %s",
	    PROP_DLG_TITLE, name
	);
	gtk_window_set_title(GTK_WINDOW(d->toplevel), buf);
	g_free(buf);

	/* Set widget sensitivity and show/hide */
	GTK_WIDGET_SET_SENSITIVE(d->ok_btn, FALSE);
	GTK_WIDGET_SET_SENSITIVE(d->cancel_btn, TRUE);
	GTK_WIDGET_SET_SENSITIVE(d->close_btn, TRUE);
	gtk_widget_show(d->ok_btn);
	gtk_widget_hide(d->cancel_btn);
	gtk_widget_show(d->close_btn);


	/* Map the Properties Dialog */
	w = d->toplevel;
	gtk_widget_show(w); 

	/* Push main loop and wait for user response */
	d->block_loop_level++;
	gtk_main();

	/* Unmap the Properties Dialog */
	w = d->toplevel;
	gtk_widget_hide(w);

	return(d->user_response);
}


/*
 *	Shuts down the Properties Dialog.
 */
void PropDlgShutdown(void)
{
	PropDlg *d = propdlg;
	if(d == NULL)
	    return;

	/* Break out of any main loops */
	while(d->block_loop_level > 0)
	{
	    gtk_main_quit();
	    d->block_loop_level--;
	}
	d->block_loop_level = 0;

	GTK_WIDGET_DESTROY(d->name_icon_pm)
	GTK_WIDGET_DESTROY(d->name_label)
	GTK_WIDGET_DESTROY(d->change_icon_btn)
	GTK_WIDGET_DESTROY(d->type_icon_pm)
	GTK_WIDGET_DESTROY(d->type_label)
	GTK_WIDGET_DESTROY(d->read_only_label)
	GTK_WIDGET_DESTROY(d->parent_icon_pm)
	GTK_WIDGET_DESTROY(d->parent_label)
	GTK_WIDGET_DESTROY(d->size_label)
	GTK_WIDGET_DESTROY(d->last_modified_label)
	GTK_WIDGET_DESTROY(d->name_parent)

	GTK_WIDGET_DESTROY(d->arguments_entry)
	GTK_WIDGET_DESTROY(d->browse_program_btn)
	GTK_WIDGET_DESTROY(d->program_entry)
	GTK_WIDGET_DESTROY(d->command_parent)

	GTK_WIDGET_DESTROY(d->accelkey_keyscan)
	GTK_WIDGET_DESTROY(d->accelkey_parent)

	GTK_WIDGET_DESTROY(d->contents_label)
	GTK_WIDGET_DESTROY(d->contents_parent)

	GTK_WIDGET_DESTROY(d->link_entry)
	GTK_WIDGET_DESTROY(d->link_parent)  

	GTK_WIDGET_DESTROY(d->special_value_entry)
	GTK_WIDGET_DESTROY(d->special_parent)

	GTK_WIDGET_DESTROY(d->ok_btn)
	GTK_WIDGET_DESTROY(d->cancel_btn)
	GTK_WIDGET_DESTROY(d->close_btn)

	GTK_WIDGET_DESTROY(d->main_vbox)
	GTK_WIDGET_DESTROY(d->toplevel)
	GTK_ACCEL_GROUP_UNREF(d->accelgrp)

	g_free(d->icon_name);

	g_free(d);
	propdlg = d = NULL;
}
