#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "guiutils.h"
#include "menubutton.h"
#include "toolbar.h"


/* Callbacks */
static gint ToolbarEnterNotifyCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
static gint ToolbarLeaveNotifyCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
static void ToolbarClickedCB(GtkWidget *widget, gpointer data);


/* Tool Bar Item List */
toolbar_item_struct *ToolBarItemNew(
	const toolbar_item_type type,
	toolbar_struct *tb,
	const gchar *text,
	guint8 **icon_data,
	const gchar *tooltip,
	const gint id,
	void (*func_cb)(toolbar_item_struct *, gint, gpointer),
	gpointer func_data,
	void (*enter_cb)(toolbar_item_struct *, gint, gpointer),
	gpointer enter_data,
	void (*leave_cb)(toolbar_item_struct *, gint, gpointer),
	gpointer leave_data
);
void ToolBarItemDelete(toolbar_item_struct *item);

toolbar_item_struct *ToolBarItemListMatchByID(
	GList *items_list, const gint id
);


/* Tool Bar */
toolbar_struct *ToolBarNew(
	GtkWidget *parent,
	GList *items_list,
	const toolbar_display display,
	const toolbar_relief relief,
	const GtkOrientation orientation
);
void ToolBarSetDisplay(
	toolbar_struct *tb, const toolbar_display display
);
void ToolBarSetRelief(
	toolbar_struct *tb, const toolbar_relief relief
);
void ToolBarItemUpdateByID(
	toolbar_struct *tb, const gint id,
	const gchar *text,
	guint8 **icon_data,
	const gchar *tooltip
);
void ToolBarMap(toolbar_struct *tb);
void ToolBarUnmap(toolbar_struct *tb);
void ToolBarDelete(toolbar_struct *tb);


/* Tool Bar Item */
GtkWidget *ToolBarItemGetWidgetID(
	toolbar_struct *tb, const gint id
);
void ToolBarItemSetSensitiveID(
	toolbar_struct *tb, const gint id,
	const gboolean sensitive
);
void ToolBarItemSetToggleID(
	toolbar_struct *tb, const gint id,
	const gboolean toggled
);
gboolean ToolBarItemGetToggleID(
	toolbar_struct *tb, const gint id
);
void ToolBarItemSetMenuID(
	toolbar_struct *tb, const gint id,
	GtkWidget *menu
);
GtkWidget *ToolBarItemGetMenuID(
	toolbar_struct *tb, const gint id
);
gboolean ToolBarItemIsMappedID(
	toolbar_struct *tb, const gint id
);
void ToolBarItemMapID(
	toolbar_struct *tb, const gint id
);
void ToolBarItemUnmapID(
	toolbar_struct *tb, const gint id
);


#define TOOLBAR_BUTTON_PICTURE_AND_TEXT_WIDTH   GUI_BUTTON_VLABEL_WIDTH
#define TOOLBAR_BUTTON_PICTURE_AND_TEXT_HEIGHT  GUI_BUTTON_VLABEL_HEIGHT

#define TOOLBAR_BUTTON_PICTURE_WIDTH            30
#define TOOLBAR_BUTTON_PICTURE_HEIGHT           30

#define TOOLBAR_BUTTON_TEXT_WIDTH               -1
#define TOOLBAR_BUTTON_TEXT_HEIGHT              30


#define SET_BUTTON_LAYOUT(_w_,_d_)			\
{ if((_w_) != NULL) { switch(_d_) {			\
   case TOOLBAR_DISPLAY_PICTURES_AND_TEXT:		\
    GUIButtonChangeLayout((_w_), 1, 1);			\
    gtk_widget_set_usize(				\
     (_w_),						\
     TOOLBAR_BUTTON_PICTURE_AND_TEXT_WIDTH,		\
     TOOLBAR_BUTTON_PICTURE_AND_TEXT_HEIGHT		\
    );							\
    break;						\
   case TOOLBAR_DISPLAY_PICTURES:			\
    GUIButtonChangeLayout((_w_), 1, 0);			\
    gtk_widget_set_usize(				\
     (_w_),						\
     TOOLBAR_BUTTON_PICTURE_WIDTH,			\
     TOOLBAR_BUTTON_PICTURE_HEIGHT			\
    );							\
    break;						\
   case TOOLBAR_DISPLAY_TEXT:				\
    GUIButtonChangeLayout((_w_), 0, 1);			\
    gtk_widget_set_usize(				\
     (_w_),						\
     TOOLBAR_BUTTON_TEXT_WIDTH,				\
     TOOLBAR_BUTTON_TEXT_HEIGHT				\
    );							\
    break;						\
} } }


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Tool Bar Item "enter_notify_event" signal callback.
 */
static gint ToolbarEnterNotifyCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	toolbar_item_struct *item = TOOLBAR_ITEM(data);
	if(item == NULL)
	    return(TRUE);

	if(item->freeze_count > 0)
	    return(TRUE);

	if(item->enter_cb != NULL)
	    item->enter_cb(
		item,			/* Tool Bar Item */
		item->id,		/* ID */
		item->enter_data	/* Data */
	    );

	return(TRUE);
}

/*
 *	Tool Bar Item "leave_notify_event" signal callback.
 */
static gint ToolbarLeaveNotifyCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	toolbar_item_struct *item = TOOLBAR_ITEM(data);
	if(item == NULL)
	    return(TRUE);

	if(item->freeze_count > 0)
	    return(TRUE);

	if(item->leave_cb != NULL)
	    item->leave_cb(
		item,			/* Tool Bar Item */
		item->id,		/* ID */
		item->leave_data	/* Data */
	    );

	return(TRUE);
}

/*
 *	Tool Bar Item "clicked" or "toggled" signal callback.
 */
static void ToolbarClickedCB(GtkWidget *widget, gpointer data)
{
	toolbar_item_struct *item = TOOLBAR_ITEM(data);
	if(item == NULL)
	    return;

	if(item->freeze_count > 0)
	    return;

	if(item->func_cb != NULL)
	    item->func_cb(
		item,			/* Tool Bar Item */
		item->id,		/* ID */
		item->func_data		/* Data */
	    );
}


/*
 *	Creates a new Tool Bar Item.
 *
 *	The item's GtkWidget will not be created.
 */
toolbar_item_struct *ToolBarItemNew(
	const toolbar_item_type type,
	toolbar_struct *tb,
	const gchar *text,
	guint8 **icon_data,
	const gchar *tooltip,
	const gint id,
	void (*func_cb)(toolbar_item_struct *, gint, gpointer),
	gpointer func_data,
	void (*enter_cb)(toolbar_item_struct *, gint, gpointer),
	gpointer enter_data,
	void (*leave_cb)(toolbar_item_struct *, gint, gpointer),
	gpointer leave_data
)
{
	toolbar_item_struct *item = TOOLBAR_ITEM(g_malloc0(
	    sizeof(toolbar_item_struct)
	));
	if(item == NULL)
	    return(item);

	item->type = type;
	item->freeze_count = 0;
	item->tb = tb;
	item->w = NULL;		/* Do not create this item's GtkWidget */
 	item->text = STRDUP(text);
	item->icon_data = icon_data;
	item->tooltip = STRDUP(tooltip);
	item->id = id;
	item->func_cb = func_cb;
	item->func_data = func_data;
	item->enter_cb = enter_cb;
	item->enter_data = enter_data;
	item->leave_cb = leave_cb;
	item->leave_data = leave_data;

	return(item);
}

/*
 *	Deletes the Tool Bar Item.
 */
void ToolBarItemDelete(toolbar_item_struct *item)
{
	if(item == NULL)
	    return;

	item->freeze_count++;

	GTK_WIDGET_DESTROY(item->w);
	g_free(item->text);
	g_free(item->tooltip);

	item->freeze_count--;

	g_free(item);
}


/*
 *	Matches the item by id.
 */
toolbar_item_struct *ToolBarItemListMatchByID(
	GList *items_list, const gint id
)
{
	GList *glist;
	toolbar_item_struct *item;

	for(glist = items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOLBAR_ITEM(glist->data);
	    if(item == NULL)
		continue;

	    if(item->id == id)
		return(item);
	}

	return(NULL);
}


/*
 *	Creates a new Tool Bar.
 *
 *	The items_list specifies a list of toolbar_item_struct *
 *	items to create. This list and each item is not modified or
 *	deleted by this function.
 *
 */
toolbar_struct *ToolBarNew(
	GtkWidget *parent,
	GList *items_list,
	const toolbar_display display,
	const toolbar_relief relief,
	const GtkOrientation orientation
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GList *glist;
	GtkWidget	*w, *parent2, *parent3,
			*label_rtn, *menu_rtn;
	toolbar_item_struct *item;
	toolbar_struct *tb = TOOLBAR(g_malloc0(
	    sizeof(toolbar_struct)
	));
	if(tb == NULL)
	    return(tb);

	tb->freeze_count = 0;
	tb->display = display;
	tb->relief = relief;
	tb->orientation = orientation;
	tb->items_list = NULL;

	/* Toplevel GtkBox */
	if(orientation == GTK_ORIENTATION_VERTICAL)
	    w = gtk_vbox_new(FALSE, 0);
	else
	    w = gtk_hbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	if(parent != NULL)
	{
	    if(GTK_IS_BOX(parent))
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    else if(GTK_IS_CONTAINER(parent))
		gtk_container_add(GTK_CONTAINER(parent), w);
	}
	tb->toplevel = w;
	parent2 = w;

	/* Create the Tool Bar Items */
	for(glist = items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOLBAR_ITEM(glist->data);
	    if(item == NULL)
		continue;

	    /* Copy this item and add it to the Toolbar's Items list */
	    item = ToolBarItemNew(
		item->type,
		tb,			/* Use our Tool Bar */
		item->text,
		item->icon_data,
		item->tooltip,
		item->id,
		item->func_cb,
		item->func_data,
		item->enter_cb,
		item->enter_data,
		item->leave_cb,
		item->leave_data
	    );
	    if(item == NULL)
		continue;

	    tb->items_list = g_list_append(
		tb->items_list,
		item
	    );

	    /* Create this item's widgets based on its type */
	    switch(item->type)
	    {
	      case TOOLBAR_ITEM_SEPARATOR:
		item->w = w = gtk_vbox_new(TRUE, 0);
		gtk_widget_set_usize(
		    w,
		    (orientation == GTK_ORIENTATION_VERTICAL) ? -1 : 5,
		    (orientation == GTK_ORIENTATION_VERTICAL) ? 5 : -1
		);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		if(orientation == GTK_ORIENTATION_VERTICAL)
		    w = gtk_hseparator_new();
		else
		    w = gtk_vseparator_new();
		gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, border_major);
		gtk_widget_show(w);
		break;

	      case TOOLBAR_ITEM_BUTTON:
		item->w = w = GUIButtonPixmapLabelV(
		    (guint8 **)item->icon_data,
		    item->text,
		    &label_rtn
		);
		switch(relief)
		{
		  case TOOLBAR_RELIEF_NONE:
		    gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
		    break;
		  case TOOLBAR_RELIEF_HALF:
		    gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_HALF);
		    break;
		  case TOOLBAR_RELIEF_NORMAL:
		    gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NORMAL);
		    break;
		}
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		if(item->func_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "clicked",
			GTK_SIGNAL_FUNC(ToolbarClickedCB), item
		    );
		if(item->enter_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "enter_notify_event",
			GTK_SIGNAL_FUNC(ToolbarEnterNotifyCB), item
		    );
		if(item->leave_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "leave_notify_event",
			GTK_SIGNAL_FUNC(ToolbarLeaveNotifyCB), item
		    );
		if(item->tooltip != NULL)
		    GUISetWidgetTip(w, item->tooltip);
		SET_BUTTON_LAYOUT(w, display)
		gtk_widget_show(w);
		break;

	      case TOOLBAR_ITEM_TOGGLE_BUTTON:
		item->w = w = GUIToggleButtonPixmapLabelV(
		    (guint8 **)item->icon_data,
		    item->text,
		    &label_rtn
		);
		switch(relief)
		{
		  case TOOLBAR_RELIEF_NONE:
		    gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
		    break;
		  case TOOLBAR_RELIEF_HALF:
		    gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_HALF);
		    break;
		  case TOOLBAR_RELIEF_NORMAL:
		    gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NORMAL);
		    break;
		}
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		if(item->func_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "toggled",
			GTK_SIGNAL_FUNC(ToolbarClickedCB), item
		    );
		if(item->enter_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "enter_notify_event",
			GTK_SIGNAL_FUNC(ToolbarEnterNotifyCB), item
		    );
		if(item->leave_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "leave_notify_event",
			GTK_SIGNAL_FUNC(ToolbarLeaveNotifyCB), item
		    );
		if(item->tooltip != NULL)
		    GUISetWidgetTip(w, item->tooltip);
		SET_BUTTON_LAYOUT(w, display)
		gtk_widget_show(w);
		break;

	      case TOOLBAR_ITEM_MENU_BUTTON:
		item->w = MenuButtonNewV(
		    item->text,
		    (guint8 **)item->icon_data,
		    &menu_rtn
		);
		switch(relief)
		{
		  case TOOLBAR_RELIEF_NONE:
		    gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
		    break;
		  case TOOLBAR_RELIEF_HALF:
		    gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_HALF);
		    break;
		  case TOOLBAR_RELIEF_NORMAL:
		    gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NORMAL);
		    break;
		}
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		if(item->func_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "toggled",
			GTK_SIGNAL_FUNC(ToolbarClickedCB), item
		    );
		if(item->enter_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "enter_notify_event",
			GTK_SIGNAL_FUNC(ToolbarEnterNotifyCB), item
		    );
		if(item->leave_cb != NULL)
		    gtk_signal_connect(
			GTK_OBJECT(w), "leave_notify_event",
			GTK_SIGNAL_FUNC(ToolbarLeaveNotifyCB), item
		    );
		if(item->tooltip != NULL)
		    GUISetWidgetTip(w, item->tooltip);
		SET_BUTTON_LAYOUT(w, display)
		gtk_widget_show(w);
		break;
	    }
	}

	return(tb);
}

/*
 *	Sets the Tool Bar's display.
 */
void ToolBarSetDisplay(
	toolbar_struct *tb, const toolbar_display display
)
{
	GList *glist;
	GtkWidget *w;
	toolbar_item_struct *item;

	if(tb == NULL)
	    return;

	/* Set the display on each item */
	for(glist = tb->items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOLBAR_ITEM(glist->data);
	    if(item == NULL)
		continue;

	    w = item->w;
	    if(w == NULL)
		continue;

	    switch(item->type)
	    {
	      case TOOLBAR_ITEM_SEPARATOR:
		break;
	      case TOOLBAR_ITEM_BUTTON:
		SET_BUTTON_LAYOUT(w, display)
		break;
	      case TOOLBAR_ITEM_TOGGLE_BUTTON:
		SET_BUTTON_LAYOUT(w, display)
		break;
	      case TOOLBAR_ITEM_MENU_BUTTON:
		SET_BUTTON_LAYOUT(w, display)
		break;
	    }
	}

#if 0
	/* Remind tool bar that it needs to resize */
	gtk_widget_queue_resize(tb->toplevel);
#endif
}

/*
 *	Sets the Tool Bar's relief.
 */
void ToolBarSetRelief(
	toolbar_struct *tb, const toolbar_relief relief
)
{
	GList *glist;
	GtkWidget *w;
	toolbar_item_struct *item;

	if(tb == NULL)
	    return;

	/* Set the relief on each item */
	for(glist = tb->items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOLBAR_ITEM(glist->data);
	    if(item == NULL)
		continue;

	    w = item->w;
	    if(w == NULL)
		continue;

	    switch(item->type)
	    {
	      case TOOLBAR_ITEM_SEPARATOR:
		break;
	      case TOOLBAR_ITEM_BUTTON:
		if(GTK_IS_BUTTON(w))
		{
		    switch(relief)
		    {
		      case TOOLBAR_RELIEF_NONE:
			gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
			break;
		      case TOOLBAR_RELIEF_HALF:
			gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_HALF);
			break;
		      case TOOLBAR_RELIEF_NORMAL:
			gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NORMAL);
			break;
		    }
		}
		break;
	      case TOOLBAR_ITEM_TOGGLE_BUTTON:
		if(GTK_IS_BUTTON(w))
		{
		    switch(relief)
		    {
		      case TOOLBAR_RELIEF_NONE:
			gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
			break;
		      case TOOLBAR_RELIEF_HALF:
			gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_HALF);
			break;
		      case TOOLBAR_RELIEF_NORMAL:
			gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NORMAL);
			break;
		    }
		}
		break;
	      case TOOLBAR_ITEM_MENU_BUTTON:
		if(GTK_IS_BUTTON(w))
		{
		    switch(relief)
		    {
		      case TOOLBAR_RELIEF_NONE:
			gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
			break;
		      case TOOLBAR_RELIEF_HALF:
			gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_HALF);
			break;
		      case TOOLBAR_RELIEF_NORMAL:
			gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NORMAL);
			break;
		    }
		}
		break;
	    }
	}

#if 0
	/* Remind tool bar that it needs to resize */
	gtk_widget_queue_resize(tb->toplevel);
#endif
}


/*
 *	Updates the Tool Bar Item specified by id with the given
 *	text, icon, and tooltip.
 */
void ToolBarItemUpdateByID(
	toolbar_struct *tb, const gint id,
	const gchar *text,
	guint8 **icon_data,
	const gchar *tooltip
)
{
	GList *glist;
	toolbar_item_struct *item;

	if(tb == NULL)
	    return;

	/* Update each item in the items list who's ID matches the
	 * specified ID
	 */
	for(glist = tb->items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = TOOLBAR_ITEM(glist->data);
	    if(item == NULL)
		continue;

	    /* This item's ID matches the specified ID? */
	    if(item->id == id)
	    {
		GtkWidget *w = item->w;

		/* Update text, icon data, and tool tip message */
		if(text != NULL)
		{
		    g_free(item->text);
		    item->text = STRDUP(text);
		}

		if(icon_data != NULL)
		{
		    item->icon_data = icon_data;
		}

		if(tooltip != NULL)
		{
		    g_free(item->tooltip);
		    item->tooltip = STRDUP(tooltip);
		}

		/* Update widget, handle by item type */
		switch(item->type)
		{
		  case TOOLBAR_ITEM_SEPARATOR:
		    break;
		  case TOOLBAR_ITEM_BUTTON:
		    if(w != NULL)
			GUIButtonPixmapUpdate(
			    w, icon_data, text
			);
		    break;
		  case TOOLBAR_ITEM_TOGGLE_BUTTON:
		    if(w != NULL)
			GUIButtonPixmapUpdate(
			    w, icon_data, text
			);
		    break;
		  case TOOLBAR_ITEM_MENU_BUTTON:
		    if(w != NULL)
			GUIButtonPixmapUpdate(
			    w, icon_data, text
			);
		    break;
		}

		/* Do not break after matching this item, there may
		 * be other items with the same ID that need to be
		 * updated
		 */
	    }
	}
}

/*
 *	Maps the Tool Bar.
 */
void ToolBarMap(toolbar_struct *tb)
{
	GtkWidget *w = (tb != NULL) ? tb->toplevel : NULL; 
	if(w == NULL)    
	    return;

	gtk_widget_show(w);
}

/*
 *	Unmaps the Tool Bar.
 */
void ToolBarUnmap(toolbar_struct *tb)
{
	GtkWidget *w = (tb != NULL) ? tb->toplevel : NULL;
	if(w == NULL)
	    return;

	gtk_widget_hide(w);
}

/*
 *	Deletes the Tool Bar.
 */
void ToolBarDelete(toolbar_struct *tb)
{
	if(tb == NULL)
	    return;

	tb->freeze_count++;

	if(tb->items_list != NULL)
	{
	    g_list_foreach(tb->items_list, (GFunc)ToolBarItemDelete, NULL);
	    g_list_free(tb->items_list);
	}
	GTK_WIDGET_DESTROY(tb->toplevel);

	tb->freeze_count--;

	g_free(tb);
}


/*
 *	Gets the item's GtkWidget by id.
 */
GtkWidget *ToolBarItemGetWidgetID(
	toolbar_struct *tb, const gint id
)
{
	toolbar_item_struct *item;

	if(tb == NULL)
	    return(NULL);

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return(NULL);

	return(item->w);
}


/*
 *	Sets the Tool Bar Item specified by id as sensitive or
 *	insensitive.
 */
void ToolBarItemSetSensitiveID(
	toolbar_struct *tb, const gint id,
	const gboolean sensitive
)
{
	GtkWidget *w = ToolBarItemGetWidgetID(tb, id);
	if(w == NULL)
	    return;

	gtk_widget_set_sensitive(w, sensitive);
}


/*
 *	Sets the Tool Bar Item specified by id as toggled or untoggled.
 *
 *	The Tool Bar Item must be of type TOOLBAR_ITEM_TOGGLE_BUTTON.
 *
 *	No signal will be emitted.
 */
void ToolBarItemSetToggleID(
	toolbar_struct *tb, const gint id,
	const gboolean toggled
)
{
	toolbar_item_struct *item;

	if(tb == NULL)
	    return;

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return;

	if((item->type == TOOLBAR_ITEM_TOGGLE_BUTTON) &&
	   (item->w != NULL)
	)
	{
	    GtkToggleButton *tb = GTK_TOGGLE_BUTTON(item->w);
	    if(tb != NULL)
	    {
		/* Check if toggle button state has changed by comparing
		 * the current value with the specified value
		 *
		 * If there is a change then first record the function
		 * callback from the item and then set it NULL
		 *
		 * Next, do the toggle, in which case the toggle
		 * callback will be called but the func_cb will not be
		 * called
		 *
		 * After the callback is made we set the func_cb back
		 * to its original value
		 */
		if((gboolean)tb->active != (gboolean)toggled)
		{
		    item->freeze_count++;
		    gtk_toggle_button_set_active(tb, toggled);
		    item->freeze_count--;
		}
	    }
	}
}

/*
 *	Gets the Tool Bar Item's toggle state.
 */
gboolean ToolBarItemGetToggleID(
	toolbar_struct *tb, const gint id
)
{
	toolbar_item_struct *item;

	if(tb == NULL)
	    return(FALSE);

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return(FALSE);

	if((item->type != TOOLBAR_ITEM_TOGGLE_BUTTON) ||
	   (item->w == NULL)
	)
	    return(FALSE);

	return(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(item->w)));
}

/*
 *	Sets the Tool Bar Item's menu.
 */
void ToolBarItemSetMenuID(
	toolbar_struct *tb, const gint id,
	GtkWidget *menu
)
{
	toolbar_item_struct *item;

	if(tb == NULL)
	    return;

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return;

	if((item->type != TOOLBAR_ITEM_MENU_BUTTON) ||
	   (item->w == NULL)
	)
	    return;

	MenuButtonSetMenu(item->w, GTK_MENU(menu));
}

/*
 *	Gets the Tool Bar Item's menu.
 */
GtkWidget *ToolBarItemGetMenuID(
	toolbar_struct *tb, const gint id
)
{
	toolbar_item_struct *item;

	if(tb == NULL)
	    return(NULL);

	item = ToolBarItemListMatchByID(tb->items_list, id);
	if(item == NULL)
	    return(NULL);

	if((item->type != TOOLBAR_ITEM_MENU_BUTTON) ||
	   (item->w == NULL)
	)
	    return(NULL);

	return(MenuButtonGetMenu(item->w));
}


/*
 *	Checks if the Tool Bar Item specified by id is mapped.
 */
gboolean ToolBarItemIsMappedID(
	toolbar_struct *tb, const gint id
)
{
	GtkWidget *w = ToolBarItemGetWidgetID(tb, id);
	if(w == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(w));
}


/*
 *	Maps the Tool Bar Item specified by id.
 */
void ToolBarItemMapID(
	toolbar_struct *tb, const gint id
)
{
	GtkWidget *w = ToolBarItemGetWidgetID(tb, id);
	if(w == NULL)
	    return;

	gtk_widget_show(w);
}

/*
 *	Unmaps the Tool Bar Item specified by id.
 */
void ToolBarItemUnmapID(
	toolbar_struct *tb, const gint id
)
{
	GtkWidget *w = ToolBarItemGetWidgetID(tb, id);
	if(w == NULL)
	    return;

	gtk_widget_hide(w);
}
